/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_thread_2ch.c,v 1.14 2003/11/08 23:44:33 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_board_2ch.h"
#include "ochusha_thread_2ch.h"
#include "ochusha_utils_2ch.h"

#include <glib-object.h>
#include <glib.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>


#define DEBUG_DAT	0
#define DEBUG_DAT_MOST	0

#define MAX_RESPONSE		1024
#define ANALYSIS_INTERVAL	20


static void ochusha_thread_2ch_class_init(OchushaThread2chClass *klass);
static void ochusha_thread_2ch_init(OchushaThread2ch *thread);
static void ochusha_thread_2ch_finalize(GObject *object);

static void ochusha_thread_2ch_read_threadlist_element(
						OchushaBBSThread *thread,
						GHashTable *thread_attributes);
static void ochusha_thread_2ch_write_threadlist_element(
						OchushaBBSThread *thread,
						FILE *threadlist_xml);

static void ochusha_thread_2ch_property_change_notify(OchushaThread2ch *thread);

static gboolean ochusha_thread_2ch_check_url(OchushaBBSThread *thread,
					     const char *url,
					     unsigned int *from_p,
					     unsigned int *to_p);
static void ochusha_thread_2ch_remove_cache(OchushaBBSThread *thread,
					    OchushaConfig *config);

static OchushaAsyncBuffer *ochusha_thread_2ch_get_responses_source(
					OchushaBBSThread *thread,
					OchushaNetworkBroker *broker,
					OchushaNetworkBrokerCacheMode mode);
static gboolean ochusha_thread_2ch_parse_responses(OchushaBBSThread *thread,
						   OchushaAsyncBuffer *buffer,
						   int start, int number,
						   StartThreadCallback *st_cb,
						   EachResponseCallback *r_cb,
						   BrokenResponseCallback *bcb,
						   EndThreadCallback *end_cb,
						   gpointer callback_data);

static time_t ochusha_thread_2ch_get_last_modified_utc(OchushaBBSThread *thread);

static gboolean ochusha_thread_2ch_preview_response(OchushaBBSThread *thread,
					const OchushaBBSResponse *response,
					StartThreadCallback *start_cb,
					EachResponseCallback *response_cb,
					EndThreadCallback *end_cb,
					gpointer callback_data);

static gboolean ochusha_thread_2ch_post_response(OchushaBBSThread *thread,
					OchushaConfig *config,
					const OchushaBBSResponse *response);

static char *ochusha_thread_2ch_get_url_for_response(OchushaBBSThread *thread,
						     int number);
static const char *ochusha_thread_2ch_get_url_to_post(OchushaBBSThread *thread);


GType
ochusha_thread_2ch_get_type(void)
{
  static GType thread_2ch_type = 0;

  if (thread_2ch_type == 0)
    {
      static const GTypeInfo thread_2ch_info =
	{
	  sizeof(OchushaThread2chClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_thread_2ch_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaThread2ch),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_thread_2ch_init,
	};

      thread_2ch_type = g_type_register_static(OCHUSHA_TYPE_BBS_THREAD,
					       "OchushaThread2ch",
					       &thread_2ch_info, 0);
    }

  return thread_2ch_type;
}


static OchushaBBSThreadClass *parent_class = NULL;


static void
ochusha_thread_2ch_class_init(OchushaThread2chClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  OchushaBBSThreadClass *b_class = OCHUSHA_BBS_THREAD_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ochusha_thread_2ch_finalize;

  b_class->read_threadlist_element
    = ochusha_thread_2ch_read_threadlist_element;
  b_class->write_threadlist_element
    = ochusha_thread_2ch_write_threadlist_element;
  b_class->get_responses_source = ochusha_thread_2ch_get_responses_source;
  b_class->parse_responses = ochusha_thread_2ch_parse_responses;
  b_class->get_last_modified_utc = ochusha_thread_2ch_get_last_modified_utc;
  b_class->get_url
    = (const char *(*)(OchushaBBSThread *))ochusha_thread_2ch_get_base_url;
  b_class->get_url_for_response = ochusha_thread_2ch_get_url_for_response;
  b_class->get_url_to_post = ochusha_thread_2ch_get_url_to_post;
  b_class->check_url = ochusha_thread_2ch_check_url;
  b_class->remove_cache = ochusha_thread_2ch_remove_cache;

  b_class->preview_response = ochusha_thread_2ch_preview_response;
  b_class->post_response = ochusha_thread_2ch_post_response;
}


static void
ochusha_thread_2ch_init(OchushaThread2ch *thread)
{
  thread->dat_url = NULL;
  thread->base_url = NULL;
  thread->base_path = NULL;
  thread->post_url = NULL;

  thread->responses = NULL;

  thread->alive = FALSE;

  g_signal_connect(thread,
		   "notify::board",
		   G_CALLBACK(ochusha_thread_2ch_property_change_notify),
		   thread);
  g_signal_connect(thread,
		   "notify::id",
		   G_CALLBACK(ochusha_thread_2ch_property_change_notify),
		   thread);
}


static void
ochusha_thread_2ch_finalize(GObject *object)
{
  OchushaThread2ch *thread = OCHUSHA_THREAD_2CH(object);

  if (thread->dat_url != NULL)
    {
      G_FREE(thread->dat_url);
      thread->dat_url = NULL;
    }

  if (thread->base_url != NULL)
    {
      G_FREE(thread->base_url);
      thread->base_url = NULL;
    }

  if (thread->base_path != NULL)
    {
      G_FREE(thread->base_path);
      thread->base_path = NULL;
    }

  if (thread->responses != NULL)
    {
      G_FREE(thread->responses);
      thread->responses = NULL;
    }

  if (thread->last_modified != NULL)
    {
      G_FREE(thread->last_modified);
      thread->last_modified = NULL;
    }

  if (thread->date != NULL)
    {
      G_FREE(thread->date);
      thread->date = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static void
ochusha_thread_2ch_read_threadlist_element(OchushaBBSThread *thread,
					   GHashTable *thread_attributes)
{
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(thread);
  thread_2ch->last_modified
    = ochusha_utils_get_attribute_string(thread_attributes, "last_modified");

  if (parent_class->read_threadlist_element != NULL)
    (*parent_class->read_threadlist_element)(thread, thread_attributes);
}


#define OUTPUT_THREAD_ATTRIBUTE_STRING(file, thread, attribute)		\
  do {									\
    if ((thread)->attribute != NULL)					\
      {									\
        gchar *text = g_markup_escape_text((thread)->attribute, -1);	\
        fprintf(file,							\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <string>%s</string>\n"				\
	  "      </attribute>\n", text);				\
        g_free(text);							\
      }									\
  } while (0)


static void
ochusha_thread_2ch_write_threadlist_element(OchushaBBSThread *thread,
					    FILE *threadlist_xml)
{
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(thread);

  if (parent_class->write_threadlist_element != NULL)
    (*parent_class->write_threadlist_element)(thread, threadlist_xml);

  OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, thread_2ch, last_modified);
}


static void
ochusha_thread_2ch_property_change_notify(OchushaThread2ch *thread)
{
  OchushaThread2ch *thread_2ch;
#if DEBUG_LIB_OCHUSHA
  fprintf(stderr, "thread(%p)->board changed to board(%p).\n",
	  thread, thread->board);
#endif
  g_return_if_fail(OCHUSHA_IS_THREAD_2CH(thread));

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  if (thread->dat_url != NULL)
    {
      G_FREE(thread->dat_url);
      thread->dat_url = NULL;
    }

  if (thread->base_url != NULL)
    {
      G_FREE(thread->base_url);
      thread->base_url = NULL;
    }

  if (thread->base_path != NULL)
    {
      G_FREE(thread->base_path);
      thread->base_path = NULL;
    }

  if (thread->post_url != NULL)
    {
      G_FREE(thread->post_url);
      thread->post_url = NULL;
    }
}


static gboolean
ochusha_thread_2ch_check_url(OchushaBBSThread *thread, const char *url,
			     unsigned int *from_p, unsigned int *to_p)
{
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), FALSE);
  if (strncmp(url, "../", 3) == 0)
    {
      int len;
      const char *base_path;
      unsigned int from = 0;
      unsigned int to = 0;
      OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(thread);

      base_path = ochusha_thread_2ch_get_base_path(thread_2ch);
      g_return_val_if_fail(base_path != NULL, FALSE);

      len = strlen(base_path);
      url += 2;	/* Ƭ".."ɤФ */
      if (strncmp(base_path, url, len) != 0)
	return FALSE;

      sscanf(url + len, "%u-%u", &from, &to);
      if (from < 0)
	{
	  to = -from;
	  from = 0;
	}
      else if (to < 0)
	to = -to;

      if (from_p != NULL)
	*from_p = from;
      if (to_p != NULL)
	*to_p = to;

      return TRUE;
    }

  /* XXX: http://̾URLǤå٤ */

  return FALSE;
}


static void
ochusha_thread_2ch_remove_cache(OchushaBBSThread *thread,
				OchushaConfig *config)
{
  const char *dat_url;
  g_return_if_fail(OCHUSHA_IS_THREAD_2CH(thread) && config != NULL);

  dat_url = ochusha_thread_2ch_get_dat_url(OCHUSHA_THREAD_2CH(thread));
  if (dat_url != NULL)
    ochusha_config_cache_unlink_file(config, dat_url);
}


OchushaBBSThread *
ochusha_thread_2ch_new(OchushaBoard2ch *board,
		       const char *dat_filename,
		       const gchar *title)
{
  return OCHUSHA_BBS_THREAD(g_object_new(OCHUSHA_TYPE_THREAD_2CH,
					 "board", board,
					 "id", dat_filename,
					 "title", title,
					 NULL));
}


const char *
ochusha_thread_2ch_get_dat_url(OchushaThread2ch *thread_2ch)
{
  char url[PATH_MAX];
  OchushaBBSThread *thread;
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread_2ch), NULL);

  if (thread_2ch->dat_url != NULL)
    return thread_2ch->dat_url;

  thread = OCHUSHA_BBS_THREAD(thread_2ch);

  g_return_val_if_fail(thread->board != NULL && thread->board->base_url != NULL
		       && thread->id != NULL,
		       NULL);

  if (snprintf(url, PATH_MAX, "%sdat/%s",
	       thread->board->base_url, thread->id) >= PATH_MAX)
    return NULL;

  thread_2ch->dat_url = G_STRDUP(url);
  return thread_2ch->dat_url;
}


const char *
ochusha_thread_2ch_get_base_url(OchushaThread2ch *thread_2ch)
{
  char url[PATH_MAX];
  const char *base_path;
  OchushaBBSThread *thread;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread_2ch), NULL);

  if (thread_2ch->base_url != NULL)
    return thread_2ch->base_url;

  base_path = ochusha_thread_2ch_get_base_path(thread_2ch);

  thread = OCHUSHA_BBS_THREAD(thread_2ch);

  g_return_val_if_fail(base_path != NULL &&
		       thread->board != NULL && thread->board->server != NULL,
		       NULL);
  if (snprintf(url, PATH_MAX, "http://%s%s",
	       thread->board->server, base_path) >= PATH_MAX)
    return NULL;

  thread_2ch->base_url = G_STRDUP(url);
  return thread_2ch->base_url;
}


const char *
ochusha_thread_2ch_get_base_path(OchushaThread2ch *thread_2ch)
{
  char base_path[PATH_MAX];
  char *tmp_pos;
  OchushaBBSThread *thread;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread_2ch), NULL);

  if (thread_2ch->base_path != NULL)
    return thread_2ch->base_path;

  thread = OCHUSHA_BBS_THREAD(thread_2ch);

  g_return_val_if_fail(thread->board != NULL && thread->id != NULL, NULL);

  if (snprintf(base_path, PATH_MAX, "/test/read.cgi%s%s",
	       thread->board->base_path, thread->id) >= PATH_MAX)
    return NULL;

  tmp_pos = strstr(base_path + 14, ".dat");
  if (tmp_pos != NULL)
    {
      tmp_pos[0] = '/';
      tmp_pos[1] = '\0';
    }

  thread_2ch->base_path = G_STRDUP(base_path);
  return thread_2ch->base_path;
}


static OchushaAsyncBuffer *
ochusha_thread_2ch_get_responses_source(OchushaBBSThread *thread,
					OchushaNetworkBroker *broker,
					OchushaNetworkBrokerCacheMode mode)
{
  OchushaThread2ch *thread_2ch;
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread)
		       && OCHUSHA_IS_NETWORK_BROKER(broker), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  return ochusha_network_broker_read_from_url(broker,
				ochusha_thread_2ch_get_dat_url(thread_2ch),
				thread_2ch->last_modified, mode);
}


/*
 * extract_response
 *
 * cur_posʸ2chDATǡ1쥹ʬȸʤƲϤ
 * 1쥹ɽOchushaBBSResponse¤Τ˳Ǽ롣
 * δؿϡߤΰ(cur_pos)龯ʤȤ1ʬDATǡ
 * ¸ߤꤹΤǸƤӽФ¦դɬסʾ֤ǸƤӽФȡ
 * Хåեۤǡ˥뤳Ȥˤʤ롣
 *
 * 1쥹ʬΥǡ줿硢쥹ľΥǥߥ"<>"
 * ľʸؤݥ󥿤֤ԻˤNULL֤
 */
static char *
extract_response(char *start_pos, char *eol_pos, OchushaBBSResponse *response)
{
  /*
   * DATեγƹԤ
   * ̾<>ᥤ<><>쥹<>[åɥȥ]
   * ȤʤäƤ롣
   */
  char *clone_buffer = NULL;
  char *cur_pos = start_pos;
  char *tmp_pos;
  gboolean comma_delimiter = FALSE;
  int i = 0;

  /* DATǡƤ뤳Ȥ⤢Τǥå */
  tmp_pos = memchr(start_pos, '\0', eol_pos - cur_pos);
  if (tmp_pos != NULL)
    {
      int rest;
      /* 줿DATǡ */
      clone_buffer = G_MALLOC(eol_pos - cur_pos);
      memcpy(clone_buffer, start_pos, eol_pos - cur_pos);
      /*
       * XXX: ԡƤ'\0'ŬʸѤΤȡ
       *      '\0'ФʤͤƥԡΤΤɤ餬ɤΤġġ
       *      ҤȤޤԡơƬ'\0'Ф
       */
      rest = eol_pos - cur_pos;

      cur_pos = clone_buffer;
      eol_pos = clone_buffer + rest;
      while (*cur_pos == '\0')
	cur_pos++;

      rest -= (cur_pos - clone_buffer);

      tmp_pos = cur_pos + 1;
      while (--rest > 0)
	if (*tmp_pos == '\0')
	  *tmp_pos++ = '*';

#if DEBUG_DAT
      {
	char *broken_line = G_STRNDUP(cur_pos, eol_pos - cur_pos);
	fprintf(stderr, "broken line: %s\n", broken_line);
	G_FREE(broken_line);
      }
#endif
    }


  tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  while (tmp_pos != NULL)
    {
      i++;
      tmp_pos = g_strstr_len(tmp_pos + 2, eol_pos - tmp_pos - 2, "<>");
    }

  if (i == 0)
    goto error_exit;

  if (i == 1)
    {
      i = 0;
      tmp_pos = memchr(start_pos, ',', eol_pos - start_pos);
      while (tmp_pos != NULL)
	{
	  i++;
	  tmp_pos = memchr(tmp_pos + 1, ',', eol_pos - tmp_pos - 1);
	}
      if (i != 3)
	goto error_exit;	/* ȤΤʤ */
      comma_delimiter = TRUE;
    }

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  response->name = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);
  /* DATǡƤ뤳Ȥ⤢ */
  while (*cur_pos == '\0')
    cur_pos++;

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos - cur_pos > 0)
    response->mailto = G_STRNDUP(cur_pos, tmp_pos - cur_pos);
  else
    response->mailto = NULL;

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);
  /* DATǡƤ뤳Ȥ⤢ */
  while (*cur_pos == '\0')
    cur_pos++;

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  response->date_id = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);
  /* DATǡƤ뤳Ȥ⤢ */
  while (*cur_pos == 0)
    cur_pos++;

  tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  response->content = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  if (clone_buffer != NULL)
    {
      G_FREE(clone_buffer);
      return start_pos + (tmp_pos - clone_buffer) + 2;
    }
  else
    return tmp_pos + 2;

 error_exit:
#if DEBUG_DAT_MOST
  tmp_pos = G_STRNDUP(start_pos, eol_pos - start_pos);
  fprintf(stderr, "DAT Parsing failed: \"%s\"", tmp_pos);
  G_FREE(tmp_pos);
#endif
  if (response->name != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "name=%s\n", response->name);
#endif
      G_FREE((gpointer)response->name);
      response->name = NULL;
    }

  if (response->mailto != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "mailto=%s\n", response->mailto);
#endif
      G_FREE((gpointer)response->mailto);
      response->mailto = NULL;
    }

  if (response->date_id != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "date_id=%s\n", response->date_id);
#endif
      G_FREE((gpointer)response->date_id);
      response->date_id = NULL;
    }

  if (response->content != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "content=%s\n", response->content);
#endif
      G_FREE((gpointer)response->content);
      response->content = NULL;
    }

  if (clone_buffer != NULL)
    G_FREE(clone_buffer);

  return NULL;
}


/*
 * ochusha_thread_2ch_parse_responses
 *
 * bufferƤåɤDATǡǤȸʤƲϤåɤ
 * ɽOchushaThread2ch¤Τˤ륹åɤʬ(responses)ۤ롣
 *
 * åɤγϻåɥȥˤstart_thread_cb
 * ġΥ쥹Фeach_response_cbåΤβϽλˤ
 * end_thread_cb򤽤줾ƤӽФΥХåFALSE
 * 顢λǲϤŪˤ롣
 *
 * start+1ܤnumberʬΥ쥹ϤġĤȤؿˤͽꡣ
 * åã硢number˴ؤ餺Ϥλ롣
 *
 * number == -1ʤ顢start饹åޤǡ
 * start == -1ʤ顢ånumberʬΥ쥹
 *
 * Ĥޤꡢstart=0,number=-1ǡ쥹ϡ
 *         start=-1,number=100Ǻǿ100쥹Ȥä
 * 褺start=-1ξ̤Τޤ֢ʤͽ
 */
static gboolean
ochusha_thread_2ch_parse_responses(OchushaBBSThread *thread,
				   OchushaAsyncBuffer *buffer,
				   int start, int number,
				   StartThreadCallback *start_thread_cb,
				   EachResponseCallback *each_response_cb,
				   BrokenResponseCallback *broken_response_cb,
				   EndThreadCallback *end_thread_cb,
				   gpointer callback_data)
{
  OchushaThread2ch *thread_2ch;
  unsigned int *responses;
  gboolean result = TRUE;
  OchushaNetworkBrokerBufferStatus *status;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);

  status = g_object_get_data(G_OBJECT(buffer),
			     "OchushaNetworkBroker::BufferStatus");

  g_return_val_if_fail(status != NULL, FALSE);

  if (!ochusha_async_buffer_active_ref(buffer, "ochusha_thread_2ch.c: ochusha_thread_2ch_parse_responses"))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      return FALSE;
    }

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  responses = thread_2ch->responses;

  if (responses == NULL)
    {
      /*
       * 1000쥹ʬǤ⡹4000ХȤʤΤǸĹ϶Ȥޤ衣
       */
      responses = G_NEW0(unsigned int, MAX_RESPONSE);
      thread_2ch->responses = responses;
    }

  if (number == -1 || number > MAX_RESPONSE)
    number = MAX_RESPONSE;

  ochusha_async_buffer_lock(buffer);
  {
    int rest_of_responses;
    unsigned int offset;
    int i;
    gboolean buffer_fixed = FALSE;

  retry_analysis:
    rest_of_responses = number;
    offset = 0;
    i = 0;
    if (status->state != OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
      {
	if (start > 1 && start < MAX_RESPONSE)
	  {
	    offset = responses[start];
	    if (offset != 0)
	      i = start;
	    else
	      {
		/*
		 * nʬΥ쥹ɤ߹n+1ܰʹߤɤ
		 * ġĤȤΤ̰ƤХʥꥵ
		 * ȤơǤᤤϺʬõȤǤ뤬
		 * ޤǤɬפʤ
		 */
		offset = responses[start - 1];
		if (offset != 0)
		  i = start - 1;
		else
		  offset = 0;
	      }
	  }
      }

    while (result)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	int rest_of_data = buffer->length - offset;
	char *eol_pos = NULL;
	int interval = ANALYSIS_INTERVAL;

#if DEBUG_DAT_MOST
	fprintf(stderr, "1: ochusha_thread_2ch.c: length=%d\n",
		buffer->length);
#endif
	while (rest_of_data > 0 && rest_of_responses > 0
	       && (buffer_fixed || interval-- > 0)
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    OchushaBBSResponse response = { NULL, NULL, NULL, NULL };
	    char *title_pos;

#if DEBUG_DAT_MOST
	    fprintf(stderr, "i=%d, offset=%d, rest_of_data=%d, cur_pos=%p\n",
		    i, offset, rest_of_data, cur_pos);
#endif

	    title_pos = extract_response(cur_pos, eol_pos, &response);

#if DEBUG_DAT_MOST
	    fprintf(stderr, "title_pos=%p\n", title_pos);
#endif

	    if (i == 0 && start == 0 && title_pos != NULL)
	      {
		char *title = G_STRNDUP(title_pos, eol_pos - title_pos);
		if (start_thread_cb != NULL)
		  result = (*start_thread_cb)(thread, title, callback_data);
		G_FREE(title);
		if (!result)
		  break;
	      }
	    responses[i] = offset;

	    i++;
#if 1
	    if (i > start)
	      {
		result = TRUE;
		if (title_pos != NULL)
		  {
		    if (each_response_cb != NULL)
		      result = (*each_response_cb)(thread, i, &response,
						   callback_data);
		  }
		else
		  {
		    if (broken_response_cb != NULL)
		      result = (*broken_response_cb)(thread, i, callback_data);
		  }
		rest_of_responses--;
		if (!result)
		  break;
	      }
#else
	    if (title_pos != NULL)
	      {
		if (i > start)
		  {
		    if (each_response_cb != NULL)
		      result = (*each_response_cb)(thread, i, &response,
						   callback_data);
		    rest_of_responses--;
		  }
		if (!result)
		  break;
	      }
#endif

	    offset = (eol_pos + 1) - buffer_top;
	    buffer_top = (char *)buffer->buffer;
	    cur_pos = buffer_top + offset;
	    rest_of_data = buffer->length - offset;

#if DEBUG_DAT
	    if (title_pos == NULL)
	      {
		char *line;
		fprintf(stderr, "response #%d: Couldn't analyze DAT file...ignoring parts.\n", i);
		line = G_STRNDUP(buffer_top + responses[i], offset - responses[i]);
		fprintf(stderr, "failed: \"%s\"\n", line);
		G_FREE(line);
		fprintf(stderr, "offset=%d\n", responses[i]);
	      }
#endif

	    if (response.name != NULL)
	      G_FREE((gpointer)response.name);
	    if (response.mailto != NULL)
	      G_FREE((gpointer)response.mailto);
	    if (response.date_id != NULL)
	      G_FREE((gpointer)response.date_id);
	    if (response.content != NULL)
	      G_FREE((gpointer)response.content);
	  }

	if (!buffer_fixed && buffer->fixed)
	  {
	    buffer_fixed = TRUE;
	    eol_pos = NULL;
	    continue;
	  }

	if (buffer_fixed)
	  break;

	/*
	 * XXX: ͥåȥ󥰥åɤĥäƤʤȤˤwaitʤ
	 *      褦ˤ٤
	 */
	if (interval > 0
	    || ochusha_async_buffer_is_busy(buffer,
					    "ochusha_thread_2ch.c: ochusha_thread_2ch_parse_responses"))
	  if (!ochusha_async_buffer_wait(buffer, "ochusha_thread_2ch.c: ochusha_thread_2ch_parse_responses"))
	    {
	      /* bufferϤ޲٤callerˤޤ */
#if DEBUG_ASYNC_BUFFER_MOST
	      fprintf(stderr, "ochusha_thread_2ch_parse_responses(): buffer has been terminated.\n");
#endif
	      break;
	    }

	if (status->state
	    == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    if (i > start && end_thread_cb != NULL)
	      result = (*end_thread_cb)(thread, FALSE, callback_data);
	    /* å夬äƤ뤳ȤϤ⤦狼ä */
	    status->state
	      = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;
	    goto retry_analysis;
	  }
      }

    if (number == MAX_RESPONSE)
      thread->number_of_responses_read = i;
  }
  ochusha_async_buffer_unlock(buffer);

  if (status->last_modified != NULL)
    {
      if (thread_2ch->last_modified != NULL)
	G_FREE(thread_2ch->last_modified);
      thread_2ch->last_modified = G_STRDUP(status->last_modified);
    }

  if (status->date != NULL)
    {
      if (thread_2ch->date != NULL)
	G_FREE(thread_2ch->date);
      thread_2ch->date = G_STRDUP(status->date);
    }

  ochusha_async_buffer_active_unref(buffer, "ochusha_thread_2ch.c: ochusha_thread_2ch_parse_responses");

  if (end_thread_cb != NULL)
    result = (*end_thread_cb)(thread, TRUE, callback_data);

  return TRUE;
}


static time_t
ochusha_thread_2ch_get_last_modified_utc(OchushaBBSThread *thread)
{
  OchushaThread2ch *thread_2ch;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), 0);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);
  if (thread_2ch->last_modified == NULL)
    return 0;

  return ochusha_utils_get_utc_time(thread_2ch->last_modified);
}


static char *
ochusha_thread_2ch_get_url_for_response(OchushaBBSThread *thread, int number)
{
  OchushaThread2ch *thread_2ch;
  char url[PATH_MAX];
  const char *base_url;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  base_url = ochusha_thread_2ch_get_base_url(thread_2ch);
  base_url = strstr(base_url, "/test/read.cgi");

  if (base_url == NULL)
    return NULL;

  snprintf(url, PATH_MAX, "..%s%d", base_url, number);

  return G_STRDUP(url);
}


static const char *
ochusha_thread_2ch_get_url_to_post(OchushaBBSThread *thread)
{
  OchushaThread2ch *thread_2ch;
  char url[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);
  if (thread_2ch->post_url != NULL)
    return thread_2ch->post_url;

  snprintf(url, PATH_MAX, "%sl10",
	   ochusha_thread_2ch_get_base_url(thread_2ch));
  thread_2ch->post_url = G_STRDUP(url);
  return thread_2ch->post_url;
}


static gboolean
ochusha_thread_2ch_preview_response(OchushaBBSThread *thread,
				    const OchushaBBSResponse *response,
				    StartThreadCallback *start_cb,
				    EachResponseCallback *response_cb,
				    EndThreadCallback *end_cb,
				    gpointer callback_data)
{
  return FALSE;
}


static gboolean
ochusha_thread_2ch_post_response(OchushaBBSThread *thread,
				 OchushaConfig *config,
				 const OchushaBBSResponse *response)
{
  OchushaThread2ch *thread_2ch;
  OchushaBoard2ch *board_2ch;
  char *tmp_string = NULL;
  char *from = NULL;
  char *mail = NULL;
  char *message = NULL;
  const char *bbs;
  char *key = NULL;
  char *query = NULL;
  time_t time;
  iconv_t converter;
  OchushaUtils2chPostResult post_result;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), FALSE);
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(thread->board), FALSE);
  g_return_val_if_fail(response->name != NULL, FALSE);
  g_return_val_if_fail(response->mailto != NULL, FALSE);
  g_return_val_if_fail(response->content != NULL, FALSE);

  converter = iconv_open(ochusha_bulletin_board_get_response_character_encoding(thread->board), "UTF-8");
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  board_2ch = OCHUSHA_BOARD_2CH(thread->board);
  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  tmp_string = convert_string(converter, NULL, response->name, -1);
  from = ochusha_utils_url_encode_string(tmp_string);
  if (from == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->mailto, -1);
  mail = ochusha_utils_url_encode_string(tmp_string);
  if (mail == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->content, -1);
  message = ochusha_utils_url_encode_string(tmp_string);
  if (message == NULL)
    goto error_exit;
  G_FREE(tmp_string);
  tmp_string = NULL;

  bbs = ochusha_bulletin_board_get_id(thread->board);

  tmp_string = (char *)ochusha_bbs_thread_get_id(thread);
  g_assert(strstr(tmp_string, ".dat"));
  key = g_strndup(tmp_string, strstr(tmp_string, ".dat") - tmp_string);
  tmp_string = NULL;
  if (key == NULL)
    goto error_exit;

  if (thread_2ch->date != NULL)
    time = ochusha_utils_get_utc_time(thread_2ch->date);
  else
    {
#if DEBUG_POST
      fprintf(stderr, "thread_2ch->date == NULL!\n");
#endif
      time = ochusha_utils_get_utc_time(thread_2ch->last_modified);
    }

  if (time == -1)
    goto error_exit;

  /* XXX: ϡɥǥ󥰤㤦ΤϤäʤġġ*/
  query = g_strdup_printf("submit=%%8F%%91%%82%%AB%%8D%%9E%%82%%DE&FROM=%s&mail=%s&MESSAGE=%s&bbs=%s&key=%s&time=%ld", from, mail, message, bbs, key, time);

  post_result = ochusha_utils_2ch_try_post(thread, query);
  if (post_result == OCHUSHA_UTILS_2CH_POST_NO_COOKIE)
    {
      sleep(10);
      post_result = ochusha_utils_2ch_try_post(thread, query);
    }

#if 0
  if (post_result == OCHUSHA_UTILS_2CH_POST_MOCHITSUKE)
    {
      /* XXX: äݤΤȤͭʤΤ֡ */
      sleep(30);
      post_result = ochusha_utils_2ch_try_post(thread, query);
    }
#endif

  if (post_result != OCHUSHA_UTILS_2CH_POST_SUCCESS)
    goto error_exit;

  /* ｪλ */
  if (tmp_string != NULL)
    G_FREE(tmp_string);
  if (from != NULL)
    G_FREE(from);
  if (mail != NULL)
    G_FREE(mail);
  if (message != NULL)
    G_FREE(message);
  if (key != NULL)
    G_FREE(key);
  if (query != NULL)
    G_FREE(query);
  iconv_close(converter);

  return TRUE;

 error_exit:
#if DEBUG_POST
  fprintf(stderr, "Error happen\n");
#endif

  if (tmp_string != NULL)
    G_FREE(tmp_string);
  if (from != NULL)
    G_FREE(from);
  if (mail != NULL)
    G_FREE(mail);
  if (message != NULL)
    G_FREE(message);
  if (key != NULL)
    G_FREE(key);
  if (query != NULL)
    G_FREE(query);
  iconv_close(converter);

  return FALSE;
}
