/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_async_buffer.c,v 1.1.1.1 2003/05/10 16:34:28 fuyu Exp $
 */

#include "ochusha_async_buffer.h"

#include <glib.h>

#include <pthread.h>	/* GThread֤Ĥ */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


#define OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE	4096


struct _BufferSyncObject
{
  pthread_mutex_t mutex;
  pthread_cond_t cond;
};


static GObjectClass *parent_class = NULL;

static pthread_mutex_t ochusha_async_buffer_global_lock;
static pthread_cond_t ochusha_async_buffer_global_condition;

/*
 * pthreadϢΥ֥ȤλȤΤƤƧߤʤΤǡѤ
 * ݤ롣
 */
static GSList *sync_object_list = NULL;

/*
 * 饤֤(newƤfinalizeޤǤδ֤)Ƥ
 * OchushaAsyncBufferݻ
 */
static GSList *ochusha_async_buffer_list = NULL;


#define LOCK_GLOBAL_LOCK						\
  if (pthread_mutex_lock(&ochusha_async_buffer_global_lock) != 0)	\
    {									\
      fprintf(stderr, "Couldn't lock a mutex.\n");			\
      abort();								\
    }

#define UNLOCK_GLOBAL_LOCK						\
  if (pthread_mutex_unlock(&ochusha_async_buffer_global_lock) != 0)	\
    {									\
      fprintf(stderr, "Couldn't unlock a mutex.\n");			\
      abort();								\
    }

#define WAIT_GLOBAL_CONDITION						\
  if (pthread_cond_wait(&ochusha_async_buffer_global_condition,		\
			&ochusha_async_buffer_global_lock) != 0)	\
    {									\
      fprintf(stderr, "Couldn't wait a condition.\n");			\
      abort();								\
    }

#define SIGNAL_GLOBAL_CONDITION						\
  if (pthread_cond_signal(&ochusha_async_buffer_global_condition) != 0)	\
    {									\
      fprintf(stderr, "Couldn't unlock a mutex.\n");			\
      abort();								\
    }

#define LOCK_BUFFER_LOCK(buffer)				\
  if (pthread_mutex_lock(&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't lock a mutex.\n");		\
      abort();							\
    }

#define UNLOCK_BUFFER_LOCK(buffer)				\
  if (pthread_mutex_unlock(&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't unlock a mutex.\n");		\
      abort();							\
    }

#define WAIT_BUFFER_CONDITION(buffer)				\
  if (pthread_cond_wait(&buffer->sync_object->cond,		\
			&buffer->sync_object->mutex) != 0)	\
    {								\
      fprintf(stderr, "Couldn't wait a condition.\n");		\
      abort();							\
    }

#define SIGNAL_BUFFER_CONDITION(buffer)				\
  if (pthread_cond_signal(&buffer->sync_object->cond) != 0)	\
    {								\
      fprintf(stderr, "Couldn't siganl a condition.\n");	\
      abort();							\
    }

#define BROADCAST_BUFFER_CONDITION(buffer)			\
  if (pthread_cond_broadcast(&buffer->sync_object->cond) != 0)	\
    {								\
      fprintf(stderr, "Couldn't siganl a condition.\n");	\
      abort();							\
    }


static void ochusha_async_buffer_class_init(OchushaAsyncBufferClass *klass);
static void ochusha_async_buffer_init(OchushaAsyncBuffer *buffer);
static void ochusha_async_buffer_finalize(GObject *object);

static BufferSyncObject *get_sync_object(void);


GType
ochusha_async_buffer_get_type(void)
{
  static GType async_buffer_type = 0;

  if (async_buffer_type == 0)
    {
      static const GTypeInfo async_buffer_info =
	{
	  sizeof(OchushaAsyncBufferClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_async_buffer_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaAsyncBuffer),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_async_buffer_init,
	};

      async_buffer_type = g_type_register_static(G_TYPE_OBJECT,
						 "OchushaAsyncBuffer",
						 &async_buffer_info, 0);
    }

  return async_buffer_type;
}


static void
ochusha_async_buffer_class_init(OchushaAsyncBufferClass *klass)
{
  GObjectClass *o_class = (GObjectClass *)klass;

  if (pthread_mutex_init(&ochusha_async_buffer_global_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't initialize a mutex.\n");
      abort();
    }

  if (pthread_cond_init(&ochusha_async_buffer_global_condition, NULL) != 0)
    {
      fprintf(stderr, "Couldn't initialize a condvar.\n");
      abort();
    }

  parent_class = g_type_class_peek_parent(klass);
  o_class->finalize = ochusha_async_buffer_finalize;
}


static void
ochusha_async_buffer_init(OchushaAsyncBuffer *buffer)
{
  buffer->fixed = FALSE;
}


static void
ochusha_async_buffer_finalize(GObject *object)
{
  OchushaAsyncBuffer *buffer = OCHUSHA_ASYNC_BUFFER(object);
#if DEBUG_ASYNC_BUFFER
  fprintf(stderr, "ochusha_async_buffer_finalize: OchushaAsyncBuffer(0x%x)\n",
	  (int)object);
#endif
  if (buffer->destructor != NULL)
    (*buffer->destructor)(buffer, buffer->user_data);

  LOCK_GLOBAL_LOCK;
  if (buffer->sync_object != NULL)
    sync_object_list = g_slist_append(sync_object_list, buffer->sync_object);
  ochusha_async_buffer_list = g_slist_remove(ochusha_async_buffer_list, buffer);
  UNLOCK_GLOBAL_LOCK;

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


static BufferSyncObject *
get_sync_object(void)
{
  GSList *recycled_sync_object;
  BufferSyncObject *sync_object;

  recycled_sync_object = sync_object_list;
  sync_object_list = g_slist_remove_link(sync_object_list,
					 recycled_sync_object);
  if (recycled_sync_object != NULL)
    {
      sync_object = (BufferSyncObject *)recycled_sync_object->data;
      g_slist_free_1(recycled_sync_object);
      return sync_object;
    }

  sync_object = (BufferSyncObject *)g_malloc(sizeof(BufferSyncObject));
  if (pthread_mutex_init(&sync_object->mutex, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }
  if (pthread_cond_init(&sync_object->cond, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a condition variable.\n");
      abort();
    }
  return sync_object;
}


OchushaAsyncBuffer *
ochusha_async_buffer_new(char *buffer, int length,
			 gpointer user_data, DestructFunc *destructor)
{
  BufferSyncObject *sync_object;
  OchushaAsyncBuffer *buf
    = OCHUSHA_ASYNC_BUFFER(g_object_new(OCHUSHA_TYPE_ASYNC_BUFFER, NULL));

  LOCK_GLOBAL_LOCK
    {
      sync_object = get_sync_object();
      ochusha_async_buffer_list = g_slist_append(ochusha_async_buffer_list,
						 buf);
    }
  UNLOCK_GLOBAL_LOCK

  if (buffer == NULL && destructor == NULL)
    {
      if (length != 0)
	buffer = (char *)g_malloc(length);
      destructor = ochusha_async_buffer_free_when_finished;
      buf->length = 0;
    }
  else
    buf->length = length;
  
  buf->fixed = FALSE;
  buf->buffer = buffer;

  buf->buffer_length = length;	/* XXX: Ǥ餦٤ġġ*/

  buf->sync_object = sync_object;
  buf->user_data = user_data;
  buf->destructor = destructor;

  return buf;
}


/* ƱߴϢ */
/*
 * MEMO: OchushaAsyncBufferȤäƤƤΥåɤ⤷Υå
 *       Фᥤ󥹥åɤƱŪ˳ߤ򤫤ȻפäΤǡ
 *       ʲAPIѰդۤɰΤΤǤʤġġ
 *       pthreadUNIXʥ뤬ꤷƻȤФȤ⤢Τ
 *       pthreadUNIXʥȹ碌ϵʤΤ򤱤롣
 *
 * active user: OchushaAsyncBufferproducer⤷consumeråɡ
 *              ХåեݻƤΥƥȤϴޤޤʤ
 *              active userϥХåե˿ˡ
 *		ochusha_async_buffer_active_ref()ƤӽФ⤦Хåե
 *		ʤȤochusha_async_buffer_active_unref()
 *		ӽФȤˤ롣
 *
 *              δؿg_object_{ref,unref}Ƥǡ
 *              number_of_active_usersĴ롣
 *
 *              active userϻsuspend/resume/terminate׵᤬Ƥ
 *              ʤɤǧ뤳Ȥ롣
 */
gboolean
ochusha_async_buffer_active_ref(OchushaAsyncBuffer *buffer, const char *user)
{
  gboolean result;

#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr, "OchushaAsyncBuffer(0x%x) is actively refered by %s\n",
	  (int)buffer, user);
#endif
  g_object_ref(G_OBJECT(buffer));

  LOCK_BUFFER_LOCK(buffer);

  buffer->number_of_active_users++;
  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
    }

  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (!result)
    {
      buffer->number_of_active_users--;
      g_object_unref(G_OBJECT(buffer));
    }

  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


void
ochusha_async_buffer_active_unref(OchushaAsyncBuffer *buffer, const char *user)
{
#if DEBUG_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "OchushaAsyncBuffer(0x%x) actively refered by %s is unrefered\n",
	  (int)buffer, user);
#endif

  LOCK_BUFFER_LOCK(buffer);

  buffer->number_of_active_users--;

  if (buffer->state != OCHUSHA_ASYNC_BUFFER_OK)
    {
      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;
    }

  UNLOCK_BUFFER_LOCK(buffer);

  g_object_unref(G_OBJECT(buffer));
}


void
ochusha_async_buffer_suspend_all(void)
{
  g_slist_foreach(ochusha_async_buffer_list,
		  (GFunc)ochusha_async_buffer_suspend, NULL);
}


void
ochusha_async_buffer_resume_all(void)
{
  g_slist_foreach(ochusha_async_buffer_list,
		  (GFunc)ochusha_async_buffer_resume, NULL);
}


void
ochusha_async_buffer_suspend(OchushaAsyncBuffer *buffer)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "suspend threads refering OchushaAsyncBuffer(0x%x)\n",
	  (int)buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->number_of_suspended_users = 0;
  buffer->state = OCHUSHA_ASYNC_BUFFER_SUSPENDED;

  LOCK_GLOBAL_LOCK;

  BROADCAST_BUFFER_CONDITION(buffer);
  UNLOCK_BUFFER_LOCK(buffer);		/* νǤʤȤʤʤ
					 * UNLOCK_BUFFER_LOCKִ֤
					 * ΥåԤäƤåɤ
					 * ưϤ롣åɤξ
					 * åSUSPENDED֤Ǥ
					 * ȤΤΤǡSUSPENDED֤
					 * ֲ򤱤ΤԤġwait롣
					 * BROADCASTϿSUSPENDED֤
					 * ʤäȤΤʤåɤ
					 * 뤿ΤΤǤ
					 * ա
					 */
  while (buffer->number_of_suspended_users < buffer->number_of_active_users)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "ochusha_async_buffer_suspend: suspended=%d, active_users=%d\n",
	      buffer->number_of_suspended_users,
	      buffer->number_of_active_users);
#endif
      WAIT_GLOBAL_CONDITION;
    }
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_suspend: suspended=%d, active_users=%d\n",
	  buffer->number_of_suspended_users,
	  buffer->number_of_active_users);
#endif

  UNLOCK_GLOBAL_LOCK;
}


void
ochusha_async_buffer_resume(OchushaAsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);
  buffer->state = OCHUSHA_ASYNC_BUFFER_OK;
  BROADCAST_BUFFER_CONDITION(buffer);
  UNLOCK_BUFFER_LOCK(buffer);
}


void
ochusha_async_buffer_terminate(OchushaAsyncBuffer *buffer)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_terminate: buffer=0x%x\n", (int)buffer);
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->state = OCHUSHA_ASYNC_BUFFER_TERMINATED;
  BROADCAST_BUFFER_CONDITION(buffer);

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_terminate: buffer condition broadcast done\n");
#endif

  LOCK_GLOBAL_LOCK;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_terminate: global lock locked\n");
#endif

  UNLOCK_BUFFER_LOCK(buffer);

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_terminate: buffer lock unlocked\n");
#endif

  while (buffer->number_of_active_users > 0)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "ochusha_async_buffer_terminate: active_users=%d\n",
	      buffer->number_of_active_users);
#endif
      WAIT_GLOBAL_CONDITION;
    }
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_terminate: active_users=%d\n",
	  buffer->number_of_active_users);
#endif

  UNLOCK_GLOBAL_LOCK;
}


/*
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_update_length(OchushaAsyncBuffer *buffer, int length,
				   const char *user)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called ochusha_async_buffer_update_length\n", user);
#endif
  LOCK_BUFFER_LOCK(buffer);
  buffer->length = length;
  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer, user);
}


/*
 * ХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_fix(OchushaAsyncBuffer *buffer, const char *user)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called ochusha_async_buffer_fix\n", user);
#endif

  LOCK_BUFFER_LOCK(buffer);
  buffer->fixed = TRUE;
  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer, user);
}


/*
 * Out of memory⤷ϥХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_resize(OchushaAsyncBuffer *buffer, int length,
			    const char *user)
{
  char *new_buf;
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called ochusha_async_buffer_resize\n", user);
#endif

  LOCK_BUFFER_LOCK(buffer);

  if (buffer->fixed)
    {
      fprintf(stderr, "Invalid use of ochusha_async_buffer: Fixed buffer isn't resizable.\n");
      abort();
    }

  new_buf = (char *)realloc((void *)buffer->buffer, length);
  if (new_buf == NULL)
    result = FALSE;
  else
    {
      buffer->buffer = new_buf;
      buffer->buffer_length = length;
      result = TRUE;
    }

  UNLOCK_BUFFER_LOCK(buffer);

  return ochusha_async_buffer_broadcast(buffer, user) && result;
}


/*
 * Out of memory⤷ϥХåեTERMINATED֤ξFALSE֤
 */
gboolean
ochusha_async_buffer_ensure_free_space(OchushaAsyncBuffer *buffer, int length,
				       const char *user)
{
  char *new_buf;
  gboolean result = TRUE;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_ensure_free_space(0x%x, %d, \"%s\") called.\n",
	  (int)buffer, length, user);
#endif

  LOCK_BUFFER_LOCK(buffer);
  if (buffer->fixed)
    {
      fprintf(stderr,
	      "ochusha_async_buffer_ensure_free_space(): buffer is fixed.\n");
      abort();
    }

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "\"%s\" is suspended at ochusha_async_buffer_free_space.\n",
	      user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
      /* λbuffer->state != OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ
       * ʥsuspend򤯤˥ᥤ󥹥åɤäƤΡ
       *
       * buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ顢waitƤ֤
       * ⤦ٿSUSPENDED֤ˤʤäȤȡ
       */
    }

  while (buffer->buffer_length - buffer->length < length)
    {
      size_t new_buf_len
	= (buffer->buffer_length > 0)
	? (buffer->buffer_length * 2) : OCHUSHA_INITIAL_ASYNC_BUFFER_SIZE;
      new_buf = (char *)realloc((void *)buffer->buffer, new_buf_len);
      if (new_buf != NULL)
	{
	  buffer->buffer = new_buf;
	  buffer->buffer_length = new_buf_len;
	}
      else
	{
	  result = FALSE;
	  break;
	}
    }
  result = (buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED) && result;
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_append_data(OchushaAsyncBuffer *buffer, const char *data, int length,
			 const char *user)
{
  gboolean result = TRUE;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "\"%s\" called ochusha_async_buffer_append_data\n", user);
#endif

  if (ochusha_async_buffer_ensure_free_space(buffer, length, user))
    {
      memcpy((char *)buffer->buffer + buffer->length, data, length);
      buffer->length += length;
    }
  else
    result = FALSE;

  return ochusha_async_buffer_broadcast(buffer, user) && result;
}


void
ochusha_async_buffer_lock(OchushaAsyncBuffer *buffer)
{
  LOCK_BUFFER_LOCK(buffer);
}


void
ochusha_async_buffer_unlock(OchushaAsyncBuffer *buffer)
{
  UNLOCK_BUFFER_LOCK(buffer);
}


gboolean
ochusha_async_buffer_check_active(OchushaAsyncBuffer *buffer, const char *user)
{
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr,
	  "ochusha_async_buffer_check_active(0x%x, \"%s\")\n",
	  (int)buffer, user);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "\"%s\" is suspended at ochusha_async_buffer_check_signal.\n",
	      user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" resumed.\n", user);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   * ޤλǤOCHUSHA_ASYNC_BUFFER_OKʤΤǡbuffercondvarԤäƤ
   * ΤconsumerΤߤǤ뤳ȤݾڤƤ롣
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;

  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_signal(OchushaAsyncBuffer *buffer, const char *user)
{
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_signal(0x%x, \"%s\")\n",
	  (int)buffer, user);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "\"%s\" is suspended at ochusha_async_buffer_signal.\n", user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" resumed.\n", user);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   * ޤλǤOCHUSHA_ASYNC_BUFFER_OKʤΤǡbuffercondvarԤäƤ
   * ΤconsumerΤߤǤ뤳ȤݾڤƤ롣
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (result)
    SIGNAL_BUFFER_CONDITION(buffer);	/* λǡbufferwaitƤ
					 * åɤԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΤǤϤʤ
					 */
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_broadcast(OchushaAsyncBuffer *buffer, const char *user)
{
  gboolean result;

#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_broadcast(0x%x, \"%s\")\n",
	  (int)buffer, user);
#endif

  LOCK_BUFFER_LOCK(buffer);

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "\"%s\" is suspended at ochusha_async_buffer_broadcast.\n",
	      user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);	/* ԤäƤ륷ʥ
					 * ᥤ󥹥åɤΤΡ
					 */
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" resumed.\n", user);
#endif
    }
  /* ХåեmutexunlockǤstateѤäƤǽΤ
   * λξ֤ǺοȤο롣Ĥޤꡢλ
   * AYNC_BUFFER_TERMINATEDǤʤä顢ĴޤǤϾư
   */
  result = buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
  if (result)
      BROADCAST_BUFFER_CONDITION(buffer);/* λǡbufferwaitƤ
					  * åɤԤäƤ륷ʥ
					  * ᥤ󥹥åɤΤΤǤϤʤ
					  */
  UNLOCK_BUFFER_LOCK(buffer);

  return result;
}


gboolean
ochusha_async_buffer_wait(OchushaAsyncBuffer *buffer, const char *user)
{
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
  fprintf(stderr, "ochusha_async_buffer_wait(0x%x, \"%s\")\n",
	  (int)buffer, user);
#endif

  while (buffer->state == OCHUSHA_ASYNC_BUFFER_SUSPENDED)
    {
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr,
	      "\"%s\" is suspended at ochusha_async_buffer_wait.\n",
	      user);
#endif
      buffer->number_of_suspended_users++;

      LOCK_GLOBAL_LOCK;
      SIGNAL_GLOBAL_CONDITION;
      UNLOCK_GLOBAL_LOCK;

      WAIT_BUFFER_CONDITION(buffer);
      /* λbuffer->state != OCHUSHA_ASYNC_BUFFER_SUSPENDEDʤ
       * ʥsuspend򤯤˥ᥤ󥹥åɤäƤΡ
       */
#if DEBUG_OCHUSHA_ASYNC_BUFFER_MOST
      fprintf(stderr, "\"%s\" resumed.\n", user);
#endif
    }

  if (buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED)
    WAIT_BUFFER_CONDITION(buffer);

  /* λOCHUSHA_ASYNC_BUFFER_SUSPENDEDʾ⤢뤬
   * ξsuspendϼȤˤ롣
   */

  return buffer->state != OCHUSHA_ASYNC_BUFFER_TERMINATED;
}


void
ochusha_async_buffer_free_when_finished(OchushaAsyncBuffer *buffer,
					gpointer unused)
{
  if (buffer->buffer != NULL)
    free((void *)buffer->buffer);
}
