/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_ui.c,v 1.8 2003/05/15 19:35:01 fuyu Exp $
 */

#include "config.h"

#include "ochusha.h"
#define OCHUSHA_UI_PRIVATE
#include "ochusha_ui.h"

#include "ochusha_utils_2ch.h"

#include "bbs_thread_ui.h"
#include "boardlist_ui.h"
#include "bulletin_board_ui.h"

#include "icon_label.h"
#include "paned_notebook.h"

#include "ugly_gtk2utils.h"

#include "worker.h"

#include <glib.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixdata.h>

#include <pthread.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "blue0.xpm"
#include "blue1.xpm"
#include "blue2.xpm"
#include "blue3.xpm"
#include "orange.xpm"
#include "black.xpm"
#include "red.xpm"

#include "ochusha48.xpm"
#include "ochusha-banner.xpm"

static void setup_tab_label_icons(void);
static void delete_event_cb(GtkWidget *widget, GdkEventAny *event,
			    gpointer user_data);
#if ENABLE_MAIN_TOOLBAR
static void go_forward_button_cb(gpointer unused,
				 OchushaApplication *application);
static void go_back_button_cb(gpointer unused,
			      OchushaApplication *application);
static void refresh_button_cb(gpointer unused,
			      OchushaApplication *application);
static void cancel_button_cb(gpointer unused, OchushaApplication *application);
#endif

static void on_quit(gpointer data, guint action, GtkWidget *widget);
static void main_menu_callback(gpointer data, guint action, GtkWidget *widget);
static void write_optional_prefs(FILE *file, gpointer user_data);
static void configure_application(OchushaApplication *application);
static void read_optional_prefs(GHashTable *pref_attrs, gpointer user_data);

static void access_started_cb(gpointer user_data);
static void access_progressed_cb(gpointer user_data);
static void access_completed_cb(gpointer user_data);
static void access_terminated_cb(gpointer user_data);
static void access_failed_cb(gpointer user_data);

static void open_about_dialog(OchushaApplication *application);
static void about_dialog_response_cb(GtkWidget *dialog, gint response_id,
				     gpointer unused);

static void initialize_open_url_dialog(OchushaApplication *application);
static void open_url_combo_entry_activate_cb(GtkWidget *entry,
					     GtkDialog *dialog);
static void open_url_dialog_response_cb(GtkWidget *dialog, gint response_id,
					OchushaApplication *application);
static void popup_open_url_dialog(OchushaApplication *application);


OchushaNetworkCallbacks icon_indicator_funcs =
{
  access_started_cb,
  access_progressed_cb,
  access_completed_cb,
  access_terminated_cb,
  access_failed_cb
};


static GtkItemFactory *main_menu_item_factory;
static GdkPixbuf *tab_label_icons[7];
static GQuark tab_label_icon_counter_id;


static pthread_mutex_t main_ui_lock;

#define MAIN_UI_LOCK					\
  if (pthread_mutex_lock(&main_ui_lock) != 0)		\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define MAIN_UI_UNLOCK					\
  if (pthread_mutex_unlock(&main_ui_lock) != 0)		\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }

static GtkWidget *about_dialog = NULL;
static GdkPixbuf *banner_pixbuf = NULL;
static GtkWidget *open_url_dialog = NULL;
static GtkWidget *open_url_combo = NULL;
static GtkWidget *open_url_in_tab_button = NULL;
static GList *open_url_history = NULL;


void
ochusha_init_application(OchushaApplication *application)
{
  GtkItemFactoryEntry menu_items[] = {
    {
      _("/_File"),		/* menu path */
      NULL,			/* accelerator */
      NULL,			/* callback_func */
      0,			/* act */
      "<Branch>"		/* type */
    },
    {
      _("/File/_Open"),
      NULL,
      main_menu_callback,
      MAIN_MENU_OPEN_URL,
      NULL
    },
    {
      _("/File/-------"),
      NULL,
      NULL,
      0,
      "<Separator>"
    },
    {
      _("/File/_Quit"),
      "<control>Q",
      on_quit,
      MAIN_MENU_QUIT,
      NULL
    },
    {
      _("/_Board"),
      NULL,
      NULL,
      0,
      "<Branch>"
    },
    {
      _("/Board/_Refresh"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_BOARDLIST,
      NULL
    },
    {
      _("/_Thread"),
      NULL,
      NULL,
      0,
      "<Branch>"
    },
    {
      _("/Thread/_Refresh"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_THREAD,
      NULL
    },
    {
      _("/Thread/Refresh _All"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_THREAD_ALL,
      NULL
    },
    {
      _("/Thread/-------"),
      NULL,
      NULL,
      0,
      "<Separator>"
    },
    {
      _("/Thread/Refresh Threadlist"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_THREADLIST,
      NULL
    },
    {
      _("/Thread/Refresh Threadlist All"),
      NULL,
      main_menu_callback,
      MAIN_MENU_REFRESH_THREADLIST_ALL,
      NULL
    },
    {
      _("/_Help"),
      NULL,
      NULL,
      0,
      "<Branch>"
    },
    {
      _("/Help/_About"),
      NULL,
      main_menu_callback,
      MAIN_MENU_ABOUT,
      NULL
    }
  };
  GtkAccelGroup *accel_group;
  gint nmenu_items = sizeof(menu_items) / sizeof(menu_items[0]);

  GdkGeometry geometry = { 0, 0, 0, 0, 0, 0, 0, 0, 0.0, 0.0, 0 };

  if (pthread_mutex_init(&main_ui_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  setup_tab_label_icons();

  banner_pixbuf
    = gdk_pixbuf_new_from_xpm_data((const char **)ochusha_banner_xpm);

  configure_application(application);

  ugly_gtk_tree_view_unbind_keys();

  application->top_level = GTK_WINDOW(gtk_window_new(GTK_WINDOW_TOPLEVEL));
  g_signal_connect(G_OBJECT(application->top_level), "delete_event",
		   G_CALLBACK(delete_event_cb), application);

  gtk_window_set_title(application->top_level, _("ochusha"));

  gtk_window_set_icon(application->top_level,
		      gdk_pixbuf_new_from_xpm_data((const char **)ochusha48_xpm));


  application->main_window = GTK_BOX(gtk_vbox_new(FALSE, 0));
  gtk_container_add(GTK_CONTAINER(application->top_level),
		    GTK_WIDGET(application->main_window));

  gtk_window_set_geometry_hints(application->top_level,
				GTK_WIDGET(application->main_window),
				&geometry, GDK_HINT_MIN_SIZE);
  gtk_widget_set_size_request(GTK_WIDGET(application->top_level),
			      application->width, application->height);

  /* menu barŹ */
  accel_group = gtk_accel_group_new();
  main_menu_item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>",
						accel_group);
  gtk_item_factory_create_items(main_menu_item_factory,
				nmenu_items, menu_items, application);
  gtk_window_add_accel_group(application->top_level, accel_group);

  application->menu_bar = gtk_item_factory_get_widget(main_menu_item_factory,
						      "<main>");
  /* ̤ʵǽΥ˥塼򻦤 */
  /* ochusha_set_main_menu_sensitive(MAIN_MENU_OPEN_URL, FALSE); */
  ochusha_set_main_menu_sensitive(MAIN_MENU_REFRESH_THREAD_ALL, FALSE);
  /* ochusha_set_main_menu_sensitive(MAIN_MENU_ABOUT, FALSE); */


  gtk_box_pack_start(application->main_window,
		     application->menu_bar, FALSE, TRUE, 0);

#if ENABLE_MAIN_TOOLBAR
  /* toolbarŹ */
  application->toolbar = GTK_TOOLBAR(gtk_toolbar_new());
  gtk_toolbar_set_style(application->toolbar, GTK_TOOLBAR_ICONS);
  gtk_box_pack_start(application->main_window,
		     GTK_WIDGET(application->toolbar),
		     FALSE, TRUE, 0);

  gtk_toolbar_insert_stock(application->toolbar, GTK_STOCK_GO_FORWARD,
			   _("Go next page"),
			   "go_next_page",
			   GTK_SIGNAL_FUNC(go_forward_button_cb),
			   application,
			   0);
  gtk_toolbar_insert_stock(application->toolbar, GTK_STOCK_GO_BACK,
			   _("Go next page"),
			   "go_next_page",
			   GTK_SIGNAL_FUNC(go_back_button_cb),
			   application,
			   0);
  gtk_toolbar_insert_stock(application->toolbar, GTK_STOCK_REFRESH,
			   _("Refresh current page"),
			   "refresh_current_page",
			   GTK_SIGNAL_FUNC(refresh_button_cb),
			   application,
			   -1);
  gtk_toolbar_insert_stock(application->toolbar, GTK_STOCK_CANCEL,
			   _("Cancel current request"),
			   "cancel_current_request",
			   GTK_SIGNAL_FUNC(cancel_button_cb),
			   application,
			   -1);
  application->url_box = GTK_COMBO(gtk_combo_new());
  gtk_toolbar_append_widget(application->toolbar,
			    GTK_WIDGET(application->url_box),
			    _("Input URL to show"),
			    "input_url_to_show");
#endif
  /* ν */
  initialize_open_url_dialog(application);


  /* GUIν */
  prepare_boardlist_ui_initialization(application);
  prepare_board_ui_initialization(application);
  prepare_thread_ui_initialization(application);


  /* İν */
  initialize_boardlist(application);

  gtk_box_pack_start(application->main_window,
		     application->contents_window, TRUE, TRUE, 0);

  application->statusbar = GTK_STATUSBAR(gtk_statusbar_new());

  initialize_board_ui(application);
  initialize_thread_ui(application);

  gtk_box_pack_end(application->main_window,
		   GTK_WIDGET(application->statusbar), FALSE, TRUE, 0);

  application->clipboard = NULL;

  gtk_widget_show_all(GTK_WIDGET(application->top_level));
}


void
ochusha_set_main_menu_sensitive(OchushaMainMenuAction action,
				gboolean sensitive)
{
  GtkWidget *item;
  item = gtk_item_factory_get_widget_by_action(main_menu_item_factory,
					       action);
  if (item != NULL)
    gtk_widget_set_sensitive(item, sensitive);
}


void
ochusha_clipboard_set_text(OchushaApplication *application, const gchar *text)
{
  MAIN_UI_LOCK
    {
      if (application->clipboard == NULL)
	application->clipboard = gtk_clipboard_get(GDK_SELECTION_CLIPBOARD);
#if DEBUG_GUI_MOST
      fprintf(stderr, "setting text on clipboard: \"%s\"\n", text);
#endif
      gtk_clipboard_set_text(application->clipboard, text, -1);
    }
  MAIN_UI_UNLOCK
}


#define COMMAND_LINE_BUFFER_SIZE	4096
void
ochusha_open_url(OchushaApplication *application, const gchar *url,
		 gboolean in_tab, gboolean use_web_browser)
{
  OchushaBulletinBoard *board;
  OchushaBBSThread *thread;
  unsigned int thread_num;
  unsigned int from;
  unsigned int to;
  char dat_filename[PATH_MAX];

  if (use_web_browser
      || !ochusha_utils_2ch_parse_url(application->table, url,
				      &board, &thread_num, &thread,
				      &from, &to, NULL))
    {
      char command_line[COMMAND_LINE_BUFFER_SIZE];
      if (snprintf(command_line, COMMAND_LINE_BUFFER_SIZE,
		   application->web_browser_template, url)
	  < COMMAND_LINE_BUFFER_SIZE)
	{
	  GError *error = NULL;
	  /*
	   * MEMO: UTF-8Ǥ'~'E280BEˤʤäƤꡢ줬ޤޤ줿URL
	   *       mozillaϤƤŪΥڡɽʤȤ
	   *       餷(̤ǧδĶǤɽƤ褦ʵ)
	   *       Τᤳnative localeλꤹencodingѴ롣
	   *
	   *       ȥѥå by Motonobu Ichimura <famao@momonga-linux.org>
	   *
	   * XXX:  UTF-8ʸnativeencodingѴǽȤϸ¤ʤ
	   *       Τϥ顼åȤкפ롣
	   */
	  char *native_command_line
	    = g_locale_from_utf8(command_line, -1, NULL, NULL, NULL);
	  if (!g_spawn_command_line_async(native_command_line, &error))
	    {
	      /* TODO: 顼ɽ */
	      fprintf(stderr, "spawning error: %s (%d)\n",
		      error->message, error->code);
	    }
	  g_free(native_command_line);
	}
      return;
    }

  snprintf(dat_filename, PATH_MAX, "%d.dat", thread_num);
  ochusha_open_thread(application, board, thread, dat_filename, from, in_tab);
}


#if ENABLE_MAIN_TOOLBAR
void
set_current_url(OchushaApplication *application, const gchar *url)
{
  g_return_if_fail(application->url_box != NULL);

  gtk_entry_set_text(GTK_ENTRY(application->url_box->entry), url);
}


const gchar *
get_current_url(OchushaApplication *application)
{
  g_return_val_if_fail(application->url_box != NULL, NULL);

  return gtk_entry_get_text(GTK_ENTRY(application->url_box->entry));
}
#endif


#if DEBUG_GUI_MOST
static void
free_quark_data(gpointer data)
{
  fprintf(stderr, "freeing data(0x%x)\n", (int)data);
  g_free(data);
}
#endif


GtkWidget *
get_tab_label(const gchar *label_text)
{
  GtkWidget *label = icon_label_new(label_text,
				tab_label_icons[TAB_LABEL_ICON_PREPARATION]);
#if DEBUG_GUI_MOST
  g_object_set_qdata_full(G_OBJECT(label), tab_label_icon_counter_id,
			  g_new(int, 1), (GDestroyNotify)free_quark_data);
#else
  g_object_set_qdata_full(G_OBJECT(label), tab_label_icon_counter_id,
			  g_new(int, 1), (GDestroyNotify)g_free);
#endif
  return label;
}


static void
setup_tab_label_icons(void)
{
  tab_label_icons[TAB_LABEL_ICON_PROGRESS0]
    = gdk_pixbuf_new_from_xpm_data((const char **)blue0_xpm);
  tab_label_icons[TAB_LABEL_ICON_PROGRESS1]
    = gdk_pixbuf_new_from_xpm_data((const char **)blue1_xpm);
  tab_label_icons[TAB_LABEL_ICON_PROGRESS2]
    = gdk_pixbuf_new_from_xpm_data((const char **)blue2_xpm);
  tab_label_icons[TAB_LABEL_ICON_PROGRESS3]
    = gdk_pixbuf_new_from_xpm_data((const char **)blue3_xpm);
  tab_label_icons[TAB_LABEL_ICON_PREPARATION]
    = gdk_pixbuf_new_from_xpm_data((const char **)orange_xpm);
  tab_label_icons[TAB_LABEL_ICON_DONE]
    = gdk_pixbuf_new_from_xpm_data((const char **)black_xpm);
  tab_label_icons[TAB_LABEL_ICON_ERROR]
    = gdk_pixbuf_new_from_xpm_data((const char **)red_xpm);
  tab_label_icon_counter_id
    = g_quark_from_static_string("OchushaUI::icon-counter");
}


static void
delete_event_cb(GtkWidget *widget, GdkEventAny *event, gpointer user_data)
{
  on_quit(user_data, 0, widget);
}


#if ENABLE_MAIN_TOOLBAR
static void
go_forward_button_cb(gpointer unused, OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "go_forward_button_cb: not implemented.\n");
#endif
}


static void
go_back_button_cb(gpointer unused, OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "go_forward_button_cb: not implemented.\n");
#endif
}


static void
refresh_button_cb(gpointer unused, OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "refresh_button_cb: not implemented.\n");
#endif
}


static void
cancel_button_cb(gpointer unused, OchushaApplication *application)
{
#if DEBUG_GUI
  fprintf(stderr, "cancel_button_cb: not implemented.\n");
#endif
}
#endif


static void
on_quit(gpointer data, guint action, GtkWidget *widget)
{
  OchushaApplication *application = (OchushaApplication *)data;
  GtkAllocation *allocation = &GTK_WIDGET(application->top_level)->allocation;

  application->width = allocation->width;
  application->height = allocation->height;

  allocation = &application->boardlist_view->allocation;
  application->boardlist_width = allocation->width;

  finalize_boardlist(application);

  if (!ochusha_write_config_xml(&application->conf,
				write_optional_prefs, application))
    fprintf(stderr, "Couldn't write %s.\n", OCHUSHA_CONFIG_XML);

  gtk_main_quit();
}


static void
main_menu_callback(gpointer data, guint action, GtkWidget *widget)
{
  OchushaApplication *application = (OchushaApplication *)data;
  switch (action)
    {
    case MAIN_MENU_OPEN_URL:
      popup_open_url_dialog(application);
      return;

    case MAIN_MENU_REFRESH_BOARDLIST:
      refresh_boardlist(application);
      return;

    case MAIN_MENU_REFRESH_THREAD:
      refresh_current_thread(application);
      return;

#if 0
    case MAIN_MENU_REFRESH_THREAD_ALL:
      refresh_current_threadlist_all(application);
      return;
#endif

    case MAIN_MENU_REFRESH_THREADLIST:
      refresh_current_threadlist(application);
      return;

    case MAIN_MENU_REFRESH_THREADLIST_ALL:
      refresh_current_threadlist_all(application);
      return;

    case MAIN_MENU_ABOUT:
      open_about_dialog(application);
      return;
    }

  fprintf(stderr, "Not implemented yet.\n");
}


#define OUTPUT_CONFIG_ATTRIBUTE_INT(file, conf, attribute)	\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <int val=\"%d\"/>\n"				\
	  "    </attribute>\n", (conf)->attribute)

#define OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, conf, attribute)	\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <boolean val=\"%s\"/>\n"			\
	  "    </attribute>\n", (conf)->attribute ? "true" : "false" )

#define OUTPUT_CONFIG_ATTRIBUTE_STRING(file, conf, attribute)	\
  fprintf(file,							\
	  "    <attribute name=\"" #attribute	"\">\n"		\
	  "      <string>%s</string>\n"				\
	  "    </attribute>\n", (conf)->attribute)


static void
write_optional_prefs(FILE *file, gpointer user_data)
{
  OchushaApplication *application = (OchushaApplication *)user_data;

  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, width);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, height);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, boardlist_width);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, threadlist_height);

  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, boardlist_pane_style);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, board_tab_shrinkable);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, board_tab_minimum_size);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  board_tab_enable_tooltips);
  
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, threadlist_pane_style);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, thread_tab_shrinkable);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, thread_tab_minimum_size);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  thread_tab_enable_tooltips);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  show_threadlist_entry_when_thread_selected);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, enable_popup_title);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, popup_title_delay);

  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application, enable_popup_response);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application, popup_response_delay);
  OUTPUT_CONFIG_ATTRIBUTE_BOOLEAN(file, application,
				  enable_smart_popup_placement);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application,
			      maximum_number_of_popup_responses);

  OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application, web_browser_template);

  if (application->user_default_filter.rule != NULL)
    OUTPUT_CONFIG_ATTRIBUTE_STRING(file, application,
				   user_default_filter.rule);
  OUTPUT_CONFIG_ATTRIBUTE_INT(file, application,
			      user_default_filter.ignore_threshold);
}


static void
configure_application(OchushaApplication *application)
{
  ochusha_read_config_xml(&application->conf, read_optional_prefs,
			  application);
  /* Sets unspecified attributes */
  if (application->width == 0)
    {
      application->width = OCHUSHA_DEFAULT_WIDTH;
      application->height = OCHUSHA_DEFAULT_HEIGHT;
      application->boardlist_width = OCHUSHA_DEFAULT_BOARDLIST_WIDTH;
      application->threadlist_height = OCHUSHA_DEFAULT_THREADLIST_HEIGHT;

      application->boardlist_pane_style = HPANED_STYLE;
      application->threadlist_pane_style = VPANED_STYLE;

      application->maximum_number_of_popup_responses
	= OCHUSHA_DEFAULT_MAXIMUM_NUMBER_OF_POPUP_RESPONSES;

      application->popup_response_delay = OCHUSHA_DEFAULT_POPUP_DELAY;
      application->show_threadlist_entry_when_thread_selected
	= OCHUSHA_DEFAULT_SHOW_THREADLIST_ENTRY_WHEN_THREAD_SELECTED;
    }

  if (application->board_tab_minimum_size == 0)
    application->board_tab_minimum_size
      = OCHUSHA_DEFAULT_BOARD_TAB_MINIMUM_SIZE;

  if (application->thread_tab_minimum_size == 0)
    application->thread_tab_minimum_size
      = OCHUSHA_DEFAULT_THREAD_TAB_MINIMUM_SIZE;

  if (application->popup_title_delay == 0)
    application->popup_title_delay = OCHUSHA_DEFAULT_POPUP_DELAY;

  if (application->web_browser_template == NULL)
    application->web_browser_template = g_strdup("mozilla -raise -remote \"openURL(%s,new-window)\"");
}


static void
read_optional_prefs(GHashTable *pref_attrs, gpointer user_data)
{
  OchushaApplication *application = (OchushaApplication *)user_data;

  application->width = ochusha_utils_get_attribute_int(pref_attrs, "width");

  application->height = ochusha_utils_get_attribute_int(pref_attrs, "height");

  application->boardlist_width
    = ochusha_utils_get_attribute_int(pref_attrs, "boardlist_width");

  application->threadlist_height
    = ochusha_utils_get_attribute_int(pref_attrs, "threadlist_height");

  if (g_hash_table_lookup(pref_attrs, "boardlist_pane_style") != NULL)
    application->boardlist_pane_style
      = ochusha_utils_get_attribute_int(pref_attrs, "boardlist_pane_style");
  else
    application->boardlist_pane_style = HPANED_STYLE;

  if (g_hash_table_lookup(pref_attrs, "board_tab_shrinkable") != NULL)
    application->board_tab_shrinkable
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "board_tab_shrinkable");
  else
    application->board_tab_shrinkable = OCHUSHA_DEFAULT_BOARD_TAB_SHRINKABLE;

  if (g_hash_table_lookup(pref_attrs, "board_tab_minimum_size") != NULL)
    application->board_tab_minimum_size
      = ochusha_utils_get_attribute_int(pref_attrs, "board_tab_minimum_size");
  else
    application->board_tab_minimum_size
      = OCHUSHA_DEFAULT_BOARD_TAB_MINIMUM_SIZE;

  if (g_hash_table_lookup(pref_attrs, "board_tab_enable_tooltips") != NULL)
    application->board_tab_enable_tooltips
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "board_tab_enable_tooltips");
  else
    application->board_tab_enable_tooltips
      = OCHUSHA_DEFAULT_BOARD_TAB_ENABLE_TOOLTIPS;


  if (g_hash_table_lookup(pref_attrs, "threadlist_pane_style") != NULL)
    application->threadlist_pane_style
      = ochusha_utils_get_attribute_int(pref_attrs, "threadlist_pane_style");
  else
    application->threadlist_pane_style = VPANED_STYLE;

  if (g_hash_table_lookup(pref_attrs, "thread_tab_shrinkable") != NULL)
    application->thread_tab_shrinkable
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "thread_tab_shrinkable");
  else
    application->thread_tab_shrinkable = OCHUSHA_DEFAULT_THREAD_TAB_SHRINKABLE;

  if (g_hash_table_lookup(pref_attrs, "thread_tab_minimum_size") != NULL)
    application->thread_tab_minimum_size
      = ochusha_utils_get_attribute_int(pref_attrs, "thread_tab_minimum_size");
  else
    application->thread_tab_minimum_size
      = OCHUSHA_DEFAULT_THREAD_TAB_MINIMUM_SIZE;

  if (g_hash_table_lookup(pref_attrs, "thread_tab_enable_tooltips") != NULL)
    application->thread_tab_enable_tooltips
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "thread_tab_enable_tooltips");
  else
    application->thread_tab_enable_tooltips
      = OCHUSHA_DEFAULT_THREAD_TAB_ENABLE_TOOLTIPS;

  if (g_hash_table_lookup(pref_attrs,
			  "show_threadlist_entry_when_thread_selected")
      != NULL)
    application->show_threadlist_entry_when_thread_selected
      = ochusha_utils_get_attribute_boolean(pref_attrs,
				"show_threadlist_entry_when_thread_selected");
  else
    application->show_threadlist_entry_when_thread_selected
      = OCHUSHA_DEFAULT_SHOW_THREADLIST_ENTRY_WHEN_THREAD_SELECTED;


  if (g_hash_table_lookup(pref_attrs, "enable_popup_title") != NULL)
    application->enable_popup_title
      = ochusha_utils_get_attribute_boolean(pref_attrs, "enable_popup_title");
  else
    application->enable_popup_title = FALSE;

  application->popup_title_delay
    = ochusha_utils_get_attribute_int(pref_attrs, "popup_title_delay");

  if (g_hash_table_lookup(pref_attrs, "enable_popup_response") != NULL)
    application->enable_popup_response
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "enable_popup_response");
  else
    application->enable_popup_response = TRUE;

  if (g_hash_table_lookup(pref_attrs, "popup_response_delay") != NULL)
    application->popup_response_delay
      = ochusha_utils_get_attribute_int(pref_attrs, "popup_response_delay");
  else
    application->popup_response_delay = OCHUSHA_DEFAULT_POPUP_DELAY;

  if (g_hash_table_lookup(pref_attrs, "enable_smart_popup_placement") != NULL)
    application->enable_smart_popup_placement
      = ochusha_utils_get_attribute_boolean(pref_attrs,
					    "enable_smart_popup_placement");
  else
    application->enable_smart_popup_placement = TRUE;

  if (g_hash_table_lookup(pref_attrs, "maximum_number_of_popup_responses")
      != NULL)
    application->maximum_number_of_popup_responses
      = ochusha_utils_get_attribute_int(pref_attrs,
					"maximum_number_of_popup_responses");
  else
    application->maximum_number_of_popup_responses
      = OCHUSHA_DEFAULT_MAXIMUM_NUMBER_OF_POPUP_RESPONSES;

  application->web_browser_template
    = ochusha_utils_get_attribute_string(pref_attrs, "web_browser_template");

  application->user_default_filter.rule
    = ochusha_utils_get_attribute_string(pref_attrs,
					 "user_default_filter.rule");

  application->user_default_filter.ignore_threshold
    = ochusha_utils_get_attribute_int(pref_attrs,
				      "user_default_filter.ignore_threshold");
}


static void
access_started_cb(gpointer user_data)
{
  IconLabel *icon_label;
  int *counter;
#if DEBUG_GUI
  fprintf(stderr, "access_started_cb\n");
#endif
  icon_label = ICON_LABEL(user_data);
  counter = g_object_get_qdata(G_OBJECT(icon_label),
			       tab_label_icon_counter_id);
  *counter = 0;

  gdk_threads_enter();
  icon_label_set_icon(icon_label, tab_label_icons[TAB_LABEL_ICON_PREPARATION]);
  gdk_threads_leave();
}


static int progress_icon_number[7] =
{
  TAB_LABEL_ICON_PROGRESS0,
  TAB_LABEL_ICON_PROGRESS1,
  TAB_LABEL_ICON_PROGRESS2,
  TAB_LABEL_ICON_PROGRESS3,
  TAB_LABEL_ICON_PROGRESS2,
  TAB_LABEL_ICON_PROGRESS1,
  TAB_LABEL_ICON_PROGRESS0,
};


static void
access_progressed_cb(gpointer user_data)
{
  IconLabel *icon_label;
  int *counter;
#if DEBUG_GUI_MOST
  fprintf(stderr, "access_progressed_cb\n");
#endif

  if (user_data == NULL)
    return;

  g_return_if_fail(IS_ICON_LABEL(user_data));
  icon_label = ICON_LABEL(user_data);

  counter = g_object_get_qdata(G_OBJECT(icon_label),
			       tab_label_icon_counter_id);
  *counter = (*counter + 1) % 7;

  gdk_threads_enter();
  icon_label_set_icon(icon_label,
		      tab_label_icons[progress_icon_number[*counter]]);
  gdk_threads_leave();
}


static void
access_completed_cb(gpointer user_data)
{
  IconLabel *icon_label;

#if DEBUG_GUI
  fprintf(stderr, "access_completed_cb\n");
#endif

  if (user_data == NULL)
    return;

  g_return_if_fail(IS_ICON_LABEL(user_data));
  icon_label = ICON_LABEL(user_data);

  gdk_threads_enter();
  icon_label_set_icon(icon_label, tab_label_icons[TAB_LABEL_ICON_DONE]);
  gdk_threads_leave();
}


static void
access_terminated_cb(gpointer user_data)
{
  IconLabel *icon_label;

#if DEBUG_GUI
  fprintf(stderr, "access_terminated_cb\n");
#endif

  if (user_data == NULL)
    return;

  g_return_if_fail(IS_ICON_LABEL(user_data));
  icon_label = ICON_LABEL(user_data);

  gdk_threads_enter();
  icon_label_set_icon(icon_label, tab_label_icons[TAB_LABEL_ICON_DONE]);
  gdk_threads_leave();
}


static void
access_failed_cb(gpointer user_data)
{
  IconLabel *icon_label;

#if DEBUG_GUI
  fprintf(stderr, "access_failed_cb\n");
#endif

  if (user_data == NULL)
    return;

  g_return_if_fail(IS_ICON_LABEL(user_data));
  icon_label = ICON_LABEL(user_data);

  gdk_threads_enter();
  icon_label_set_icon(icon_label, tab_label_icons[TAB_LABEL_ICON_ERROR]);
  gdk_threads_leave();
}


static void
open_about_dialog(OchushaApplication *application)
{
  /* about_dialog */
  MAIN_UI_LOCK
    {
      if (about_dialog == NULL)
	{
	  gchar text_buffer[1024];
	  GtkWidget *label;
	  about_dialog = gtk_dialog_new_with_buttons(_("About Ochusha"),
					application->top_level,
					GTK_DIALOG_DESTROY_WITH_PARENT,
					GTK_STOCK_OK, GTK_RESPONSE_OK,
					NULL);
	  gtk_dialog_set_default_response(GTK_DIALOG(about_dialog),
					  GTK_RESPONSE_OK);
	  g_signal_connect(G_OBJECT(about_dialog), "response",
			   G_CALLBACK(about_dialog_response_cb),
			   NULL);

	  snprintf(text_buffer, 1024, "%s %s %s",
		   _("ochusha"), _("Version"), PACKAGE_VERSION);
	  label = gtk_label_new(text_buffer);
	  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(about_dialog)->vbox),
			    label);

	  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(about_dialog)->vbox),
			    gtk_image_new_from_pixbuf(banner_pixbuf));

	  label = gtk_label_new("Copyright(c) 2003 The Ochusha Project");
	  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(about_dialog)->vbox),
			    label);

	  gtk_widget_show_all(about_dialog);
	}
    }
  MAIN_UI_UNLOCK;
}


static void
about_dialog_response_cb(GtkWidget *widget, gint response_id, gpointer unused)
{
  GtkWidget *dialog;
  MAIN_UI_LOCK
    {
      dialog = about_dialog;
      about_dialog = NULL;
    }
  MAIN_UI_UNLOCK;

  if (dialog == NULL)
    return;

  gtk_widget_destroy(dialog);
}


static void
initialize_open_url_dialog(OchushaApplication *application)
{
  GtkWidget *tmp_hbox;
  open_url_dialog = gtk_message_dialog_new(application->top_level,
					   GTK_DIALOG_DESTROY_WITH_PARENT,
					   GTK_MESSAGE_QUESTION,
					   GTK_BUTTONS_OK_CANCEL,
					   _("Input the URL of the document you want to open."));
  gtk_dialog_set_default_response(GTK_DIALOG(open_url_dialog),
				  GTK_RESPONSE_OK);
  g_signal_connect(G_OBJECT(open_url_dialog), "response",
		   G_CALLBACK(open_url_dialog_response_cb),
		   application);

  open_url_combo = gtk_combo_new();
  if (open_url_history != NULL)
    gtk_combo_set_popdown_strings(GTK_COMBO(open_url_combo),
				  open_url_history);
  gtk_combo_disable_activate(GTK_COMBO(open_url_combo));
  g_signal_connect(G_OBJECT(GTK_COMBO(open_url_combo)->entry), "activate",
		   G_CALLBACK(open_url_combo_entry_activate_cb),
		   open_url_dialog);
  tmp_hbox = gtk_hbox_new(FALSE, 0);
  gtk_box_pack_start(GTK_BOX(tmp_hbox), gtk_label_new(_("URL: ")),
		     FALSE, FALSE, 0);
  gtk_box_pack_start(GTK_BOX(tmp_hbox), open_url_combo, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(open_url_dialog)->vbox),
		     tmp_hbox, FALSE, FALSE, 0);
  open_url_in_tab_button = gtk_check_button_new_with_label(_("Open in tab"));
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(open_url_dialog)->vbox),
		     open_url_in_tab_button, FALSE, FALSE, 0);
}


static void
open_url_combo_entry_activate_cb(GtkWidget *entry, GtkDialog *dialog)
{
  gtk_dialog_response(dialog, GTK_RESPONSE_OK);
}


static void
open_url_dialog_response_cb(GtkWidget *dialog, gint response_id,
			    OchushaApplication *application)
{
  if (response_id == GTK_RESPONSE_OK)
    {
      const gchar *url
	= gtk_entry_get_text(GTK_ENTRY(GTK_COMBO(open_url_combo)->entry));
      gboolean in_tab = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(open_url_in_tab_button));
      ochusha_open_url(application, url, in_tab, FALSE);
    }

  gtk_widget_hide_all(open_url_dialog);

  if (response_id == GTK_RESPONSE_DELETE_EVENT)
    {
      MAIN_UI_LOCK
	{
	  open_url_dialog = NULL;
	}
      MAIN_UI_UNLOCK;
    }
}


static void
popup_open_url_dialog(OchushaApplication *application)
{
  MAIN_UI_LOCK
    {
      if (open_url_dialog == NULL)
	initialize_open_url_dialog(application);
      gtk_widget_show_all(open_url_dialog);
    }
  MAIN_UI_UNLOCK;
}
