/*
 * Copyright (c) 2003 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: boardlist_ui.c,v 1.2 2003/05/11 20:55:12 fuyu Exp $
 */

#include "config.h"

#include "ochusha.h"
#include "ochusha_ui.h"

#include "ochusha_utils_2ch.h"

#include "bbs_thread_ui.h"
#include "board_properties.h"
#include "boardlist_ui.h"
#include "boardlist_view.h"
#include "bulletin_board_ui.h"
#include "paned_notebook.h"
#include "ugly_gtk2utils.h"

#include "worker.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <errno.h>

#include <stdio.h>
#include <stdlib.h>


#if DEBUG_WIDGET_MOST
static void select_category_cb(BoardlistView *view,
			       OchushaBoardCategory *category,
			       gpointer user_data);
#endif
static void boardlist_read_category_element_cb(OchushaBBSTable *table,
					       OchushaBoardCategory *category,
					       GHashTable *cat_attributes,
					       OchushaApplication *appl);
static void boardlist_read_board_element_cb(OchushaBBSTable *table,
					    OchushaBulletinBoard *board,
					    GHashTable *cat_attributes,
					    OchushaApplication *appl);

static void boardlist_write_category_element_cb(OchushaBBSTable *table,
						OchushaBoardCategory *category,
						FILE *boardlist_xml,
						OchushaApplication *appl);
static void boardlist_write_board_element_cb(OchushaBBSTable *table,
					     OchushaBulletinBoard *board,
					     FILE *boardlist_xml,
					     OchushaApplication *appl);

static void write_threadlist(gpointer data, gpointer user_data);

static void build_boardlist(WorkerThread *employee, gpointer args);

static void setup_boardlist(OchushaApplication *application);

static void filter_toggled_cb(GtkToggleButton *toggle_button,
			      OchushaApplication *application);
static void item_view_required_cb(PanedNotebook *paned_notebook,
				  OchushaBulletinBoard *board, GtkWidget **item_view,
				  const gchar **title, GtkWidget **tab_label,
				  OchushaApplication *application);
static void item_view_being_closed_cb(PanedNotebook *paned_notebook,
				      OchushaBulletinBoard *board,
				      GtkWidget *item_view,
				      OchushaApplication *application);
static void item_view_closed_cb(PanedNotebook *paned_notebook,
				OchushaBulletinBoard *board, GtkWidget *item_view,
				OchushaApplication *application);
static void page_switched_cb(PanedNotebook *paned_notebook,
			     GtkWidget *previous_page, GtkWidget *new_page,
			     OchushaApplication *application);

#if 0
static void make_new_thread_button_cb(GtkWidget *widget,
				      OchushaApplication *application);
#endif
static void open_board_properties_button_cb(GtkWidget *widget,
					    OchushaApplication *application);
static void refresh_board_button_cb(GtkWidget *widget,
				    OchushaApplication *application);
static void start_threadlist_search_button_cb(GtkWidget *widget,
					      OchushaApplication *application);



/* Υå˰٤ǤⳫ(OchushaBulletinBoard *)Υꥹȡ*/
static GSList *board_list;

static pthread_mutex_t board_list_lock;


#define BOARD_LIST_LOCK					\
  if (pthread_mutex_lock(&board_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define BOARD_LIST_UNLOCK				\
  if (pthread_mutex_unlock(&board_list_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


void
initialize_boardlist(OchushaApplication *application)
{
  GtkToolbar *toolbar;
  GtkWidget *filter_button;

  board_list = NULL;

  if (pthread_mutex_init(&board_list_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  application->boardlist_view = boardlist_view_new();
  gtk_widget_show(application->boardlist_view);

  application->contents_window
    = paned_notebook_new_with_selector(application->boardlist_pane_style,
				       application->boardlist_view);
  gtk_paned_set_position(GTK_PANED(application->contents_window),
			 application->boardlist_width);
  paned_notebook_set_always_show_contents_pane(PANED_NOTEBOOK(application->contents_window),
					       TRUE);
  paned_notebook_set_tab_shrinkable(PANED_NOTEBOOK(application->contents_window), application->board_tab_shrinkable);
  paned_notebook_set_minimum_tab_label_size(PANED_NOTEBOOK(application->contents_window), application->board_tab_minimum_size);
  paned_notebook_set_tooltips(PANED_NOTEBOOK(application->contents_window),
			      application->board_tab_enable_tooltips);

  toolbar = paned_notebook_get_toolbar(PANED_NOTEBOOK(application->contents_window));

#if 0
  gtk_toolbar_prepend_space(toolbar);
  gtk_toolbar_insert_stock(toolbar, GTK_STOCK_NEW,
			   _("Make a new thread at the current board"),
			   "make_new_thread_at_current_board",
			   GTK_SIGNAL_FUNC(make_new_thread_button_cb),
			   application,
			   0);
#endif

  gtk_toolbar_prepend_space(toolbar);
  gtk_toolbar_insert_stock(toolbar, GTK_STOCK_PROPERTIES,
			   _("Open the current board's properties"),
			   "open_the_current_board_properties",
			   GTK_SIGNAL_FUNC(open_board_properties_button_cb),
			   application,
			   0);
  filter_button = gtk_toggle_button_new_with_label(_("Filter"));
  gtk_widget_show(filter_button);
  application->filter_button = GTK_TOGGLE_BUTTON(filter_button);
  gtk_toolbar_prepend_widget(toolbar, filter_button,
			     _("Apply filter to the current threadlist"),
			     "apply_filter_to_the_current_threadlist");
  g_signal_connect(G_OBJECT(filter_button), "toggled",
		   G_CALLBACK(filter_toggled_cb), application);

  gtk_toolbar_prepend_space(toolbar);
  gtk_toolbar_insert_stock(toolbar, GTK_STOCK_REFRESH,
			   _("Refresh current threadlist"),
			   "refresh_current_threadlist",
			   GTK_SIGNAL_FUNC(refresh_board_button_cb),
			   application,
			   0);

  gtk_toolbar_insert_stock(toolbar, GTK_STOCK_FIND,
			   _("Start text search on current threadlist"),
			   "start_text_search_on_threadlist",
			   GTK_SIGNAL_FUNC(start_threadlist_search_button_cb),
			   application,
			   0);

#if DEBUG_WIDGET_MOST
  g_signal_connect(G_OBJECT(application->boardlist_view), "select_category",
		   G_CALLBACK(select_category_cb), application);
#endif
  g_signal_connect(G_OBJECT(application->contents_window),
		   "item_view_required", G_CALLBACK(item_view_required_cb),
		   application);
  g_signal_connect(G_OBJECT(application->contents_window),
		   "item_view_being_closed",
		   G_CALLBACK(item_view_being_closed_cb), application);
  g_signal_connect(G_OBJECT(application->contents_window),
		   "item_view_closed", G_CALLBACK(item_view_closed_cb),
		   application);
  g_signal_connect(G_OBJECT(application->contents_window),
		   "page_switched", G_CALLBACK(page_switched_cb),
		   application);

  application->table = ochusha_bbs_table_new();

  /* γĥǥ¸ */
  g_signal_connect(G_OBJECT(application->table),
		   "boardlist_read_category_element",
		   G_CALLBACK(boardlist_read_category_element_cb),
		   application);
  g_signal_connect(G_OBJECT(application->table),
		   "boardlist_read_board_element",
		   G_CALLBACK(boardlist_read_board_element_cb),
		   application);
  g_signal_connect(G_OBJECT(application->table),
		   "boardlist_write_category_element",
		   G_CALLBACK(boardlist_write_category_element_cb),
		   application);
  g_signal_connect(G_OBJECT(application->table),
		   "boardlist_write_board_element",
		   G_CALLBACK(boardlist_write_board_element_cb),
		   application);

  if (ochusha_bbs_table_read_boardlist_xml(application->table,
					   &application->conf))
    {
      setup_boardlist(application);
      return;
    }

  refresh_boardlist(application);
}


void
refresh_boardlist(OchushaApplication *application)
{
  WorkerJob *job = (WorkerJob *)calloc(1, sizeof(WorkerJob));
  if (job == NULL)
    {
      fprintf(stderr, "Couldn't start boardlist reader.\n");
      return;
    }
  job->canceled = FALSE;
  job->job = build_boardlist;
  job->args = application;

  commit_job(job);
}


void
finalize_boardlist(OchushaApplication *application)
{
  if (application->contents_window != NULL)
    {
      PanedNotebook *boardlist = PANED_NOTEBOOK(application->contents_window);
      GtkWidget *current_board;
      application->boardlist_width
	= gtk_paned_get_position(GTK_PANED(boardlist));
      current_board = paned_notebook_get_current_item_view(boardlist);

      if (current_board != NULL)
	application->threadlist_height
	  = gtk_paned_get_position(GTK_PANED(current_board));
    }

  if (application->table != NULL)
    {
      if (!ochusha_bbs_table_write_boardlist_xml(application->table,
						 &application->conf))
	fprintf(stderr, "Couldn't write boardlist2.xml.\n");
    }

  g_slist_foreach(board_list, write_threadlist, application);
}


#if 0
void
make_new_thread(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  OchushaBulletinBoard *board;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);

  contents_window = PANED_NOTEBOOK(application->contents_window);

  board = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);
  if (board == NULL)
    return;

  /*
   * TODO: 빽ݤʤΤǺ٤ˤ褦
   */
}
#endif


void
open_current_board_properties(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  OchushaBulletinBoard *board;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);

  contents_window = PANED_NOTEBOOK(application->contents_window);

  board = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);
  if (board == NULL)
    return;

  open_board_properties(application, board);
}


void
refresh_current_threadlist(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  OchushaBulletinBoard *board;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);

  contents_window = PANED_NOTEBOOK(application->contents_window);

  board = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);
  if (board == NULL)
    return;

  refresh_threadlist_view(application, board);
}


static void
refresh_threadlist_helper(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)data;
  OchushaApplication *application = (OchushaApplication *)user_data;
  refresh_threadlist_view(application, board);
}


void
refresh_current_threadlist_all(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  paned_notebook_foreach_item(contents_window, refresh_threadlist_helper,
			      application);
}


static void
redraw_threadlist_helper(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)data;
  OchushaApplication *application = (OchushaApplication *)user_data;
  redraw_threadlist_view(application, board);
}


void
redraw_current_threadlist_all(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  paned_notebook_foreach_item(contents_window, redraw_threadlist_helper,
			      application);
  fix_filter_button_state(application);
}


void
start_current_threadlist_search(OchushaApplication *application)
{
  GtkWidget *widget;

  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);

  widget = paned_notebook_get_current_item_view(
				PANED_NOTEBOOK(application->contents_window));
  if (widget == NULL)
    return;
  g_return_if_fail(IS_PANED_NOTEBOOK(widget));

  widget = paned_notebook_get_selector(PANED_NOTEBOOK(widget));
  if (widget == NULL)
    return;
  g_return_if_fail(IS_THREADLIST_VIEW(widget));

  threadlist_view_start_interactive_search(THREADLIST_VIEW(widget));
}


/*
 * ϤʤΤѰդषOchushaBulletinBoardѲưŪ
 * ˻ΤƼΧŪб٤ʤΤʡ
 */
void
fix_filter_button_state(OchushaApplication *application)
{
  PanedNotebook *contents_window;
  OchushaBulletinBoard *board;
  g_return_if_fail(application != NULL
		   && application->contents_window != NULL);
  contents_window = PANED_NOTEBOOK(application->contents_window);
  board = (OchushaBulletinBoard *)paned_notebook_get_current_item(contents_window);
  if (board == NULL)
    return;

  gtk_toggle_button_set_active(application->filter_button,
		((BulletinBoardGUIInfo *)board->user_data)->enable_filter);
}


void
update_threadlist_entry(OchushaApplication *application, OchushaBBSThread *thread)
{
  PanedNotebook *board_view;
  ThreadlistView *threadlist_view;
  PanedNotebook *boardlist = (PanedNotebook *)application->contents_window;

  g_return_if_fail(boardlist != NULL && thread != NULL);

  gdk_threads_enter();
  board_view = (PanedNotebook *)paned_notebook_get_item_view(boardlist,
							     thread->board);

  if (board_view == NULL)
    goto finish_update;	/* ĤɽƤʤ⤢ */

  threadlist_view = (ThreadlistView *)paned_notebook_get_selector(board_view);

  if (threadlist_view == NULL)
    {
      goto finish_update;	/* ե륿ߤǥɽƤʤ
				 * ⤢롣
				 */
    }

  update_threadlist_entry_style(application, threadlist_view, thread);

 finish_update:
  gdk_threads_leave();
}


static void
filter_toggled_cb(GtkToggleButton *toggle_button,
		  OchushaApplication *application)
{
  PanedNotebook *board_pane = PANED_NOTEBOOK(application->contents_window);
  OchushaBulletinBoard *board
    = (OchushaBulletinBoard *)paned_notebook_get_current_item(board_pane);
  BulletinBoardGUIInfo *info;

#if DEBUG_GUI_MOST
  fprintf(stderr, "filter_toggled_cb.\n");
#endif

  if (board == NULL)
    return;

  info = (BulletinBoardGUIInfo *)board->user_data;

  if (info == NULL)
    return;

  if (info->enable_filter == gtk_toggle_button_get_active(toggle_button))
    return;

  info->enable_filter = !info->enable_filter;

  gtk_toggle_button_set_active(toggle_button, info->enable_filter);
  gdk_threads_leave();
  redraw_threadlist_view(application, board);
  gdk_threads_enter();
}


#if DEBUG_WIDGET_MOST
static void
select_category_cb(BoardlistView *view, OchushaBoardCategory *category,
		   gpointer user_data)
{
  fprintf(stderr, "select_category_cb: 0x%x, 0x%x, 0x%x\n",
	  (int)view, (int)category, (int)user_data);
}
#endif


static void
item_view_required_cb(PanedNotebook *paned_notebook, OchushaBulletinBoard *board,
		      GtkWidget **item_view, const gchar **title,
		      GtkWidget **tab_label,
		      OchushaApplication *application)
{
  OchushaBulletinBoard *board2;
#if DEBUG_GUI_MOST
  fprintf(stderr, "boardlist_ui.c: item_view_required_cb called.\n");
#endif
  *tab_label = get_tab_label(board->name);
  *item_view = open_bulletin_board(application, board, ICON_LABEL(*tab_label));

  board2 = ochusha_bbs_table_lookup_board_by_url(application->table,
						 board->base_url);

  if (*item_view == NULL)
    {
      gtk_object_sink(GTK_OBJECT(*tab_label));
      *tab_label = NULL;
      return;
    }

  BOARD_LIST_LOCK
    {
      if (g_slist_find(board_list, board) == NULL)
	board_list = g_slist_append(board_list, board);
    }
  BOARD_LIST_UNLOCK;

  gtk_widget_show(*item_view);
  *title = board->name;
}


static void
item_view_being_closed_cb(PanedNotebook *paned_notebook, OchushaBulletinBoard *board,
			  GtkWidget *item_view,
			  OchushaApplication *application)
{
  OchushaAsyncBuffer *buffer = snatch_subject_txt_buffer_for_board(board);

#ifndef SUSPEND_ALL_FOR_DESTRUCTION	/* ¸ */
  /*
   * MEMO: ɽƤ륦åȤºݤ˼ΤƤ륿ߥ
   *       򤺤餹
   * 
   * GtkTreeViewŪidleؿϿƤꡢ줬Ƥ֤˥
   * åȤȲ줿åȤidleؿưƤޤ
   * ͡ˡʤΤǡǤϥåȤref_count1äƤ
   * Υץ饤ƥidleؿˤäơˡref_count򸺤餹
   * Υץ饤ƥidleؿưȤȤϡGTK+Ū
   * ȤäƤidleؿλŻλƤȤȤˤʤΤǡ
   * פ
   */
  g_object_ref(G_OBJECT(item_view));

  /*
   * MEMO: Ĥ褦ȤƤ륹Ρsubject.txtɤ߹ߡפ
   *       ֥󥰡פԤäƤthreadŪ˽λ롣
   */
  if (buffer != NULL)
    {
      gdk_threads_leave();

      ochusha_async_buffer_suspend(buffer);
      ochusha_async_buffer_terminate(buffer);
      ochusha_async_buffer_resume(buffer);

      gdk_threads_enter();
    }
#else
  /*
   * ΥɽĤ褦ȤƤΤǡϢthreadߤᡢ
   * ¾GUIϢthread򥵥ڥɤίäƤ륤٥Ȥ
   * Ƥ롣ǽʸ¤᤯٥Ȥò뤿ᡢƤthread
   * ڥɤˡ˴Ϣthread϶Ūߤ롣
   *
   * ġĤ褦ˤ顢ब򤱤褦ʵΤġġ
   */

  gdk_threads_leave();

  ochusha_async_buffer_suspend_all();

  if (buffer != NULL)
    ochusha_async_buffer_terminate(buffer);

  gdk_threads_enter();

  while (gtk_events_pending())
    gtk_main_iteration();

  ochusha_async_buffer_resume_all();
#endif

  if (buffer != NULL)
    g_object_unref(G_OBJECT(buffer));
  return;
}


static void
item_view_closed_cb(PanedNotebook *paned_notebook, OchushaBulletinBoard *board,
		    GtkWidget *item_view, OchushaApplication *application)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "boardlist_ui.c: item_view_closed_cb called.\n");
#endif
  if (paned_notebook_get_current_item_view(paned_notebook) == NULL)
    {
      /* ɽˤǤƤڡĤ줿 */
      if (item_view != NULL && IS_PANED_NOTEBOOK(item_view))
	{
	  application->threadlist_height
	    = gtk_paned_get_position(GTK_PANED(item_view));
	}
    }

  finalize_bulletin_board_view(board, item_view);
#ifndef SUSPEND_ALL_FOR_DESTRUCTION
  delayed_g_object_unref(G_OBJECT(item_view));
#endif
}


static void
page_switched_cb(PanedNotebook *paned_notebook,
		 GtkWidget *previous_page, GtkWidget *new_page,
		 OchushaApplication *application)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "boardlist_ui.c: page_switched_cb()\n");
#endif
  g_return_if_fail((previous_page != NULL && IS_PANED_NOTEBOOK(previous_page))
		   || (new_page != NULL && IS_PANED_NOTEBOOK(new_page)));

  if (previous_page != NULL)
    {
      application->threadlist_height
	= gtk_paned_get_position(GTK_PANED(previous_page));
    }

  if (new_page != NULL)
    {
      OchushaBulletinBoard *board
	= (OchushaBulletinBoard *)paned_notebook_get_item(paned_notebook, new_page);
      BulletinBoardGUIInfo *info;

      g_return_if_fail(board != NULL);
      info = (BulletinBoardGUIInfo *)board->user_data;
      gtk_paned_set_position(GTK_PANED(new_page),
			     application->threadlist_height);
      gtk_toggle_button_set_active(application->filter_button,
				   info->enable_filter);
    }
#if DEBUG_GUI_MOST
  fprintf(stderr, "boardlist_ui.c: page_switched_cb() done.\n");
#endif
}


static void
boardlist_read_category_element_cb(OchushaBBSTable *table,
				   OchushaBoardCategory *category,
				   GHashTable *category_attributes,
				   OchushaApplication *application)
{
  BoardCategoryInfo *info = category->user_data;
#if DEBUG_GUI
  fprintf(stderr, "boardlist_read_category_element_cb.\n");
#endif

  if (info == NULL)
    {
      info = g_new0(BoardCategoryInfo, 1);
      category->user_data = info;
    }

  info->expanded = ochusha_utils_get_attribute_boolean(category_attributes, "expanded");
  return;
}


static void
boardlist_read_board_element_cb(OchushaBBSTable *table,
				OchushaBulletinBoard *board,
				GHashTable *board_attributes,
				OchushaApplication *application)
{
  BulletinBoardGUIInfo *info = ensure_bulletin_board_info(board);
  info->last_modified = ochusha_utils_get_attribute_string(board_attributes,
							   "last_modified");
  info->enable_filter = ochusha_utils_get_attribute_boolean(board_attributes,
							    "enable_filter");
  info->filter.rule = ochusha_utils_get_attribute_string(board_attributes,
							 "filter.rule");
  info->filter.ignore_threshold
    = ochusha_utils_get_attribute_int(board_attributes,
				      "filter.ignore_threshold");
  return;
}


#define OUTPUT_CATEGORY_ATTRIBUTE_BOOLEAN(file, category, attribute)	\
  fprintf(file,								\
	  "      <attribute name=\"" #attribute	"\">\n"			\
	  "        <boolean val=\"%s\"/>\n"				\
	  "      </attribute>\n",					\
	  (category)->attribute ? "true" : "false")


static void
boardlist_write_category_element_cb(OchushaBBSTable *table,
				    OchushaBoardCategory *category,
				    FILE *boardlist_xml,
				    OchushaApplication *application)
{
  BoardCategoryInfo *info = (BoardCategoryInfo *)category->user_data;
#if DEBUG_GUI
  fprintf(stderr, "boardlist_write_category_element_cb.\n");
#endif

  OUTPUT_CATEGORY_ATTRIBUTE_BOOLEAN(boardlist_xml, info, expanded);
}


#define OUTPUT_BOARD_ATTRIBUTE_BOOLEAN(file, board, attribute)		\
  fprintf(file,								\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <boolean val=\"%s\"/>\n"				\
	  "        </attribute>\n",					\
	  (board)->attribute ? "true" : "false")

#define OUTPUT_BOARD_ATTRIBUTE_INT(file, board, attribute)		\
  fprintf(file,								\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <int val=\"%d\"/>\n"				\
	  "        </attribute>\n", (board)->attribute)

#define OUTPUT_BOARD_ATTRIBUTE_STRING(file, board, attribute)		\
  fprintf(file,								\
	  "        <attribute name=\"" #attribute	"\">\n"		\
	  "          <string>%s</string>\n"				\
	  "        </attribute>\n", (board)->attribute)


static void
boardlist_write_board_element_cb(OchushaBBSTable *table,
				 OchushaBulletinBoard *board,
				 FILE *boardlist_xml,
				 OchushaApplication *application)
{
  BulletinBoardGUIInfo *info = (BulletinBoardGUIInfo *)board->user_data;
  OUTPUT_BOARD_ATTRIBUTE_BOOLEAN(boardlist_xml, info, enable_filter);

  if (info->last_modified != NULL)
    OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, info, last_modified);

  if (info->filter.rule != NULL)
    {
      OUTPUT_BOARD_ATTRIBUTE_STRING(boardlist_xml, info, filter.rule);
      OUTPUT_BOARD_ATTRIBUTE_INT(boardlist_xml, info, filter.ignore_threshold);
    }

  return;
}


static void
write_threadlist(gpointer data, gpointer user_data)
{
  OchushaBulletinBoard *board = (OchushaBulletinBoard *)data;
  OchushaApplication *application = (OchushaApplication *)user_data;

  if (!ochusha_bulletin_board_write_threadlist_xml(board, &application->conf))
    fprintf(stderr, "Couldn't write threadlist2.xml.\n");
}


BoardCategoryInfo *
ensure_board_category_info(OchushaBoardCategory *category)
{
  BoardCategoryInfo *info = category->user_data;

  if (info == NULL)
    {
      info = g_new0(BoardCategoryInfo, 1);
      category->user_data = info;
    }

  return info;
}


static gboolean
prepare_category(OchushaBoardCategory *category, gpointer user_data)
{
  ensure_board_category_info(category);
  return TRUE;
}


static gboolean
prepare_board(OchushaBulletinBoard *board, gpointer user_data)
{
  ensure_bulletin_board_info(board);
  return TRUE;
}


static void
build_boardlist(WorkerThread *employee, gpointer args)
{
  int status;
  OchushaAsyncBuffer *buffer;

  OchushaApplication *application = (OchushaApplication *)args;
  OchushaBBSTable *table = application->table;

#if DEBUG_THREAD_MOST
  fprintf(stderr, "build_boardlist thread is invoked.\n");
  fflush(stderr);
#endif

  if ((status = pthread_mutex_trylock(&board_list_lock)) != 0)
    {
      if (status == EBUSY)
	return;		/* İι1åɤ */

      fprintf(stderr, "Internal error: invalid mutex.\n");
      abort();
    }

  buffer = ochusha_utils_2ch_get_bbsmenu(&application->conf, NULL, NULL);

  if (buffer == NULL)
    {
      fprintf(stderr, "Couldn't get %s\n", application->conf.bbsmenu_url);
      goto cleanup_before_return;
      return;
    }

  if (ochusha_utils_2ch_analyze_bbsmenu(table, buffer,
					prepare_category,
					prepare_board,
					NULL,	/* TODO: İžб */
					application))
    {
      gdk_threads_enter();
      setup_boardlist(application);
      gdk_threads_leave();

      ochusha_bbs_table_write_boardlist_xml(table, &application->conf);
    }

 cleanup_before_return:
  if (buffer != NULL)
    g_object_unref(G_OBJECT(buffer));

  if (pthread_mutex_unlock(&board_list_lock) != 0)
    {
      fprintf(stderr, "Couldn't unlock a mutex.\n");
      abort();
    }
}


static void
setup_boardlist(OchushaApplication *application)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "setup_boardlist\n");
#endif
  boardlist_view_open(BOARDLIST_VIEW(application->boardlist_view),
		      application->table->category_list, NULL,
		      (BulletinBoardInfoNewFunc *)ensure_bulletin_board_info);
  gtk_widget_show_all(application->boardlist_view);
  gtk_widget_show_all(application->contents_window);
}


#if 0
static void
make_new_thread_button_cb(GtkWidget *widget, OchushaApplication *application)
{
  make_new_thread(application);
}
#endif


static void
open_board_properties_button_cb(GtkWidget *widget,
				OchushaApplication *application)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "open_board_properties_button_cb: widget=0x%x\n", (int)widget);
#endif
  open_current_board_properties(application);
}


static void
refresh_board_button_cb(GtkWidget *widget, OchushaApplication *application)
{
#if DEBUG_GUI_MOST
  fprintf(stderr, "refresh_board_button_cb: widget=0x%x\n", (int)widget);
#endif
  refresh_current_threadlist(application);
}


static void
start_threadlist_search_button_cb(GtkWidget *widget,
				  OchushaApplication *application)
{
  start_current_threadlist_search(application);
}

