/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: download_ui.c,v 1.11.2.6 2004/11/09 23:03:23 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_async_buffer.h"
#include "ochusha_network_broker.h"
#include "utils.h"

#include "ochusha_ui.h"
#include "download_ui.h"

#include <glib.h>
#include <gtk/gtk.h>

#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>


static GQuark dialog_id;
static GQuark progress_id;
static GQuark filename_entry_id;
static GQuark buffer_id;
static GQuark download_url_id;

static pthread_mutex_t download_ui_lock;


#define DOWNLOAD_UI_LOCK				\
  if (pthread_mutex_lock(&download_ui_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define DOWNLOAD_UI_UNLOCK				\
  if (pthread_mutex_unlock(&download_ui_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


void
initialize_download_ui(OchushaApplication *application)
{
  dialog_id = g_quark_from_static_string("DownloadUI::Dialog");
  progress_id = g_quark_from_static_string("DownloadUI::Progress");
  filename_entry_id = g_quark_from_static_string("DownloadUI::FilenameEntry");
  buffer_id = g_quark_from_static_string("DownloadUI::Buffer");
  download_url_id = g_quark_from_static_string("DownloadUI::URL");
}


static void
access_progressed_cb(OchushaAsyncBuffer *buffer,
		     int bytes_read, int bytes_total,
		     GtkWidget *download_dialog)
{
  GtkProgressBar *pbar;
  if (!GTK_IS_WIDGET(download_dialog))
    return;

  gdk_threads_enter();

  pbar = (GtkProgressBar *)g_object_get_qdata(G_OBJECT(download_dialog),
					      progress_id);
  if (pbar != NULL)
    {
      if (bytes_total <= 0)
	gtk_progress_bar_set_text(pbar, _("Unknown Size"));
      else
	{
	  char message[256];
	  double frac = (double)bytes_read / (double)bytes_total;
	  snprintf(message, 256, "%d%% (%d/%d)",
		   (int)(frac * 100.0), bytes_read, bytes_total);
	  gtk_progress_bar_set_text(pbar, message);
	  gtk_progress_bar_set_fraction(pbar, frac);
	}
    }

  gdk_threads_leave();

  return;
}


static void
access_finished_cb(OchushaAsyncBuffer *buffer, GtkDialog *download_dialog)
{
  char message[256];
  GtkProgressBar *pbar;
  g_return_if_fail(GTK_IS_DIALOG(download_dialog));

  gdk_threads_enter();

  pbar = (GtkProgressBar *)g_object_get_qdata(G_OBJECT(download_dialog),
					      progress_id);
  snprintf(message, 256, "100%% (%ld/%ld)",
	   (long)buffer->length, (long)buffer->length);
  gtk_progress_bar_set_text(pbar, message);
  gtk_progress_bar_set_fraction(pbar, 1.0);
  gtk_dialog_set_response_sensitive(download_dialog,
				    GTK_RESPONSE_OK, TRUE);

  gdk_threads_leave();

  return;
}


static void
access_failed_cb(OchushaAsyncBuffer *buffer,
		 OchushaNetworkBrokerFailureReason reason_code,
		 const gchar *reason, GtkDialog *download_dialog)
{
  GtkProgressBar *pbar;

  if (!GTK_IS_DIALOG(download_dialog))
    return;

  gdk_threads_enter();

  pbar = (GtkProgressBar *)g_object_get_qdata(G_OBJECT(download_dialog),
					      progress_id);
  gtk_progress_bar_set_text(pbar, _("Download Failed"));
  gtk_dialog_set_response_sensitive(download_dialog,
				    GTK_RESPONSE_OK, FALSE);

  gdk_threads_leave();

  return;
}


static void
download_dialog_response_cb(GtkWidget *widget, int response_id,
			    OchushaApplication *application)
{
  OchushaAsyncBuffer *buffer
    = (OchushaAsyncBuffer *)g_object_get_qdata(G_OBJECT(widget), buffer_id);
  g_return_if_fail(GTK_IS_DIALOG(widget));

  if (!OCHUSHA_IS_ASYNC_BUFFER(buffer))
    goto done;

  if (response_id == GTK_RESPONSE_OK)
    {
      GtkEntry *entry
	= (GtkEntry *)g_object_get_qdata(G_OBJECT(widget), filename_entry_id);
      const char *filename = gtk_entry_get_text(entry);

      int fd = open(filename, O_CREAT | O_TRUNC | O_WRONLY, 0x1a4);
      if (fd > 0)
	{
	  const char *url
	    = g_object_get_qdata(G_OBJECT(widget), download_url_id);
	  ssize_t result = write(fd, (const void *)buffer->buffer,
				 buffer->length);
	  close(fd);
	  if (result == buffer->length)
	    ochusha_config_cache_unlink_file(&application->config, url);
	}
    }
  else
    {
#if 0
      fprintf(stderr, "Pre termination.\n");
      fprintf(stderr, "Pre gdk_threads_leave().\n");
#endif
      gdk_threads_leave();
      ochusha_async_buffer_suspend(buffer);
      ochusha_async_buffer_terminate(buffer);
      /* ochusha_async_buffer_resume(buffer); */
#if 0
      fprintf(stderr, "Post termination.\n");
#endif
      gdk_threads_enter();
#if 0
      fprintf(stderr, "Post gdk_threads_enter().\n");
#endif
    }

 done:
  OCHU_OBJECT_UNREF(buffer);
  gtk_widget_hide(widget);
  gtk_widget_unrealize(widget);
  gtk_widget_destroy(widget);
}


static void
filesel_dialog_response_cb(GtkWidget *widget, int response_id,
			   OchushaApplication *application)
{
  const char *filename = NULL;
  GtkWidget *download_dialog
    = (GtkWidget *)g_object_get_qdata(G_OBJECT(widget), dialog_id);

  if (GTK_IS_DIALOG(download_dialog))
    {
      if (response_id == GTK_RESPONSE_OK)
	{
	  GtkEntry *entry;
	  filename
	    = gtk_file_selection_get_filename(GTK_FILE_SELECTION(widget));
	  entry = (GtkEntry *)g_object_get_qdata(G_OBJECT(download_dialog),
						 filename_entry_id);
	  gtk_entry_set_text(entry, filename);
	  if (application->last_download_directory != NULL)
	    G_FREE(application->last_download_directory);
	  application->last_download_directory
	    = g_path_get_dirname(filename);

	  gtk_widget_show(download_dialog);
	}
    }
  g_return_if_fail(GTK_IS_DIALOG(download_dialog));

  if (response_id != GTK_RESPONSE_OK)
    {
      gtk_widget_hide(download_dialog);
      gtk_widget_unrealize(download_dialog);
      gtk_widget_destroy(download_dialog);
    }

  gtk_widget_hide(widget);
  gtk_widget_unrealize(widget);
  gtk_widget_destroy(widget);
}


static void
open_download_dialog(OchushaApplication *application, const char *url,
		     OchushaAsyncBuffer *buffer)
{
  char *abspath;
  char *filename;
  char pathname[PATH_MAX];
  GtkWidget *table;
  GtkWidget *tmp_widget;
  GtkWidget *filesel_dialog;
  GtkWidget *download_dialog
    = gtk_dialog_new_with_buttons(_("Save Link As"),
				  NULL,
				  GTK_DIALOG_DESTROY_WITH_PARENT,
				  GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				  GTK_STOCK_OK, GTK_RESPONSE_OK,
				  NULL);
  g_object_set_qdata(G_OBJECT(download_dialog), buffer_id, buffer);
  g_object_set_qdata_full(G_OBJECT(download_dialog), download_url_id,
			  g_strdup(url), (GDestroyNotify)g_free);
  g_signal_connect(G_OBJECT(download_dialog), "response",
		   G_CALLBACK(download_dialog_response_cb), application);
  gtk_dialog_set_default_response(GTK_DIALOG(download_dialog),
				  GTK_RESPONSE_OK);


  table = gtk_table_new(2, 2, FALSE);
  gtk_widget_show(table);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(download_dialog)->vbox), table,
		     TRUE, TRUE, 0);

  tmp_widget = gtk_label_new(_("Saving From:"));
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 0, 1, 0, 1);

  tmp_widget = gtk_entry_new();
  gtk_entry_set_width_chars(GTK_ENTRY(tmp_widget), 50);
  gtk_entry_set_text(GTK_ENTRY(tmp_widget), url);
  gtk_editable_set_editable(GTK_EDITABLE(tmp_widget), FALSE);
  GTK_WIDGET_UNSET_FLAGS(tmp_widget, GTK_CAN_FOCUS);
  gtk_widget_show(tmp_widget);
  gtk_table_attach(GTK_TABLE(table), tmp_widget, 1, 2, 0, 1,
		   GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

  tmp_widget = gtk_label_new(_("To:"));
  gtk_widget_show(tmp_widget);
  gtk_table_attach_defaults(GTK_TABLE(table), tmp_widget, 0, 1, 1, 2);

  tmp_widget = gtk_entry_new();
  gtk_entry_set_width_chars(GTK_ENTRY(tmp_widget), 50);
  g_object_set_qdata(G_OBJECT(download_dialog), filename_entry_id, tmp_widget);
  gtk_editable_set_editable(GTK_EDITABLE(tmp_widget), TRUE);
  gtk_widget_show(tmp_widget);
  gtk_table_attach(GTK_TABLE(table), tmp_widget, 1, 2, 1, 2,
		   GTK_EXPAND | GTK_FILL, GTK_EXPAND | GTK_FILL, 0, 0);

  tmp_widget = gtk_progress_bar_new();
  g_object_set_qdata(G_OBJECT(download_dialog), progress_id, tmp_widget);
  gtk_widget_show(tmp_widget);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(download_dialog)->vbox), tmp_widget,
		     FALSE, FALSE, 0);

  ochusha_async_buffer_lock(buffer);
  if (!buffer->fixed)
    {
      gtk_dialog_set_response_sensitive(GTK_DIALOG(download_dialog),
					GTK_RESPONSE_OK, FALSE);
      g_signal_connect_object(G_OBJECT(buffer), "access_progressed",
			      G_CALLBACK(access_progressed_cb),
			      download_dialog, 0);
      g_signal_connect_object(G_OBJECT(buffer), "access_finished",
			      G_CALLBACK(access_finished_cb),
			      download_dialog, 0);
      g_signal_connect_object(G_OBJECT(buffer), "access_failed",
			      G_CALLBACK(access_failed_cb),
			      download_dialog, 0);
      gtk_progress_bar_set_text(GTK_PROGRESS_BAR(tmp_widget),
				_("Download Not Started"));
      gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(tmp_widget), 0.0);
    }
  else
    {
      if (buffer->length > 0)
	{
	  char message[256];
	  snprintf(message, 256, "100%% (%ld/%ld)",
		   (long)buffer->length, (long)buffer->length);
	  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(tmp_widget), message);
	  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(tmp_widget), 1.0);
	}
      else
	{
	  gtk_progress_bar_set_text(GTK_PROGRESS_BAR(tmp_widget),
				    _("Download Failed"));
	  gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(tmp_widget), 0.0);
	  gtk_dialog_set_response_sensitive(GTK_DIALOG(download_dialog),
					    GTK_RESPONSE_OK, FALSE);
	}
    }
  ochusha_async_buffer_unlock(buffer);

  filename = NULL;
  abspath = ochusha_utils_url_extract_http_absolute_path(url);
  if (abspath != NULL)
    {
      size_t len = strlen(abspath);
      if (abspath[len - 1] != '/')
	filename = g_path_get_basename(abspath);
      G_FREE(abspath);
    }

#if 0
  fprintf(stderr, "url: %s, filename: %s\n", url, filename);
#endif
  if (filename != NULL)
    {
      snprintf(pathname, PATH_MAX, "%s/%s",
	       application->last_download_directory != NULL
	       ? application->last_download_directory : g_get_home_dir(),
	       filename);
      g_free(filename);
    }
  else
    {
      snprintf(pathname, PATH_MAX, "%s/anonymous",
	       application->last_download_directory != NULL
	       ? application->last_download_directory : g_get_home_dir());
    }

  filesel_dialog = gtk_file_selection_new(_("Save Link As"));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel_dialog),
				  pathname);
  g_object_set_qdata(G_OBJECT(filesel_dialog), dialog_id, download_dialog);
  g_signal_connect(G_OBJECT(filesel_dialog), "response",
		   G_CALLBACK(filesel_dialog_response_cb), application);
  gtk_dialog_set_default_response(GTK_DIALOG(filesel_dialog),
				  GTK_RESPONSE_OK);
  gtk_widget_show(filesel_dialog);
}


void
ochusha_download_url(OchushaApplication *application, const char *url)
{
  char *scheme;
  g_return_if_fail(url != NULL);

  if (*url == '\0')
    return;

  scheme = ochusha_utils_url_extract_scheme(url);
  if (scheme == NULL)
    return;
  if (strcmp(scheme, "http") != 0)
    {
      G_FREE(scheme);
      return;
    }
  G_FREE(scheme);

  DOWNLOAD_UI_LOCK
  {
    OchushaAsyncBuffer *buffer
      = ochusha_network_broker_read_from_url(application->broker,
					NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_AS_IS,
					FALSE,
					application->download_chunksize);
    if (buffer != NULL)
      open_download_dialog(application, url, buffer);
  }
  DOWNLOAD_UI_UNLOCK;
}
