/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: utils.c,v 1.30.2.4 2004/05/30 06:34:58 fuyu Exp $
 */

#define OCHUSHA_UTILS_IMPLEMENTATION	1

#include "config.h"

#include "ochusha_private.h"
#include "utils.h"

#include <errno.h>
#include <iconv.h>
#if HAVE_LANGINFO_CODESET
#include <langinfo.h>
#endif

#if HAVE_LOCALE_H
#include <locale.h>
#endif

#include <fcntl.h>
#include <limits.h>

#if !defined(HAVE_TIMEGM) || TRACE_MEMORY_USAGE || TRACE_REFCOUNT
#include <pthread.h>
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>


iconv_t utf8_to_native = 0;	/* ǥХå */

#define DEFAULT_BUFFER_SIZE	4096
#define MAX_HOSTNAME_LEN	256



/*
 * initialize_common_convertes
 *
 * ɤȤencodingconverter롣
 *
 * MEMO: converterϥåɥդʤȤա
 */
void
initialize_common_converters(void)
{
  if (utf8_to_native == 0)
    {
#if HAVE_LANGINFO_CODESET
      char codeset[256];
      int len = snprintf(codeset, 256, "%s//IGNORE", nl_langinfo(CODESET));
      if (len < 256)
	{
	  utf8_to_native = iconv_open(codeset, "UTF-8");
	  if (utf8_to_native == (iconv_t)-1)
	    {
	      fprintf(stderr, "Couldn't iconv_open(\"%s\", \"UTF-8\")\n",
		      codeset);
	      utf8_to_native = 0;	/* try hardcoded one */
	    }
	}

      if (utf8_to_native == 0)
	utf8_to_native = iconv_open("EUC-JP//IGNORE", "UTF-8");
#else
      utf8_to_native = iconv_open("EUC-JP//IGNORE", "UTF-8");
#endif
      if (utf8_to_native == (iconv_t)-1)
	{
	  fprintf(stderr, "iconv_open() failed.\n");
	  exit(1);
	}
    }
}


/*
 * convert_string
 *
 * iconvȤäsrcʸencodingѴѴ줿ʸ֤
 * Ѵ줿ʸheapmalloc줿ΰ˳ǼΤǡ桼
 * פˤʤ꼡freeǤ롣
 * Ϳ줿length-1ξ硢srcNULüCʸȤƼ갷졢
 * strlen(src)η̤ʸĹȤƼ갷롣ʳξ硢
 * Ϳ줿lengthĹʬѴ롣
 * NULLhelperͿ줿硢ǤiconvѴ˼Ԥ֤
 * helperѴ³ꤹ롣
 */
char *
convert_string(iconv_t converter, iconv_helper *helper,
	       const char *src, int length)
{
  char default_buffer[DEFAULT_BUFFER_SIZE];
  size_t buffer_size = DEFAULT_BUFFER_SIZE;
  char *buffer = default_buffer;
  const char *inbuf;
  char *outbuf;
  size_t outbytesleft;
  size_t inbytesleft;

  if (length < 0)
    length = strlen(src);

  while (1)
    {
      size_t size;

      inbuf = src;
      outbuf = buffer;
      outbytesleft = buffer_size;
      inbytesleft = length;

#if ENABLE_MS_IZONMOJI
      do
	{
	  size = iconv(converter,
		       &inbuf, &inbytesleft, &outbuf, &outbytesleft);
	  if (size == -1 && errno == EILSEQ && helper != NULL)
	    size = (*helper)(&inbuf, &inbytesleft, &outbuf, &outbytesleft);
	  else
	    break;
	} while (size != -1);

      if (size != -1)
	break;	/* ｪλ */

      if (errno == E2BIG)
	{
	  /* allocate 2 times larger buffer in the heap. */
	  buffer_size *= 2;
	  if (buffer != default_buffer)
	    buffer = (char *)G_REALLOC(buffer, buffer_size);
	  else
	    buffer = (char *)G_MALLOC(buffer_size);
	}
      else
	{
#if DEBUG_ICONV
	  fprintf(stderr, "iconv failed(errno = %d).\n", errno);
	  fprintf(stderr, "character code = 0x%02x\n", (int)*inbuf);
#endif
	  if (buffer != default_buffer)
	    G_FREE(buffer);
	  return NULL;
	}
#else
      size = iconv(converter, &inbuf, &inbytesleft, &outbuf, &outbytesleft);
      if (size == -1)
	{
	  if (errno == E2BIG)
	    {
	      /* allocate 2 times larger buffer in the heap. */
	      buffer_size *= 2;
	      if (buffer != default_buffer)
		buffer = (char *)G_REALLOC(buffer, buffer_size);
	      else
		buffer = (char *)G_MALLOC(buffer_size);
	    }
	  else
	    {
#if DEBUG_ICONV
	      fprintf(stderr, "iconv failed(errno = %d).\n", errno);
	      fprintf(stderr, "character code = 0x%02x\n", (int)*inbuf);
#endif
	      if (buffer != default_buffer)
		G_FREE(buffer);
	      return NULL;
	    }
	}
      else
	break;	/* ｪλ */
#endif
    }

  if (outbytesleft == 0)
    {
      if (buffer != default_buffer)
	buffer = (char *)G_REALLOC(buffer, buffer_size + 1);
      else
	{
	  buffer = (char *)G_MALLOC(buffer_size + 1);
	  memcpy(buffer, src, buffer_size);
	}
      buffer[buffer_size] = '\0';
      return buffer;
    }

  *outbuf = '\0';

  if (buffer == default_buffer)
    return G_STRDUP(buffer);

  return (char *)G_REALLOC(buffer, outbuf - buffer + 1);
}


/*
 * wipe_string:
 *   string˴ޤޤnon-printableʸȥĤζʸ
 *   ԡ֤
 */
gchar *
wipe_string(const gchar *string)
{
  gchar *result_string = g_strchomp(G_STRDUP(string));
  gchar *head = result_string;

  while (*head != '\0')
    {
      if (*head > '\0' && *head <= '\37')
	{	/* ȯˤĤʬФ */
	  int len;
	  gchar *tmp_pos;
	  for (tmp_pos = g_utf8_next_char(head);
	       *tmp_pos > '\0' && *tmp_pos <= '\37';
	       tmp_pos = g_utf8_next_char(tmp_pos));
	  len = strlen(tmp_pos);
	  memmove(head, tmp_pos, len + 1);
	}
      else
	head = g_utf8_next_char(head);
    }
  return result_string;
}


/*
 * ochusha_utils_shell_escape_text
 *
 * Ϳ줿ʸ򥷥Ϥޥɥ饤Ȥƥפɬ
 * ʸ򥨥פʸΥԡ֤
 * Ǥϡ#\#֤
 */
char *
ochusha_utils_shell_escape_text(const char *text)
{
  int count = 0;
  const char *cur_pos = text;
  char *result;
  char *tmp_pos;

  while (*cur_pos != '\0')
    {
      if (*cur_pos == '#')
	count++;
      cur_pos++;
    }

  if (count == 0)
    return G_STRDUP(text);

  result = G_MALLOC(strlen(text) + count + 1);
  cur_pos = text;
  tmp_pos = result;
  do
    {
      *tmp_pos++ = *cur_pos;
      if (*cur_pos == '#')
	{
	  tmp_pos[-1] = '\\';
	  *tmp_pos++ = '#';
	}
    } while (*cur_pos++ != '\0');

  return result;
}


/*
 * ochusha_utils_url_extract_scheme
 *
 * Ϳ줿ʸURLȤƲᤷschemeʬ֤
 * ˼ԤˤNULL֤
 */
char *
ochusha_utils_url_extract_scheme(const char *url)
{
  char *comma_pos;
  g_return_val_if_fail(url != NULL, NULL);

  comma_pos = strchr(url, ':');
  if (comma_pos == NULL)
    return NULL;

  /*  */
  return G_STRNDUP(url, comma_pos - url);
}


/*
 * ochusha_utils_url_extract_http_server
 *
 * Ϳ줿ʸhttpФURLǤȤƥ̾ʬ֤
 * ˼ԤˤNULL֤
 */
char *
ochusha_utils_url_extract_http_server(const char *url)
{
  /*
   * ݤʤΤǡurlhttp://server/.*ʷǤȷǤġ
   */
  char *cur_pos;
  char *hostpart_tail;
  int length;
  char buffer[MAX_HOSTNAME_LEN];

  cur_pos = strstr(url, "http://");
  if (cur_pos == NULL)
    {
      cur_pos = strstr(url, "https://");
      if (cur_pos == NULL)
	{
#if DEBUG_UTILS
	  fprintf(stderr, "Not a http URL: %s\n", url);
#endif
	  return NULL;
	}
      cur_pos += 8;	/* skip "https://" */
    }
  else
    cur_pos += 7;	/* skip "http://" */

  hostpart_tail = strchr(cur_pos, '/');
  if (hostpart_tail == NULL)
    {
      if (*cur_pos != '\n')
	return G_STRDUP(cur_pos);
      else
	return NULL;
    }

  length = hostpart_tail - cur_pos;
  if (length < 0 || length >= MAX_HOSTNAME_LEN)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't parse URL: %s\n", url);
#endif
      return NULL;
    }

  memcpy(buffer, cur_pos, length);
  buffer[length] = '\0';
  return G_STRDUP(buffer);
}


/*
 * ochusha_utils_url_extract_http_absolute_path
 *
 * Ϳ줿ʸhttpФURLǤȤƲϤ
 * ɥȤΥФˤХѥ֤̾
 * ex) http://pc.2ch.net/unix/  /unix/
 * ˼ԤˤNULL֤
 */
char *
ochusha_utils_url_extract_http_absolute_path(const char *url)
{
  /*
   * ݤʤΤǡurlhttp://server/.*ʷǤȷǤġ
   */
  char *cur_pos = strstr(url, "http://");

  if (cur_pos == NULL)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't parse URL: %s\n", url);
#endif
      return NULL;
    }
  cur_pos += 7; /* skip "http://" */
  cur_pos = strchr(cur_pos, '/');
  if (cur_pos == NULL)
    {
#if DEBUG_UTILS
      fprintf(stderr, "Couldn't get path parts: %s\n", url);
#endif
      return NULL;
    }

  return G_STRDUP(cur_pos);
}


char *
ochusha_utils_url_encode_string(const char *src)
{
  int src_len;
  int buffer_len;
  const char *src_pos;
  char *buf_pos;
  char *buf_tail;
  char *buffer;
  unsigned char c;
  gboolean expand_buf;

  g_return_val_if_fail(src != NULL, NULL);

  src_len = strlen(src);
  buffer_len = src_len * 3 + 1;
  buffer = (char *)G_MALLOC(buffer_len);
  buf_tail = buffer + buffer_len - 1;
  src_pos = src;
  buf_pos = buffer;
  expand_buf = FALSE;

#if DEBUG_UTILS
  fprintf(stderr, "url_encode: src=\"%s\"\n", src);
#endif

  while ((c = *src_pos) != '\0')
    {
      if (expand_buf)
	{
	  int result_len = buf_pos - buffer;
#if DEBUG_UTILS
	  fprintf(stderr, "expand_buf@url_encode: buffer_len %d => %d\n",
		  buffer_len, buffer_len * 2);
#endif
	  buffer_len *= 2;
	  buffer = (char *)G_REALLOC(buffer, buffer_len);
	  buf_tail = buffer + buffer_len - 1;
	  buf_pos = buffer + result_len;
	  expand_buf = FALSE;
	}

      if (g_ascii_isalnum(c)
	  || c == '*' || c == '-' || c == '.' || c == '@' || c == '_')
	{
	  if (buf_pos >= buf_tail)
	    {
	      expand_buf = TRUE;
	      continue;	/* bufferĥƤľ: ꤽˤʤʡ */
	    }
	  *buf_pos++ = c;
	}
      else if (c == ' ')
	{
	  if (buf_pos >= buf_tail)
	    {
	      expand_buf = TRUE;
	      continue;	/* bufferĥƤľ: ꤽˤʤʡ */
	    }
	  *buf_pos++ = '+';
	}
      else
	{
	  int left = buf_tail - buf_pos;
	  if (left < 4)
	    {
	      expand_buf = TRUE;
	      continue;	/* bufferĥƤľ: ꤽˤʤʡ */
	    }
	  buf_pos += g_snprintf(buf_pos, left, "%%%02X", c);
	}
      src_pos++;
    }

  g_assert(buf_pos <= buf_tail);
  *buf_pos = '\0';

#if DEBUG_UTILS
  fprintf(stderr, "url_encode: \"%s\"\n", buffer);
#endif

  return buffer;
}


char *
ochusha_utils_url_decode_string(const char *src)
{
  int src_len = strlen(src);
  char *buffer = (char *)G_MALLOC(src_len + 1);
  const char *cur_src = src;
  char *cur_dst = buffer;

  while (*cur_src != '\0')
    {
      if (*cur_src != '%')
	{
	  if (*cur_src == '+')
	    *cur_dst++ = ' ';
	  else
	    *cur_dst++ = *cur_src;
	  cur_src++;
	}
      else
	{
	  unsigned int code;
	  int result = sscanf(cur_src, "%%%02x", &code);
	  if (result == 1)
	    {
	      *cur_dst++ = (char)code;
	      cur_src += 3;
	    }
	  else
	    *cur_dst++ = *cur_src++;
	}
    }
  *cur_dst = '\0';
  return buffer;
}


/*
 * ochusha_utils_get_utc_time
 *
 * Ϳ줿ʸɽʸȤƲᤷUTC֤
 */
time_t
ochusha_utils_get_utc_time(const char *date_string)
{
  static char *lc_time = NULL;
  struct tm tm;
  char *parse_result;

  if (lc_time == NULL)
    {
#if HAVE_LOCALE_H
      lc_time = setlocale(LC_TIME, "C");
      tzset();
      g_return_val_if_fail(lc_time != NULL, -1);
#else
      return -1;
#endif
    }

  g_return_val_if_fail(date_string != NULL && *date_string != '\0', -1);

  memset(&tm, 0, sizeof(struct tm));
#if 0
  /*
   * GMTΤ褦timezoneλꤵ줿date_string顢
   * struct tmǤĤäΤ
   * LinuxǤtimezone̵꤬뤵Ƥ͡
   * SolarisǤ%Z"GMT"ȥ顼ˤʤäƤޤ
   */
  parse_result = strptime(date_string, "%a, %d %b %Y %T %Z", &tm);
  g_return_val_if_fail(parse_result != NULL, -1);

  return mktime(&tm);
#else
  /*
   * ȡdate_string"GMT"ʳ褿˻郎
   * 2chΥФ˴ؤ¤Linux/SolarisǤԤ褦
   * 郎֤äƤΤǡ괺Ƥ
   */
  parse_result = strptime(date_string, "%a, %d %b %Y %T", &tm);
  g_return_val_if_fail(parse_result != NULL, -1);

# ifdef HAVE_TIMEGM
  return timegm(&tm);
# else /* Ķad-hoc塼 */
  return mktime(&tm) + (strstr(date_string, "GMT") != NULL ? 32400 : 0);
# endif
#endif
}


/*
 * Ĺǽstrpbrkؿ
 * stringĹlenʸcharset˴ޤޤǽʸؤΥݥ󥿤֤
 * ʤNULL֤
 */
const char *
mempbrk(const char *string, const char *charset, size_t len)
{
  while (len > 0)
    {
      const char *c = charset;
      char s = *string;
      while (*c)
	{
	  if (*c == s)
	    return string;
	  c++;
	}
      string++;
      len--;
    }
  return NULL;
}


const char *
reverse_strpbrk(const char *string, const char *charset)
{
  const gchar *s = string;
  while (*s != '\0')
    {
      if (strchr(charset, *s) == NULL)
	return s;
      s++;
    }

  return NULL;
}


int
parse_int(const gchar *start, size_t len)
{
  int result = 0;
  const gchar *tmp_pos = start;
  const gchar *tail_pos = (len > 0 ? start + len : NULL);
  gunichar c;

  do
    {
      c = g_utf8_get_char(tmp_pos);
      if (c >= '0' && c <= '9')
	result = result * 10 + c - '0';
      else if (c >= 65296 && c <= 65305)
	result = result * 10 + c - 65296;
      else
	break;
      tmp_pos = g_utf8_find_next_char(tmp_pos, tail_pos);
    } while (tmp_pos != NULL);

  return result;
}


char *
ochusha_utils_find_cgi_query_key(const char *src, CGIQueryKeyTuple *result)
{
  const char *name_pos = src;
  char *tmp_pos;

  if (src == NULL)
    return NULL;

  while ((tmp_pos = strpbrk(name_pos, "&=")) != NULL)
    {
      char *term;
      if (*tmp_pos == '&')
	{
	  name_pos = tmp_pos + 1;
	  continue;
	}

      if (result != NULL)
	{
	  result->name = name_pos;
	  result->name_len = tmp_pos - name_pos;
	}

      term = strchr(tmp_pos + 1, '&');
      if (term == NULL)
	term = tmp_pos + strlen(tmp_pos);
      /* λǡ*value_tail'&''\0' */

      if (result != NULL)
	{
	  result->value = tmp_pos + 1;
	  result->value_len = term - tmp_pos - 1;
	}

      return term;
    }

  return NULL;
}


/*
 * mkdir_p
 *
 * mkdir -pޥɤΤ褦ˡpathǻꤵ줿ǥ쥯ȥŪ
 * 롣
 * ǥ쥯ȥ꤬¸ߤ뤫ǥ쥯ȥκˤ1
 * ʳξ0֤
 */
int
mkdir_p(const char *path)
{
  char buffer[PATH_MAX];
  char *cur_pos;

  if (path == NULL)
    return 0;

  strncpy(buffer, path, PATH_MAX);

  cur_pos = buffer;
  while (1)
    {
      int result;
      struct stat sb;
      char tmp_char;

      cur_pos = strchr(cur_pos, '/');
      if (cur_pos != NULL)
	{
	  tmp_char = *(++cur_pos);
	  *cur_pos = '\0';
	}
      else
	tmp_char = '\0';

      result = stat(buffer, &sb);

      if (result == -1)
	{
	  if (errno != ENOENT)
	    return 0;
	
	  result = mkdir(buffer, 0x1c0);
	  if (result != 0)
	    return 0;
	}
      else
	if (!S_ISDIR(sb.st_mode))
	  return 0;

      if (tmp_char == '\0')
	return 1;

      *cur_pos = tmp_char;
    }
}


/*
 * iconvϢ
 */
/* 0x87, 0x40-0x9c */
static const char *cp932_to_utf8_table_0x87[] = {
  /*====== 0x87, 0x40-0x7e ======*/
  /* 0x87 0x40 */  "\342\221\240",
  /* 0x87 0x41 */  "\342\221\241",
  /* 0x87 0x42 */  "\342\221\242",
  /* 0x87 0x43 */  "\342\221\243",
  /* 0x87 0x44 */  "\342\221\244",
  /* 0x87 0x45 */  "\342\221\245",
  /* 0x87 0x46 */  "\342\221\246",
  /* 0x87 0x47 */  "\342\221\247",
  /* 0x87 0x48 */  "\342\221\250",
  /* 0x87 0x49 */  "\342\221\251",
  /* 0x87 0x4a */  "\342\221\252",
  /* 0x87 0x4b */  "\342\221\253",
  /* 0x87 0x4c */  "\342\221\254",
  /* 0x87 0x4d */  "\342\221\255",
  /* 0x87 0x4e */  "\342\221\256",
  /* 0x87 0x4f */  "\342\221\257",
  /* 0x87 0x50 */  "\342\221\260",
  /* 0x87 0x51 */  "\342\221\261",
  /* 0x87 0x52 */  "\342\221\262",
  /* 0x87 0x53 */  "\342\221\263",
  /* 0x87 0x54 */  "\342\205\240",
  /* 0x87 0x55 */  "\342\205\241",
  /* 0x87 0x56 */  "\342\205\242",
  /* 0x87 0x57 */  "\342\205\243",
  /* 0x87 0x58 */  "\342\205\244",
  /* 0x87 0x59 */  "\342\205\245",
  /* 0x87 0x5a */  "\342\205\246",
  /* 0x87 0x5b */  "\342\205\247",
  /* 0x87 0x5c */  "\342\205\250",
  /* 0x87 0x5d */  "\342\205\251",
  /* 0x87 0x5e (invalid) */  NULL,
  /* 0x87 0x5f */  "\343\215\211",
  /* 0x87 0x60 */  "\343\214\224",
  /* 0x87 0x61 */  "\343\214\242",
  /* 0x87 0x62 */  "\343\215\215",
  /* 0x87 0x63 */  "\343\214\230",
  /* 0x87 0x64 */  "\343\214\247",
  /* 0x87 0x65 */  "\343\214\203",
  /* 0x87 0x66 */  "\343\214\266",
  /* 0x87 0x67 */  "\343\215\221",
  /* 0x87 0x68 */  "\343\215\227",
  /* 0x87 0x69 */  "\343\214\215",
  /* 0x87 0x6a */  "\343\214\246",
  /* 0x87 0x6b */  "\343\214\243",
  /* 0x87 0x6c */  "\343\214\253",
  /* 0x87 0x6d */  "\343\215\212",
  /* 0x87 0x6e */  "\343\214\273",
  /* 0x87 0x6f */  "\343\216\234",
  /* 0x87 0x70 */  "\343\216\235",
  /* 0x87 0x71 */  "\343\216\236",
  /* 0x87 0x72 */  "\343\216\216",
  /* 0x87 0x73 */  "\343\216\217",
  /* 0x87 0x74 */  "\343\217\204",
  /* 0x87 0x75 */  "\343\216\241",
  /* 0x87 0x76 (invalid) */  NULL,
  /* 0x87 0x77 (invalid) */  NULL,
  /* 0x87 0x78 (invalid) */  NULL,
  /* 0x87 0x79 (invalid) */  NULL,
  /* 0x87 0x7a (invalid) */  NULL,
  /* 0x87 0x7b (invalid) */  NULL,
  /* 0x87 0x7c (invalid) */  NULL,
  /* 0x87 0x7d (invalid) */  NULL,
  /* 0x87 0x7e */  "\343\215\273",
  /* 0x87 0x7f (invalid) */  NULL,

  /*====== 0x87, 0x80-0x9e ======*/
  /* 0x87 0x80 */  "\343\200\235",
  /* 0x87 0x81 */  "\343\200\237",
  /* 0x87 0x82 */  "\342\204\226",
  /* 0x87 0x83 */  "\343\217\215",
  /* 0x87 0x84 */  "\342\204\241",
  /* 0x87 0x85 */  "\343\212\244",
  /* 0x87 0x86 */  "\343\212\245",
  /* 0x87 0x87 */  "\343\212\246",
  /* 0x87 0x88 */  "\343\212\247",
  /* 0x87 0x89 */  "\343\212\250",
  /* 0x87 0x8a */  "\343\210\261",
  /* 0x87 0x8b */  "\343\210\262",
  /* 0x87 0x8c */  "\343\210\271",
  /* 0x87 0x8d */  "\343\215\276",
  /* 0x87 0x8e */  "\343\215\275",
  /* 0x87 0x8f */  "\343\215\274",
  /* 0x87 0x90 */  "\342\211\222",
  /* 0x87 0x91 */  "\342\211\241",
  /* 0x87 0x92 */  "\342\210\253",
  /* 0x87 0x93 */  "\342\210\256",
  /* 0x87 0x94 */  "\342\210\221",
  /* 0x87 0x95 */  "\342\210\232",
  /* 0x87 0x96 */  "\342\212\245",
  /* 0x87 0x97 */  "\342\210\240",
  /* 0x87 0x98 */  "\342\210\237",
  /* 0x87 0x99 */  "\342\212\277",
  /* 0x87 0x9a */  "\342\210\265",
  /* 0x87 0x9b */  "\342\210\251",
  /* 0x87 0x9c */  "\342\210\252",
  /* 0x87 0x9d (invalid) */  NULL,
  /* 0x87 0x9e (invalid) */  NULL,
};


/* 0xed, 0x40-0xfc */
static const char *cp932_to_utf8_table_0xed[] = {
  /*====== 0xed, 0x40-0x7e ======*/
  /* 0xed 0x40 */  "\347\272\212",
  /* 0xed 0x41 */  "\350\244\234",
  /* 0xed 0x42 */  "\351\215\210",
  /* 0xed 0x43 */  "\351\212\210",
  /* 0xed 0x44 */  "\350\223\234",
  /* 0xed 0x45 */  "\344\277\211",
  /* 0xed 0x46 */  "\347\202\273",
  /* 0xed 0x47 */  "\346\230\261",
  /* 0xed 0x48 */  "\346\243\210",
  /* 0xed 0x49 */  "\351\213\271",
  /* 0xed 0x4a */  "\346\233\273",
  /* 0xed 0x4b */  "\345\275\205",
  /* 0xed 0x4c */  "\344\270\250",
  /* 0xed 0x4d */  "\344\273\241",
  /* 0xed 0x4e */  "\344\273\274",
  /* 0xed 0x4f */  "\344\274\200",
  /* 0xed 0x50 */  "\344\274\203",
  /* 0xed 0x51 */  "\344\274\271",
  /* 0xed 0x52 */  "\344\275\226",
  /* 0xed 0x53 */  "\344\276\222",
  /* 0xed 0x54 */  "\344\276\212",
  /* 0xed 0x55 */  "\344\276\232",
  /* 0xed 0x56 */  "\344\276\224",
  /* 0xed 0x57 */  "\344\277\215",
  /* 0xed 0x58 */  "\345\201\200",
  /* 0xed 0x59 */  "\345\200\242",
  /* 0xed 0x5a */  "\344\277\277",
  /* 0xed 0x5b */  "\345\200\236",
  /* 0xed 0x5c */  "\345\201\206",
  /* 0xed 0x5d */  "\345\201\260",
  /* 0xed 0x5e */  "\345\201\202",
  /* 0xed 0x5f */  "\345\202\224",
  /* 0xed 0x60 */  "\345\203\264",
  /* 0xed 0x61 */  "\345\203\230",
  /* 0xed 0x62 */  "\345\205\212",
  /* 0xed 0x63 */  "\345\205\244",
  /* 0xed 0x64 */  "\345\206\235",
  /* 0xed 0x65 */  "\345\206\276",
  /* 0xed 0x66 */  "\345\207\254",
  /* 0xed 0x67 */  "\345\210\225",
  /* 0xed 0x68 */  "\345\212\234",
  /* 0xed 0x69 */  "\345\212\246",
  /* 0xed 0x6a */  "\345\213\200",
  /* 0xed 0x6b */  "\345\213\233",
  /* 0xed 0x6c */  "\345\214\200",
  /* 0xed 0x6d */  "\345\214\207",
  /* 0xed 0x6e */  "\345\214\244",
  /* 0xed 0x6f */  "\345\215\262",
  /* 0xed 0x70 */  "\345\216\223",
  /* 0xed 0x71 */  "\345\216\262",
  /* 0xed 0x72 */  "\345\217\235",
  /* 0xed 0x73 */  "\357\250\216",
  /* 0xed 0x74 */  "\345\222\234",
  /* 0xed 0x75 */  "\345\222\212",
  /* 0xed 0x76 */  "\345\222\251",
  /* 0xed 0x77 */  "\345\223\277",
  /* 0xed 0x78 */  "\345\226\206",
  /* 0xed 0x79 */  "\345\235\231",
  /* 0xed 0x7a */  "\345\235\245",
  /* 0xed 0x7b */  "\345\236\254",
  /* 0xed 0x7c */  "\345\237\210",
  /* 0xed 0x7d */  "\345\237\207",
  /* 0xed 0x7e */  "\357\250\217",
  /* 0xed 0x7f (invalid) */  NULL,

  /*====== 0xed, 0x80-0x9e ======*/
  /* 0xed 0x80 */  "\357\250\220",
  /* 0xed 0x81 */  "\345\242\236",
  /* 0xed 0x82 */  "\345\242\262",
  /* 0xed 0x83 */  "\345\244\213",
  /* 0xed 0x84 */  "\345\245\223",
  /* 0xed 0x85 */  "\345\245\233",
  /* 0xed 0x86 */  "\345\245\235",
  /* 0xed 0x87 */  "\345\245\243",
  /* 0xed 0x88 */  "\345\246\244",
  /* 0xed 0x89 */  "\345\246\272",
  /* 0xed 0x8a */  "\345\255\226",
  /* 0xed 0x8b */  "\345\257\200",
  /* 0xed 0x8c */  "\347\224\257",
  /* 0xed 0x8d */  "\345\257\230",
  /* 0xed 0x8e */  "\345\257\254",
  /* 0xed 0x8f */  "\345\260\236",
  /* 0xed 0x90 */  "\345\262\246",
  /* 0xed 0x91 */  "\345\262\272",
  /* 0xed 0x92 */  "\345\263\265",
  /* 0xed 0x93 */  "\345\264\247",
  /* 0xed 0x94 */  "\345\265\223",
  /* 0xed 0x95 */  "\357\250\221",
  /* 0xed 0x96 */  "\345\265\202",
  /* 0xed 0x97 */  "\345\265\255",
  /* 0xed 0x98 */  "\345\266\270",
  /* 0xed 0x99 */  "\345\266\271",
  /* 0xed 0x9a */  "\345\267\220",
  /* 0xed 0x9b */  "\345\274\241",
  /* 0xed 0x9c */  "\345\274\264",
  /* 0xed 0x9d */  "\345\275\247",
  /* 0xed 0x9e */  "\345\276\267",
  /* 0xed 0x9f (invalid) */  NULL,

  /*====== 0xed, 0x9f-0xfc ======*/
  /* 0xed 0x9f */  "\345\277\236",
  /* 0xed 0xa0 */  "\346\201\235",
  /* 0xed 0xa1 */  "\346\202\205",
  /* 0xed 0xa2 */  "\346\202\212",
  /* 0xed 0xa3 */  "\346\203\236",
  /* 0xed 0xa4 */  "\346\203\225",
  /* 0xed 0xa5 */  "\346\204\240",
  /* 0xed 0xa6 */  "\346\203\262",
  /* 0xed 0xa7 */  "\346\204\221",
  /* 0xed 0xa8 */  "\346\204\267",
  /* 0xed 0xa9 */  "\346\204\260",
  /* 0xed 0xaa */  "\346\206\230",
  /* 0xed 0xab */  "\346\210\223",
  /* 0xed 0xac */  "\346\212\246",
  /* 0xed 0xad */  "\346\217\265",
  /* 0xed 0xae */  "\346\221\240",
  /* 0xed 0xaf */  "\346\222\235",
  /* 0xed 0xb0 */  "\346\223\216",
  /* 0xed 0xb1 */  "\346\225\216",
  /* 0xed 0xb2 */  "\346\230\200",
  /* 0xed 0xb3 */  "\346\230\225",
  /* 0xed 0xb4 */  "\346\230\273",
  /* 0xed 0xb5 */  "\346\230\211",
  /* 0xed 0xb6 */  "\346\230\256",
  /* 0xed 0xb7 */  "\346\230\236",
  /* 0xed 0xb8 */  "\346\230\244",
  /* 0xed 0xb9 */  "\346\231\245",
  /* 0xed 0xba */  "\346\231\227",
  /* 0xed 0xbb */  "\346\231\231",
  /* 0xed 0xbc */  "\357\250\222",
  /* 0xed 0xbd */  "\346\231\263",
  /* 0xed 0xbe */  "\346\232\231",
  /* 0xed 0xbf */  "\346\232\240",
  /* 0xed 0xc0 */  "\346\232\262",
  /* 0xed 0xc1 */  "\346\232\277",
  /* 0xed 0xc2 */  "\346\233\272",
  /* 0xed 0xc3 */  "\346\234\216",
  /* 0xed 0xc4 */  "\357\244\251",
  /* 0xed 0xc5 */  "\346\235\246",
  /* 0xed 0xc6 */  "\346\236\273",
  /* 0xed 0xc7 */  "\346\241\222",
  /* 0xed 0xc8 */  "\346\237\200",
  /* 0xed 0xc9 */  "\346\240\201",
  /* 0xed 0xca */  "\346\241\204",
  /* 0xed 0xcb */  "\346\243\217",
  /* 0xed 0xcc */  "\357\250\223",
  /* 0xed 0xcd */  "\346\245\250",
  /* 0xed 0xce */  "\357\250\224",
  /* 0xed 0xcf */  "\346\246\230",
  /* 0xed 0xd0 */  "\346\247\242",
  /* 0xed 0xd1 */  "\346\250\260",
  /* 0xed 0xd2 */  "\346\251\253",
  /* 0xed 0xd3 */  "\346\251\206",
  /* 0xed 0xd4 */  "\346\251\263",
  /* 0xed 0xd5 */  "\346\251\276",
  /* 0xed 0xd6 */  "\346\253\242",
  /* 0xed 0xd7 */  "\346\253\244",
  /* 0xed 0xd8 */  "\346\257\226",
  /* 0xed 0xd9 */  "\346\260\277",
  /* 0xed 0xda */  "\346\261\234",
  /* 0xed 0xdb */  "\346\262\206",
  /* 0xed 0xdc */  "\346\261\257",
  /* 0xed 0xdd */  "\346\263\232",
  /* 0xed 0xde */  "\346\264\204",
  /* 0xed 0xdf */  "\346\266\207",
  /* 0xed 0xe0 */  "\346\265\257",
  /* 0xed 0xe1 */  "\346\266\226",
  /* 0xed 0xe2 */  "\346\266\254",
  /* 0xed 0xe3 */  "\346\267\217",
  /* 0xed 0xe4 */  "\346\267\270",
  /* 0xed 0xe5 */  "\346\267\262",
  /* 0xed 0xe6 */  "\346\267\274",
  /* 0xed 0xe7 */  "\346\270\271",
  /* 0xed 0xe8 */  "\346\271\234",
  /* 0xed 0xe9 */  "\346\270\247",
  /* 0xed 0xea */  "\346\270\274",
  /* 0xed 0xeb */  "\346\272\277",
  /* 0xed 0xec */  "\346\276\210",
  /* 0xed 0xed */  "\346\276\265",
  /* 0xed 0xee */  "\346\277\265",
  /* 0xed 0xef */  "\347\200\205",
  /* 0xed 0xf0 */  "\347\200\207",
  /* 0xed 0xf1 */  "\347\200\250",
  /* 0xed 0xf2 */  "\347\202\205",
  /* 0xed 0xf3 */  "\347\202\253",
  /* 0xed 0xf4 */  "\347\204\217",
  /* 0xed 0xf5 */  "\347\204\204",
  /* 0xed 0xf6 */  "\347\205\234",
  /* 0xed 0xf7 */  "\347\205\206",
  /* 0xed 0xf8 */  "\347\205\207",
  /* 0xed 0xf9 */  "\357\250\225",
  /* 0xed 0xfa */  "\347\207\201",
  /* 0xed 0xfb */  "\347\207\276",
  /* 0xed 0xfc */  "\347\212\261",
};


/* 0xee, 0x40-0xfc */
static const char *cp932_to_utf8_table_0xee[] = {
  /*====== 0xee, 0x40-0x7e ======*/
  /* 0xee 0x40 */  "\347\212\276",
  /* 0xee 0x41 */  "\347\214\244",
  /* 0xee 0x42 */  "\357\250\226",
  /* 0xee 0x43 */  "\347\215\267",
  /* 0xee 0x44 */  "\347\216\275",
  /* 0xee 0x45 */  "\347\217\211",
  /* 0xee 0x46 */  "\347\217\226",
  /* 0xee 0x47 */  "\347\217\243",
  /* 0xee 0x48 */  "\347\217\222",
  /* 0xee 0x49 */  "\347\220\207",
  /* 0xee 0x4a */  "\347\217\265",
  /* 0xee 0x4b */  "\347\220\246",
  /* 0xee 0x4c */  "\347\220\252",
  /* 0xee 0x4d */  "\347\220\251",
  /* 0xee 0x4e */  "\347\220\256",
  /* 0xee 0x4f */  "\347\221\242",
  /* 0xee 0x50 */  "\347\222\211",
  /* 0xee 0x51 */  "\347\222\237",
  /* 0xee 0x52 */  "\347\224\201",
  /* 0xee 0x53 */  "\347\225\257",
  /* 0xee 0x54 */  "\347\232\202",
  /* 0xee 0x55 */  "\347\232\234",
  /* 0xee 0x56 */  "\347\232\236",
  /* 0xee 0x57 */  "\347\232\233",
  /* 0xee 0x58 */  "\347\232\246",
  /* 0xee 0x59 */  "\357\250\227",
  /* 0xee 0x5a */  "\347\235\206",
  /* 0xee 0x5b */  "\345\212\257",
  /* 0xee 0x5c */  "\347\240\241",
  /* 0xee 0x5d */  "\347\241\216",
  /* 0xee 0x5e */  "\347\241\244",
  /* 0xee 0x5f */  "\347\241\272",
  /* 0xee 0x60 */  "\347\244\260",
  /* 0xee 0x61 */  "\357\250\230",
  /* 0xee 0x62 */  "\357\250\231",
  /* 0xee 0x63 */  "\357\250\232",
  /* 0xee 0x64 */  "\347\246\224",
  /* 0xee 0x65 */  "\357\250\233",
  /* 0xee 0x66 */  "\347\246\233",
  /* 0xee 0x67 */  "\347\253\221",
  /* 0xee 0x68 */  "\347\253\247",
  /* 0xee 0x69 */  "\357\250\234",
  /* 0xee 0x6a */  "\347\253\253",
  /* 0xee 0x6b */  "\347\256\236",
  /* 0xee 0x6c */  "\357\250\235",
  /* 0xee 0x6d */  "\347\265\210",
  /* 0xee 0x6e */  "\347\265\234",
  /* 0xee 0x6f */  "\347\266\267",
  /* 0xee 0x70 */  "\347\266\240",
  /* 0xee 0x71 */  "\347\267\226",
  /* 0xee 0x72 */  "\347\271\222",
  /* 0xee 0x73 */  "\347\275\207",
  /* 0xee 0x74 */  "\347\276\241",
  /* 0xee 0x75 */  "\357\250\236",
  /* 0xee 0x76 */  "\350\214\201",
  /* 0xee 0x77 */  "\350\215\242",
  /* 0xee 0x78 */  "\350\215\277",
  /* 0xee 0x79 */  "\350\217\207",
  /* 0xee 0x7a */  "\350\217\266",
  /* 0xee 0x7b */  "\350\221\210",
  /* 0xee 0x7c */  "\350\222\264",
  /* 0xee 0x7d */  "\350\225\223",
  /* 0xee 0x7e */  "\350\225\231",
  /* 0xee 0x7f (invalid) */  NULL,

  /*====== 0xee, 0x80-0x9e ======*/
  /* 0xee 0x80 */  "\350\225\253",
  /* 0xee 0x81 */  "\357\250\237",
  /* 0xee 0x82 */  "\350\226\260",
  /* 0xee 0x83 */  "\357\250\240",
  /* 0xee 0x84 */  "\357\250\241",
  /* 0xee 0x85 */  "\350\240\207",
  /* 0xee 0x86 */  "\350\243\265",
  /* 0xee 0x87 */  "\350\250\222",
  /* 0xee 0x88 */  "\350\250\267",
  /* 0xee 0x89 */  "\350\251\271",
  /* 0xee 0x8a */  "\350\252\247",
  /* 0xee 0x8b */  "\350\252\276",
  /* 0xee 0x8c */  "\350\253\237",
  /* 0xee 0x8d */  "\357\250\242",
  /* 0xee 0x8e */  "\350\253\266",
  /* 0xee 0x8f */  "\350\255\223",
  /* 0xee 0x90 */  "\350\255\277",
  /* 0xee 0x91 */  "\350\263\260",
  /* 0xee 0x92 */  "\350\263\264",
  /* 0xee 0x93 */  "\350\264\222",
  /* 0xee 0x94 */  "\350\265\266",
  /* 0xee 0x95 */  "\357\250\243",
  /* 0xee 0x96 */  "\350\273\217",
  /* 0xee 0x97 */  "\357\250\244",
  /* 0xee 0x98 */  "\357\250\245",
  /* 0xee 0x99 */  "\351\201\247",
  /* 0xee 0x9a */  "\351\203\236",
  /* 0xee 0x9b */  "\357\250\246",
  /* 0xee 0x9c */  "\351\204\225",
  /* 0xee 0x9d */  "\351\204\247",
  /* 0xee 0x9e */  "\351\207\232",

  /*====== 0xee, 0x9f-0xfc ======*/
  /* 0xee 0x9f */  "\351\207\227",
  /* 0xee 0xa0 */  "\351\207\236",
  /* 0xee 0xa1 */  "\351\207\255",
  /* 0xee 0xa2 */  "\351\207\256",
  /* 0xee 0xa3 */  "\351\207\244",
  /* 0xee 0xa4 */  "\351\207\245",
  /* 0xee 0xa5 */  "\351\210\206",
  /* 0xee 0xa6 */  "\351\210\220",
  /* 0xee 0xa7 */  "\351\210\212",
  /* 0xee 0xa8 */  "\351\210\272",
  /* 0xee 0xa9 */  "\351\211\200",
  /* 0xee 0xaa */  "\351\210\274",
  /* 0xee 0xab */  "\351\211\216",
  /* 0xee 0xac */  "\351\211\231",
  /* 0xee 0xad */  "\351\211\221",
  /* 0xee 0xae */  "\351\210\271",
  /* 0xee 0xaf */  "\351\211\247",
  /* 0xee 0xb0 */  "\351\212\247",
  /* 0xee 0xb1 */  "\351\211\267",
  /* 0xee 0xb2 */  "\351\211\270",
  /* 0xee 0xb3 */  "\351\213\247",
  /* 0xee 0xb4 */  "\351\213\227",
  /* 0xee 0xb5 */  "\351\213\231",
  /* 0xee 0xb6 */  "\351\213\220",
  /* 0xee 0xb7 */  "\357\250\247",
  /* 0xee 0xb8 */  "\351\213\225",
  /* 0xee 0xb9 */  "\351\213\240",
  /* 0xee 0xba */  "\351\213\223",
  /* 0xee 0xbb */  "\351\214\245",
  /* 0xee 0xbc */  "\351\214\241",
  /* 0xee 0xbd */  "\351\213\273",
  /* 0xee 0xbe */  "\357\250\250",
  /* 0xee 0xbf */  "\351\214\236",
  /* 0xee 0xc0 */  "\351\213\277",
  /* 0xee 0xc1 */  "\351\214\235",
  /* 0xee 0xc2 */  "\351\214\202",
  /* 0xee 0xc3 */  "\351\215\260",
  /* 0xee 0xc4 */  "\351\215\227",
  /* 0xee 0xc5 */  "\351\216\244",
  /* 0xee 0xc6 */  "\351\217\206",
  /* 0xee 0xc7 */  "\351\217\236",
  /* 0xee 0xc8 */  "\351\217\270",
  /* 0xee 0xc9 */  "\351\220\261",
  /* 0xee 0xca */  "\351\221\205",
  /* 0xee 0xcb */  "\351\221\210",
  /* 0xee 0xcc */  "\351\226\222",
  /* 0xee 0xcd */  "\357\247\234",
  /* 0xee 0xce */  "\357\250\251",
  /* 0xee 0xcf */  "\351\232\235",
  /* 0xee 0xd0 */  "\351\232\257",
  /* 0xee 0xd1 */  "\351\234\263",
  /* 0xee 0xd2 */  "\351\234\273",
  /* 0xee 0xd3 */  "\351\235\203",
  /* 0xee 0xd4 */  "\351\235\215",
  /* 0xee 0xd5 */  "\351\235\217",
  /* 0xee 0xd6 */  "\351\235\221",
  /* 0xee 0xd7 */  "\351\235\225",
  /* 0xee 0xd8 */  "\351\241\227",
  /* 0xee 0xd9 */  "\351\241\245",
  /* 0xee 0xda */  "\357\250\252",
  /* 0xee 0xdb */  "\357\250\253",
  /* 0xee 0xdc */  "\351\244\247",
  /* 0xee 0xdd */  "\357\250\254",
  /* 0xee 0xde */  "\351\246\236",
  /* 0xee 0xdf */  "\351\251\216",
  /* 0xee 0xe0 */  "\351\253\231",
  /* 0xee 0xe1 */  "\351\253\234",
  /* 0xee 0xe2 */  "\351\255\265",
  /* 0xee 0xe3 */  "\351\255\262",
  /* 0xee 0xe4 */  "\351\256\217",
  /* 0xee 0xe5 */  "\351\256\261",
  /* 0xee 0xe6 */  "\351\256\273",
  /* 0xee 0xe7 */  "\351\260\200",
  /* 0xee 0xe8 */  "\351\265\260",
  /* 0xee 0xe9 */  "\351\265\253",
  /* 0xee 0xea */  "\357\250\255",
  /* 0xee 0xeb */  "\351\270\231",
  /* 0xee 0xec */  "\351\273\221",
  /* 0xee 0xed (invalid) */  NULL,
  /* 0xee 0xee (invalid) */  NULL,
  /* 0xee 0xef */  "\342\205\260",
  /* 0xee 0xf0 */  "\342\205\261",
  /* 0xee 0xf1 */  "\342\205\262",
  /* 0xee 0xf2 */  "\342\205\263",
  /* 0xee 0xf3 */  "\342\205\264",
  /* 0xee 0xf4 */  "\342\205\265",
  /* 0xee 0xf5 */  "\342\205\266",
  /* 0xee 0xf6 */  "\342\205\267",
  /* 0xee 0xf7 */  "\342\205\270",
  /* 0xee 0xf8 */  "\342\205\271",
  /* 0xee 0xf9 */  "\357\277\242",
  /* 0xee 0xfa */  "\357\277\244",
  /* 0xee 0xfb */  "\357\274\207",
  /* 0xee 0xfc */  "\357\274\202",
};


/* 0xfa, 0x40-0xfc */
static const char *cp932_to_utf8_table_0xfa[] = {
  /*====== 0xfa, 0x40-0x7e ======*/
  /* 0xfa 0x40 */  "\342\205\260",
  /* 0xfa 0x41 */  "\342\205\261",
  /* 0xfa 0x42 */  "\342\205\262",
  /* 0xfa 0x43 */  "\342\205\263",
  /* 0xfa 0x44 */  "\342\205\264",
  /* 0xfa 0x45 */  "\342\205\265",
  /* 0xfa 0x46 */  "\342\205\266",
  /* 0xfa 0x47 */  "\342\205\267",
  /* 0xfa 0x48 */  "\342\205\270",
  /* 0xfa 0x49 */  "\342\205\271",
  /* 0xfa 0x4a */  "\342\205\240",
  /* 0xfa 0x4b */  "\342\205\241",
  /* 0xfa 0x4c */  "\342\205\242",
  /* 0xfa 0x4d */  "\342\205\243",
  /* 0xfa 0x4e */  "\342\205\244",
  /* 0xfa 0x4f */  "\342\205\245",
  /* 0xfa 0x50 */  "\342\205\246",
  /* 0xfa 0x51 */  "\342\205\247",
  /* 0xfa 0x52 */  "\342\205\250",
  /* 0xfa 0x53 */  "\342\205\251",
  /* 0xfa 0x54 */  "\357\277\242",
  /* 0xfa 0x55 */  "\357\277\244",
  /* 0xfa 0x56 */  "\357\274\207",
  /* 0xfa 0x57 */  "\357\274\202",
  /* 0xfa 0x58 */  "\343\210\261",
  /* 0xfa 0x59 */  "\342\204\226",
  /* 0xfa 0x5a */  "\342\204\241",
  /* 0xfa 0x5b */  "\342\210\265",
  /* 0xfa 0x5c */  "\347\272\212",
  /* 0xfa 0x5d */  "\350\244\234",
  /* 0xfa 0x5e */  "\351\215\210",
  /* 0xfa 0x5f */  "\351\212\210",
  /* 0xfa 0x60 */  "\350\223\234",
  /* 0xfa 0x61 */  "\344\277\211",
  /* 0xfa 0x62 */  "\347\202\273",
  /* 0xfa 0x63 */  "\346\230\261",
  /* 0xfa 0x64 */  "\346\243\210",
  /* 0xfa 0x65 */  "\351\213\271",
  /* 0xfa 0x66 */  "\346\233\273",
  /* 0xfa 0x67 */  "\345\275\205",
  /* 0xfa 0x68 */  "\344\270\250",
  /* 0xfa 0x69 */  "\344\273\241",
  /* 0xfa 0x6a */  "\344\273\274",
  /* 0xfa 0x6b */  "\344\274\200",
  /* 0xfa 0x6c */  "\344\274\203",
  /* 0xfa 0x6d */  "\344\274\271",
  /* 0xfa 0x6e */  "\344\275\226",
  /* 0xfa 0x6f */  "\344\276\222",
  /* 0xfa 0x70 */  "\344\276\212",
  /* 0xfa 0x71 */  "\344\276\232",
  /* 0xfa 0x72 */  "\344\276\224",
  /* 0xfa 0x73 */  "\344\277\215",
  /* 0xfa 0x74 */  "\345\201\200",
  /* 0xfa 0x75 */  "\345\200\242",
  /* 0xfa 0x76 */  "\344\277\277",
  /* 0xfa 0x77 */  "\345\200\236",
  /* 0xfa 0x78 */  "\345\201\206",
  /* 0xfa 0x79 */  "\345\201\260",
  /* 0xfa 0x7a */  "\345\201\202",
  /* 0xfa 0x7b */  "\345\202\224",
  /* 0xfa 0x7c */  "\345\203\264",
  /* 0xfa 0x7d */  "\345\203\230",
  /* 0xfa 0x7e */  "\345\205\212",
  /* 0xfa 0x7f (invalid) */  NULL,

  /*====== 0xfa, 0x80-0xfc ======*/
  /* 0xfa 0x80 */  "\345\205\244",
  /* 0xfa 0x81 */  "\345\206\235",
  /* 0xfa 0x82 */  "\345\206\276",
  /* 0xfa 0x83 */  "\345\207\254",
  /* 0xfa 0x84 */  "\345\210\225",
  /* 0xfa 0x85 */  "\345\212\234",
  /* 0xfa 0x86 */  "\345\212\246",
  /* 0xfa 0x87 */  "\345\213\200",
  /* 0xfa 0x88 */  "\345\213\233",
  /* 0xfa 0x89 */  "\345\214\200",
  /* 0xfa 0x8a */  "\345\214\207",
  /* 0xfa 0x8b */  "\345\214\244",
  /* 0xfa 0x8c */  "\345\215\262",
  /* 0xfa 0x8d */  "\345\216\223",
  /* 0xfa 0x8e */  "\345\216\262",
  /* 0xfa 0x8f */  "\345\217\235",
  /* 0xfa 0x90 */  "\357\250\216",
  /* 0xfa 0x91 */  "\345\222\234",
  /* 0xfa 0x92 */  "\345\222\212",
  /* 0xfa 0x93 */  "\345\222\251",
  /* 0xfa 0x94 */  "\345\223\277",
  /* 0xfa 0x95 */  "\345\226\206",
  /* 0xfa 0x96 */  "\345\235\231",
  /* 0xfa 0x97 */  "\345\235\245",
  /* 0xfa 0x98 */  "\345\236\254",
  /* 0xfa 0x99 */  "\345\237\210",
  /* 0xfa 0x9a */  "\345\237\207",
  /* 0xfa 0x9b */  "\357\250\217",
  /* 0xfa 0x9c */  "\357\250\220",
  /* 0xfa 0x9d */  "\345\242\236",
  /* 0xfa 0x9e */  "\345\242\262",
  /* 0xfa 0x9f */  "\345\244\213",
  /* 0xfa 0xa0 */  "\345\245\223",
  /* 0xfa 0xa1 */  "\345\245\233",
  /* 0xfa 0xa2 */  "\345\245\235",
  /* 0xfa 0xa3 */  "\345\245\243",
  /* 0xfa 0xa4 */  "\345\246\244",
  /* 0xfa 0xa5 */  "\345\246\272",
  /* 0xfa 0xa6 */  "\345\255\226",
  /* 0xfa 0xa7 */  "\345\257\200",
  /* 0xfa 0xa8 */  "\347\224\257",
  /* 0xfa 0xa9 */  "\345\257\230",
  /* 0xfa 0xaa */  "\345\257\254",
  /* 0xfa 0xab */  "\345\260\236",
  /* 0xfa 0xac */  "\345\262\246",
  /* 0xfa 0xad */  "\345\262\272",
  /* 0xfa 0xae */  "\345\263\265",
  /* 0xfa 0xaf */  "\345\264\247",
  /* 0xfa 0xb0 */  "\345\265\223",
  /* 0xfa 0xb1 */  "\357\250\221",
  /* 0xfa 0xb2 */  "\345\265\202",
  /* 0xfa 0xb3 */  "\345\265\255",
  /* 0xfa 0xb4 */  "\345\266\270",
  /* 0xfa 0xb5 */  "\345\266\271",
  /* 0xfa 0xb6 */  "\345\267\220",
  /* 0xfa 0xb7 */  "\345\274\241",
  /* 0xfa 0xb8 */  "\345\274\264",
  /* 0xfa 0xb9 */  "\345\275\247",
  /* 0xfa 0xba */  "\345\276\267",
  /* 0xfa 0xbb */  "\345\277\236",
  /* 0xfa 0xbc */  "\346\201\235",
  /* 0xfa 0xbd */  "\346\202\205",
  /* 0xfa 0xbe */  "\346\202\212",
  /* 0xfa 0xbf */  "\346\203\236",
  /* 0xfa 0xc0 */  "\346\203\225",
  /* 0xfa 0xc1 */  "\346\204\240",
  /* 0xfa 0xc2 */  "\346\203\262",
  /* 0xfa 0xc3 */  "\346\204\221",
  /* 0xfa 0xc4 */  "\346\204\267",
  /* 0xfa 0xc5 */  "\346\204\260",
  /* 0xfa 0xc6 */  "\346\206\230",
  /* 0xfa 0xc7 */  "\346\210\223",
  /* 0xfa 0xc8 */  "\346\212\246",
  /* 0xfa 0xc9 */  "\346\217\265",
  /* 0xfa 0xca */  "\346\221\240",
  /* 0xfa 0xcb */  "\346\222\235",
  /* 0xfa 0xcc */  "\346\223\216",
  /* 0xfa 0xcd */  "\346\225\216",
  /* 0xfa 0xce */  "\346\230\200",
  /* 0xfa 0xcf */  "\346\230\225",
  /* 0xfa 0xd0 */  "\346\230\273",
  /* 0xfa 0xd1 */  "\346\230\211",
  /* 0xfa 0xd2 */  "\346\230\256",
  /* 0xfa 0xd3 */  "\346\230\236",
  /* 0xfa 0xd4 */  "\346\230\244",
  /* 0xfa 0xd5 */  "\346\231\245",
  /* 0xfa 0xd6 */  "\346\231\227",
  /* 0xfa 0xd7 */  "\346\231\231",
  /* 0xfa 0xd8 */  "\357\250\222",
  /* 0xfa 0xd9 */  "\346\231\263",
  /* 0xfa 0xda */  "\346\232\231",
  /* 0xfa 0xdb */  "\346\232\240",
  /* 0xfa 0xdc */  "\346\232\262",
  /* 0xfa 0xdd */  "\346\232\277",
  /* 0xfa 0xde */  "\346\233\272",
  /* 0xfa 0xdf */  "\346\234\216",
  /* 0xfa 0xe0 */  "\357\244\251",
  /* 0xfa 0xe1 */  "\346\235\246",
  /* 0xfa 0xe2 */  "\346\236\273",
  /* 0xfa 0xe3 */  "\346\241\222",
  /* 0xfa 0xe4 */  "\346\237\200",
  /* 0xfa 0xe5 */  "\346\240\201",
  /* 0xfa 0xe6 */  "\346\241\204",
  /* 0xfa 0xe7 */  "\346\243\217",
  /* 0xfa 0xe8 */  "\357\250\223",
  /* 0xfa 0xe9 */  "\346\245\250",
  /* 0xfa 0xea */  "\357\250\224",
  /* 0xfa 0xeb */  "\346\246\230",
  /* 0xfa 0xec */  "\346\247\242",
  /* 0xfa 0xed */  "\346\250\260",
  /* 0xfa 0xee */  "\346\251\253",
  /* 0xfa 0xef */  "\346\251\206",
  /* 0xfa 0xf0 */  "\346\251\263",
  /* 0xfa 0xf1 */  "\346\251\276",
  /* 0xfa 0xf2 */  "\346\253\242",
  /* 0xfa 0xf3 */  "\346\253\244",
  /* 0xfa 0xf4 */  "\346\257\226",
  /* 0xfa 0xf5 */  "\346\260\277",
  /* 0xfa 0xf6 */  "\346\261\234",
  /* 0xfa 0xf7 */  "\346\262\206",
  /* 0xfa 0xf8 */  "\346\261\257",
  /* 0xfa 0xf9 */  "\346\263\232",
  /* 0xfa 0xfa */  "\346\264\204",
  /* 0xfa 0xfb */  "\346\266\207",
  /* 0xfa 0xfc */  "\346\265\257",
};


/* 0xfb, 0x40-0xfc */
static const char *cp932_to_utf8_table_0xfb[] = {
  /*====== 0xfb, 0x40-0x7e ======*/
  /* 0xfb 0x40 */  "\346\266\226",
  /* 0xfb 0x41 */  "\346\266\254",
  /* 0xfb 0x42 */  "\346\267\217",
  /* 0xfb 0x43 */  "\346\267\270",
  /* 0xfb 0x44 */  "\346\267\262",
  /* 0xfb 0x45 */  "\346\267\274",
  /* 0xfb 0x46 */  "\346\270\271",
  /* 0xfb 0x47 */  "\346\271\234",
  /* 0xfb 0x48 */  "\346\270\247",
  /* 0xfb 0x49 */  "\346\270\274",
  /* 0xfb 0x4a */  "\346\272\277",
  /* 0xfb 0x4b */  "\346\276\210",
  /* 0xfb 0x4c */  "\346\276\265",
  /* 0xfb 0x4d */  "\346\277\265",
  /* 0xfb 0x4e */  "\347\200\205",
  /* 0xfb 0x4f */  "\347\200\207",
  /* 0xfb 0x50 */  "\347\200\250",
  /* 0xfb 0x51 */  "\347\202\205",
  /* 0xfb 0x52 */  "\347\202\253",
  /* 0xfb 0x53 */  "\347\204\217",
  /* 0xfb 0x54 */  "\347\204\204",
  /* 0xfb 0x55 */  "\347\205\234",
  /* 0xfb 0x56 */  "\347\205\206",
  /* 0xfb 0x57 */  "\347\205\207",
  /* 0xfb 0x58 */  "\357\250\225",
  /* 0xfb 0x59 */  "\347\207\201",
  /* 0xfb 0x5a */  "\347\207\276",
  /* 0xfb 0x5b */  "\347\212\261",
  /* 0xfb 0x5c */  "\347\212\276",
  /* 0xfb 0x5d */  "\347\214\244",
  /* 0xfb 0x5e */  "\357\250\226",
  /* 0xfb 0x5f */  "\347\215\267",
  /* 0xfb 0x60 */  "\347\216\275",
  /* 0xfb 0x61 */  "\347\217\211",
  /* 0xfb 0x62 */  "\347\217\226",
  /* 0xfb 0x63 */  "\347\217\243",
  /* 0xfb 0x64 */  "\347\217\222",
  /* 0xfb 0x65 */  "\347\220\207",
  /* 0xfb 0x66 */  "\347\217\265",
  /* 0xfb 0x67 */  "\347\220\246",
  /* 0xfb 0x68 */  "\347\220\252",
  /* 0xfb 0x69 */  "\347\220\251",
  /* 0xfb 0x6a */  "\347\220\256",
  /* 0xfb 0x6b */  "\347\221\242",
  /* 0xfb 0x6c */  "\347\222\211",
  /* 0xfb 0x6d */  "\347\222\237",
  /* 0xfb 0x6e */  "\347\224\201",
  /* 0xfb 0x6f */  "\347\225\257",
  /* 0xfb 0x70 */  "\347\232\202",
  /* 0xfb 0x71 */  "\347\232\234",
  /* 0xfb 0x72 */  "\347\232\236",
  /* 0xfb 0x73 */  "\347\232\233",
  /* 0xfb 0x74 */  "\347\232\246",
  /* 0xfb 0x75 */  "\357\250\227",
  /* 0xfb 0x76 */  "\347\235\206",
  /* 0xfb 0x77 */  "\345\212\257",
  /* 0xfb 0x78 */  "\347\240\241",
  /* 0xfb 0x79 */  "\347\241\216",
  /* 0xfb 0x7a */  "\347\241\244",
  /* 0xfb 0x7b */  "\347\241\272",
  /* 0xfb 0x7c */  "\347\244\260",
  /* 0xfb 0x7d */  "\357\250\230",
  /* 0xfb 0x7e */  "\357\250\231",
  /* 0xfb 0x7f (invalid) */  NULL,

  /*====== 0xfb, 0x80-0xfc ======*/
  /* 0xfb 0x80 */  "\357\250\232",
  /* 0xfb 0x81 */  "\347\246\224",
  /* 0xfb 0x82 */  "\357\250\233",
  /* 0xfb 0x83 */  "\347\246\233",
  /* 0xfb 0x84 */  "\347\253\221",
  /* 0xfb 0x85 */  "\347\253\247",
  /* 0xfb 0x86 */  "\357\250\234",
  /* 0xfb 0x87 */  "\347\253\253",
  /* 0xfb 0x88 */  "\347\256\236",
  /* 0xfb 0x89 */  "\357\250\235",
  /* 0xfb 0x8a */  "\347\265\210",
  /* 0xfb 0x8b */  "\347\265\234",
  /* 0xfb 0x8c */  "\347\266\267",
  /* 0xfb 0x8d */  "\347\266\240",
  /* 0xfb 0x8e */  "\347\267\226",
  /* 0xfb 0x8f */  "\347\271\222",
  /* 0xfb 0x90 */  "\347\275\207",
  /* 0xfb 0x91 */  "\347\276\241",
  /* 0xfb 0x92 */  "\357\250\236",
  /* 0xfb 0x93 */  "\350\214\201",
  /* 0xfb 0x94 */  "\350\215\242",
  /* 0xfb 0x95 */  "\350\215\277",
  /* 0xfb 0x96 */  "\350\217\207",
  /* 0xfb 0x97 */  "\350\217\266",
  /* 0xfb 0x98 */  "\350\221\210",
  /* 0xfb 0x99 */  "\350\222\264",
  /* 0xfb 0x9a */  "\350\225\223",
  /* 0xfb 0x9b */  "\350\225\231",
  /* 0xfb 0x9c */  "\350\225\253",
  /* 0xfb 0x9d */  "\357\250\237",
  /* 0xfb 0x9e */  "\350\226\260",
  /* 0xfb 0x9f */  "\357\250\240",
  /* 0xfb 0xa0 */  "\357\250\241",
  /* 0xfb 0xa1 */  "\350\240\207",
  /* 0xfb 0xa2 */  "\350\243\265",
  /* 0xfb 0xa3 */  "\350\250\222",
  /* 0xfb 0xa4 */  "\350\250\267",
  /* 0xfb 0xa5 */  "\350\251\271",
  /* 0xfb 0xa6 */  "\350\252\247",
  /* 0xfb 0xa7 */  "\350\252\276",
  /* 0xfb 0xa8 */  "\350\253\237",
  /* 0xfb 0xa9 */  "\357\250\242",
  /* 0xfb 0xaa */  "\350\253\266",
  /* 0xfb 0xab */  "\350\255\223",
  /* 0xfb 0xac */  "\350\255\277",
  /* 0xfb 0xad */  "\350\263\260",
  /* 0xfb 0xae */  "\350\263\264",
  /* 0xfb 0xaf */  "\350\264\222",
  /* 0xfb 0xb0 */  "\350\265\266",
  /* 0xfb 0xb1 */  "\357\250\243",
  /* 0xfb 0xb2 */  "\350\273\217",
  /* 0xfb 0xb3 */  "\357\250\244",
  /* 0xfb 0xb4 */  "\357\250\245",
  /* 0xfb 0xb5 */  "\351\201\247",
  /* 0xfb 0xb6 */  "\351\203\236",
  /* 0xfb 0xb7 */  "\357\250\246",
  /* 0xfb 0xb8 */  "\351\204\225",
  /* 0xfb 0xb9 */  "\351\204\247",
  /* 0xfb 0xba */  "\351\207\232",
  /* 0xfb 0xbb */  "\351\207\227",
  /* 0xfb 0xbc */  "\351\207\236",
  /* 0xfb 0xbd */  "\351\207\255",
  /* 0xfb 0xbe */  "\351\207\256",
  /* 0xfb 0xbf */  "\351\207\244",
  /* 0xfb 0xc0 */  "\351\207\245",
  /* 0xfb 0xc1 */  "\351\210\206",
  /* 0xfb 0xc2 */  "\351\210\220",
  /* 0xfb 0xc3 */  "\351\210\212",
  /* 0xfb 0xc4 */  "\351\210\272",
  /* 0xfb 0xc5 */  "\351\211\200",
  /* 0xfb 0xc6 */  "\351\210\274",
  /* 0xfb 0xc7 */  "\351\211\216",
  /* 0xfb 0xc8 */  "\351\211\231",
  /* 0xfb 0xc9 */  "\351\211\221",
  /* 0xfb 0xca */  "\351\210\271",
  /* 0xfb 0xcb */  "\351\211\247",
  /* 0xfb 0xcc */  "\351\212\247",
  /* 0xfb 0xcd */  "\351\211\267",
  /* 0xfb 0xce */  "\351\211\270",
  /* 0xfb 0xcf */  "\351\213\247",
  /* 0xfb 0xd0 */  "\351\213\227",
  /* 0xfb 0xd1 */  "\351\213\231",
  /* 0xfb 0xd2 */  "\351\213\220",
  /* 0xfb 0xd3 */  "\357\250\247",
  /* 0xfb 0xd4 */  "\351\213\225",
  /* 0xfb 0xd5 */  "\351\213\240",
  /* 0xfb 0xd6 */  "\351\213\223",
  /* 0xfb 0xd7 */  "\351\214\245",
  /* 0xfb 0xd8 */  "\351\214\241",
  /* 0xfb 0xd9 */  "\351\213\273",
  /* 0xfb 0xda */  "\357\250\250",
  /* 0xfb 0xdb */  "\351\214\236",
  /* 0xfb 0xdc */  "\351\213\277",
  /* 0xfb 0xdd */  "\351\214\235",
  /* 0xfb 0xde */  "\351\214\202",
  /* 0xfb 0xdf */  "\351\215\260",
  /* 0xfb 0xe0 */  "\351\215\227",
  /* 0xfb 0xe1 */  "\351\216\244",
  /* 0xfb 0xe2 */  "\351\217\206",
  /* 0xfb 0xe3 */  "\351\217\236",
  /* 0xfb 0xe4 */  "\351\217\270",
  /* 0xfb 0xe5 */  "\351\220\261",
  /* 0xfb 0xe6 */  "\351\221\205",
  /* 0xfb 0xe7 */  "\351\221\210",
  /* 0xfb 0xe8 */  "\351\226\222",
  /* 0xfb 0xe9 */  "\357\247\234",
  /* 0xfb 0xea */  "\357\250\251",
  /* 0xfb 0xeb */  "\351\232\235",
  /* 0xfb 0xec */  "\351\232\257",
  /* 0xfb 0xed */  "\351\234\263",
  /* 0xfb 0xee */  "\351\234\273",
  /* 0xfb 0xef */  "\351\235\203",
  /* 0xfb 0xf0 */  "\351\235\215",
  /* 0xfb 0xf1 */  "\351\235\217",
  /* 0xfb 0xf2 */  "\351\235\221",
  /* 0xfb 0xf3 */  "\351\235\225",
  /* 0xfb 0xf4 */  "\351\241\227",
  /* 0xfb 0xf5 */  "\351\241\245",
  /* 0xfb 0xf6 */  "\357\250\252",
  /* 0xfb 0xf7 */  "\357\250\253",
  /* 0xfb 0xf8 */  "\351\244\247",
  /* 0xfb 0xf9 */  "\357\250\254",
  /* 0xfb 0xfa */  "\351\246\236",
  /* 0xfb 0xfb */  "\351\251\216",
  /* 0xfb 0xfc */  "\351\253\231",
};


/* 0xfc, 0x40-0x4b */
static const char *cp932_to_utf8_table_0xfc[] = {
  /*====== 0xfc, 0x40-0x4b ======*/
  /* 0xfc 0x40 */  "\351\253\234",
  /* 0xfc 0x41 */  "\351\255\265",
  /* 0xfc 0x42 */  "\351\255\262",
  /* 0xfc 0x43 */  "\351\256\217",
  /* 0xfc 0x44 */  "\351\256\261",
  /* 0xfc 0x45 */  "\351\256\273",
  /* 0xfc 0x46 */  "\351\260\200",
  /* 0xfc 0x47 */  "\351\265\260",
  /* 0xfc 0x48 */  "\351\265\253",
  /* 0xfc 0x49 */  "\357\250\255",
  /* 0xfc 0x4a */  "\351\270\231",
  /* 0xfc 0x4b */  "\351\273\221",
};


#define CP932_TO_UTF8_0X87_MIN	0x40
#define CP932_TO_UTF8_0X87_MAX	0x93
#define CP932_TO_UTF8_0XED_MIN	0x40
#define CP932_TO_UTF8_0XED_MAX	0xfc
#define CP932_TO_UTF8_0XEE_MIN	0x40
#define CP932_TO_UTF8_0XEE_MAX	0xfc
#define CP932_TO_UTF8_0XFA_MIN	0x40
#define CP932_TO_UTF8_0XFA_MAX	0xfc
#define CP932_TO_UTF8_0XFB_MIN	0x40
#define CP932_TO_UTF8_0XFB_MAX	0xfc
#define CP932_TO_UTF8_0XFC_MIN	0x40
#define CP932_TO_UTF8_0XFC_MAX	0x4b


size_t
cp932_to_utf8_helper(const char **inbuf, size_t *inbytesleft,
		     char **outbuf, size_t *outbytesleft)
{
  const char *utf8_string = NULL;
  unsigned char first;
  unsigned char second;

  if (*inbytesleft <= 2)
    {
      errno = EINVAL;
      return -1;
    }

  first = (*inbuf)[0];
  second = (*inbuf)[1];

#if DEBUG_ICONV
  fprintf(stderr, "first=%02x second=%02x\n", first, second);
#endif

  switch (first)
    {
    case 0x87:
      if (second >= CP932_TO_UTF8_0X87_MIN && second <= CP932_TO_UTF8_0X87_MAX)
	utf8_string = cp932_to_utf8_table_0x87[second - CP932_TO_UTF8_0X87_MIN];
      break;

    case 0xed:
      if (second >= CP932_TO_UTF8_0XED_MIN && second <= CP932_TO_UTF8_0XED_MAX)
	utf8_string = cp932_to_utf8_table_0xed[second - CP932_TO_UTF8_0XED_MIN];
      break;

    case 0xee:
      if (second >= CP932_TO_UTF8_0XEE_MIN && second <= CP932_TO_UTF8_0XEE_MAX)
	utf8_string = cp932_to_utf8_table_0xee[second - CP932_TO_UTF8_0XEE_MIN];
      break;

    case 0xfa:
      if (second >= CP932_TO_UTF8_0XFA_MIN && second <= CP932_TO_UTF8_0XFA_MAX)
	utf8_string = cp932_to_utf8_table_0xfa[second - CP932_TO_UTF8_0XFA_MIN];
      break;

    case 0xfb:
      if (second >= CP932_TO_UTF8_0XFB_MIN && second <= CP932_TO_UTF8_0XFB_MAX)
	utf8_string = cp932_to_utf8_table_0xfb[second - CP932_TO_UTF8_0XFB_MIN];
      break;

    case 0xfc:
      if (second >= CP932_TO_UTF8_0XFC_MIN && second <= CP932_TO_UTF8_0XFC_MAX)
	utf8_string = cp932_to_utf8_table_0xfc[second - CP932_TO_UTF8_0XFC_MIN];
      break;

    }

  if (*outbytesleft < 3)
    {
      errno = E2BIG;
      return -1;
    }

  if (utf8_string == NULL)
    {
      /* Ʀ(={ 0xe2, 0x96, 0xa1 })ɤ*/
#if DEBUG_ICONV
      fprintf(stderr, "first=%02x second=%02x\n", first, second);
#endif
      utf8_string = "\342\226\241";
      first = '\0';
    }

  if ((first & 0x80) != 0)
    {
      *inbuf += 2;
      *inbytesleft -= 2;
    }
  else
    {
      *inbuf += 1;
      *inbytesleft -= 1;
    }

  (*outbuf)[0] = utf8_string[0];
  (*outbuf)[1] = utf8_string[1];
  (*outbuf)[2] = utf8_string[2];
  *outbuf += 3;
  *outbytesleft -= 3;

  return 0;
}


#if TRACE_MEMORY_USAGE
/*
 * ꡼åϢ
 */
static pthread_mutex_t memory_trace_lock;
static GHashTable *memory_trace;


#define MEMORY_TRACE_LOCK				\
  if (pthread_mutex_lock(&memory_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define MEMORY_TRACE_UNLOCK				\
  if (pthread_mutex_unlock(&memory_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


void
init_memory_trace(void)
{
  if (pthread_mutex_init(&memory_trace_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  memory_trace = g_hash_table_new_full(g_direct_hash, g_direct_equal,
				       NULL, (GDestroyNotify)g_free);
}


void
register_pointer(gpointer pointer, const char *where)
{
  gpointer previous;
  MEMORY_TRACE_LOCK
    {
      previous = g_hash_table_lookup(memory_trace, pointer);
      g_hash_table_insert(memory_trace, pointer, g_strdup(where));
    }
  MEMORY_TRACE_UNLOCK;

  if (previous != NULL)
    fprintf(stderr, "%p has already been registered: \"%s\" -> \"%s\"\n",
	    pointer, (char *)previous, where);
}


void
unregister_pointer(gpointer pointer, const char *where)
{
  gpointer usage;
  MEMORY_TRACE_LOCK
    {
      usage = g_hash_table_lookup(memory_trace, pointer);
      g_hash_table_remove(memory_trace, pointer);
    }
  MEMORY_TRACE_UNLOCK;

  if (usage == NULL && pointer != NULL)
    fprintf(stderr, "%p not registered: %s\n", pointer, where);
}


static void
print_pointer_info(gpointer key, gpointer value, gpointer user_data)
{
  if (strstr((const char *)value, "g_str") != NULL)
    fprintf(stderr, "%p \"%s\": %s\n", key, (char *)key, (char *)value);
  else
    fprintf(stderr, "%p: %s\n", key, (char *)value);
}


void
dump_pointers()
{
  MEMORY_TRACE_LOCK
    {
      g_hash_table_foreach(memory_trace, print_pointer_info, NULL);
    }
  MEMORY_TRACE_UNLOCK;
}
#endif


#if TRACE_REFCOUNT
static pthread_mutex_t refcount_trace_lock;
static GHashTable *refcount_trace;
static GSList *consistent_object_list;


#define REFCOUNT_TRACE_LOCK				\
  if (pthread_mutex_lock(&refcount_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't lock a mutex.\n");	\
      abort();						\
    }

#define REFCOUNT_TRACE_UNLOCK				\
  if (pthread_mutex_unlock(&refcount_trace_lock) != 0)	\
    {							\
      fprintf(stderr, "Couldn't unlock a mutex.\n");	\
      abort();						\
    }


static void
delete_refcount_history(GSList *history)
{
  g_slist_foreach(history, (GFunc)g_free, NULL);
  g_slist_free(history);
}


void
init_refcount_trace(void)
{
  if (pthread_mutex_init(&refcount_trace_lock, NULL) != 0)
    {
      fprintf(stderr, "Couldn't init a mutex.\n");
      abort();
    }

  refcount_trace = g_hash_table_new(g_direct_hash, g_direct_equal);
  consistent_object_list = NULL;
}


void
trace_register_g_object(gpointer pointer, const char *file, int line)
{
  GSList *previous;
  char where[1024];
  GObject *object = (GObject *)pointer;
  snprintf(where, 1024, "register_g_object(%p) at %s, line %d: ref_count=%d",
	   object, file, line, object->ref_count);
  REFCOUNT_TRACE_LOCK
  {
    previous = g_hash_table_lookup(refcount_trace, object);
    if (previous != NULL)
      {
	if (g_slist_find(consistent_object_list, object) != NULL)
	  {
	    consistent_object_list = g_slist_remove(consistent_object_list,
						    object);
	    delete_refcount_history(previous);
	    previous = NULL;
	  }
      }
    g_hash_table_insert(refcount_trace, object,
			g_slist_append(previous, g_strdup(where)));
  }
  REFCOUNT_TRACE_UNLOCK;
}


static void
output_warning(GObject *object)
{
  fprintf(stderr, "%p seems not to be a valid pointer to GObject\n", object);
}


void
trace_g_object_ref(gpointer pointer, const char *file, int line)
{
  GSList *history;
  char where[1024];
  GObject *object = (GObject *)pointer;
  snprintf(where, 1024, "g_object_ref(%p) at %s, line %d: ref_count=%d",
	   object, file, line, object->ref_count);
  if (object->ref_count < 1 || object->ref_count >= 100)
    output_warning(object);
  REFCOUNT_TRACE_LOCK
  {
    history = g_hash_table_lookup(refcount_trace, object);
    g_hash_table_insert(refcount_trace, object,
			g_slist_append(history, g_strdup(where)));
  }
  REFCOUNT_TRACE_UNLOCK;
  g_object_ref(object);
}


void
trace_g_object_unref(gpointer pointer, const char *file, int line)
{
  GSList *history;
  char where[1024];
  GObject *object = (GObject *)pointer;
  snprintf(where, 1024, "g_object_unref(%p) at %s, line %d: ref_count=%d",
	   object, file, line, object->ref_count);
  if (object->ref_count < 1 || object->ref_count >= 100)
    output_warning(object);
  REFCOUNT_TRACE_LOCK
  {
    history = g_hash_table_lookup(refcount_trace, object);
    g_hash_table_insert(refcount_trace, object,
			g_slist_append(history, g_strdup(where)));
    if (object->ref_count == 1)
      consistent_object_list = g_slist_append(consistent_object_list, object);
    else if (object->ref_count == 0)
      consistent_object_list = g_slist_remove(consistent_object_list, object);
  }
  REFCOUNT_TRACE_UNLOCK;
  if (object->ref_count > 0 && object->ref_count < 100)
    g_object_unref(object);
}


static void
print_history(const char *where)
{
  fprintf(stderr, "%s\n", where);
}


static void
remove_consistent_object(gpointer object, gpointer unused)
{
  GSList *history = g_hash_table_lookup(refcount_trace, object);
#if 0
  fprintf(stderr, "*** Consistent object %p\n", object);
  g_slist_foreach(history, (GFunc)print_history, NULL);
  fprintf(stderr, "******\n\n");
#endif
  delete_refcount_history(history);
  g_hash_table_remove(refcount_trace, object);
}


static void
print_refcount_info(GObject *object, GSList *history, gpointer unused)
{
  fprintf(stderr, "*** Inconsistent object %p\n", object);
  g_slist_foreach(history, (GFunc)print_history, NULL);
  fprintf(stderr, "******\n\n");
  delete_refcount_history(history);
}


void
dump_objects(void)
{
  REFCOUNT_TRACE_LOCK
  {
    g_slist_foreach(consistent_object_list, remove_consistent_object, NULL);
    g_hash_table_foreach(refcount_trace, (GHFunc)print_refcount_info, NULL);
  }
  REFCOUNT_TRACE_UNLOCK;
}


void
ochu_object_ref(gpointer pointer)
{
  GObject *object = (GObject *)pointer;
  OCHU_OBJECT_REF(object);
}


void
ochu_object_unref(gpointer pointer)
{
  GObject *object = (GObject *)pointer;
  OCHU_OBJECT_UNREF(object);
}
#endif


