/*
 * Copyright (c) 2003-2004 The Ochusha Project.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 * $Id: ochusha_thread_2ch.c,v 1.71.2.3 2004/09/22 14:31:28 fuyu Exp $
 */

#include "config.h"

#include "ochusha_private.h"
#include "ochusha.h"
#include "ochusha_board_2ch.h"
#include "ochusha_thread_2ch.h"
#include "ochusha_utils_2ch.h"

#include "htmlutils.h"
#include "worker.h"

#include <glib-object.h>
#include <glib.h>

#include <errno.h>
#include <fcntl.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#include <zlib.h>


#define DEBUG_DAT	0
#define DEBUG_DAT_MOST	0

#define ANALYSIS_INTERVAL	100


static void ochusha_thread_2ch_class_init(OchushaThread2chClass *klass);
static void ochusha_thread_2ch_init(OchushaThread2ch *thread);
static void ochusha_thread_2ch_finalize(GObject *object);

static const char *ochusha_thread_2ch_get_base_url(OchushaBBSThread *thread);
static void ochusha_thread_2ch_read_threadlist_element(
						OchushaBBSThread *thread,
						GHashTable *thread_attributes);
static void ochusha_thread_2ch_write_threadlist_element(
						OchushaBBSThread *thread,
						gzFile threadlist_xml);

static void ochusha_thread_2ch_property_change_notify(OchushaThread2ch *thread);

static gboolean ochusha_thread_2ch_check_url(OchushaBBSThread *thread,
					     const char *url,
					     unsigned int *from_p,
					     unsigned int *to_p);
static void ochusha_thread_2ch_remove_cache(OchushaBBSThread *thread,
					    OchushaConfig *config);
static OchushaAsyncBuffer *ochusha_thread_2ch_get_responses_source(
					OchushaBBSThread *thread,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode);
static char *extract_2ch_response(char *start_pos, char *eol_pos,
				  OchushaBBSResponse *response);
static gboolean ochusha_thread_2ch_parse_responses(OchushaBBSThread *thread,
						   OchushaAsyncBuffer *buffer,
						   int start, int number,
						   gboolean no_wait,
						   StartThreadCallback *st_cb,
						   EachResponseCallback *r_cb,
						   BrokenResponseCallback *bcb,
						   EndThreadCallback *end_cb,
						   StartParsingCallback *start_parsing_cb,
						   BeforeWaitCallback *before_wait_cb,
						   AfterWaitCallback *after_wait_cb,
						   EndParsingCallback *end_parsing_cb,
						   gpointer callback_data);

static time_t ochusha_thread_2ch_get_last_modified_utc(OchushaBBSThread *thread);

static gboolean ochusha_thread_2ch_preview_response(OchushaBBSThread *thread,
					const OchushaBBSResponse *response,
					StartThreadCallback *start_cb,
					EachResponseCallback *response_cb,
					EndThreadCallback *end_cb,
					gpointer callback_data);

static gboolean ochusha_thread_2ch_post_supported(OchushaBBSThread *thread);

static gboolean ochusha_thread_2ch_post_response(OchushaBBSThread *thread,
					OchushaNetworkBroker *broker,
					const OchushaBBSResponse *response,
					gboolean use_id);

static char *ochusha_thread_2ch_get_url_for_response(OchushaBBSThread *thread,
						     int from, int to);
static const char *ochusha_thread_2ch_get_url_to_post(OchushaBBSThread *thread);


GType
ochusha_thread_2ch_get_type(void)
{
  static GType thread_2ch_type = 0;

  if (thread_2ch_type == 0)
    {
      static const GTypeInfo thread_2ch_info =
	{
	  sizeof(OchushaThread2chClass),
	  NULL, /* base_init */
	  NULL, /* base_finalize */
	  (GClassInitFunc)ochusha_thread_2ch_class_init,
	  NULL, /* class_finalize */
	  NULL, /* class_data */
	  sizeof(OchushaThread2ch),
	  0,	/* n_preallocs */
	  (GInstanceInitFunc)ochusha_thread_2ch_init,
	};

      thread_2ch_type = g_type_register_static(OCHUSHA_TYPE_BBS_THREAD,
					       "OchushaThread2ch",
					       &thread_2ch_info, 0);
    }

  return thread_2ch_type;
}


static OchushaBBSThreadClass *parent_class = NULL;
static GQuark idle_checker_id;


static void
ochusha_thread_2ch_class_init(OchushaThread2chClass *klass)
{
  GObjectClass *o_class = G_OBJECT_CLASS(klass);
  OchushaBBSThreadClass *b_class = OCHUSHA_BBS_THREAD_CLASS(klass);

  parent_class = g_type_class_peek_parent(klass);

  o_class->finalize = ochusha_thread_2ch_finalize;

  b_class->read_threadlist_element
    = ochusha_thread_2ch_read_threadlist_element;
  b_class->write_threadlist_element
    = ochusha_thread_2ch_write_threadlist_element;

  b_class->get_board = NULL;
  b_class->get_number_of_responses_on_server = NULL;
  b_class->get_number_of_responses_read = NULL;
  b_class->set_number_of_responses_read = NULL;
  b_class->get_flags = NULL;
  b_class->set_flags = NULL;

  b_class->get_responses_source = ochusha_thread_2ch_get_responses_source;
  b_class->parse_responses = ochusha_thread_2ch_parse_responses;
  b_class->get_last_modified_utc = ochusha_thread_2ch_get_last_modified_utc;
  b_class->get_url = ochusha_thread_2ch_get_base_url;
  b_class->get_url_for_response = ochusha_thread_2ch_get_url_for_response;
  b_class->get_url_to_post = ochusha_thread_2ch_get_url_to_post;
  b_class->check_url = ochusha_thread_2ch_check_url;
  b_class->remove_cache = ochusha_thread_2ch_remove_cache;

  b_class->preview_response = ochusha_thread_2ch_preview_response;
  b_class->post_supported = ochusha_thread_2ch_post_supported;
  b_class->post_response = ochusha_thread_2ch_post_response;

  klass->get_base_path = NULL;
  klass->make_post_response_message = NULL;

  idle_checker_id
    = g_quark_from_static_string("OchushaThread2ch::IdleChecker");
}


static void
ochusha_thread_2ch_init(OchushaThread2ch *thread)
{
  thread->dat_url = NULL;
  thread->base_url = NULL;
  thread->base_path = NULL;
  thread->post_url = NULL;

  thread->responses = NULL;

  thread->alive = FALSE;

  g_signal_connect(thread,
		   "notify::board",
		   G_CALLBACK(ochusha_thread_2ch_property_change_notify),
		   thread);
  g_signal_connect(thread,
		   "notify::id",
		   G_CALLBACK(ochusha_thread_2ch_property_change_notify),
		   thread);
}


static void
ochusha_thread_2ch_finalize(GObject *object)
{
  OchushaThread2ch *thread = OCHUSHA_THREAD_2CH(object);

  if (thread->dat_url != NULL)
    {
      G_FREE(thread->dat_url);
      thread->dat_url = NULL;
    }

  if (thread->base_url != NULL)
    {
      G_FREE(thread->base_url);
      thread->base_url = NULL;
    }

  if (thread->base_path != NULL)
    {
      G_FREE(thread->base_path);
      thread->base_path = NULL;
    }

  if (thread->kako_html != NULL)
    {
      G_FREE(thread->kako_html);
      thread->kako_html = NULL;
    }

  if (thread->oyster_url != NULL)
    {
      G_FREE(thread->oyster_url);
      thread->oyster_url = NULL;
    }

  if (thread->responses != NULL)
    {
      G_FREE(thread->responses);
      thread->responses = NULL;
    }

  if (thread->last_modified != NULL)
    {
      G_FREE(thread->last_modified);
      thread->last_modified = NULL;
    }

  if (thread->date != NULL)
    {
      G_FREE(thread->date);
      thread->date = NULL;
    }

  if (G_OBJECT_CLASS(parent_class)->finalize)
    (*G_OBJECT_CLASS(parent_class)->finalize)(object);
}


void
ochusha_thread_2ch_set_kako_html_url(OchushaThread2ch *thread_2ch,
				     const gchar *url)
{
  OchushaBBSThread *thread;
  g_return_if_fail(OCHUSHA_IS_THREAD_2CH(thread_2ch));

  if (thread_2ch->kako_html != NULL)
    G_FREE(thread_2ch->kako_html);

  thread_2ch->kako_html = url != NULL ? G_STRDUP(url) : NULL;

  thread = OCHUSHA_BBS_THREAD(thread_2ch);
  thread->flags |= OCHUSHA_BBS_THREAD_KAKO;
}


static void
ochusha_thread_2ch_read_threadlist_element(OchushaBBSThread *thread,
					   GHashTable *thread_attributes)
{
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(thread);
  thread_2ch->kako_html
    = ochusha_utils_get_attribute_string(thread_attributes, "kako_html");
  thread_2ch->last_modified
    = ochusha_utils_get_attribute_string(thread_attributes, "last_modified");

  if (parent_class->read_threadlist_element != NULL)
    (*parent_class->read_threadlist_element)(thread, thread_attributes);
}


#define OUTPUT_THREAD_ATTRIBUTE_STRING(gzfile, thread, attribute)	\
  do									\
    {									\
      if ((thread)->attribute != NULL)					\
	{								\
	  gchar *text = g_markup_escape_text((thread)->attribute, -1);	\
	  gzprintf(gzfile,						\
		   "      <attribute name=\"" #attribute	"\">\n"	\
		   "        <string>%s</string>\n"			\
		   "      </attribute>\n", text);			\
	  g_free(text);							\
	}								\
    } while (0)


static void
ochusha_thread_2ch_write_threadlist_element(OchushaBBSThread *thread,
					    gzFile threadlist_xml)
{
  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(thread);

  if (thread->board->bbs_type == OCHUSHA_BBS_TYPE_2CH_HEADLINE)
    thread->number_of_responses_read = 0;

  if (parent_class->write_threadlist_element != NULL)
    (*parent_class->write_threadlist_element)(thread, threadlist_xml);

  OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, thread_2ch, kako_html);
  if (!(thread->flags
	& (OCHUSHA_BBS_THREAD_DAT_DROPPED | OCHUSHA_BBS_THREAD_KAKO)))
    OUTPUT_THREAD_ATTRIBUTE_STRING(threadlist_xml, thread_2ch, last_modified);
}


static void
ochusha_thread_2ch_property_change_notify(OchushaThread2ch *thread)
{
  OchushaThread2ch *thread_2ch;
#if DEBUG_LIB_OCHUSHA
  fprintf(stderr, "thread(%p)->board changed to board(%p).\n",
	  thread, thread->board);
#endif
  g_return_if_fail(OCHUSHA_IS_THREAD_2CH(thread));

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  if (thread->dat_url != NULL)
    {
      G_FREE(thread->dat_url);
      thread->dat_url = NULL;
    }

  if (thread->base_url != NULL)
    {
      G_FREE(thread->base_url);
      thread->base_url = NULL;
    }

  if (thread->base_path != NULL)
    {
      G_FREE(thread->base_path);
      thread->base_path = NULL;
    }

  if (thread->post_url != NULL)
    {
      G_FREE(thread->post_url);
      thread->post_url = NULL;
    }
}


static gboolean
ochusha_thread_2ch_check_url(OchushaBBSThread *thread, const char *url,
			     unsigned int *from_p, unsigned int *to_p)
{
  char *board_id = NULL;
  char *thread_id = NULL;
  gboolean result;

  if (!ochusha_utils_2ch_check_url(url, NULL, NULL, &board_id,
				   &thread_id, from_p, to_p, NULL))
    return FALSE;

#if 0
  fprintf(stderr, "check_url: board_id=\"%s\" thread_id=\"%s\"\n",
	  board_id, thread_id);
#endif
  if (strcmp(ochusha_bulletin_board_get_id(thread->board), board_id) == 0
      && thread_id != NULL
      && (*thread_id == '\0' || strcmp(thread->id, thread_id) == 0))
    result = TRUE;
  else
    result = FALSE;

  if (board_id != NULL)
    G_FREE(board_id);

  if (thread_id != NULL)
    G_FREE(thread_id);

  return result;
}


static void
ochusha_thread_2ch_remove_cache(OchushaBBSThread *thread,
				OchushaConfig *config)
{
  const char *dat_url;
  OchushaThread2ch *thread_2ch;
  g_return_if_fail(OCHUSHA_IS_THREAD_2CH(thread) && config != NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  dat_url = ochusha_thread_2ch_get_dat_url(thread_2ch);
  if (dat_url != NULL)
    ochusha_config_cache_unlink_file(config, dat_url);

  if (thread_2ch->last_modified != NULL)
    {
      G_FREE(thread_2ch->last_modified);
      thread_2ch->last_modified = NULL;
    }
}


OchushaBBSThread *
ochusha_thread_2ch_new(OchushaBoard2ch *board,
		       const char *dat_filename,
		       const gchar *title)
{
  return OCHUSHA_BBS_THREAD(g_object_new(OCHUSHA_TYPE_THREAD_2CH,
					 "board", board,
					 "id", dat_filename,
					 "title", title,
					 NULL));
}


const char *
ochusha_thread_2ch_get_dat_url(OchushaThread2ch *thread_2ch)
{
  OchushaBBSThread *thread;
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread_2ch), NULL);

  if (thread_2ch->dat_url != NULL)
    return thread_2ch->dat_url;

  thread = OCHUSHA_BBS_THREAD(thread_2ch);

  g_return_val_if_fail(thread->board != NULL && thread->board->base_url != NULL
		       && thread->id != NULL,
		       NULL);

  if (thread_2ch->kako_html == NULL)
    {
      char url[PATH_MAX];
      if (snprintf(url, PATH_MAX, "%sdat/%s.dat",
		   thread->board->base_url, thread->id) >= PATH_MAX)
	return NULL;

      thread_2ch->dat_url = G_STRDUP(url);
    }
  else
    {
      char *dat_url = G_STRNDUP(thread_2ch->kako_html,
				strlen(thread_2ch->kako_html) + 2);
      char *html_pos = strstr(dat_url, ".html");
      if (html_pos != NULL)
	{
	  html_pos[1] = 'd';
	  html_pos[2] = 'a';
	  html_pos[3] = 't';
	  html_pos[4] = '.';
	  html_pos[5] = 'g';
	  html_pos[6] = 'z';
	  html_pos[7] = '\0';

	  thread_2ch->dat_url = dat_url;
	}
      else
	G_FREE(dat_url);
    }

  return thread_2ch->dat_url;
}


static const char *
ochusha_thread_2ch_get_base_url(OchushaBBSThread *thread)
{
  char url[PATH_MAX];
  const char *base_path;
  const char *server;
  OchushaThread2ch *thread_2ch;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);
  if (thread_2ch->base_url != NULL && thread_2ch->kako_html == NULL)
    return thread_2ch->base_url;

  base_path = ochusha_thread_2ch_get_base_path(thread_2ch);

  thread = OCHUSHA_BBS_THREAD(thread_2ch);
  server = ochusha_bulletin_board_get_server(thread->board);

  g_return_val_if_fail(base_path != NULL && server != NULL, NULL);
  if (snprintf(url, PATH_MAX, "http://%s%s", server, base_path) >= PATH_MAX)
    return NULL;

  thread_2ch->base_url = G_STRDUP(url);

  if (thread_2ch->kako_html != NULL)
    return thread_2ch->kako_html;
  return thread_2ch->base_url;
}


const char *
ochusha_thread_2ch_get_base_path(OchushaThread2ch *thread_2ch)
{
  OchushaThread2chClass *klass;
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread_2ch), NULL);

  if (thread_2ch->base_path != NULL)
    return thread_2ch->base_path;

  klass = OCHUSHA_THREAD_2CH_GET_CLASS(thread_2ch);
  if (klass->get_base_path != NULL)
    thread_2ch->base_path = (*klass->get_base_path)(thread_2ch);
  else
    {
      OchushaBBSThread *thread = OCHUSHA_BBS_THREAD(thread_2ch);
      char base_path[PATH_MAX];
      g_return_val_if_fail(thread->board != NULL && thread->id != NULL, NULL);

#if 1
      if (snprintf(base_path, PATH_MAX, "%stest/read.cgi/%s/%s/",
		   ochusha_bulletin_board_get_base_path(thread->board),
		   ochusha_bulletin_board_get_id(thread->board),
		   thread->id) < PATH_MAX)
	thread_2ch->base_path = G_STRDUP(base_path);
#else
      if (snprintf(base_path, PATH_MAX, "/test/read.cgi%s%s/%s/",
		   ochusha_bulletin_board_get_base_path(thread->board),
		   ochusha_bulletin_board_get_id(thread->board),
		   thread->id) < PATH_MAX)
	thread_2ch->base_path = G_STRDUP(base_path);
#endif
    }

  return thread_2ch->base_path;
}


static const char *
ochusha_thread_2ch_get_oyster_url(OchushaThread2ch *thread_2ch,
				  const char *sid)
{
  OchushaBulletinBoard *board;
  OchushaBBSThread *thread;
  char url_buffer[PATH_MAX];

  if (thread_2ch->oyster_url != NULL)
    return thread_2ch->oyster_url;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread_2ch), NULL);

  thread = OCHUSHA_BBS_THREAD(thread_2ch);
  board = ochusha_bbs_thread_get_board(thread);
  g_return_val_if_fail(board->bbs_type == OCHUSHA_BBS_TYPE_2CH, NULL);

  snprintf(url_buffer, PATH_MAX,
	   "http://%s/test/offlaw.cgi/%s/%s/?raw=0.0&sid=%s",
	   ochusha_bulletin_board_get_server(board),
	   ochusha_bulletin_board_get_id(board),
	   ochusha_bbs_thread_get_id(thread),
	   sid);
  thread_2ch->oyster_url = G_STRDUP(url_buffer);
  return thread_2ch->oyster_url;
}


typedef struct _ProcessOfflawResultJobArgs
{
  OchushaAsyncBuffer *dat_buffer;	/* ʪDATǡǼ */
  OchushaAsyncBuffer *cgi_buffer;	/* offlaw.cgiη̤Ǽ */
  OchushaBBSThread *thread;
  OchushaNetworkBroker *broker;
} ProcessOfflawResultJobArgs;


static void
process_offlaw_result(WorkerThread *employee, ProcessOfflawResultJobArgs *args)
{
  OchushaAsyncBuffer *dat_buffer = args->dat_buffer;
  OchushaAsyncBuffer *cgi_buffer = args->cgi_buffer;
  OchushaBBSThread *thread = args->thread;
  OchushaNetworkBroker *broker = args->broker;
  size_t offset = 0;
  char *cur_pos;
  char *eol_pos;
  char *title_pos;
  int fd;
  const char *dat_url;
  gchar message[4096];
  OchushaBBSResponse response = { NULL, NULL, NULL, NULL };

  if (!ochusha_async_buffer_active_ref(cgi_buffer))
    {
      goto terminated;
    }

  if (!ochusha_async_buffer_active_ref(dat_buffer))
    {
      ochusha_async_buffer_active_unref(cgi_buffer);
      goto terminated;
    }

  ochusha_async_buffer_lock(cgi_buffer);
  /* 1ܤɤǡofflaw.cgiη̤Ĵ٤롣 */
  while ((eol_pos = memchr((const char *)cgi_buffer->buffer, '\n',
			   cgi_buffer->length))
	 == NULL)
    if (cgi_buffer->fixed || !ochusha_async_buffer_wait(cgi_buffer))
      {
#if DEBUG_ASYNC_BUFFER_MOST
	fprintf(stderr, "process_offlaw_result(): buffer has been terminated.\n");
#endif
	ochusha_async_buffer_unlock(cgi_buffer);
	ochusha_async_buffer_active_unref(cgi_buffer);
	ochusha_async_buffer_active_unref(dat_buffer);
	goto terminated;
      }

  /* ޤ褿顢offlaw.cgiη̤1ܤɤ߹ߺ */
  if (cgi_buffer->length >= 3 && strncmp((const char *)cgi_buffer->buffer,
					 "+OK", 3) == 0)
    {
      /* eol_pos + 1ʹߤDATե */
      cur_pos = eol_pos + 1;
      offset = cur_pos - cgi_buffer->buffer;
      eol_pos = NULL;
    }
  else
    {
      char *tmp_pos;
      *eol_pos = '\0';
      eol_pos = NULL;
      /* 顼 */
      /* (HTML)ҸˤäƤ뤫åIDδڤΤߡ */
      tmp_pos = strstr((const char *)cgi_buffer->buffer, "kako/");
      if (tmp_pos != NULL)
	{
	  /* (HTML)ҸˤäƤ */
	  OchushaAsyncBuffer *old_buffer = cgi_buffer;
	  char kako_url[PATH_MAX];
	  snprintf(kako_url, PATH_MAX, "%s%s",
		   ochusha_bulletin_board_get_base_url(thread->board),
		   tmp_pos);
	  /* cgi_bufferkako_url
	   * ɤ߹ޤǡΤbuffer֤
	   */
	  cgi_buffer = ochusha_network_broker_read_from_url(broker, NULL,
					kako_url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_IGNORE,
					FALSE,
					broker->config->thread_chunksize);
	  if (cgi_buffer == NULL)
	    {
	      /*  */
	      cgi_buffer = old_buffer;
	      ochusha_async_buffer_unlock(cgi_buffer);
	      ochusha_async_buffer_active_unref(cgi_buffer);
	      ochusha_async_buffer_active_unref(dat_buffer);
	      goto terminated;
	    }
	  ochusha_async_buffer_unlock(old_buffer);
	  ochusha_async_buffer_active_unref(old_buffer);
	  OCHU_OBJECT_UNREF(old_buffer);

	  if (!ochusha_async_buffer_active_ref(cgi_buffer))
	    {
	      ochusha_async_buffer_active_unref(dat_buffer);
	      goto terminated;
	    }
	  ochusha_async_buffer_lock(cgi_buffer);

	  g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_started",
			G_CALLBACK(ochusha_async_buffer_emit_access_started),
			dat_buffer);
	  g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_progressed",
			G_CALLBACK(ochusha_async_buffer_emit_access_progressed),
			dat_buffer);
	  g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_finished",
			G_CALLBACK(ochusha_async_buffer_emit_access_finished),
			dat_buffer);
	  g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_failed",
			G_CALLBACK(ochusha_async_buffer_emit_access_failed),
			dat_buffer);

	  cur_pos = NULL;
	  /* HTMLѲѤthread򹹿 */
	  tmp_pos = g_strrstr(kako_url, ".dat");
	  if (tmp_pos != NULL)
	    {
	      *tmp_pos = '\0';
	      g_strlcat(kako_url, ".html", PATH_MAX);
	      ochusha_thread_2ch_set_kako_html_url(OCHUSHA_THREAD_2CH(thread),
						   kako_url);
	    }
#if 0
	  else
	    {
	      /* ä֤뤷ʤ */
	    }
#endif
	}
      else
	{
	  /*
	   * Υ顼
	   * İžؤбԽʬʤΤǡžΥФ¸ߤ
	   * ɤ⤦ȤȤ⤳롣
	   * åIDδڤξloginȤľ
	   * ٤žΥФۤ٤ġĤȤ
	   * ĤȤϤΤ0.5ϤǤ֡
	   */
	  OchushaAsyncBuffer *old_buffer = cgi_buffer;
	  OchushaThread2ch *thread_2ch = OCHUSHA_THREAD_2CH(thread);
	  const char *dat_url;
	  char *old_url;
	  const char *encoding
	    = ochusha_bbs_thread_get_response_character_encoding(thread);
	  iconv_helper *helper
	    = ochusha_bbs_thread_get_response_iconv_helper(thread);
	  iconv_t converter = iconv_open(helper ? "UTF-8" : "UTF-8//IGNORE",
					 encoding);
	  if (converter != (iconv_t)-1)
	    {
	      gchar *result = convert_string(converter, helper,
					     (const char *)cgi_buffer->buffer,
					     -1);

	      snprintf(message, 4096, _("Error by offlaw.cgi: %s\n"), result);
	      ochusha_network_broker_output_log(broker, message);
	      G_FREE(result);
	      iconv_close(converter);
	    }
	  else
	    ochusha_network_broker_output_log(broker,
					      _("Error by offlaw.cgi\n"));

	  dat_url = ochusha_thread_2ch_get_dat_url(thread_2ch);
	  cgi_buffer = ochusha_network_broker_read_from_url(broker, NULL,
					dat_url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_ONLY,
					FALSE,
					broker->config->thread_chunksize);
	  if (cgi_buffer == NULL)
	    {
	      /*  */
	      cgi_buffer = old_buffer;
	      ochusha_async_buffer_unlock(cgi_buffer);
	      ochusha_async_buffer_active_unref(cgi_buffer);
	      ochusha_async_buffer_active_unref(dat_buffer);

	      old_url = thread_2ch->oyster_url;
	      thread_2ch->oyster_url = NULL;
	      G_FREE(old_url);

	      goto terminated;
	    }
	  ochusha_async_buffer_unlock(old_buffer);
	  ochusha_async_buffer_active_unref(old_buffer);
	  OCHU_OBJECT_UNREF(old_buffer);

	  if (!ochusha_async_buffer_active_ref(cgi_buffer))
	    {
	      ochusha_async_buffer_active_unref(dat_buffer);
	      goto terminated;
	    }
	  ochusha_async_buffer_lock(cgi_buffer);

	  g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_started",
			G_CALLBACK(ochusha_async_buffer_emit_access_started),
			dat_buffer);
	  g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_progressed",
			G_CALLBACK(ochusha_async_buffer_emit_access_progressed),
			dat_buffer);
	  g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_finished",
			G_CALLBACK(ochusha_async_buffer_emit_access_finished),
			dat_buffer);
	  g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_failed",
			G_CALLBACK(ochusha_async_buffer_emit_access_failed),
			dat_buffer);

	  cur_pos = NULL;
	  old_url = thread_2ch->oyster_url;
	  thread_2ch->oyster_url = NULL;
	  G_FREE(old_url);
	}
    }

  /* DATǡ饹쥿ơOchushaBBSThreadtitle򹹿 */
  while ((eol_pos = memchr((const char *)cgi_buffer->buffer + offset,
			   '\n', cgi_buffer->length)) == NULL)
    if (cgi_buffer->fixed || !ochusha_async_buffer_wait(cgi_buffer))
      {
#if DEBUG_ASYNC_BUFFER_MOST
	fprintf(stderr, "process_offlaw_result(): buffer has been terminated.\n");
#endif
	ochusha_async_buffer_unlock(cgi_buffer);
	ochusha_async_buffer_active_unref(cgi_buffer);
	ochusha_async_buffer_active_unref(dat_buffer);
	goto terminated;
      }
  cur_pos = (char *)cgi_buffer->buffer + offset;

  title_pos = extract_2ch_response(cur_pos, eol_pos, &response);
  if (title_pos != NULL)
    {
      const char *encoding
	= ochusha_bbs_thread_get_response_character_encoding(thread);
      iconv_helper *helper
	= ochusha_bbs_thread_get_response_iconv_helper(thread);
      iconv_t converter = iconv_open(helper ? "UTF-8" : "UTF-8//IGNORE",
				     encoding);
      if (converter != (iconv_t)-1)
	{
	  gchar *title = simple_string_canon(title_pos, eol_pos - title_pos,
					     converter, helper);
	  ochusha_bbs_thread_set_title(thread, title);
	  G_FREE(title);
	  iconv_close(converter);
	}
    }

  /* DATǡΤҤdat_buffer˥ԡ */
  while (TRUE)
    {
      if (cgi_buffer->length > offset)
	{
	  ochusha_async_buffer_append_data(dat_buffer,
					   (const char *)(cgi_buffer->buffer
							  + offset),
					   cgi_buffer->length - offset);
	  offset = cgi_buffer->length;
	}

      if (cgi_buffer->fixed)
	break;

      if (!ochusha_async_buffer_wait(cgi_buffer))
	{
#if DEBUG_ASYNC_BUFFER_MOST
	  fprintf(stderr, "process_offlaw_result(): buffer has been terminated.\n");
#endif
	  ochusha_async_buffer_unlock(cgi_buffer);
	  ochusha_async_buffer_active_unref(cgi_buffer);
	  ochusha_async_buffer_active_unref(dat_buffer);
	  goto terminated;
	}
    }
  ochusha_async_buffer_unlock(cgi_buffer);

  ochusha_async_buffer_active_unref(cgi_buffer);
  ochusha_async_buffer_active_unref(dat_buffer);

  ochusha_async_buffer_fix(dat_buffer);

  /* ХåեƤ򥭥å˽񤭹 */
  dat_url = ochusha_thread_2ch_get_dat_url(OCHUSHA_THREAD_2CH(thread));
  fd = ochusha_config_cache_open_file(broker->config, dat_url,
				      O_WRONLY | O_TRUNC | O_CREAT);
  if (fd >= 0)
    {
      ssize_t len = write(fd, (void *)dat_buffer->buffer, dat_buffer->length);
      close(fd);
      if (len == dat_buffer->length)
	{
	  /* ｪλ */
	  if ((thread->flags & OCHUSHA_BBS_THREAD_KAKO) == 0)
	    thread->flags |= OCHUSHA_BBS_THREAD_DAT_OYSTER;
	}
      else
	{
	  ochusha_config_cache_unlink_file(broker->config, dat_url);
	  fd = -1;
	}
#if DEBUG_NETWORK_MOST
      if (fd < 0)
	fprintf(stderr, "Couldn't write cache file for %s: %s (%d)\n",
		dat_url, strerror(errno), errno);
#endif	
    }

 terminated:
  OCHU_OBJECT_UNREF(cgi_buffer);
  OCHU_OBJECT_UNREF(dat_buffer);
  OCHU_OBJECT_UNREF(broker);
}


static void
buffer_status_free(OchushaNetworkBrokerBufferStatus *status)
{
  if (status->last_modified != NULL)
    G_FREE(status->last_modified);
  if (status->date != NULL)
    G_FREE(status->date);
  G_FREE(status);
}


static OchushaAsyncBuffer *
ochusha_thread_2ch_read_dat_by_offlaw(OchushaThread2ch *thread_2ch,
				      OchushaNetworkBroker *broker,
				      OchushaAsyncBuffer *buffer)
{
  OchushaAsyncBuffer *dat_buffer = buffer;
  OchushaAsyncBuffer *cgi_buffer;
  OchushaNetworkBrokerBufferStatus *status;
  const char *url
    = ochusha_thread_2ch_get_oyster_url(thread_2ch,
					broker->config->session_id_2ch);

  if (dat_buffer != NULL && !ochusha_async_buffer_reset(dat_buffer))
    {
      OCHU_OBJECT_UNREF(dat_buffer);
      dat_buffer = NULL;
    }

  if (dat_buffer == NULL)
    dat_buffer = ochusha_async_buffer_new(NULL, 0, NULL);

  status = G_NEW0(OchushaNetworkBrokerBufferStatus, 1);
  status->state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY;
  g_object_set_data_full(G_OBJECT(dat_buffer),
		"OchushaNetworkBroker::BufferStatus", status,
		(GDestroyNotify)buffer_status_free);

  cgi_buffer = ochusha_network_broker_read_from_url(broker, NULL, url, NULL,
					OCHUSHA_NETWORK_BROKER_CACHE_IGNORE,
					FALSE,
					broker->config->thread_chunksize);
  if (cgi_buffer != NULL)
    {
      /* åɵư */
      WorkerJob *job = G_NEW0(WorkerJob, 1);
      ProcessOfflawResultJobArgs *args = G_NEW0(ProcessOfflawResultJobArgs, 1);

      g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_started",
			G_CALLBACK(ochusha_async_buffer_emit_access_started),
			dat_buffer);
      g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_progressed",
			G_CALLBACK(ochusha_async_buffer_emit_access_progressed),
			dat_buffer);
      g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_finished",
			G_CALLBACK(ochusha_async_buffer_emit_access_finished),
			dat_buffer);
      g_signal_connect_swapped(G_OBJECT(cgi_buffer), "access_failed",
			G_CALLBACK(ochusha_async_buffer_emit_access_failed),
			dat_buffer);

      args->dat_buffer = dat_buffer;
      args->cgi_buffer = cgi_buffer;
      args->thread = OCHUSHA_BBS_THREAD(thread_2ch);
      args->broker = broker;

      job->canceled = FALSE;
      job->job = (JobFunc *)process_offlaw_result;
      job->args = args;

      OCHU_OBJECT_REF(dat_buffer);
      OCHU_OBJECT_REF(broker);

      commit_job(job);
    }

  return dat_buffer;
}


static OchushaAsyncBuffer *
ochusha_thread_2ch_get_responses_source(OchushaBBSThread *thread,
					OchushaNetworkBroker *broker,
					OchushaAsyncBuffer *buffer,
					OchushaNetworkBrokerCacheMode mode)
{
  OchushaThread2ch *thread_2ch;
  const char *url;
  const char *cache_url = NULL;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread)
		       && OCHUSHA_IS_NETWORK_BROKER(broker), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  url = ochusha_thread_2ch_get_dat_url(thread_2ch);

  if (thread->board->bbs_type == OCHUSHA_BBS_TYPE_2CH_HEADLINE)
    mode = OCHUSHA_NETWORK_BROKER_CACHE_IGNORE;
  else if (thread->flags & OCHUSHA_BBS_THREAD_KAKO)
    mode = OCHUSHA_NETWORK_BROKER_CACHE_AS_IS;
  else if (thread->flags & OCHUSHA_BBS_THREAD_DAT_DROPPED)
    {
      if (thread->board->bbs_type == OCHUSHA_BBS_TYPE_2CH
	  && broker->config->login_2ch
	  && (thread->flags & OCHUSHA_BBS_THREAD_DAT_OYSTER) == 0)
	{
	  /* ɤ */
	  return ochusha_thread_2ch_read_dat_by_offlaw(thread_2ch, broker,
						       buffer);
	}
      else
	{
	  /*
	   * loginƤʤˡֲҸˤURLͽۤɤǤߤפä
	   * ΤϤϤݤʤΤǡ褺󤷡
	   */
	  mode = OCHUSHA_NETWORK_BROKER_CACHE_ONLY;
	}
    }
  else if ((thread->flags & OCHUSHA_BBS_THREAD_STOPPED)
	   && thread->number_of_responses_read <= thread->number_of_responses_on_server)
    mode = OCHUSHA_NETWORK_BROKER_CACHE_ONLY;

  return ochusha_network_broker_read_from_url_full(broker, buffer,
					url, cache_url,
					thread_2ch->last_modified,
					mode, FALSE,
					broker->config->thread_chunksize);
}


/*
 * extract_2ch_response
 *
 * cur_posʸ2chDATǡ1쥹ʬȸʤƲϤ
 * 1쥹ɽOchushaBBSResponse¤Τ˳Ǽ롣
 * δؿϡߤΰ(cur_pos)龯ʤȤ1ʬDATǡ
 * ¸ߤꤹΤǸƤӽФ¦դɬסʾ֤ǸƤӽФȡ
 * Хåեۤǡ˥뤳Ȥˤʤ롣
 *
 * 1쥹ʬΥǡ줿硢쥹ľΥǥߥ"<>"
 * ľʸؤݥ󥿤֤ԻˤNULL֤
 */
static char *
extract_2ch_response(char *start_pos, char *eol_pos,
		     OchushaBBSResponse *response)
{
  /*
   * DATեγƹԤ
   * ̾<>ᥤ<><>쥹<>[åɥȥ]
   * ȤʤäƤ롣
   */
  char *clone_buffer = NULL;
  char *cur_pos = start_pos;
  char *tmp_pos;
  gboolean comma_delimiter = FALSE;
  int i = 0;

  /* DATǡƤ뤳Ȥ⤢Τǥå */
  tmp_pos = memchr(start_pos, '\0', eol_pos - start_pos);
  if (tmp_pos != NULL)
    {
      int rest;
      int clone_len = eol_pos - start_pos + 1;
      /* 줿DATǡ */
      clone_buffer = G_MALLOC(clone_len);
      memcpy(clone_buffer, start_pos, clone_len);
      clone_buffer[clone_len - 1] = '\n';	/* פʤϤǰΤ */
      rest = clone_len;

      cur_pos = clone_buffer;
      eol_pos = clone_buffer + clone_len;
      /*
       * DATե1ʬ˴ޤޤƤ롢
       * ٤Ǥʤ'\0''*'֤롣
       */
      tmp_pos = cur_pos;
      while (*tmp_pos != '\n')
	{
	  if (*tmp_pos == '\0')
	    *tmp_pos = '*';
	  tmp_pos++;
	}

#if DEBUG_DAT
      clone_buffer[clone_len - 1] = '\0';
      fprintf(stderr, "broken line: %s\n", clone_buffer);
#endif
      clone_buffer[clone_len - 1] = '\n';
    }

  tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  while (tmp_pos != NULL)
    {
      i++;
      tmp_pos = g_strstr_len(tmp_pos + 2, eol_pos - tmp_pos - 2, "<>");
    }

  if (i == 0)
    goto error_exit;

  if (i == 1)
    {
      i = 0;
      tmp_pos = memchr(cur_pos, ',', eol_pos - cur_pos);
      while (tmp_pos != NULL)
	{
	  i++;
	  tmp_pos = memchr(tmp_pos + 1, ',', eol_pos - tmp_pos - 1);
	}
      if (i != 3)
	goto error_exit;	/* ȤΤʤ */
      comma_delimiter = TRUE;
    }

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  response->name = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos - cur_pos > 0)
    response->mailto = G_STRNDUP(cur_pos, tmp_pos - cur_pos);
  else
    response->mailto = NULL;

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);

  tmp_pos = comma_delimiter ? memchr(cur_pos, ',', eol_pos - cur_pos)
                            : g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  response->date_id = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  cur_pos = tmp_pos + (comma_delimiter ? 1 : 2);

  tmp_pos = g_strstr_len(cur_pos, eol_pos - cur_pos, "<>");
  if (tmp_pos == NULL)
    goto error_exit;

  response->content = G_STRNDUP(cur_pos, tmp_pos - cur_pos);

  if (clone_buffer != NULL)
    {
      G_FREE(clone_buffer);
      return start_pos + (tmp_pos - clone_buffer) + 2;
    }
  else
    return tmp_pos + 2;

 error_exit:
  if (response->name != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "name=%s\n", response->name);
#endif
      G_FREE((gpointer)response->name);
      response->name = NULL;
    }

  if (response->mailto != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "mailto=%s\n", response->mailto);
#endif
      G_FREE((gpointer)response->mailto);
      response->mailto = NULL;
    }

  if (response->date_id != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "date_id=%s\n", response->date_id);
#endif
      G_FREE((gpointer)response->date_id);
      response->date_id = NULL;
    }

  if (response->content != NULL)
    {
#if DEBUG_DAT
      fprintf(stderr, "content=%s\n", response->content);
#endif
      G_FREE((gpointer)response->content);
      response->content = NULL;
    }

  if (clone_buffer != NULL)
    G_FREE(clone_buffer);

  return NULL;
}


static gboolean
advance_parsing(gpointer data)
{
  g_object_set_qdata(G_OBJECT(data), idle_checker_id, NULL);
  ochusha_async_buffer_broadcast((OchushaAsyncBuffer *)data);
  OCHU_OBJECT_UNREF(data);
  return FALSE;
}


/*
 * ochusha_thread_2ch_parse_responses
 *
 * bufferƤåɤDATǡǤȸʤƲϤåɤ
 * ɽOchushaThread2ch¤Τˤ륹åɤʬ(responses)ۤ롣
 *
 * åɤγϻåɥȥˤstart_thread_cb
 * ġΥ쥹Фeach_response_cbåΤβϽλˤ
 * end_thread_cb򤽤줾ƤӽФΥХåFALSE
 * 顢λǲϤŪˤ롣
 *
 * start+1ܤnumberʬΥ쥹ϤġĤȤؿˤͽꡣ
 * åã硢number˴ؤ餺Ϥλ롣
 *
 * number == -1ʤ顢start饹åޤǡ
 * start == -1ʤ顢ånumberʬΥ쥹
 *
 * Ĥޤꡢstart=0,number=-1ǡ쥹ϡ
 *         start=-1,number=100Ǻǿ100쥹Ȥä
 * 褺start=-1ξ̤Τޤ֢ʤͽ
 */
static gboolean
ochusha_thread_2ch_parse_responses(OchushaBBSThread *thread,
				   OchushaAsyncBuffer *buffer,
				   int start, int number, gboolean no_wait,
				   StartThreadCallback *start_thread_cb,
				   EachResponseCallback *each_response_cb,
				   BrokenResponseCallback *broken_response_cb,
				   EndThreadCallback *end_thread_cb,
				   StartParsingCallback *start_parsing_cb,
				   BeforeWaitCallback *before_wait_cb,
				   AfterWaitCallback *after_wait_cb,
				   EndParsingCallback *end_parsing_cb,
				   gpointer callback_data)
{
  OchushaThread2ch *thread_2ch;
  unsigned int *responses;
  gboolean result = TRUE;
  OchushaNetworkBrokerBufferStatus *status;
  OchushaNetworkBrokerBufferStatus fake_status;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread)
		       && OCHUSHA_IS_ASYNC_BUFFER(buffer), FALSE);

  status = g_object_get_data(G_OBJECT(buffer),
			     "OchushaNetworkBroker::BufferStatus");

  if (status == NULL)
    {
      fake_status.state = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_HIT;
      fake_status.status_code = 304;
      fake_status.last_modified = NULL;
      fake_status.date = NULL;
      status = &fake_status;
    }

  if (!ochusha_async_buffer_active_ref(buffer))
    {
#if DEBUG_ASYNC_BUFFER_MOST
      fprintf(stderr, "buffer has been terminated.\n");
#endif
      return FALSE;
    }

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  responses = thread_2ch->responses;

  if (responses == NULL)
    {
      /*
       * 1000쥹ʬǤ⡹4000ХȤʤΤǡ
       * ǽ餫MAX_RESPONSE쥹ʬ϶Ȥ롣
       */
      responses = G_MALLOC0(sizeof(unsigned int) * MAX_RESPONSE);
      thread_2ch->responses = responses;
      thread_2ch->responses_length = MAX_RESPONSE;
    }

  if (number < -1)
    number = thread_2ch->responses_length;

  ochusha_async_buffer_lock(buffer);
  {
    int rest_of_responses;
    unsigned int offset;
    int i;
    gboolean buffer_fixed = FALSE;

    if (start_parsing_cb != NULL)
      (*start_parsing_cb)(callback_data);

  retry_analysis:
    rest_of_responses = number;
    offset = 0;
    i = 0;
    if (status->state == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
      {
	memset(thread_2ch->responses, 0,
	       thread_2ch->responses_length * sizeof(unsigned int));
      }
    else
      {
	if (start > 1 && start < thread_2ch->responses_length)
	  {
	    offset = responses[start];
	    if (offset != 0)
	      i = start;
	    else
	      {
		/*
		 * nʬΥ쥹ɤ߹n+1ܰʹߤɤ
		 * ġĤȤΤ̰ƤХʥꥵ
		 * ȤơǤᤤϺʬõȤǤ뤬
		 * ޤǤɬפʤ
		 */
		offset = responses[start - 1];
		if (offset != 0)
		  i = start - 1;
		else
		  offset = 0;
	      }
	  }
      }

    while (result)
      {
	char *buffer_top = (char *)buffer->buffer;
	char *cur_pos = buffer_top + offset;
	int rest_of_data = buffer->length - offset;
	char *eol_pos = NULL;
	int interval = ANALYSIS_INTERVAL;

#if DEBUG_DAT_MOST
	fprintf(stderr, "1: ochusha_thread_2ch.c: offset=%d, rest_of_data=%d, length=%d\n",
		offset, rest_of_data, buffer->length);
#endif
	while (rest_of_data > 0
	       && (rest_of_responses > 0 || rest_of_responses == -1)
	       && interval-- > 0
	       && (eol_pos = memchr(cur_pos, '\n', rest_of_data)) != NULL)
	  {
	    OchushaBBSResponse response = { NULL, NULL, NULL, NULL };
	    char *title_pos;

	    if (i >= thread_2ch->responses_length)
	      {
		unsigned int new_len = thread_2ch->responses_length * 2;
		thread_2ch->responses
		  = G_REALLOC(thread_2ch->responses,
			      new_len * sizeof(unsigned int));
		memset(thread_2ch->responses + thread_2ch->responses_length, 0,
		       (new_len - thread_2ch->responses_length)
		       * sizeof(unsigned int));
		thread_2ch->responses_length = new_len;
		responses = thread_2ch->responses;
	      }
#if DEBUG_DAT_MOST
	    fprintf(stderr, "i=%d, offset=%d, rest_of_data=%d, cur_pos=%p\n",
		    i, offset, rest_of_data, cur_pos);
#endif

	    title_pos = extract_2ch_response(cur_pos, eol_pos, &response);

#if DEBUG_DAT_MOST
	    fprintf(stderr, "title_pos=%p\n", title_pos);
#endif

	    if (i == 0 && start == 0 && title_pos != NULL)
	      {
		char *title = G_STRNDUP(title_pos, eol_pos - title_pos);
		if (start_thread_cb != NULL)
		  result = (*start_thread_cb)(thread, title, callback_data);
		G_FREE(title);
		if (!result)
		  goto terminated;
	      }
	    responses[i] = offset;

	    i++;
	    if (i > start)
	      {
		result = TRUE;
		if (title_pos != NULL)
		  {
		    if (each_response_cb != NULL)
		      result = (*each_response_cb)(thread, i, &response,
						   callback_data);
#if 0
		    /*
		     * Ƴ()ߤʤΤǡᡣ
		     */
		    if (!(thread->flags & OCHUSHA_BBS_THREAD_STOPPED)
			&& response.date_id != NULL)
		      {
			/*
			 * "\222\342\216\176"""Shift_JISɽ
			 */
			if (strcmp(response.date_id, "\222\342\216\176") == 0
			    || (i > 1000
				&& strcmp(response.date_id,
					  "Over 1000 Thread") == 0))
			  thread->flags |= OCHUSHA_BBS_THREAD_STOPPED;
		      }
#else
		    if (i > 1000
			&& !(thread->flags & OCHUSHA_BBS_THREAD_STOPPED)
			&& response.date_id != NULL
			&& strcmp(response.date_id, "Over 1000 Thread") == 0)
		      {
			thread->flags |= OCHUSHA_BBS_THREAD_STOPPED;
		      }
#endif
		  }
		else
		  {
		    if (broken_response_cb != NULL)
		      result = (*broken_response_cb)(thread, i, callback_data);
		  }
		if (rest_of_responses != -1)
		  rest_of_responses--;
		if (!result)
		  goto terminated;
	      }

	    offset = (eol_pos + 1) - buffer_top;
	    buffer_top = (char *)buffer->buffer;
	    cur_pos = buffer_top + offset;
	    rest_of_data = buffer->length - offset;

#if DEBUG_DAT
	    if (title_pos == NULL)
	      {
		char *line;
		fprintf(stderr, "response #%d: Couldn't analyze DAT file...ignoring parts.\n", i);
		line = G_STRNDUP(buffer_top + responses[i -1],
				 offset - responses[i - 1]);
		fprintf(stderr, "failed: \"%s\"\n", line);
		G_FREE(line);
		fprintf(stderr, "offset=%d\n", responses[i - 1]);
	      }
#endif

	    if (response.name != NULL)
	      G_FREE((gpointer)response.name);
	    if (response.mailto != NULL)
	      G_FREE((gpointer)response.mailto);
	    if (response.date_id != NULL)
	      G_FREE((gpointer)response.date_id);
	    if (response.content != NULL)
	      G_FREE((gpointer)response.content);

	    eol_pos = NULL;
	  }

	if (!buffer_fixed && buffer->fixed)
	  {
	    buffer_fixed = TRUE;
	    eol_pos = NULL;
	    continue;
	  }

	if (buffer_fixed && eol_pos == NULL && interval > 0)
	  goto terminated;

	if (interval > 0)
	  {
	    /*
	     * ɤ٤ǡʤ硣
	     */
	    unsigned int old_len = buffer->length;
	    if (no_wait)
	      goto terminated;	/* ξ */

	    if (before_wait_cb != NULL)
	      (*before_wait_cb)(callback_data);
	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  {
#if DEBUG_ASYNC_BUFFER_MOST
		    fprintf(stderr, "ochusha_thread_2ch_parse_responses(): buffer has been terminated.\n");
#endif
		    if (after_wait_cb != NULL)
		      (*after_wait_cb)(callback_data);
		    goto terminated;
		  }
	      } while (buffer->length == old_len && !buffer->fixed);
	    if (after_wait_cb != NULL)
	      (*after_wait_cb)(callback_data);
	  }
	else
	  {
	    /*
	     * GUIΥ쥹ݥ󥹤βܻؤ¸
	     */
	    OCHU_OBJECT_REF(buffer);
	    g_object_set_qdata(G_OBJECT(buffer), idle_checker_id, buffer);
	    g_idle_add_full(G_PRIORITY_HIGH_IDLE + 15,
			    advance_parsing, buffer, NULL);
	    if (before_wait_cb != NULL)
	      (*before_wait_cb)(callback_data);
	    do
	      {
		if (!ochusha_async_buffer_wait(buffer))
		  {
		    if (after_wait_cb != NULL)
		      (*after_wait_cb)(callback_data);
		    goto terminated;
		  }
	      } while (g_object_get_qdata(G_OBJECT(buffer), idle_checker_id)
		       != NULL);
	    if (after_wait_cb != NULL)
	      (*after_wait_cb)(callback_data);
	    interval = ANALYSIS_INTERVAL;
	  }

	if (status->state
	    == OCHUSHA_NETWORK_BROKER_BUFFER_STATE_CACHE_IS_DIRTY)
	  {
	    if (i > start && end_thread_cb != NULL)
	      result = (*end_thread_cb)(thread, FALSE, callback_data);
	    /* å夬äƤ뤳ȤϤ⤦狼ä */
	    status->state
	      = OCHUSHA_NETWORK_BROKER_BUFFER_STATE_DIRECT_ACCESS;
	    memset(thread_2ch->responses, 0,
		   thread_2ch->responses_length * sizeof(unsigned int));
	    goto retry_analysis;
	  }
      }

  terminated:
    if (thread->number_of_responses_read < i || number == -1)
      thread->number_of_responses_read = i;
  }
  ochusha_async_buffer_unlock(buffer);

  if (!(thread->flags
	& (OCHUSHA_BBS_THREAD_DAT_DROPPED | OCHUSHA_BBS_THREAD_KAKO)))
    {
      if (status->last_modified != NULL)
	{
	  if (thread_2ch->last_modified != NULL)
	    G_FREE(thread_2ch->last_modified);
	  thread_2ch->last_modified = G_STRDUP(status->last_modified);
	}

      if (status->date != NULL)
	{
	  if (thread_2ch->date != NULL)
	    G_FREE(thread_2ch->date);
	  thread_2ch->date = G_STRDUP(status->date);
	}
    }

  ochusha_async_buffer_active_unref(buffer);

  if (end_thread_cb != NULL)
    result = (*end_thread_cb)(thread, TRUE, callback_data);

  if (end_parsing_cb != NULL)
    (*end_parsing_cb)(callback_data);

  if (thread->board->bbs_type == OCHUSHA_BBS_TYPE_2CH_HEADLINE)
    thread->number_of_responses_read = 0;

  return TRUE;
}


static time_t
ochusha_thread_2ch_get_last_modified_utc(OchushaBBSThread *thread)
{
  OchushaThread2ch *thread_2ch;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), 0);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);
  if (thread_2ch->last_modified == NULL)
    return 0;

  return ochusha_utils_get_utc_time(thread_2ch->last_modified);
}


static char *
ochusha_thread_2ch_get_url_for_response(OchushaBBSThread *thread, int from,
					int to)
{
  OchushaThread2ch *thread_2ch;
  char url[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  switch (thread->board->bbs_type)
    {
    case OCHUSHA_BBS_TYPE_2CH:
    case OCHUSHA_BBS_TYPE_2CH_COMPATIBLE:
    case OCHUSHA_BBS_TYPE_2CH_HEADLINE:
    case OCHUSHA_BBS_TYPE_2CHLIKE_EUCJP:
      {
	if (from >= to)
	  snprintf(url, PATH_MAX, "../test/read.cgi/%s/%s/%d",
		   ochusha_bulletin_board_get_id(thread->board),
		   thread->id, from);
	else
	  snprintf(url, PATH_MAX, "../test/read.cgi/%s/%s/%d-%d",
		   ochusha_bulletin_board_get_id(thread->board),
		   thread->id, from, to);
	break;
      }

    default:	/* ̤ݡ */
      return NULL;
    }

  return G_STRDUP(url);
}


static const char *
ochusha_thread_2ch_get_url_to_post(OchushaBBSThread *thread)
{
  OchushaThread2ch *thread_2ch;
  char url[PATH_MAX];

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), NULL);

  thread_2ch = OCHUSHA_THREAD_2CH(thread);
  if (thread_2ch->post_url != NULL)
    return thread_2ch->post_url;

  snprintf(url, PATH_MAX, "%sl10",
	   ochusha_thread_2ch_get_base_url(thread));
  thread_2ch->post_url = G_STRDUP(url);

  return thread_2ch->post_url;
}


static gboolean
ochusha_thread_2ch_preview_response(OchushaBBSThread *thread,
				    const OchushaBBSResponse *response,
				    StartThreadCallback *start_cb,
				    EachResponseCallback *response_cb,
				    EndThreadCallback *end_cb,
				    gpointer callback_data)
{
  return FALSE;
}


static gboolean
ochusha_thread_2ch_post_supported(OchushaBBSThread *thread)
{
  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), FALSE);
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(thread->board), FALSE);

  return thread->board->bbs_type != OCHUSHA_BBS_TYPE_2CH_HEADLINE;
}


static gboolean
ochusha_thread_2ch_post_response(OchushaBBSThread *thread,
				 OchushaNetworkBroker *broker,
				 const OchushaBBSResponse *response,
				 gboolean use_id)
{
  OchushaThread2ch *thread_2ch;
  OchushaThread2chClass *thread_2ch_class;
  OchushaBoard2ch *board_2ch;
  char *tmp_string = NULL;
  char *from = NULL;
  char *mail = NULL;
  char *message = NULL;
  const char *bbs;
  const char *key;
  char *query = NULL;
  long time;	/* time_tˤ٤10ʿʸؤѴ
		 * ԤˡʤΤǡ褺longѴƤޤ
		 * long32bitʥƥˤ2038ǯ()¸ߤ롣
		 */
  iconv_t converter;
  OchushaUtils2chPostResult post_result;

  g_return_val_if_fail(OCHUSHA_IS_THREAD_2CH(thread), FALSE);
  g_return_val_if_fail(OCHUSHA_IS_BOARD_2CH(thread->board), FALSE);
  g_return_val_if_fail(response->name != NULL, FALSE);
  g_return_val_if_fail(response->mailto != NULL, FALSE);
  g_return_val_if_fail(response->content != NULL, FALSE);

  converter = iconv_open(ochusha_bulletin_board_get_response_character_encoding(thread->board), "UTF-8");
  g_return_val_if_fail(converter != (iconv_t)-1, FALSE);

  board_2ch = OCHUSHA_BOARD_2CH(thread->board);
  thread_2ch = OCHUSHA_THREAD_2CH(thread);

  tmp_string = convert_string(converter, NULL, response->name, -1);
  from = ochusha_utils_url_encode_string(tmp_string);
  if (from == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->mailto, -1);
  mail = ochusha_utils_url_encode_string(tmp_string);
  if (mail == NULL)
    goto error_exit;
  G_FREE(tmp_string);

  tmp_string = convert_string(converter, NULL, response->content, -1);
  message = ochusha_utils_url_encode_string(tmp_string);
  if (message == NULL)
    goto error_exit;
  G_FREE(tmp_string);
  tmp_string = NULL;

  bbs = ochusha_bulletin_board_get_id(thread->board);
  key = thread->id;

  if (thread_2ch->date != NULL)
    time = ochusha_utils_get_utc_time(thread_2ch->date);
  else
    {
#if DEBUG_POST
      char message[4096];
      snprintf(message, 4096, "thread_2ch(%s)->date == NULL\n",
	       thread->title);
      ochusha_network_broker_output_log(broker, message);
#endif
#if DEBUG_POST_MOST
      fprintf(stderr, "thread_2ch->date == NULL!\n");
#endif
      time = ochusha_utils_get_utc_time(thread_2ch->last_modified);
    }

  if (time == -1)
    goto error_exit;

  thread_2ch_class = OCHUSHA_THREAD_2CH_GET_CLASS(thread_2ch);
  if (thread_2ch_class->make_post_response_message)
    query = (*thread_2ch_class->make_post_response_message)(thread_2ch,
							    from, mail,
							    message, bbs,
							    key, time);
  else
    {
      if (use_id && broker->config->login_2ch
	  && thread->board->bbs_type == OCHUSHA_BBS_TYPE_2CH)
	query = g_strdup_printf("submit=%%8F%%91%%82%%AB%%8D%%9E%%82%%DE&FROM=%s&mail=%s&MESSAGE=%s&bbs=%s&key=%s&sid=%s&time=%ld", from, mail, message, bbs, key, broker->config->session_id_2ch, time);
      else
	query = g_strdup_printf("submit=%%8F%%91%%82%%AB%%8D%%9E%%82%%DE&FROM=%s&mail=%s&MESSAGE=%s&bbs=%s&key=%s&time=%ld", from, mail, message, bbs, key, time);
    }

  if (query == NULL)
    goto error_exit;

  post_result = ochusha_utils_2ch_try_post(broker, thread->board, query);
  if (post_result == OCHUSHA_UTILS_2CH_POST_NO_COOKIE)
    {
      if (ochusha_board_2ch_get_cookie(board_2ch) != NULL)
	{
	  if (thread->board->bbs_type == OCHUSHA_BBS_TYPE_2CH)
	    {
	      g_free(query);
	      if (use_id && broker->config->login_2ch)
		query = g_strdup_printf("submit=%%91%%53%%90%%D3%%94%%43%%82%%F0%%95%%89%%82%%A4%%82%%B1%%82%%C6%%82%%F0%%8F%%B3%%91%%F8%%82%%B5%%82%%C4%%8F%%91%%82%%AB%%8D%%9E%%82%%DE&FROM=%s&mail=%s&MESSAGE=%s&bbs=%s&key=%s&sid=%s&time=%ld", from, mail, message, bbs, key, broker->config->session_id_2ch, time);
	      else
		query = g_strdup_printf("submit=%%91%%53%%90%%D3%%94%%43%%82%%F0%%95%%89%%82%%A4%%82%%B1%%82%%C6%%82%%F0%%8F%%B3%%91%%F8%%82%%B5%%82%%C4%%8F%%91%%82%%AB%%8D%%9E%%82%%DE&FROM=%s&mail=%s&MESSAGE=%s&bbs=%s&key=%s&time=%ld", from, mail, message, bbs, key, time);
	    }
	  sleep(10);
	  post_result = ochusha_utils_2ch_try_post(broker, thread->board,
						   query);
	}
      else
	{
	  ochusha_network_broker_output_log(broker, "No Cookie!\n");
	}
    }

#if 0
  if (post_result == OCHUSHA_UTILS_2CH_POST_MOCHITSUKE)
    {
      /* XXX: äݤΤȤͭʤΤ֡ */
      sleep(30);
      post_result = ochusha_utils_2ch_try_post(broker, thread->board, query);
    }
#endif

  if (post_result != OCHUSHA_UTILS_2CH_POST_SUCCESS)
    goto error_exit;

  /* ｪλ */
  if (tmp_string != NULL)
    G_FREE(tmp_string);
  if (from != NULL)
    G_FREE(from);
  if (mail != NULL)
    G_FREE(mail);
  if (message != NULL)
    G_FREE(message);
  if (query != NULL)
    g_free(query);
  iconv_close(converter);

  return TRUE;

 error_exit:
#if DEBUG_POST_MOST
  fprintf(stderr, "Error happen\n");
#endif

  if (tmp_string != NULL)
    G_FREE(tmp_string);
  if (from != NULL)
    G_FREE(from);
  if (mail != NULL)
    G_FREE(mail);
  if (message != NULL)
    G_FREE(message);
  if (query != NULL)
    g_free(query);
  iconv_close(converter);

  return FALSE;
}
