/*
 * Oaks
 * Copyright (c) 2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に
 * 同意できる場合にのみ利用可能です。
 */
package oaks.analysis;
import java.util.*;

/**
 * SQL情報。
 */
public class SqlInformation {
	private static String CR = System.getProperty("line.separator");

	/** 発行SQL。*/
	public String SQL;
	/** バインド値。バインド値が無い場合は0個の配列です。 */
	public Object[] BIND;
	/** SQL実行開始時刻。 */
	public Date START;
	/** SQL終了時刻。
	 * SELECT文の場合、フェッチ時間は含みません。
	 */
	public Date END;
	/** 発生例外。無ければnull。 */
	public java.sql.SQLException	EXCEPTION = null;
	/** 呼び出しスレッド。 */
	public Thread THREAD;
	/** 呼び出しスタックトレース。 */
	public StackTraceElement[] STACK;

	/**
	 * コンストラクタ。
	 * @param sql 発行SQL。
	 * @param obj バインド値。
	 */
	public SqlInformation( String sql, Object ... obj ) {
		START = END = new Date();
		SQL = sql;
		BIND = obj;
		if ( BIND == null )	BIND = new Object[0];
		THREAD = Thread.currentThread();

		ArrayList<StackTraceElement>	list = new ArrayList<StackTraceElement>();
		boolean flag = false;
		for ( StackTraceElement st: THREAD.getStackTrace() ) {
			if ( !flag ) {
				String	name = st.getClassName();
				if ( "java.lang.Thread".equals( name ) )	continue;
				if ( name.length() > 5 ) {
					name = name.substring( 0, 5 );
					if ( !"oaks.".equals( name ) )	flag = true;
				}
			}
			if ( flag )	list.add( st );
		}
		STACK = list.toArray( new StackTraceElement[0] );
	}

	/**
	 * SQL終了登録。
	 * @param ex 発生例外。
	 */
	public void close( java.sql.SQLException ex ) {
		if ( EXCEPTION != null )	return;
		END = new Date();
		EXCEPTION = ex;
	}

	/**
	 * 実行時間を返す。SQL終了時刻とSQL実行開始時刻の差をミリ秒で返します。
	 * @return 実行時間。ミリ秒。
	 */
	public long getExecMsec() {
		return END.getTime() - START.getTime();
	}

	/**
	 * 文字列化。全ての情報を文字列にして返します。
	 * @return 全情報。
	 */
	public String toString() {
		StringBuilder	buf = new StringBuilder( "-- SqlInformation ---------------------------------------------------" );
		buf = buf.append( CR );
		buf = buf.append( START.toString() ).append( " ~ " ).append( END.toString() ).append( "  [" ).append( getExecMsec() ).append( "ms]" ).append( CR );
		buf = buf.append( SQL ).append( CR );
		if ( BIND.length > 0 ) {
			buf = buf.append( "<BIND>" ).append( CR );
			for( Object o : BIND ) {
				if ( o == null ) {
					buf = buf.append( "NULL" );
				}
				else {
					buf = buf.append( o.getClass().toString() ).append( " / " ).append( o.toString() );
				}
				buf = buf.append( CR );
			}
		}
		buf = buf.append( "<" ).append( THREAD.toString() ).append( ">" ).append( CR );
		for ( StackTraceElement st: STACK ) {
			buf = buf.append( st.toString() ).append( CR );
		}
		if ( EXCEPTION != null ) {
			buf = buf.append( "<SqlException>" ).append( CR );
			buf = buf.append( EXCEPTION.toString() ).append( CR );
		}
		return buf.toString();
	}
}

