/*
 * Oaks
 * Copyright (c) 2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に
 * 同意できる場合にのみ利用可能です。
 */
package oaks;

import java.util.*;

/**
 * where句を表します。
 */
public class Where {
	/**
	 * 比較演算子。
	 */
	public enum RO {
		/** フィールド名 = 比較値 */
		EQ( "=" ),
		/** フィールド名 &lt;&gt; 比較値 */
		NE( "<>" ),
		/** フィールド名 &lt; 比較値 */
		LT( "<" ),
		/** フィールド名 &lt;= 比較値 */
		LE( "<=" ),
		/** フィールド名 &gt; 比較値 */
		GT( ">" ),
		/** フィールド名 &gt;= 比較値 */
		GE( ">=" );

		private String str;
		private RO( String s ) {
			str = s;
		}
	}

	/**
	 * 論理演算子。
	 */
	public enum LO {
		/** and */
		AND( " and " ),
		/** or */
		OR( " or " );

		private String str;
		private LO( String s ) {
			str = s;
		}
	}

	/**
	 * 文字列演算子。
	 */
	public enum STR {
		/** like */
		LIKE( " like " ),
		/** not like */
		NOT_LIKE( " not like " );

		private String str;
		private STR( String s ) {
			str = s;
		}
	}
	/**
	 * 文字列フィールド。
	 */
	public interface Likeable {}

	private RO	ope = null;
	private Field	fld = null;
	private Field.GRP	grp = null;

	/**
	 * AND 一致条件の生成。
	 * 指定された引数が Where.RO.EQ で、その全てが Where.LO.AND となる
	 * Where句を返します。
	 * @param field Where.RO.EQ となるフィールド。
	 * @return Where句。field が 0件の場合、nullを返します。
	 */
	public static Where makeAndEq( Field ... field ) {
		switch( field.length ) {
		case 0:
			return null;
		case 1:
			return new Where( Where.RO.EQ, field[0] );
		}
		ArrayList<Where>	list = new ArrayList<Where>();
		for ( Field f: field ) {
			list.add( new Where( Where.RO.EQ, f ) );
		}
		return new Where( Where.LO.AND, list.toArray( new Where[0] ) );
	}

	/**
	 * コンストラクタ。
	 * フィールド値を比較する式を作ります。
	 * @param relational 比較演算子。
	 * @param field フィールド。
	 */
	public Where( RO relational, Field field ) {
		ope = relational;
		fld = field;
	}

	/**
	 * Having句用コンストラクタ。
	 * フィールド集計値を比較する式を作ります。
	 * @param relational 比較演算子。
	 * @param field フィールド。
	 */
	public Where( RO relational, Field field, Field.GRP group ) {
		this( relational, field );
		grp = group;
	}

	private LO	logi = null;
	private Where[]	wh = null;

	/**
	 * コンストラクタ。
	 * 複数のWhereインスタンスを論理演算比較する式を作ります。
	 * @param logical 論理演算子。
	 * @param where where句。
	 */
	public Where( LO logical, Where ... where ) {
		logi = logical;
		ArrayList<Where>	list = new ArrayList<Where>();
		for ( Where w: where ) {
			if ( w != null )	list.add( w );
		}
		if ( list.size() == 0 )	wh = null;
		else	wh = list.toArray( new Where[0] );
	}

	private STR	str_cond = null;
	/**
	 * コンストラクタ。
	 * Like条件の式を作ります。
	 * @param str like演算子。
	 * @param field フィールド。
	 */
	@SuppressWarnings("unchecked")
	public Where( STR str, Likeable field ) {
		str_cond = str;
		fld = (Field)field;
	}

	/**
	 * SQL文字列変換。
	 * @return where句文字列。
	 */
	public String toString() {
		StringBuilder	buf = new StringBuilder();
		boolean	flag = false;
		if ( logi != null && wh != null ) {
			buf = buf.append( "(" );
			for ( int i = 0; i < wh.length; i++ ) {
				String	ret = wh[i].toString();
				if ( ret == null )	continue;
				if ( flag == false )	flag = true;
				else	buf = buf.append( logi.str );
				buf = buf.append( ret );
			}
			buf = buf.append( ")" );
		}
		else if ( fld != null ) {
			Object	obj = fld.getObject();
			if ( obj == Field.NO_CARE )	return null;
			buf = buf.append( grp == null?	fld.getName(): fld.toString( grp ) );
			if ( ope != null ) {
				flag = true;
				if ( obj != null ) {
					buf = buf.append( ope.str ).append( "?" );
				}
				else {
					buf = buf.append( ( ope == RO.EQ )?	" IS NULL":	" IS NOT NULL" );
				}
			}
			else if ( str_cond != null && obj != null ) {
				flag = true;
				buf = buf.append( str_cond.str ).append( "?" );
			}
		}
		return flag?	buf.toString(): null;
	}

	String getString( boolean where_f ) {
		StringBuilder	buf = new StringBuilder(
			where_f?	" where ":	" having " );
		String	ret = toString();
		if ( ret == null )	return "";
		return buf.append( ret ).toString();
	}

	private ArrayList<Object> getParam( ArrayList<Object> list ) {
		if ( wh != null ) {
			for ( int i = 0; i < wh.length; i++ ) {
				list = wh[i].getParam( list );
			}
			return list;
		}
		if ( fld == null )	return list;
		Object	obj = fld.getObject();
		if ( obj == null || obj == Field.NO_CARE )	return list;
		list.add( obj );
		return list;
	}

	Object[] getParam() {
		return getParam( new ArrayList<Object>() ).toArray( new Object[0] );
	}
}




