/*
 * Oaks
 * Copyright (c) 2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に
 * 同意できる場合にのみ利用可能です。
 */
package oaks;

import java.sql.*;
import java.util.*;


/**
 * ビューを表します。<br>
 * 派生クラスにはビューを参照するメソッドが準備されます。<br>
 * この派生クラスはメモリ確保動作が多くなるため、インスタンスはコンテナへ
 * 事前にキャッシュされ、コンテナからインスタンスを提供します。
 */
public abstract class View<E extends View> implements java.io.Serializable {
	private static final long serialVersionUID = 1;
	private String	name;

	protected Field[]	fields;

	/**
	 * 文字列化。
	 * @return 内容。
	 */
	public String toString() {
		StringBuilder	buf = new StringBuilder( name );
		buf = buf.append( "{" ).append( DB.LF );
		for ( Field f: fields ) {
			buf = buf.append( f.toString() ).append( DB.LF );
		}
		buf = buf.append( "}" );
		return buf.toString();
	}

	static volatile int max_view_limit = 200;
	/**
	 * 事前ロードインスタンス上限数の設定。初期値は 200 です。
	 * 全ての View, Table 派生インスタンス(以下「テーブルインスタンス」と
	 * 呼称します)に影響します。<br>
	 * テーブルインスタンスは、Farmクラスから取得に備え、予めインスタンスが
	 * 作成されています。<br>
	 * この時の、事前にメモリに作成されるインスタンス数の上限を設定します。<br>
	 * メモリに多くのインスタンスを準備した場合、レコード件数が多い
	 * SELECT のフェッチでのパフォーマンス向上に繋がりますが、その分、
	 * メモリ使用量が増加します。<br>
	 * 例えば、テーブル数が 100、この上限値が 200 ならば、最大 20000
	 * インスタンスがメモリ上に作成される事になります。<br>
	 * ただし、使用頻度の少ないテーブルは上限値未満だけ作成されますので、
	 * 最大値まで使い切ることは稀でしょう。<br>
	 * 処理速度とメモリ使用量とがトレードオフの関係となるため、
	 * 適切な値を指定して下さい。
	 * @param count 事前ロードインスタンス上限数。
	 */
	public static void setPreLoadViewLimitCount( int count ) {
		max_view_limit = count;
	}
	/**
	 * 事前ロードインスタンス上限数の取得。
	 * @return 事前ロードインスタンス上限数。
	 */
	public static int getPreLoadViewLimitCount() {
		return max_view_limit;
	}

	private static class ViewHolder {
		private View parent;
		private LinkedList<View> list = new LinkedList<View>();
		private volatile int max_count = 3;
		private volatile int once_count = 0;

		ViewHolder( View p ) {
			parent = p;
			for ( int i = 0; i < 5; i++ ) {
				list.add( p.getInstance() );
			}
		}

		View get() {
			View	v = null;
			synchronized( list ) {
				once_count++;
				v = list.poll();
				if ( v == null )	v = parent.getInstance();
			}
			return v;
		}

		void make() {
			if ( once_count > max_count ) {
				max_count = once_count;
			}
			int	cnt = 0;
			synchronized( list ) {
				once_count = 0;
				cnt = list.size();
			}
			if ( cnt >= View.max_view_limit )	return;
			cnt = (int)(max_count * 1.2) - cnt;
			for ( int i = 0; i < cnt; i++ ) {
				synchronized( list ) {
					list.add( parent.getInstance() );
				}
			}
		}
	}

	private static class ViewMake extends Thread {
		private HashMap<Class<? extends View>, ViewHolder>	map;
		private ViewMake( HashMap<Class<? extends View>, ViewHolder> m ) {
			map = m;
			setDaemon( true );
			setPriority( MIN_PRIORITY );
			start();
		}
		public void run() {
			while( true ) {
				try {
					sleep( 1000 );
					ArrayList<ViewHolder>	list = new ArrayList<ViewHolder>();
					synchronized( map ) {
						for ( Class<?> key: map.keySet() ) {
							list.add( map.get( key ) );
						}
					}
					for ( ViewHolder vh: list ) {
						try {
							vh.make();
						}
						catch( Exception ie ) {}
					}
				}
				catch( Exception e ){}
			}
		}
	}

	private static HashMap<Class<? extends View>, ViewHolder>	map = new HashMap<Class<? extends View>, ViewHolder>();
	private static ViewMake maker = new ViewMake( map );

	/**
	 * クラス初期化。
	 * @param v クラス生成用インスタンス。
	 */
	protected static void init( View v ) {
		synchronized( map ) {
			map.put( v.getClass(), new ViewHolder( v ) );
		}
	}

	/**
	 * コンテナからインスタンスを提供します。
	 * @param key 提供クラス。
	 * @return 新しいインスタンス。
	 */
	protected static View newInstance( Class<? extends View> key ) {
		ViewHolder	vh = null;
		synchronized( map ) {
			vh = map.get( key );
		}
		return vh.get();
	}

	View(){}

	/**
	 * コンストラクタ。
	 * @param n テーブル名。
	 */
	protected View( String n ) {
		name = n;
	}

	/**
	 * テーブル名の取得。
	 * @return テーブル名。
	 */
	public String getName() {
		return name;
	}

	/**
	 * テーブル名の設定。
	 * @param n テーブル名。
	 */
	public void setName( String n ) {
		name = n;
	}

	/**
	 * フィールド一覧の取得。
	 * @return フィールド一覧。
	 */
	public Field[] getFields() {
		return fields;
	};


	/**
	 * インスタンスの取得。
	 * @return インスタンス。
	 */
	protected abstract View getInstance(); 

	private transient SelectGroup	sg;

	String makeSelect( Where where, GroupBy group, Where have, OrderBy ... order ) throws DB.DBException {
		StringBuilder	buf = new StringBuilder( "select " );
		sg = new SelectGroup( fields, group, order );
		buf = buf.append( sg.getSelect() );
		buf = buf.append( " from " ).append( getName() );
		if ( where != null ) {
			String	str = where.getString( true );
			if ( str != null ) {
				buf = buf.append( str );
			}
		}
		if ( group != null ) {
			String	str = group.toString();
			if ( str != null ) {
				buf = buf.append( " " ).append( str );
			}
		}
		if ( have != null ) {
			String	str = have.getString( false );
			if ( str != null ) {
				buf = buf.append( " " ).append( str );
			}
		}
		buf = buf.append( sg.getOrderBy() );
		return buf.toString();
	}

	SelectList<E> select( DB db, String sql, Where where, Where have ) throws DB.DBException {
		SelectList<E>	list = new SelectList<E>( db, sg );
		sg = null;
		ArrayList<Object>	ol = new ArrayList<Object>();
		if ( where != null ) {
			Object[]	obj = where.getParam();
			for ( Object o: obj )	ol.add( o );
		}
		if ( have != null ) {
			Object[]	obj = have.getParam();
			for ( Object o: obj )	ol.add( o );
		}
		list.start( this, db.executeQuery( sql, ol.toArray( new Object[0] ) ) );
		return list;
	}

	/**
	 * selectの実行。
	 * @param db DB接続。
	 * @param where 検索条件。条件が無い場合はnull。
	 * @param order ソート指定。
	 */
	public SelectList<E> select( DB db, Where where, OrderBy ... order ) throws DB.DBException {
		return select( db, makeSelect( where, null, null, order ), where, null );
	}

	/**
	 * selectの実行。集計処理されます。<br>
	 * 集計関数を指定したフィールドと、group指定されたフィールドのみが
	 * 取得対象になります。
	 * @param db DB接続。
	 * @param where 検索条件。条件が無い場合はnull。
	 * @param group 集計対象。対象が無い場合はnull。
	 * @param having 集計検索条件。条件が無い場合はnull。
	 * @param order ソート指定。
	 */
	public SelectList<E> selectGroup( DB db, Where where, GroupBy group, Where having, OrderBy ... order ) throws DB.DBException {
		if ( group == null )	group = new GroupBy();
		return select( db, makeSelect( where, group, having, order ), where, having );
	}
}

