/*
 * Oaks
 * Copyright (c) 2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に
 * 同意できる場合にのみ利用可能です。
 */
package oaks;

import java.util.*;
import java.sql.*;
import java.io.*;

class Record<E extends View> implements java.io.Serializable {
	private static final long serialVersionUID = 1;
	private String tail = null;
	private volatile int limit;
	private volatile int all_count = 0;
	private volatile int one_count = 0;
	private transient volatile Object[]	write_data = null;
	private transient volatile ArrayList<File>	tmp = new ArrayList<File>();

	private transient volatile int read_no = -1;
	private transient volatile int write_no = -1;
	private transient volatile Object[]	read_data = null;

	private void writeObject( ObjectOutputStream stream ) throws IOException, DB.DBException {
		stream.defaultWriteObject();
		for ( int i = 0; i < all_count; i++ ) {
			stream.writeObject( get( i ) );
		}
	}

	@SuppressWarnings("unchecked")
	private void readObject( ObjectInputStream stream ) throws IOException, ClassNotFoundException, DB.DBException {
		stream.defaultReadObject();
		init();
		for ( int i = 0; i < all_count; i++ ) {
			add( (E)stream.readObject() );
		}
	}

	protected void finalize() throws Throwable {
		if ( tmp == null )	return;
		for ( File f: tmp ) {
			f.delete();
		}
	}

	private Record() {
		tail = Integer.toString( hashCode(), Character.MAX_RADIX );
	}

	Record( int lim ) {
		this();
		limit = lim;
		init();
	}

	private void init() {
		write_data = new Object[limit];
		tmp = new ArrayList<File>();
		read_no = -1;
		write_no = -1;
	}

	void add( E v ) throws DB.DBException {
		ObjectOutputStream	out = null;
		try {
			synchronized( tmp ) {
				if ( one_count >= limit ) {
					File	swap = File.createTempFile( "oaks", tail );
					swap.deleteOnExit();
					out = new ObjectOutputStream( new FileOutputStream( swap ) );
					out.writeObject( write_data );
					write_data = new Object[limit];
					one_count = 0;
					tmp.add( swap );
				}
				write_data[one_count] = v;
				write_no = all_count / limit;
				one_count++;
				all_count++;
			}
		}
		catch( IOException e ) {
			throw new DB.DBException( e );
		}
		finally {
			if ( out != null ) {
				try {
					out.close();
				}
				catch( Exception e ){}
			}
		}
	}

	int size() {
		return all_count;
	}

	@SuppressWarnings("unchecked")
	E get( int index ) throws DB.DBException {
		if ( index < 0 )	return null;
		if ( all_count <= index )	return null;
		int	grp = index / limit;
		int offset = index % limit;
		synchronized( tail ) {
			synchronized( tmp ) {
				if ( write_no == grp )	return (E)write_data[offset];
			}
			ObjectInputStream	in = null;
			try {
				if ( read_no == grp && read_data != null )	return (E)read_data[offset];
				synchronized( tmp ) {
					in = new ObjectInputStream( new FileInputStream( tmp.get( grp ) ) );
				}
				read_data = (Object[])in.readObject();
				read_no = grp;
				return (E)read_data[offset];
			}
			catch( ClassNotFoundException cnfe ) {}
			catch( IOException ie ) {
				throw new DB.DBException( ie );
			}
			finally {
				if ( in != null ) {
					try {
						in.close();
					}
					catch( Exception e ){}
				}
			}
		}
		return null;
	}
}

/**
 * select結果を保持します。<br>
 * フェッチは別スレッドにて非同期に行われます。<br>
 * そのため、全結果のフェッチ完了を待たずに、別処理を実施可能です。<br><br>
 * また、一定数以上のレコードをフェッチすると、全てのデータをメモリに持たず、
 * 一時ファイルにフェッチ済みレコードを待避させながら処理します。<br>
 * 一時ファイルは、OS標準の一時ファイル用ディレクトリに作成されます。
 * 一時ファイルの削除タイミングは、Java VMの停止時か、このインスタンス自体が
 * ガベージコレクトによりメモリ解放された時になります。<br>
 * 一時ファイル動作の詳細については{@link SelectList#setOnMemoryLimit(int)}
 * を参照して下さい。
 */
public class SelectList<E extends View> extends Thread implements Iterable<E>, Serializable, DB.Resource {
	private static final long serialVersionUID = 1;
	private static int LIMIT = 1000;

	/**
	 * メモリ保持レコード数の設定。初期値は1000です。<br>
	 * フェッチしたレコード数が、ここで指定した値を上回ると一時ファイル利用を
	 * 開始します。<br>
	 * ここで指定した値は、全てのDB接続、全てのテーブル参照で共有されます。
	 * <ul>
	 * <li>一時ファイルは、ここで指定したレコード数につき１ファイル作成します。<br>
	 * つまり、ここで指定するレコード数が小さいと、小さいサイズの一時ファイルが大量に
	 * 発生する事になります。
	 * <li>メインルーチン側からフェッチ済みレコード参照を行い、それが一時ファイルに
	 * 待避されていた場合、メモリ上の全データを捨てて一時ファイル内容一つを全て
	 * 読み込みます。
	 * <br>つまり、ここで指定したレコード数を１ブロックとして、メモリとファイル間で
	 * データのスワップ動作を行います。
	 * </ul>
	 * このような動作のため、メインルーチンからのレコード参照は、ランダムアクセス
	 * よりもシーケンシャルアクセスである方が効率的です。<br>
	 * また、メモリ保持レコード数を大きく取るとディスクI/Oの発生回数が減るため
	 * 速度向上に繋がりますが、メモリの占有量が増加することになります。<br>
	 * 処理速度とメモリ使用量とがトレードオフの関係となるため、
	 * 適切な値を指定して下さい。
	 * @param limit メモリ保持レコード数。5未満を指定すると 5 となります。
	 */
	public static void setOnMemoryLimit( int limit ) {
		if ( limit < 5 )	limit = 5;
		LIMIT = limit;
	}

	/**
	 * メモリ保持レコード数の取得。
	 * @return メモリ保持レコード数。
	 */
	public static int getOnMemoryLimit() {
		return LIMIT;
	}

	private Record<E>	list = null;
	private transient View view = null;
	private transient SelectGroup	sg = null;
	private transient ResultSet	rs = null;
	private transient DB	mydb;
	private volatile transient DB.DBException ex = null;
	private volatile transient boolean end_f = true;
	private volatile int size = 0;

	private SelectList(){}

	SelectList( DB db, SelectGroup sel ) {
		mydb = db;
		mydb.addResource( this );
		sg = sel;
		list = new Record<E>( LIMIT );
	}

	private void writeObject( ObjectOutputStream stream ) throws IOException {
		synchronized( list ) {
			while( !end_f ) {
				try {
					list.wait( 1000 );
				}
				catch( Exception e ) {}
			}
		}
		stream.defaultWriteObject();
	}

	/**
	 * レコード取得。<br>
	 * フェッチ動作はバックグラウンドで行われており、
	 * 取得対象がまだフェッチされていなければ、フェッチされるまで待機します。<br>
	 * バックグラウンドフェッチで例外が発生すると、このメソッドからスローします。
	 * @param index 0～。取得レコードのインデックス。
	 * @return select結果。総数範囲外のインデックスを渡した場合、null。
	 */
	public E get( int index ) throws DB.DBException {
		if ( index < 0 )	return null;
		waitClose( index + 1 );
		synchronized( list ) {
			if ( ex != null )	throw ex;
			if ( size > index )	return list.get( index );
		}
		return null;
	}

	/**
	 * フェッチ完了状態。<br>
	 * バックグラウンドフェッチで例外が発生すると、このメソッドからスローします。
	 * @return true:フェッチが終わっている、false:フェッチ中。
	 */
	public boolean isClosed() throws DB.DBException {
		if ( ex != null )	throw ex;
		return end_f;
	}

	/**
	 * 全フェッチの完了まで待機します。
	 * <br>バックグラウンドフェッチで例外が発生すると、このメソッドからスローします。
	 */
	public void waitClose() throws DB.DBException {
		waitClose( 0 );
	}

	/**
	 * 指定レコード数のフェッチまたは全フェッチの完了まで待機します。
	 * <br>バックグラウンドフェッチで例外が発生すると、このメソッドからスローします。
	 * @param s レコード数。0を指定すると、全て取得するまで待機します。
	 */
	public void waitClose( int s ) throws DB.DBException {
		int	ms = 1;
		while( !end_f ) {
			if ( s <= size && s > 0 )	break;
			synchronized( list ) {
				try {
					list.wait( ms );
				}
				catch( Exception e ) {}
			}
			if ( ms < 50 )	ms <<= 1;
		}
		if ( ex != null )	throw ex;
	}

	private class Next implements Iterator<E> {
		private SelectList<E>	list;
		private int count = 0;
		private Next( SelectList<E> sl ) {
			list = sl;
		}
		public void remove() throws UnsupportedOperationException, IllegalStateException {
			throw new UnsupportedOperationException();
		}
		public E next() throws NoSuchElementException {
			if ( !hasNext() )	throw new NoSuchElementException();
			try {
				count++;
				return list.get( count - 1 );
			}
			catch( DB.DBException e ) {
				throw new NoSuchElementException( e.getMessage() );
			}
		}
		public boolean hasNext() {
			try {
				list.waitClose( count + 1 );
				return list.size() > count;
			}
			catch( DB.DBException e ) {
				return true;
			}
		}
	}

	/**
	 * 反復子の取得。
	 * @return 反復子。
	 */
	public Iterator<E> iterator() {
		return new Next( this );
	}

	/**
	 * レコード数取得。<br>
	 * フェッチ完了まで待機し、そのレコード数を返します。
	 * @return レコード数。
	 */
	public int size() throws DB.DBException {
		if ( ex != null )	throw ex;
		waitClose();
		return size;
	}

	private transient ArrayList<Field.GRP[]>	org_grp = new ArrayList<Field.GRP[]>();

	void start( View v, ResultSet r ) {
		view = v;
		for ( Field f: v.getFields() ) {
			org_grp.add( f.getGroup() );
		}
		rs = r;
		end_f = false;
		setDaemon( true );
		super.start();
	}

	/**
	 * フェッチスレッド。
	 */
	public void run() {
		if ( view == null || rs == null )	return;
		try {
			while( rs.next() ) {
				@SuppressWarnings("unchecked")
				E	v = (E)view.newInstance( view.getClass() );
				Field[]	fld = v.getFields();
				for ( int i = 0; i < fld.length; i++ ) {
					fld[i].setGroup( org_grp.get( i ) );
				}
				sg.set( v, rs );
				synchronized( list ) {
					list.add( v );
					size++;
					list.notifyAll();
				}
			}
		}
		catch( DB.DBException dbe ) {
			ex = dbe;
		}
		catch( SQLException e ) {
			ex = new DB.SQLExecException( mydb, e );
			return;
		}
		finally {
			try {
				rs.close();
				synchronized( rs ) {
					rs.notify();
				}
			}
			catch( Exception e ) {}
			view = null;
			mydb = null;
			rs = null;
			sg = null;
			end_f = true;
		}
	}
}

