/*
 * Oaks
 * Copyright (c) 2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に
 * 同意できる場合にのみ利用可能です。
 */
package oaks;

import java.util.*;

/**
 * フィールドを表します。<br>
 * また、フィールドの設定値を保持します。
 */
public abstract class Field<E> implements java.io.Serializable {
	/**
	 * フィールドのメタデータ。
	 */
	public static class MetaData {
		private Class<?>	cls;
		private int	len;
		private int	scale;
		/**
		 * コンストラクタ。
		 * @param c クラス。
		 * @param p 精度。
		 * @param s スケール。
		 */
		public MetaData( Class<?> c, int p, int s ) {
			cls = c;
			len = p;
			scale = s;
		}
		/**
		 * 入出力クラスの取得。<br>DBフィールドの型ではありません。
		 * このフィールドで扱われる型です。例えば enum の可能性もあります。
		 * @return フィールドの入出力で使用されるクラス。
		 */
		public Class<?> getSQLClass() { return cls; }
		/**
		 * 入出力クラスが数値型であるか？
		 * @return true:数値型、false:数値以外。
		 */
		public boolean isNumber() { return Number.class.isAssignableFrom( cls ); }
		/**
		 * 入出力クラスがStringであるか？
		 * @return true:String、false:String以外。
		 */
		public boolean isString() { return String.class.isAssignableFrom( cls ); }
		/**
		 * フィールドサイズ。<br>
		 * 文字列長や、バイト数を示します。<br>
		 * サイズが得られないフィールドであれば 0 を返します。
		 * @return フィールドサイズ。
		 */
		public int getPrecision() { return len; }
		/**
		 * フィールドスケール。<br>
		 * 小数を扱うフィールドの小数点以下桁数を返します。<br>
		 * 小数を扱わないフィールドであれば 0 を返します。
		 */
		public int getScale() { return scale; }
	}
	/**
	 * メタデータの取得。
	 * @return メタデータ。
	 */
	public abstract MetaData getMetaData();

	private static final long serialVersionUID = 1;
	static final Object	NO_CARE = new Object();
	private static HashMap<Class<? extends Field>, String>	map =
		new HashMap<Class<? extends Field>, String>();

	private String	name;
	private Object	data = NO_CARE;

	/**
	 * 集計関数。
	 */
	public interface GRP {
		/**
		 * 文字列化。
		 * @return 文字列。
		 */
		public String toString();
	}

	/**
	 * 数値型集計関数。
	 */
	public enum NumberGRP implements GRP {
		/** count */
		COUNT( "count" );
		private String str;
		private NumberGRP( String s ) {
			str = s;
		}
		public String toString() { return str; }
	}

	/**
	 * 同値型集計関数。
	 */
	public enum SameGRP implements GRP {
		/** min */
		MIN( "min" ),
		/** max */
		MAX( "max" ),
		/** sum */
		SUM( "sum" );

		private String str;
		private SameGRP( String s ) {
			str = s;
		}
		public String toString() { return str; }
	}

	static String toString( GRP _grp, Class<? extends Field> f ) {
		StringBuilder	buf = new StringBuilder( _grp.toString() );
		buf = buf.append( "(" ).append( Field.getName( f ) );
		buf = buf.append( ")" );
		return buf.toString();
	}

	@SuppressWarnings( "unchecked" )
	String toString( GRP _grp ) {
		return toString( _grp, (Class<? extends Field>)getClass() );
	}

	private HashMap<GRP, Object>	grp = null;

	GRP[] getGroup() {
		if ( grp == null )	return null;
		return grp.keySet().toArray( new GRP[0] );
	}

	/**
	 * 初期化。
	 * @param cl クラス。
	 * @param n フィールド名。
	 */
	protected static void init( Class<? extends Field> cl, String n ) {
		synchronized( map ) {
			map.put( cl, n );
		}
	}

	static String getName( Class<? extends Field> cl ) {
		synchronized( map ) {
			return map.get( cl );
		}
	}

	private Field(){}

	/**
	 * コンストラクタ。
	 * @param n フィールド名。
	 */
	protected Field( String n ) {
		name = n;
	}

	/**
	 * フィールド名の取得。
	 * @return フィールド名。
	 */
	public String getName() {
		return name;
	}

	Object getObject() {
		return data;
	}

	/**
	 * 値の取得。
	 * @return 設定値。
	 */
	public E get() {
		if ( data == NO_CARE )	return null;
		return convert( data );
	}

	/**
	 * 値の取得。設定値を toString() して返します。
	 * @return 設定値の文字列。
	 */
	public String getString() {
		E ret = get();
		if ( ret == null )	return null;
		return ret.toString();
	}

	/**
	 * 値の設定。
	 * @param v 値。
	 */
	public void set( E v ) {
		data = toObject( v );
	}

	/**
	 * 値の複写。別フィールドの値を取得して設定します。
	 * @param v 別フィールド。
	 */
	public void setFrom( Field<E> v ) {
		set( v.get() );
	}

	/**
	 * 型変換。
	 * @param obj 元の値。
	 * @return 変換後の値。
	 */
	@SuppressWarnings("unchecked")
	protected E convert( Object obj ) {
		return (E)obj;
	}

	/**
	 * 型変換。
	 * @param v 論理表現値。
	 * @return 変換後の値。
	 */
	protected Object toObject( E v ) {
		return v;
	}

	/**
	 * 集計関数の指定。
	 * @param function 集計関数。
	 */
	public void setGroup( GRP ... function ) {
		if ( function == null ) {
			grp = null;
			return;
		}
		int	cnt = function.length;
		if ( cnt == 0 ) {
			grp = null;
			return;
		}
		HashMap<GRP, Object>	g = new HashMap<GRP, Object>();
		for ( int i = 0; i < cnt; i++ ) {
			g.put( function[i], null );
		}
		grp = g;
	}

	/**
	 * 集計結果の取得。
	 * @param function 集計関数。
	 * @return 集計値。
	 */
	public E getGroup( SameGRP function ) {
		if ( grp == null )	return null;
		return convert( grp.get( function ) );
	}

	/**
	 * 集計結果の取得。
	 * @param function 集計関数。
	 * @return 集計値。
	 */
	public Integer getGroup( NumberGRP function ) {
		if ( grp == null )	return null;
		return new Integer( grp.get( function ).toString() );
	}

	void setObject( GRP function, Object obj ) {
		if ( function == null || grp == null )	data = obj;
		else	grp.put( function, obj );
	}

	/**
	 * 文字列化。
	 * @return 内容。
	 */
	public String toString() {
		StringBuilder	buf = new StringBuilder( name );
		buf = buf.append( " = " );
		if ( data == null )	buf = buf.append( "null" );
		else if ( data == NO_CARE )	buf = buf.append( "not initialized" );
		else {
			buf = buf.append( "[" ).append( data.getClass().toString() ).append( "][" );
			buf = buf.append( data.toString() ).append( "]" );
		}
		return buf.toString();
	}
}




