/*
 * Oaks
 * Copyright (c) 2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に
 * 同意できる場合にのみ利用可能です。
 */
package oaks;

import java.util.*;
import java.sql.*;

/**
 * DB接続。
 * java.sql.Connectionのラッパークラスです。
 * また、DB接続プール識別子の設定値に従った接続プーリングを行います。
 */
public class DB extends Thread implements java.io.Closeable {
	/**
	 * DBアクセス例外を表します。
	 */
	public static class DBException extends Exception {
		DBException( Throwable e ) {
			super( e );
		}
		DBException( String s ) {
			super( s );
		}
	}
	/**
	 * SQL実行時例外を表します。
	 */
	public static class SQLExecException extends DBException {
		private volatile DB	err;
		private volatile SQLException ex;
		SQLExecException( DB db, SQLException e ) {
			super( e );
			err = db;
			ex = e;
		}
		public String getLocalizedMessage() {
			StringBuilder	buf = new StringBuilder( super.getLocalizedMessage() );
			buf = buf.append( DB.LF ).append( err.getLastSQL() );
			return buf.toString();
		}
	}
	/**
	 * close後のDBインスタンスを操作すると発生します。
	 */
	public static class DBClosedException extends DBException {
		private DBClosedException() {
			super( "DB is already closed and cannot be used." );
		}
	}
	/**
	 * 接続タイムアウト({@link ConnectionKey#setTimeout(int)}参照)。
	 */
	public static class ConnectTimeoutException extends DBException {
		/**
		 * 例外発生時の接続状況。
		 */
		public class Info {
			private volatile java.util.Date	start;
			private volatile StackTraceElement[]	locate;
			private Info( java.util.Date d, StackTraceElement[] l ) {
				start = d;
				locate = l;
			}
			/**
			 * 接続取得時間。
			 * @return 接続取得時間。
			 */
			public java.util.Date getOpenTime() { return start; }
			/**
			 * 接続取得時のスタックトレース。
			 * @return スタックトレース。
			 */
			public StackTraceElement[] getStackTrace() { return locate; }
			/**
			 * 文字列化。
			 * @return 接続取得時間とスタックトレース内容。
			 */
			public String toString() {
				StringBuilder	buf = new StringBuilder( start.toString() );
				buf = buf.append( DB.LF );
				for ( StackTraceElement st: locate ) {
					buf = buf.append( st.toString() ).append( DB.LF );
				}
				return buf.toString();
			}
		}
		private volatile java.util.Date now;
		private volatile Info[]	info;
		private ConnectTimeoutException( LinkedList<DB> list ) {
			super( "connect timeout." );
			now = new java.util.Date();
			synchronized( list ) {
				info = new Info[list.size()];
				int	i = 0;
				for( DB db: list ) {
					info[i] = new Info( db.open_time, db.open_locate );
					i++;
				}
			}
		}
		/**
		 * 例外発生時間。
		 * @return 例外発生時間。
		 */
		public java.util.Date getDate() { return now; }
		/**
		 * 接続状況の取得。例外発生時の全接続状況を返します。
		 * @return 接続状況。
		 */
		public Info[] getInfo() {
			return info;
		}
		public String getLocalizedMessage() {
			StringBuilder	buf = new StringBuilder( super.getLocalizedMessage() );
			buf = buf.append( DB.LF );
			buf = buf.append( now.toString() );
			buf = buf.append( DB.LF );
			for ( Info i: info ) {
				buf = buf.append( DB.LF );
				buf = buf.append( "-------------------------" ).append( DB.LF );
				buf = buf.append( i.toString() );
				buf = buf.append( DB.LF );
			}
			return buf.toString();
		}
	}

	/**
	 * DBリソース。
	 */
	public interface Resource {
		/**
		 * リソース解放待ち。
		 */
		void waitClose() throws DBException;
	}

	private static volatile HashMap<ConnectionKey, LinkedList<DB>>	wait_map = new HashMap<ConnectionKey, LinkedList<DB>>();
	private static volatile HashMap<ConnectionKey, LinkedList<DB>>	active_map = new HashMap<ConnectionKey, LinkedList<DB>>();

	private LinkedList<DB>	pool;
	private LinkedList<DB>	lease;
	private Connection	con;
	private ConnectionKey con_key;
	private volatile boolean open_f = false;
	private volatile int exe_count = 0;
	private volatile ArrayList<Resource>	res = null;
	volatile java.util.Date	open_time = null;
	volatile StackTraceElement[]	open_locate = null;

	private DB( ConnectionKey key, LinkedList<DB> plist, LinkedList<DB> llist ) throws DBException {
		this( key );
		synchronized( llist ) {
			if ( llist.size() >= key.getMaxConnect() ) {
				try {
					llist.wait( key.getTimeout() * 1000 );
				}
				catch( Exception e ){}
			}
			if ( llist.size() >= key.getMaxConnect() )	throw new ConnectTimeoutException( llist );
		}
		try {
			set( key.getConnection(), plist, llist );
		}
		catch( Exception e ) {
			throw new DBException( e );
		}
	}

	private DB( DB old ) {
		this( old.con_key );
		set( old.con, old.pool, old.lease );
		old.con = null;
		{
			HashMap<String, CallableStatement>	tmp = cas;
			cas = old.cas;
			old.cas = tmp;
		}
		{
			HashMap<String, PreparedStatement>	tmp = pps;
			pps = old.pps;
			old.pps = tmp;
		}
	}

	private void set( Connection c, LinkedList<DB> plist, LinkedList<DB> llist ) {
		con = c;
		pool = plist;
		lease = llist;
	}

	private DB( ConnectionKey k ) {
		con_key = k;
	}

	private DB() {}

	/**
	 * DB接続取得。
	 * 接続はオートコミットされません。常にcommit/rollbackを要します。
	 * @param key 識別子。
	 * @return DB接続。
	 */
	public static DB open( ConnectionKey key ) throws DBException {
		DB	db = null;
		LinkedList<DB>	plist = null;
		synchronized( wait_map ) {
			plist = wait_map.get( key );
			if ( plist == null ) {
				plist = new LinkedList<DB>();
				wait_map.put( key, plist );
			}
		}
		while( true ) {
			synchronized( plist ) {
				db = plist.poll();
			}
			if ( db == null )	break;
			try {
				db.con.rollback();
				break;
			}
			catch( SQLException e ){
				db.allClose();
				db = null;
			}
		}
		LinkedList<DB>	llist = null;
		synchronized( active_map ) {
			llist = active_map.get( key );
			if ( llist == null ) {
				llist = new LinkedList<DB>();
				active_map.put( key, llist );
			}
		}
		if ( db == null ) {
			db = new DB( key, plist, llist );
		}
		synchronized( llist ) {
			llist.offer( db );
		}
		db.open_f = true;
		db.exe_count = 0;
		db.last_sql = "";
		db.last_param = new Object[0];
		db.res = new ArrayList<Resource>();
		db.open_time = new java.util.Date();
		db.open_locate = currentThread().getStackTrace();
		return db;
	}

	/**
	 * DBMSのメタデータ取得。
	 * @return メタデータ。
	 */
	public DatabaseMetaData getMetaData() throws DB.DBException {
		try {
			return con.getMetaData();
		}
		catch( SQLException e ) {
			throw new DB.DBException( e );
		}
	}

	void addResource( Resource r ) {
		res.add( r );
	}

	private static volatile LinkedList<DB>	closed = new LinkedList<DB>();
	private static DB	cleanner = new DB();

	static {
		cleanner.setDaemon( true );
		cleanner.setPriority( Thread.MIN_PRIORITY );
		cleanner.start();
	}

	/**
	 * プール回収スレッド。
	 */
	public void run() {
		while ( true ) {
			DB	db = null;
			synchronized( closed ) {
				try {
					if ( closed.size() < 1 ) {
						closed.wait( 1000 );
					}
					db = closed.poll();
				}
				catch( Exception e ){}
			}
			if ( db == null )	continue;
			for ( Resource r: db.res ) {
				try {
					r.waitClose();
				}
				catch( DBException e ){}
			}
			db.res = null;
			try {
				int	max = db.con_key.getMaxConnect();
				int	cnt = 0;
				synchronized( db.lease ) {
					db.lease.remove( db );
					cnt = db.lease.size();
					db.lease.notify();
				}
				db.con.rollback();
				synchronized( db.pool ) {
					cnt += db.pool.size();
					if ( cnt < max ) {
						db.pool.offer( new DB( db ) );
					}
				}
			}
			catch( SQLException e ) {}
			db.allClose();
		}
	}

	/**
	 * 接続返却。プールへ返却します。<br>
	 * 使用されていたpreparedStatement等のリソースははプール返却タイミングで
	 * 解放されます。
	 */
	public void close() {
		open_f = false;
		synchronized( closed ) {
			closed.offer( this );
			closed.notifyAll();
		}
	}

	protected void finalize() throws Throwable {
		allClose();
	}

	private void checkOpen() throws DBClosedException {
		if ( !open_f )	throw new DBClosedException();
	}

	private void allClose() {
		for ( String k : pps.keySet() ) {
			try {
				pps.get( k ).close();
			}
			catch( Exception e ){}
		}
		for ( String k : cas.keySet() ) {
			try {
				cas.get( k ).close();
			}
			catch( Exception e ){}
		}
		try {
			if ( con != null )	con.close();
		}
		catch( Exception e ){}
	}

	private HashMap<String, PreparedStatement>	pps = new HashMap<String, PreparedStatement>();
	/**
	 * PreparedStatementの作成。<br>
	 * 作成されたPreparedStatementはキャッシュされ２回目からはキャッシュを返します。
	 * 戻り値のPreparedStatementはcloseしないで下さい。
	 * @param sql SQL。
	 * @return プリコンパイル済みSQL。
	 */
	public PreparedStatement prepareStatement( String sql ) throws DBException {
		checkOpen();
		PreparedStatement	ret = pps.get( sql );
		try {
			if ( ret == null ) {
				ret = con.prepareStatement( sql );
				pps.put( sql, ret );
			}
			ret.clearParameters();
		}
		catch( SQLException e ) {
			throw new DBException( e );
		}
		return ret;
	}

	static final Calendar	TIMEZONE = Calendar.getInstance();

	private PreparedStatement prepareStatement( String sql, Object ... param ) throws DBException {
		PreparedStatement	ret = prepareStatement( sql );
		for ( int i = 0; i < param.length; i++ ) {
			try {
				if ( param[i] instanceof java.sql.Date ) {
					ret.setDate( i + 1, (java.sql.Date)param[i], TIMEZONE );
				}
				else if ( param[i] instanceof java.sql.Time ) {
					ret.setTime( i + 1, (java.sql.Time)param[i], TIMEZONE );
				}
				else if ( param[i] instanceof java.sql.Timestamp ) {
					ret.setTimestamp( i + 1, (java.sql.Timestamp)param[i], TIMEZONE );
				}
				else {
					ret.setObject( i + 1, param[i] );
				}
			}
			catch( SQLException e ) {
				throw new DBException( e );
			}
		}
		return ret;
	}

	private HashMap<String, CallableStatement>	cas = new HashMap<String, CallableStatement>();
	/**
	 * CallableStatementの作成。
	 * 作成されたCallableStatementはキャッシュされ２回目からはキャッシュを返します。
	 * 戻り値のCallableStatementはcloseしないで下さい。
	 * @param sql SQL。
	 * @return プリコンパイル済みSQL。
	 */
	public CallableStatement prepareCall( String sql ) throws DBException {
		checkOpen();
		CallableStatement	ret = cas.get( sql );
		try {
			if ( ret == null ) {
				ret = con.prepareCall( sql );
				cas.put( sql, ret );
			}
			ret.clearParameters();
		}
		catch( SQLException e ) {
			throw new DBException( e );
		}
		return ret;
	}

	private String last_sql;
	private Object[]	last_param;
	static final String	LF =  System.getProperty( "line.separator" );

	/**
	 * 最後に実行したSQLとバインド変数を出力します。
	 * @return SQLとバインド変数。
	 */
	public String getLastSQL() {
		StringBuilder	buf = new StringBuilder( last_sql );
		for ( int i = 0; i < last_param.length; i++ ) {
			if ( last_param[i] != null ) {
				buf = buf.append( LF ).append(
					String.format( "    No.%02d:[%s](%s)", i + 1, last_param[i].toString(), last_param[i].getClass().toString() )
				);
			}
			else {
				buf = buf.append( LF ).append(
					String.format( "    No.%02d:null", i + 1 )
				);
			}
		}
		buf = buf.append( LF );
		return buf.toString();
	}

	/**
	 * DML文の実行。
	 * @param sql SQL。
	 * @param param バインド変数。
	 * @return 更新行数。
	 */
	public int executeUpdate( String sql, Object ... param ) throws DBException {
		exe_count++;
		if ( sql == null )	sql = "";
		if ( param == null )	param = new Object[0];
		last_sql = sql;
		last_param = param;
		try {
			return prepareStatement( sql, param ).executeUpdate();
		}
		catch( SQLException e ) {
			throw new SQLExecException( this, e );
		}
	}

	private HashMap<PreparedStatement, ResultSet>	selecting = new HashMap<PreparedStatement, ResultSet>();

	/**
	 * クエリの実行。戻り値はメインルーチン側で必ずcloseして下さい。<br>
	 * 同一SQLを繰り返す呼び出した場合、直前呼び出し時の ResultSet が
	 * close されるまで、処理がブロックされます。
	 * @param sql SQL。
	 * @param param バインド変数。
	 * @return セレクト結果。
	 */
	public ResultSet executeQuery( String sql, Object ... param ) throws DBException {
		if ( sql == null )	sql = "";
		if ( param == null )	param = new Object[0];
		last_sql = sql;
		last_param = param;
		try {
			PreparedStatement	st = prepareStatement( sql, param );
			ResultSet	rs = null;
			synchronized( selecting ) {
				rs = selecting.get( st );
				if ( rs != null ) {
					synchronized( rs ) {
						while( !rs.isClosed() ) {
							try {
								rs.wait( 10 );
							}
							catch( Exception e ) {}
						}
					}
				}
				rs = st.executeQuery();
				selecting.put( st, rs );
			}
			return rs;
		}
		catch( SQLException e ) {
			throw new SQLExecException( this, e );
		}
	}

	/**
	 * コミット。
	 */
	public void commit() throws DBException {
		checkOpen();
		try {
			con.commit();
		}
		catch( SQLException e ) {
			throw new DBException( e );
		}
	}

	/**
	 * コミット。
	 * 指定回数のexecuteUpdate()が実行されていればコミットします。<br>
	 * 大量にinsertする場合の高速化等を目的とします。
	 * @param cnt SQL実行回数。
	 */
	public void commit( int cnt ) throws DBException {
		if ( exe_count >= cnt ) {
			commit();
			exe_count = 0;
		}
	}

	/**
	 * ロールバック。
	 */
	public void rollback() throws DBException {
		checkOpen();
		try {
			con.rollback();
		}
		catch( SQLException e ) {
			throw new DBException( e );
		}
	}

	/**
	 * ロールバック用トランザクション開始。
	 * @return ロールバック開始位置。
	 */
	public Savepoint setSavepoint() throws DBException {
		checkOpen();
		try {
			return con.setSavepoint();
		}
		catch( SQLException e ) {
			throw new DBException( e );
		}
	}

	/**
	 * セーブポイント以降のロールバック。
	 * @param sp ロールバック開始位置。
	 */
	public void rollback( Savepoint sp ) throws DBException {
		checkOpen();
		try {
			con.rollback( sp );
		}
		catch( SQLException e ) {
			throw new DBException( e );
		}
	}

	private static long getNow() {
		return new java.util.Date().getTime();
	}

	/**
	 * 現在日時の生成。
	 * @return 現在日時。
	 */
	public static java.sql.Date nowDate() {
		return new java.sql.Date( getNow() );
	}

	/**
	 * 現在日時の生成。
	 * @return 現在日時。
	 */
	public static java.sql.Time nowTime() {
		return new java.sql.Time( getNow() );
	}

	/**
	 * 現在日時の生成。
	 * @return 現在日時。
	 */
	public static java.sql.Timestamp nowTimestamp() {
		return new java.sql.Timestamp( getNow() );
	}
}

