/*
 * Oaks
 * Copyright (c) 2012  Akira Terasaki
 * このファイルは同梱されているLicense.txtに定めた条件に
 * 同意できる場合にのみ利用可能です。
 */
package oaks;

import java.sql.*;
import java.util.*;

/**
 * DB接続プール識別子。
 */
public class ConnectionKey implements java.io.Serializable {
	private static final long serialVersionUID = 1;
	private static final Integer	DEF_MAX = 20;
	private static final Integer	DEF_TIMEOUT = 15;
	private static volatile HashMap<ConnectionKey, Integer>	max = new HashMap<ConnectionKey, Integer>();
	private static volatile HashMap<ConnectionKey, Integer>	timeout = new HashMap<ConnectionKey, Integer>();
	private int	hash;
	private String	driver = null;
	private String	url;
	private String	user;
	private String	passwd;

	private ConnectionKey(){}

	/**
	 * コンストラクタ。JDBC4.0以上であれば、こちらを使用して下さい。
	 * @param ur 接続URL。
	 * @param us 接続ユーザー。
	 * @param pw 接続パスワード。
	 */
	public ConnectionKey( String ur, String us, String pw ) {
		url = ur;
		user = us;
		passwd = pw;
		StringBuilder	buf = new StringBuilder( url );
		hash = buf.append( "<oaks>" ).append( user ).toString().hashCode();
		getMaxConnect();
	}

	/**
	 * コンストラクタ。JDBC4.0未満であれば、こちらを使用して下さい。
	 * @param jdbc JDBCドライバークラス名。
	 * @param ur 接続URL。
	 * @param us 接続ユーザー。
	 * @param pw 接続パスワード。
	 */
	public ConnectionKey( String jdbc, String ur, String us, String pw ) {
		this( ur, us, pw );
		driver = jdbc;
	}

	/**
	 * 最大接続数の設定。同じ識別子で設定は共用されます。<br>
	 * 接続数カウントの処理排他は処理速度を優先しているため若干緩くなっており、
	 * 物理接続数がこの指定数を上回る可能性があります。<br>
	 * 初期値は20です。
	 * @param count 最大接続数。
	 */
	public void setMaxConnect( int count ) {
		synchronized( max ) {
			max.put( this, count );
		}
	}

	/**
	 * 最大接続数の取得。
	 * @return 最大接続数。
	 */
	public int getMaxConnect() {
		Integer	count = null;
		synchronized( max ) {
			count = max.get( this );
			if ( count == null || count < 1 ) {
				count = DEF_MAX;
				max.put( this, count );
			}
		}
		return count;
	}

	/**
	 * 接続タイムアウトの設定。同じ識別子で設定は共用されます。<br>
	 * この値は物理通信のタイムアウトではなく、
	 * 接続プール解放待ちのタイムアウトです。<br>
	 * 初期値は15秒です。
	 * @param sec タイムアウト(秒)。
	 */
	public void setTimeout( int sec ) {
		synchronized( timeout ) {
			timeout.put( this, sec );
		}
	}

	/**
	 * 接続タイムアウトの取得。
	 * @return タイムアウト(秒)。
	 */
	public int getTimeout() {
		Integer	sec = null;
		synchronized( timeout ) {
			sec = timeout.get( this );
			if ( sec == null || sec < 1 ) {
				sec = DEF_TIMEOUT;
				timeout.put( this, sec );
			}
		}
		return sec;
	}

	/**
	 * 同じ識別子であるか判定します。
	 * ただし、接続URLと接続ユーザーが一致した場合をtrueとし、
	 * JDBCドライバーや接続パスワードは比較条件ではありません。
	 * @return true:一致、false:不一致。
	 */
	public boolean equals( Object o ) {
		if ( !(o instanceof ConnectionKey) )	return false;
		ConnectionKey	key = (ConnectionKey)o;
		return url.equals( key.url ) && user.equals( key.user );
	}

	/**
	 * ハッシュ値を返します。
	 * 接続URLと接続ユーザーよりハッシュ値を作成します。
	 * @return ハッシュ値。
	 */
	public int hashCode() {
		return hash;
	}

	Connection getConnection() throws SQLException, ClassNotFoundException {
		if ( driver != null )	Class.forName( driver );
		Connection	con = DriverManager.getConnection( url, user, passwd );
		con.setAutoCommit( false );
		return con;
	}
}

