/* vgrab.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <pthread.h>
#include <math.h>

#include "csp.h"
#include "v4l.h"
#include "vgrab.h"
#include "util.h"
#include "vframe_mgr.h"
#include "mgr.h"
#include "aread.h"

#define LOOP_COUNT 4096

static VGRAB_CONFIG vgrab_conf;

static int vgrab_initialized = 0;
static int vgrab_thread_running = 0;

/* Thread helper functions */

static void
init_frame_time(VGRAB_CONFIG *conf)
{
  double fps;
  if (conf->fps == 0) {
    switch (v4l_get_norm ()) {
      case NORM_NTSC:
        fps = 29.97;
        break;
      default:
        fps = 25;
        break;
    }
  }
  else
    fps = conf->fps;
  conf->loop_count = LOOP_COUNT;
  conf->interval = 1000000.0 / fps;
  conf->interval_count = 0;
  conf->diff_time = 0;
  conf->next_time = conf->base_time = get_cur_u_time();
  conf->diff_time += conf->interval;
  conf->next_next_time = conf->base_time + conf->diff_time;

//printf("get_next_frame_time: interval %f, diff_time %f, fps %f\n", conf->interval, conf->diff_time, fps);
}

static void
set_next_frame_time(VGRAB_CONFIG *conf)
{
  conf->next_time = conf->next_next_time;
  if (conf->interval_count++ > conf->loop_count) {
    conf->interval_count = 0;
    conf->diff_time = 0;
    conf->base_time = conf->next_next_time;
  }
  conf->diff_time += conf->interval;
  conf->next_next_time = conf->base_time + conf->diff_time;
}

static void
init_grab_time(VGRAB_CONFIG *conf)
{
  double fps = 0.0;
  double t;

  switch (v4l_get_norm()) {
    case NORM_NTSC:
      fps = 29.97;
      break;
    default:
      fps = 25;
      break;
  }
  conf->time_chunk_num = 16;
  t = 16.0 * 1000000.0 / fps;
  conf->time_chunk = t;
  conf->time_length = conf->time_chunk / conf->time_chunk_num;
  conf->time_chunk = conf->time_chunk - conf->time_length;
  conf->grab_time = conf->last_grab_time = get_cur_u_time();
}

static void
set_grab_time(VGRAB_CONFIG *conf)
{
  u_time cur_time = get_cur_u_time();
  if (cur_time < conf->grab_time)
    conf->grab_time = cur_time;
  conf->last_grab_time = conf->grab_time;
  conf->time_chunk = conf->time_chunk + (cur_time - conf->grab_time);
  conf->time_length = conf->time_chunk / conf->time_chunk_num;
  conf->time_chunk = conf->time_chunk - conf->time_length;
  conf->grab_time = conf->grab_time + conf->time_length;
//  conf->frame_time = conf->grab_time - conf->last_time;
//  conf->grab_time=conf->grab_time+((cur_time-conf->grab_time)/conf->time_chunk_num);
}

void
init_grab_pic(VGRAB_CONFIG *conf)
{
  unsigned char *buf = NULL;
  int error_count = 0;

  for (error_count = 0; buf == NULL; error_count++) {
    buf = v4l_grab_pic_sync ();
    if (buf != NULL) {
      conf->grab_time = conf->last_grab_time = get_cur_u_time();
      if (v4l_grab_pic_capture() < 0) {
        fprintf(stderr, "init_grab_pic: v4l capture error.\n");
      }
      break;
    } else if (error_count > 30) {
      fprintf(stderr, "vgrab_thread(): (2) buf NULL count %d\n", error_count);
      pthread_mutex_lock(&vgrab_conf.mutex);
      conf->state = STATE_ERROR;
      pthread_mutex_unlock(&vgrab_conf.mutex);
      mgr_error_quit();
      break;
    }
  }
}

unsigned char*
get_grab_pic_buf(VGRAB_CONFIG *conf)
{
  unsigned char *buf = NULL;
  int error_count = 0;

  for (error_count = 0; buf == NULL; error_count++) {
    buf = v4l_grab_pic_sync();
    if (buf != NULL) {
      set_grab_time(conf);
      if (v4l_grab_pic_capture() < 0) {
        fprintf(stderr, "vgrab_thread: v4l capture error.\n");
      }
      break;
    } else if (error_count > 30) {
      fprintf(stderr, "vgrab_thread: (2) buf NULL count %d\n", error_count);
      pthread_mutex_lock(&vgrab_conf.mutex);
      conf->state = STATE_ERROR;
      pthread_mutex_unlock(&vgrab_conf.mutex);
      mgr_error_quit();
      break;
    }
  }
  conf->pic_data = buf;
  return buf;
}

static void
pic_cpy(VGRAB_CONFIG *conf, uint8_t *data)
{
#if 0
  if (conf->csp == CSP_YV12_INTERLACE) {
    //csp_rgb24_to_yv12(
    csp_rgb32_to_yv12(
	  conf->yv12_interlace_width,
	  conf->yv12_interlace_height,
	  data + conf->yv12_interlace_y_offset,
	  conf->yv12_interlace_y_stride,
	  data + conf->yv12_interlace_u_offset,
	  data + conf->yv12_interlace_v_offset,
	  conf->yv12_interlace_uv_stride,
	  conf->pic_data,
	  conf->yv12_interlace_rgb_stride);
  } else {
    memcpy(data, conf->pic_data, conf->pic_data_size);
  }
#else
  if (conf->csp == CSP_YV12) {
    memcpy(data, conf->pic_data, conf->y_size);
    memcpy(data+conf->v_offset, conf->pic_data+conf->u_offset, conf->uv_size);
    memcpy(data+conf->u_offset, conf->pic_data+conf->v_offset, conf->uv_size);
  } else {
    memcpy(data, conf->pic_data, conf->pic_data_size);
  }
#endif
}

static int
put_grab_pic_buf(VGRAB_CONFIG *conf)
{
  VFRAME *vfrm;

  vfrm = vframe_src_frame_get();
  if (vfrm == NULL) {
#ifdef DEBUG
    fprintf(stderr, "put_grab_pic_buf: vframe_src_frame_get() failed.\n");
    fprintf(stderr, "put_grab_pic_buf: frame %d drop, total %d\n", conf->frame_num, conf->drop_frame_num);
#endif /* DEBUG */
    vfrm = vframe_ext_frame_get();
    if (vfrm == NULL) {
      fprintf(stderr, "put_grab_pic_buf: vframe_ext_frame_get() failed.\n");
      pthread_mutex_lock(&vgrab_conf.mutex);
      conf->state = STATE_ERROR;
      pthread_mutex_unlock(&vgrab_conf.mutex);
      mgr_error_quit();
      return VGRAB_FAIL;
    }
    vfrm->flag |= DROP_VFRAME;
    vfrm->data_length = 0;
    vfrm->csp = CSP_UNKNOWN;
    conf->drop_frame_num++;
  } else {
    pic_cpy(conf, vfrm->data);
    vfrm->data_length = conf->pic_data_size;
    vfrm->csp = conf->csp;
  }
  if (conf->real_time_flag)
    vfrm->time = conf->grab_time - conf->last_grab_time;
  else
    vfrm->time = conf->next_next_time - conf->next_time;
  conf->frame_num++;
  //fprintf(stderr, "put_grab_pic_buf: frame time %llu\n", conf->vfrm->time);
  vframe_src_frame_update(vfrm);
  return VGRAB_OK;
}

static int
put_grab_pic_buf_end(VGRAB_CONFIG *conf)
{
  VFRAME *vfrm;

#ifdef DEBUG
  if (conf->restart_frame_num >= conf->restart_overlap_frame_num) {
    fprintf(stderr, "put_grab_pic_buf_end: overlap frame overrun.\n");
    exit(1);
  }
#endif /* DEBUG */

  vfrm = vframe_src_frame_get();
  if (vfrm == NULL) {
#ifdef DEBUG
    fprintf(stderr, "put_grab_pic_buf_end: frame %d drop, total %d\n", conf->frame_num, conf->drop_frame_num);
#endif /* DEBUG */
    vfrm = vframe_ext_frame_get();
    if (vfrm == NULL) {
      fprintf(stderr, "put_grab_pic_buf_end: vframe_ext_frame_get() failed.\n");
      pthread_mutex_lock(&vgrab_conf.mutex);
      conf->state = STATE_ERROR;
      pthread_mutex_unlock(&vgrab_conf.mutex);
      mgr_error_quit();
      return VGRAB_FAIL;
    }
    vfrm->flag |= DROP_VFRAME;
    vfrm->data_length = 0;
    vfrm->csp = CSP_UNKNOWN;
    conf->drop_frame_num++;

    pic_cpy(conf, conf->restart_frame[conf->restart_frame_num].data);
  } else {
    pic_cpy(conf, vfrm->data);
    memcpy(conf->restart_frame[conf->restart_frame_num].data, vfrm->data,
       	conf->pic_data_size);
    vfrm->data_length = conf->pic_data_size;
    vfrm->csp = conf->csp;
  }
  if (conf->real_time_flag) {
    conf->restart_frame[conf->restart_frame_num].frame_time = vfrm->time =
      (conf->grab_time - conf->last_grab_time);
    conf->restart_frame[conf->restart_frame_num].time = conf->last_grab_time;
  } else {
    conf->restart_frame[conf->restart_frame_num].frame_time = vfrm->time =
      (conf->next_next_time - conf->next_time);
    conf->restart_frame[conf->restart_frame_num].time = conf->next_time;
  }
  conf->restart_frame_num++;
  conf->frame_num++;
  //fprintf(stderr, "put_grab_pic_buf: frame time %llu\n", conf->vfrm->time);
  vframe_src_frame_update(vfrm);
  return VGRAB_OK;
}

static int
put_grab_pic_buf_start(VGRAB_CONFIG *conf, int num)
{
  VFRAME *vfrm;

#ifdef DEBUG
  if (conf->restart_frame_num > conf->restart_overlap_frame_num ||
      conf->restart_frame_num <= num) {
    fprintf(stderr, "put_grab_pic_buf_end: overlap frame overrun.\n");
    exit(1);
  }
#endif /* DEBUG */

  vfrm = vframe_src_frame_get();
  if (vfrm == NULL) {
#ifdef DEBUG
    fprintf(stderr, "put_grab_pic_buf_start: frame %d drop, total %d\n", conf->frame_num, conf->drop_frame_num);
#endif /* DEBUG */
    vfrm = vframe_ext_frame_get();
    if (vfrm == NULL) {
      fprintf(stderr, "put_grab_pic_buf_start: vframe_ext_frame_get() failed.\n");
      pthread_mutex_lock(&vgrab_conf.mutex);
      conf->state = STATE_ERROR;
      pthread_mutex_unlock(&vgrab_conf.mutex);
      mgr_error_quit();
      return VGRAB_FAIL;
    }
    vfrm->flag |= DROP_VFRAME;
    vfrm->data_length = 0;
    vfrm->csp = CSP_UNKNOWN;
    conf->drop_frame_num++;
  } else {
    memcpy(vfrm->data, conf->restart_frame[num].data,
         	conf->pic_data_size);
    vfrm->data_length = conf->pic_data_size;
    vfrm->csp = conf->csp;
  }
  vfrm->time = conf->restart_frame[num].frame_time;
  conf->frame_num++;
  vframe_src_frame_update(vfrm);

  return VGRAB_OK;
}

static int
put_ext_vframe(VGRAB_CONFIG *conf, uint32_t flag)
{
  VFRAME *vfrm;

  if (!conf) {
    fprintf(stderr, "put_ext_vframe: conf is NULL.\n");
  }

  vfrm = vframe_ext_frame_get();
#ifndef NDEBUG
  if (!vfrm || !(vfrm->flag & EXT_VFRAME)) {
    fprintf(stderr, "put_ext_vframe: invalid frame. %u !\n", vfrm->flag);
    exit(1);
  }
#else
  if (!vfrm)
    return VGRAB_FAIL;
#endif /* NDEBUG */
  vfrm->flag |= flag;
  vfrm->time = 0;
  vframe_src_frame_update(vfrm);
  return VGRAB_OK;
}

static inline int
get_state(void)
{
  int state;
  pthread_mutex_lock(&vgrab_conf.mutex);
  state = vgrab_conf.state;
  pthread_mutex_unlock(&vgrab_conf.mutex);
  return state;
}

static inline void
state_change_done(VGRAB_CONFIG *conf)
{
  pthread_mutex_lock(&conf->mutex);
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

static inline void
set_state(VGRAB_CONFIG *conf, int state)
{
  pthread_mutex_lock(&conf->mutex);
  conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP|STATE_QUIT);
  conf->state |= state;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

/* thread function */

static void*
vgrab_thread(void *data)
{
  int state;
  int real_time = 0;
  unsigned char *grab_buf;
  int repeat_count = 0;
  VGRAB_CONFIG *conf = &vgrab_conf;
  uint32_t vinfo_flag = 0;
  int start_flag = 0;
  int restart_count = 0;

  // avoid compile warnning, unnecessary
  data = NULL;
  // avoid compile warnning, unnecessary

  v4l_grab_pic_start ();

  init_grab_pic(conf);
  init_grab_time(conf);

  if (!real_time) {
    init_frame_time(conf);
  }

  pthread_mutex_lock(&conf->mutex);
  real_time = conf->real_time_flag;
  vgrab_thread_running = 1;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);

  while (1) {
    grab_buf = get_grab_pic_buf(conf);

    state = get_state();

    if (state & STATE_ERROR) {
#ifdef DEBUG
      fprintf(stderr, "vgrab_thread: STATE_ERROR break\n");
#endif /* DEBUG */
      if (aread_set_vinfo(AREAD_FLAG_ERROR, conf->grab_time) != AREAD_OK) {
	fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
      }
      break;
    } else if (state & STATE_QUIT) {
      if (aread_set_vinfo(AREAD_FLAG_QUIT, conf->grab_time) != AREAD_OK) {
	fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
      }
      if (put_ext_vframe(conf, QUIT_VFRAME) == VGRAB_FAIL) {
	fprintf(stderr, "vgrab_thread: put_ext_vframe() failed.\n");
	set_state(conf, STATE_ERROR);
      }
      break;
    } else if (state & (STATE_START|STATE_PAUSE)) {
      state_change_done(conf);

      pthread_mutex_lock(&conf->mutex);
      if (!conf->rec_start_flag) {
	conf->rec_start_flag = 1;
      }
      pthread_mutex_unlock(&conf->mutex);

      conf->frame_num = 0;
      conf->drop_frame_num = 0;
      restart_count = 0;
      conf->restart_frame_num = 0;

      vinfo_flag = AREAD_FLAG_DROP;
      if (real_time) {
        grab_buf = get_grab_pic_buf(conf);
        grab_buf = get_grab_pic_buf(conf);
        if (aread_set_vinfo(vinfo_flag, conf->last_grab_time) != AREAD_OK) {
	  fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
        }
      } else {
        grab_buf = get_grab_pic_buf(conf);
	while (conf->grab_time < conf->next_time)
          grab_buf = get_grab_pic_buf(conf);
	while (conf->grab_time >= conf->next_time) {
          if (aread_set_vinfo(vinfo_flag, conf->next_time) != AREAD_OK) {
	    fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
          }
	  set_next_frame_time(conf);
	}
      }

      while (state & (STATE_START|STATE_PAUSE)) {
        int rec_start_flag;
        int i;
        grab_buf = get_grab_pic_buf(conf);

	if (real_time) {
	  if (state & STATE_START) {
            pthread_mutex_lock(&conf->mutex);
	    rec_start_flag = conf->rec_start_flag;
            pthread_mutex_unlock(&conf->mutex);
	    vinfo_flag = AREAD_FLAG_RECORD;
            if (rec_start_flag) {
	      if (restart_count > 0) {
                if (aread_set_vinfo(vinfo_flag,conf->last_grab_time)!=AREAD_OK){
	          fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
                }
		put_grab_pic_buf_end(conf);
		restart_count--;
	      } else {
                if (start_flag) {
                  if (aread_set_vinfo(AREAD_FLAG_STOP, conf->last_grab_time) !=
		      AREAD_OK){
	            fprintf(stderr,"vgrab_thread: aread_set_vinfo() failed.\n");
                  }
 		  if (put_ext_vframe(conf, END_VFRAME) == VGRAB_FAIL) {
	            fprintf(stderr, "vgrab_thread: put_ext_vframe() failed.\n");
	            set_state(conf, STATE_ERROR);
	            break;
		  }
                }
                if (put_ext_vframe(conf, START_VFRAME) == VGRAB_FAIL) {
	          fprintf(stderr, "vgrab_thread: put_ext_vframe() failed.\n");
	          set_state(conf, STATE_ERROR);
	          break;
                }
                conf->frame_num = 0;
                conf->drop_frame_num = 0;
	        vinfo_flag = AREAD_FLAG_START;
		for (i = 0; i < conf->restart_frame_num; i++) {
                  if (aread_set_vinfo(vinfo_flag,conf->restart_frame[i].time)!=
		      AREAD_OK){
	            fprintf(stderr,"vgrab_thread: aread_set_vinfo() failed.\n");
                  }
	          vinfo_flag = AREAD_FLAG_RECORD;
		  put_grab_pic_buf_start(conf, i);
		}
                if (aread_set_vinfo(vinfo_flag,conf->last_grab_time)!=AREAD_OK){
	          fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
                }
                put_grab_pic_buf(conf);
	        start_flag = 1;
                //pthread_mutex_lock(&conf->mutex);
	        conf->rec_start_flag = 0;
                //pthread_mutex_unlock(&conf->mutex);
	        restart_count = conf->restart_overlap_frame_num;
		conf->restart_frame_num = 0;
	      }
	    } else {
              if (aread_set_vinfo(vinfo_flag,conf->last_grab_time) != AREAD_OK){
	        fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
              }
              put_grab_pic_buf(conf);
	    }
	  } else {
	    if (start_flag && conf->restart_frame_num > 0) {
	      restart_count = conf->restart_overlap_frame_num;
	      conf->restart_frame_num = 0;
	    }
	    vinfo_flag = AREAD_FLAG_DROP;
            if (aread_set_vinfo(vinfo_flag, conf->last_grab_time) != AREAD_OK) {
	      fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
            }
	  }
	} else {
	  repeat_count = 0;
	  while (conf->grab_time >= conf->next_time) {
	    if (state & STATE_START) {
              pthread_mutex_lock(&conf->mutex);
	      rec_start_flag = conf->rec_start_flag;
              pthread_mutex_unlock(&conf->mutex);
	      vinfo_flag = AREAD_FLAG_RECORD;
              if (rec_start_flag) {
	        if (restart_count > 0) {
                  if (aread_set_vinfo(vinfo_flag,conf->next_time) !=
		      AREAD_OK){
	            fprintf(stderr,"vgrab_thread: aread_set_vinfo() failed.\n");
                  }
		  put_grab_pic_buf_end(conf);
		  restart_count--;
	        } else {
                  if (start_flag) {
                    if (aread_set_vinfo(AREAD_FLAG_STOP, conf->next_time) !=
		        AREAD_OK){
	              fprintf(stderr,"vgrab_thread: aread_set_vinfo failed.\n");
                    }
 		    if (put_ext_vframe(conf, END_VFRAME) == VGRAB_FAIL) {
	             fprintf(stderr,"vgrab_thread: put_ext_vframe() failed.\n");
	              set_state(conf, STATE_ERROR);
	              break;
		    }
                  }
                  if (put_ext_vframe(conf, START_VFRAME) == VGRAB_FAIL) {
	            fprintf(stderr, "vgrab_thread: put_ext_vframe() failed.\n");
	            set_state(conf, STATE_ERROR);
	            break;
                  }
                  conf->frame_num = 0;
                  conf->drop_frame_num = 0;
	          vinfo_flag = AREAD_FLAG_START;
		  for (i = 0; i < conf->restart_frame_num; i++) {
                    if(aread_set_vinfo(vinfo_flag,conf->restart_frame[i].time)!=
		      AREAD_OK){
	              fprintf(stderr,"vgrab_thread: aread_set_vinfo failed.\n");
                    }
	            vinfo_flag = AREAD_FLAG_RECORD;
		    put_grab_pic_buf_start(conf, i);
		  }
                  if (aread_set_vinfo(vinfo_flag, conf->next_time) !=
		      AREAD_OK){
	            fprintf(stderr,"vgrab_thread: aread_set_vinfo() failed.\n");
                  }
                  put_grab_pic_buf(conf);
	          start_flag = 1;
                  //pthread_mutex_lock(&conf->mutex);
	          conf->rec_start_flag = 0;
                  //pthread_mutex_unlock(&conf->mutex);
	          restart_count = conf->restart_overlap_frame_num;
		  conf->restart_frame_num = 0;
	        }
	      } else {
                if (aread_set_vinfo(vinfo_flag,conf->next_time)!=AREAD_OK){
	          fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
                }
                put_grab_pic_buf(conf);
	      }
	    } else {
	      if (start_flag && conf->restart_frame_num > 0) {
	        restart_count = conf->restart_overlap_frame_num;
	        conf->restart_frame_num = 0;
	      }
	      vinfo_flag = AREAD_FLAG_DROP;
              if (aread_set_vinfo(vinfo_flag,conf->next_time)!=AREAD_OK) {
	        fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
              }
	    }
	    set_next_frame_time(conf);
	    if (repeat_count > 0) {
	      fprintf(stderr,"vgrab_thread: repeat_count %d\n", repeat_count);
	    }
	    repeat_count++;
	  }
	}
	pthread_mutex_lock(&conf->mutex);
	if ((state & (STATE_START|STATE_PAUSE)) &&
	    (state & STATE_INTERFACE_WAIT)) {
          pthread_cond_signal(&conf->interface_wait_cond);
	}
	state = conf->state;
	pthread_mutex_unlock(&conf->mutex);
      }
      vinfo_flag = AREAD_FLAG_STOP;
      if (real_time) {
        if (aread_set_vinfo(vinfo_flag, conf->grab_time) != AREAD_OK) {
          fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
        }
      } else {
        if (aread_set_vinfo(vinfo_flag, conf->next_time) != AREAD_OK) {
          fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
        }
      }
      if (put_ext_vframe(conf, END_VFRAME) == VGRAB_FAIL) {
        fprintf(stderr, "vgrab_thread: put_ext_vframe() failed.\n");
        set_state(conf, STATE_ERROR);
        break;
      }
      start_flag = 0;
    } else if (state & STATE_STOP) {
      state_change_done(conf);

      vinfo_flag = AREAD_FLAG_DROP;
      while (state & STATE_STOP) {
        grab_buf = get_grab_pic_buf(conf);
        if (real_time) {
          if (aread_set_vinfo(vinfo_flag, conf->last_grab_time) != AREAD_OK) {
	    fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
          }
        } else {
          while (conf->grab_time >= conf->next_time) {
            if (aread_set_vinfo(vinfo_flag, conf->next_time) != AREAD_OK) {
	      fprintf(stderr, "vgrab_thread: aread_set_vinfo() failed.\n");
            }
	    set_next_frame_time(conf);
	  }
        }
        state = get_state();
      }
    }
  }

  v4l_grab_pic_stop();

  pthread_mutex_lock(&conf->mutex);
  vgrab_thread_running = 0;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
  //state_change_done(conf);

#ifdef DEBUG
  fprintf(stderr, "vgrab_thread: done\n");
#endif /* DEBUG */
  return NULL;
}

/* interface functions */

void
vgrab_set_restart(void)
{
  VGRAB_CONFIG *conf = &vgrab_conf;

  if (!vgrab_initialized)
    return;

  pthread_mutex_lock(&conf->mutex);
  conf->rec_start_flag = 1;
  pthread_mutex_unlock(&conf->mutex);
}

int
vgrab_state_error(void)
{
  VGRAB_CONFIG *conf = &vgrab_conf;
  int ret = VGRAB_FAIL;

  if (!vgrab_initialized)
    return VGRAB_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!vgrab_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VGRAB_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if (conf->state & STATE_ERROR) {
    ret = VGRAB_OK;
  } else {
    conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP);
    conf->state |= STATE_ERROR;
#if 0
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT)
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
#endif
    ret = VGRAB_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
vgrab_state_start(void)
{
  int ret = VGRAB_FAIL;
  VGRAB_CONFIG *conf = &vgrab_conf;

  if (!vgrab_initialized)
    return VGRAB_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!vgrab_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VGRAB_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if (conf->state & (STATE_START|STATE_QUIT|STATE_ERROR)) {
    ret = VGRAB_OK;
  } else {
    conf->state &= ~(STATE_PAUSE|STATE_STOP);
    conf->state |= STATE_START;
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT)
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    ret = VGRAB_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
vgrab_state_stop(void)
{
  VGRAB_CONFIG *conf = &vgrab_conf;
  int ret = VGRAB_FAIL;

  if (!vgrab_initialized)
    return VGRAB_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!vgrab_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VGRAB_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if (conf->state & (STATE_STOP|STATE_QUIT|STATE_ERROR)) {
    ret = VGRAB_OK;
  } else {
    conf->state &= ~(STATE_START|STATE_PAUSE);
    conf->state |= STATE_STOP;
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT)
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    ret = VGRAB_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
vgrab_state_quit(void)
{
  VGRAB_CONFIG *conf = &vgrab_conf;
  int ret = VGRAB_FAIL;

  if (!vgrab_initialized)
    return VGRAB_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!vgrab_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
//    printf("!thread_running\n");
    return VGRAB_OK;
  }
//    printf("thread_running\n");
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if (conf->state & (STATE_QUIT|STATE_ERROR)) {
    ret = VGRAB_OK;
  } else {
    conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP);
    conf->state |= STATE_QUIT;
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT)
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    ret = VGRAB_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
vgrab_state_pause(void)
{
  VGRAB_CONFIG *conf = &vgrab_conf;
  int ret = VGRAB_FAIL;

  if (!vgrab_initialized)
    return VGRAB_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!vgrab_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VGRAB_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if ((conf->state & (STATE_PAUSE|STATE_STOP|STATE_QUIT|STATE_ERROR))) {
    ret = VGRAB_OK;
  } else {
    conf->state &= ~STATE_START;
    conf->state |= STATE_PAUSE;
#if 0
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT)
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
#endif
    ret = VGRAB_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
vgrab_state_toggle_pause(void)
{
  VGRAB_CONFIG *conf = &vgrab_conf;
  int ret = VGRAB_FAIL;

  if (!vgrab_initialized)
    return VGRAB_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!vgrab_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VGRAB_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if ((conf->state & (STATE_STOP|STATE_QUIT|STATE_ERROR))) {
    ret = VGRAB_OK;
  } else {
    if (conf->state & STATE_START) {
      conf->state &= ~STATE_START;
      conf->state |= STATE_PAUSE;
    } else if (conf->state & STATE_PAUSE) {
      conf->state &= ~STATE_PAUSE;
      conf->state |= STATE_START;
    }
#if 0
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT)
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
#endif
    ret = VGRAB_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
vgrab_csp_to_vdev_csp(int csp)
{
  int ret;
//  if (csp == CSP_YV12_INTERLACE) {
//    ret = CSP_RGB32;
  if (csp == CSP_YV12) {
    ret = CSP_YUV420P;
  } else {
    ret = csp;
  }
  return ret;
}

int
vgrab_error_quit (void)
{
  VGRAB_CONFIG *conf = &vgrab_conf;
  int retcode;
  void* retval;
  int i;

  if (!vgrab_initialized) {
    if (conf->restart_frame)
      fprintf (stderr, "vgrab_quit: restart_frame remains.\n");
    return VGRAB_OK;
  }

  vgrab_state_error();

  retcode = pthread_join(conf->vgrab_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "vgrab_quit: thread join error %d\n", retcode);
//  pthread_mutex_destroy(&conf->mutex);

  if (conf->restart_frame) {
    for (i = 0; i < conf->restart_overlap_frame_num; i++)
      free(conf->restart_frame[i].bufp);
    free(conf->restart_frame);
  }
  conf->restart_frame = NULL;
  conf->restart_overlap_frame_num = 0;

  vgrab_initialized = 0;
//  fprintf (stderr, "vgrab_error_quit: done\n");
  return VGRAB_OK;
}

int
vgrab_quit (void)
{
  VGRAB_CONFIG *conf = &vgrab_conf;
  int retcode;
  void* retval;
  int i;

  if (!vgrab_initialized) {
#ifdef DEBUG
    if (conf->restart_frame)
      fprintf (stderr, "vgrab_quit: restart_frame remains.\n");
#endif /* DEBUG */
    return VGRAB_OK;
  }

  vgrab_state_quit();

  retcode = pthread_join(conf->vgrab_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "vgrab_quit: thread join error %d\n", retcode);
//  pthread_mutex_destroy(&conf->mutex);

  if (conf->restart_frame) {
    for (i = 0; i < conf->restart_overlap_frame_num; i++)
      free(conf->restart_frame[i].bufp);
    free(conf->restart_frame);
  }
  conf->restart_frame = NULL;
  conf->restart_overlap_frame_num = 0;

  vgrab_initialized = 0;

//  fprintf (stderr, "vgrab_quit: done\n");
  return VGRAB_OK;
}

int
vgrab_init (int width, int height, double fps, int csp)
{
  VGRAB_CONFIG *conf = &vgrab_conf;
  int retcode;
  int need_config = 0;
  int depth;
  int restart_frame_num;
  int v4l_csp;
  int i;
static int first = 1;

#if 0
  if (v4l_init () < 0) {
    return VGRAB_FAIL;
  }
#endif

#if 0
//  printf ("vgrab_init: width %d, height %d, fps %f, csp %d\n",
//      *width, *height, fps, csp);
#endif

  if (first) {
    vgrab_initialized = 0;
    vgrab_thread_running = 0;
    conf->restart_frame = NULL;
    conf->width = conf->height = 0;
    conf->fps = 0;
    conf->csp = CSP_UNKNOWN;
    conf->pic_data_size = 0;
    conf->pic_data = NULL;
    conf->frame_num = conf->drop_frame_num = 0;
    first = 0;
    need_config = 1;
  }

  if (!vgrab_initialized)
    need_config = 1;
  else if (conf->width != width)
    need_config = 1;
  else if (conf->height != height)
    need_config = 1;
  else if (conf->fps != fps)
    need_config = 1;
  else if (conf->csp != csp)
    need_config = 1;

  if (!need_config) {
    return VGRAB_OK;
  }

  if (vgrab_initialized) {
    fprintf(stderr, "vgrab_init: ERROR vgrab_initialized TRUE !!\n");
    vgrab_quit();
  }

  conf->width = width;
  conf->height = height;
  conf->fps = fps;

  if (v4l_capture_cap() == V4L_FAIL) {
    fprintf(stderr, "vgrab_init: video device cannot capture.\n");
    return VGRAB_FAIL;
  }

  if (v4l_get_window_size(&width, &height) == V4L_FAIL) {
    fprintf(stderr, "vgrab_init: v4l_get_window_size() failed.\n");
    return VGRAB_FAIL;
  }
  if (width != conf->width || height != conf->height) {
    fprintf(stderr, "vgrab_init: can't set picture size %dx%d, use %dx%d\n",
	conf->width, conf->height, width, height);
    conf->width = width;
    conf->height = height;
  }

  if ((v4l_csp = v4l_get_csp()) == V4L_FAIL) {
    fprintf(stderr, "vgrab_init: v4l_get_csp() failed.\n");
    return VGRAB_FAIL;
  }

#if 0
  if (csp == CSP_YV12_INTERLACE) {
    if (v4l_csp != CSP_RGB32) {
      fprintf(stderr, "vgrab_init: csp invalid.\n");
      return VGRAB_FAIL;
    }
  } else 
#endif
  if (csp == CSP_I420 || csp == CSP_YUV420P || csp == CSP_YV12) {
    if (v4l_csp != CSP_I420 && v4l_csp != CSP_YUV420P) {
      fprintf(stderr, "vgrab_init: csp invalid.\n");
      return VGRAB_FAIL;
    }
  } else if (csp != v4l_csp) {
    fprintf(stderr, "vgrab_init: csp invalid.\n");
    return VGRAB_FAIL;
  }

  conf->csp = csp;
  conf->fps = fps;
  conf->real_time_flag = 0;

  if (fps == 0) {
    conf->real_time_flag = 1;
    switch (v4l_get_norm()) {
      case NORM_PAL:
	fps = 25;
	break;
      case NORM_NTSC:
	fps = 29.97;
	break;
      default:
	fps = 25;
	break;
    }
  }

  conf->rec_start_flag = 0;
  conf->frame_num = 0;
  conf->drop_frame_num = 0;

#if 0
  if (csp == CSP_YV12_INTERLACE) {
    depth = csp_to_pixel_depth(csp);
    conf->pic_data_size = conf->width * conf->height * depth / 8;
    conf->yv12_interlace_width = conf->width * 2;
    conf->yv12_interlace_height = conf->height / 2;
    conf->yv12_interlace_y_stride = conf->width * 2;
    conf->yv12_interlace_uv_stride = conf->width;
    conf->yv12_interlace_rgb_stride = conf->width * 2;
    conf->yv12_interlace_y_offset = 0;
    conf->yv12_interlace_v_offset = conf->width * conf->height;
    conf->yv12_interlace_u_offset = (conf->width * conf->height) / 4 * 5;
  } else {
    depth = csp_to_pixel_depth(csp);
    conf->pic_data_size = conf->width * conf->height * depth / 8;
  }
#else
  if (csp == CSP_YV12) {
    conf->y_size   = conf->width * conf->height;
    conf->uv_size  = conf->y_size / 4;
    conf->v_offset = conf->y_size;
    conf->u_offset = conf->y_size + conf->uv_size;
  }
  depth = csp_to_pixel_depth(csp);
  conf->pic_data_size = conf->width * conf->height * depth / 8;
#endif

  restart_frame_num = (int)(fps * 0.3 + 0.5);
  if (restart_frame_num <= 0)
    restart_frame_num = 1;

  conf->restart_frame = (RESTART_FRAME*)mem_malloc(
      sizeof(RESTART_FRAME) * restart_frame_num, "vgrab_init");

  for (i = 0; i < restart_frame_num; i++) {
    conf->restart_frame[i].time = 0;
    conf->restart_frame[i].frame_time = 0;
#ifdef ARCH_X86
    conf->restart_frame[i].bufp =
      (uint8_t*) mem_malloc(conf->pic_data_size + 15, "vgrab_init");
    conf->restart_frame[i].data =
      (uint8_t*)((((uint32_t)(conf->restart_frame[i].bufp)) + 15) & 0xfffffff0);
#else
    conf->restart_frame[i].bufp =
      (uint8_t*) mem_malloc(conf->pic_data_size, "vgrab_init");
    conf->restart_frame[i].data = conf->restart_frame[i].bufp;
#endif
  }

  conf->restart_overlap_frame_num = restart_frame_num;
  conf->restart_frame_num = 0;
  conf->pic_data = NULL;
  conf->state = STATE_STOP;

  pthread_mutex_init (&conf->mutex, NULL);
  pthread_cond_init (&conf->interface_wait_cond, NULL);
  pthread_cond_init (&conf->thread_wait_cond, NULL);
  retcode = pthread_create (&conf->vgrab_th, NULL, vgrab_thread, NULL);
  if (retcode != 0) {
    fprintf (stderr, "vgrab_init: thread create failed %d\n", retcode);
    for (i = 0; i < restart_frame_num; i++) {
      free(conf->restart_frame[i].bufp);
    }
    free(conf->restart_frame);
    conf->restart_frame = NULL;
    return VGRAB_FAIL;
  }

  pthread_mutex_lock(&conf->mutex);
  if (!vgrab_thread_running) {
    conf->state |= STATE_INTERFACE_WAIT;
    pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
  }
  pthread_mutex_unlock(&conf->mutex);

  vgrab_initialized = 1;

  return VGRAB_OK;
}

