/* venc.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <inttypes.h>
#include <pthread.h>
#include <errno.h>

#include "vcodec.h"
#include "venc.h"
#include "csp.h"
#include "state.h"
#include "vframe_mgr.h"

#ifdef DEBUG
#if 0
static void
print_vflag(uint32_t flag)
{
  printf("venc: ");
  if (flag & NORMAL_VFRAME)
    printf("NORMAL_VFRAME ");
  if (flag & EXT_VFRAME)
    printf("EXT_VFRAME ");
  if (flag & START_VFRAME)
    printf("START_VFRAME ");
  if (flag & END_VFRAME)
    printf("END_VFRAME ");
  if (flag & QUIT_VFRAME)
    printf("QUIT_VFRAME ");
  if (flag & DROP_VFRAME)
    printf("DROP_VFRAME ");
  printf("!! \n");
}
#endif
#endif /* DEBUG */

static VENC_CONFIG venc_conf;

static int venc_initialized = 0;
static int venc_thread_running = 0;

static inline int
get_state(VENC_CONFIG *conf)
{
  int state;
  pthread_mutex_lock(&conf->mutex);
  state = conf->state;
  pthread_mutex_unlock(&conf->mutex);
  return state;
}

static inline void
state_change_done(VENC_CONFIG *conf)
{
  pthread_mutex_lock(&conf->mutex);
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

static inline void
set_state(VENC_CONFIG *conf, int state)
{
  pthread_mutex_lock(&conf->mutex);
  conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP|STATE_QUIT);
  conf->state |= state;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

static int
put_enc_vframe(VENC_CONFIG *conf, VFRAME *svfrm)
{
  VFRAME *dvfrm;
  int data_length;
  unsigned int ext_flag;
  int ret;

#ifdef DEBUG
  if (svfrm->flag & EXT_VFRAME) {
    fprintf(stderr, "put_enc_vframe: error EXT_VFRAME, flag %u.\n",svfrm->flag);
    exit(1);
  }
#endif /* DEBUG */

  while((dvfrm = vframe_enc_dst_frame_get()) == NULL) {
#ifdef DEBUG
    fprintf(stderr, "put_enc_vframe: dvfrm == NULL\n");
#endif /* DEBUG */
    pthread_mutex_lock(&conf->mutex);
    if (!conf->wait_cb_called && !(conf->state & STATE_ERROR)) {
      conf->state |= STATE_THREAD_WAIT;
      pthread_cond_wait(&conf->thread_wait_cond, &conf->mutex);
    }
    conf->wait_cb_called = 0;
    if (conf->state & STATE_ERROR) {
      pthread_mutex_unlock(&conf->mutex);
      return VENC_FAIL;
    }
    pthread_mutex_unlock(&conf->mutex);
  }

  ext_flag = 0;
  data_length = vcodec_encode(svfrm->data, dvfrm->data,
      dvfrm->data_buf_size, &ext_flag);
  if (data_length < 0) {
    fprintf(stderr, "put_enc_vframe: vcodec_encode() error %d.\n",data_length);
#ifdef DEBUG
    exit(1);
#else
    data_length = 0;
    ret = VENC_FAIL;
#endif /* DEBUG */
  } else {
    ret = VENC_OK;
  }

  if (ext_flag & VCODEC_IS_INTRA) {
    dvfrm->flag = svfrm->flag | INTRA_VFRAME;
  } else {
    dvfrm->flag = svfrm->flag | INTER_VFRAME;
  }
  dvfrm->data_length = data_length;
  dvfrm->time = svfrm->time;
//#ifdef DEBUG
//	fprintf(stderr, "venc_thread: data_size %d\n", data_size);
//#endif /* DEBUG */

  vframe_enc_dst_frame_update(dvfrm);

  return ret;
}

/* thread function */

static void*
venc_thread (void *data)
{
  VENC_CONFIG *conf = &venc_conf;
  int state;
  VFRAME *svfrm = NULL;
  int rec_start_flag = 0;
  int quit_flag = 0;
  int drop_frame_num = 0;

  // avoid compile warnning, unnecessary
  data = NULL;
  // avoid compile warnning, unnecessary

  pthread_mutex_lock(&conf->mutex);
  venc_thread_running = 1;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);

  while (1) {
    state = get_state(conf);

    if (quit_flag || state & STATE_ERROR) {
      break;
    }
//    else
//      state_change_done(conf);

    while (!(state & STATE_ERROR) &&
	(svfrm = vframe_enc_src_frame_get()) == NULL) {
      pthread_mutex_lock(&conf->mutex);
      if (!conf->wait_cb_called) {
        conf->state |= STATE_THREAD_WAIT;
        pthread_cond_wait(&conf->thread_wait_cond, &conf->mutex);
      }
      conf->wait_cb_called = 0;
      state = conf->state;
      pthread_mutex_unlock(&conf->mutex);
    }

    if (!svfrm) {
      continue;
    } else if (state & STATE_ERROR) {
#ifdef DEBUG
      if (state & STATE_ERROR) {
        fprintf(stderr, "venc_thread: state STATE_ERROR, svfrm %p.\n", svfrm);
      }
#endif /* DEBUG */
      vframe_enc_src_frame_update(svfrm);
      continue;
    }

    //print_vflag(svfrm->flag);
#ifdef DEBUG
    if (svfrm->flag & NORMAL_VFRAME &&
        svfrm->flag & (START_VFRAME|END_VFRAME|QUIT_VFRAME)) {
      fprintf(stderr, "venc_thread: invalid ext frame.\n");
      exit(1);
    }
#endif /* DEBUG */

    if (svfrm->flag & START_VFRAME) {
#ifdef DEBUG
      if (rec_start_flag) {
        fprintf(stderr, "venc_thread: invalid start.\n");
        exit(1);
      }
#endif /* DEBUG */
      rec_start_flag = 1;
      set_state(conf, STATE_START);
      drop_frame_num = 0;
    } else if (svfrm->flag & END_VFRAME) {
      if (rec_start_flag) {
        if (vcodec_encode_reset() != VCODEC_OK) {
          fprintf(stderr, "venc_thread: vcodec_reset() error.\n");
          conf->state = STATE_ERROR;
          break;
        }
      }
      rec_start_flag = 0;
      set_state(conf, STATE_STOP);
    } else if (svfrm->flag & QUIT_VFRAME) {
      rec_start_flag = 0;
      set_state(conf, STATE_QUIT);
      quit_flag = 1;
      drop_frame_num = 0;
    } else if (svfrm->flag & DROP_VFRAME) {
      drop_frame_num++;
//      fprintf(stderr, "venc_thread: drop_frame_num %d.\n", drop_frame_num);
    } else {
#ifdef DEBUG
      if (svfrm->flag & EXT_VFRAME) {
        fprintf(stderr, "venc_thread: invalid src frame.\n");
        exit(1);
      }
      if (!rec_start_flag) {
        fprintf(stderr, "venc_thread: invalid state frame.\n");
        exit(1);
      }
#endif /* DEBUG */
      if (put_enc_vframe(conf, svfrm) == VENC_FAIL) {
        set_state(conf, STATE_ERROR);
#ifdef DEBUG
        fprintf(stderr, "venc_thread: put_enc_vframe() failed.\n");
        exit(1);
#endif /* DEBUG */
      }
    }

    vframe_enc_src_frame_update(svfrm);
  }

  pthread_mutex_lock(&conf->mutex);
  venc_thread_running = 0;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
//  state_change_done(conf);

#ifdef DEBUG
  fprintf(stderr, "venc_thread: done\n");
#endif /* DEBUG */
  return NULL;
}

/* interface functions */

void
venc_enc_wait_cb(void)
{
  VENC_CONFIG *conf = &venc_conf;
  
  if (!venc_initialized)
    return;

  pthread_mutex_lock(&conf->mutex);
  conf->wait_cb_called = 1;
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

int
venc_state_error(void)
{
  VENC_CONFIG *conf = &venc_conf;
  int ret = VENC_FAIL;

  if (!venc_initialized)
    return VENC_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!venc_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VENC_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if (conf->state & STATE_ERROR) {
    ret = VENC_OK;
  } else {
    conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP);
    conf->state |= STATE_ERROR;
#if 0
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT) {
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
#endif
    ret = VENC_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
venc_state_start(void)
{
  VENC_CONFIG *conf = &venc_conf;
  int ret = VENC_FAIL;

  if (!venc_initialized)
    return VENC_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!venc_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VENC_OK;
  }
  if (conf->state & (STATE_START|STATE_QUIT|STATE_ERROR))
    ret = VENC_OK;
  else {
    while (!(conf->state & STATE_START)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = VENC_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
venc_state_stop(void)
{
  VENC_CONFIG *conf = &venc_conf;
  int ret = VENC_FAIL;

  if (!venc_initialized)
    return VENC_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!venc_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VENC_OK;
  }
  if (conf->state & (STATE_STOP|STATE_QUIT|STATE_ERROR))
    ret = VENC_OK;
  else {
    while (!(conf->state & STATE_STOP)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = VENC_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
venc_state_quit(void)
{
  VENC_CONFIG *conf = &venc_conf;
  int ret = VENC_FAIL;

  if (!venc_initialized)
    return VENC_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!venc_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return VENC_OK;
  }
  if (conf->state & (STATE_QUIT|STATE_ERROR))
    ret = VENC_OK;
  else {
    while (!(conf->state & STATE_QUIT)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = VENC_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
venc_state_pause(void)
{
  if (!venc_initialized)
    return VENC_OK;
  return VENC_OK;
}

int
venc_state_toggle_pause(void)
{
  if (!venc_initialized)
    return VENC_OK;
  return VENC_OK;
}

int
venc_error_quit (void)
{
  VENC_CONFIG *conf = &venc_conf;
  int retcode;
  void* retval;

  if (!venc_initialized)
    return VENC_OK;

  venc_state_error();

  retcode = pthread_join (conf->venc_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "venc_quit() thread join error %d\n", retcode);

//  pthread_mutex_destroy(&conf->mutex);
//  vframe_mgr_quit();
//  vcodec_quit();

  venc_initialized = 0;
//  fprintf (stderr, "venc_quit: done\n");
  return VENC_OK;
}

int
venc_quit (void)
{
  VENC_CONFIG *conf = &venc_conf;
  int retcode;
  void* retval;

  if (!venc_initialized)
    return VENC_OK;

  venc_state_quit();

  retcode = pthread_join (conf->venc_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "venc_quit() thread join error %d\n", retcode);

//  pthread_mutex_destroy(&conf->mutex);
//  vframe_mgr_quit();
//  vcodec_quit();

  venc_initialized = 0;
//  fprintf (stderr, "venc_quit: done\n");
  return VENC_OK;
}

int
venc_init (int width, int height, double fps, unsigned long fourcc, int in_csp)
{
  VENC_CONFIG *conf = &venc_conf;
  int retcode;
  int need_config = 0;
static int first = 1;
//  int in_depth, out_depth, max_pixel_depth;

  if (first) {
    venc_initialized = 0;
    venc_thread_running = 0;
    conf->fourcc = 0;
    conf->width = conf->height = 0;
    conf->fps = 0;
    conf->in_csp = CSP_UNKNOWN;
    conf->wait_cb_called = 0;
    need_config = 1;
    first = 0;
  }

  if (width == 0)
    width = 320;
  if (height == 0)
    height = 240;
  if (fps == 0)
    fps = 29.97;

  if (!venc_initialized)
    need_config = 1;
  else if (width != conf->width)
    need_config = 1;
  else if (height != conf->height)
    need_config = 1;
  else if (fps != conf->fps)
    need_config = 1;
  else if (fourcc != conf->fourcc)
    need_config = 1;
  else if (in_csp != conf->in_csp)
    need_config = 1;

  if (!need_config) {
    return VENC_OK;
  }

  if (venc_initialized) {
    fprintf(stderr, "venc_init: ERROR venc_initialized TRUE !!\n");
    venc_quit();
  }

  conf->width = width;
  conf->height = height; 
  conf->fps = fps;
  conf->fourcc = fourcc;
  conf->in_csp = in_csp;

  vframe_set_enc_wait_cb(venc_enc_wait_cb);

  conf->wait_cb_called = 0;
  conf->state = STATE_STOP;
  pthread_mutex_init (&conf->mutex, NULL);
  pthread_cond_init (&conf->interface_wait_cond, NULL);
  pthread_cond_init (&conf->thread_wait_cond, NULL);

  retcode = pthread_create (&conf->venc_th, NULL, venc_thread, NULL);
  if (retcode != 0) {
    fprintf (stderr, "venc_init() thread create failed %d\n", retcode);
    return VENC_FAIL;
  }

  pthread_mutex_lock(&conf->mutex);
  if (!venc_thread_running) {
    conf->state |= STATE_INTERFACE_WAIT;
    pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
  }
  pthread_mutex_unlock(&conf->mutex);

  venc_initialized = 1;

  return VENC_OK;
}

void
venc_print_param(void)
{
  VENC_CONFIG *conf = &venc_conf;
//  SEL_COMPO *sel;
//  int i;

  fprintf (stderr, "VIDEO PARAM\n");
#if 0
  sel = codecs;
  for (i=0; i < sizeof(codecs)/sizeof(SEL_COMPO); i++, sel++) {
    if (sel->id == conf->fourcc) {
      fprintf (stderr, "video codec:            %s\n", sel->name);
      break;
    }
  }
#endif
  fprintf (stderr, "fps:                    %f\n", conf->fps);
  fprintf (stderr, "dimmention:             %dx%d\n",conf->width,conf->height);
}

