/* vc_xvid.c */

/* References:
 * MPlayer-0.90rc5
 *   ve_xvid.c   < Kim Minh Kaplan & Ri Guyomarch >
 *   xvid_vbr.c  <ed.gomez@wanadoo.fr>
 *   xvid_vbr.h  <ed.gomez@wanadoo.fr>
 */


#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_LIBXVIDCORE

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <limits.h>

#include "xvid.h"

#include "util.h"
#include "vcodec.h"
#include "parseopt.h"
#include "csp.h"
#include "vc_xvid.h"

#include "xvid_vbr.c"

static int vc_xvid_init_called = 0;

static VC_XVID_PARAM vc_xvid_param;

static SEL_COMPO modes[] = {
  { "CBR",    VC_XVID_MODE_CBR },
  { "QUANT",  VC_XVID_MODE_VBR_QUANT },
  { "2PASS1", VC_XVID_MODE_2PASS_1 },
  { "2PASS2", VC_XVID_MODE_2PASS_2 },
};

#if 0
static SEL_COMPO me_methods[] = {
  { "DEFAULT",     0 }
  { "ZERO",        XVID_ME_ZERO },
  { "LOGARITHMIC", XVID_ME_LOGARITHMIC }
  { "FULLSEARCH",  XVID_ME_FULLSEARCH }
  { "PMVFAST",     XVID_ME_PMVFAST }
  { "EPZS",        XVID_ME_EPZS }
};
#endif

static SEL_COMPO quant_types[] = {
  { "DEFAULT",    VC_XVID_QUANT_TYPE_DEFAULT },
  { "H263",       VC_XVID_QUANT_TYPE_H263 },
  { "MPEG",       VC_XVID_QUANT_TYPE_MPEG },
  { "MODULATE",   VC_XVID_QUANT_TYPE_MOD },
//  { "CUSTOM",     VC_XVID_QUANT_TYPE_CUSTOM },
};

#define SWITCH_DEFAULT  0
#define SWITCH_ON       1
#define SWITCH_OFF      2
static SEL_COMPO switch_sel[] = {
  { "DEFAULT", SWITCH_DEFAULT },
  { "ON",      SWITCH_ON },
  { "OFF",     SWITCH_OFF },
};

OptionDef xvid_param[] = {
  {"xvid-mode", "XVID_MODE", HAS_ARG|OPT_SEL, {(void*)&vc_xvid_param.mode}, {VC_XVID_MODE_CBR}, {(int)modes}, sizeof(modes)/sizeof(SEL_COMPO), "XVID: Encoding mode", "sel"},
  {"xvid-bitrate", "XVID_BITRATE", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.bitrate}, {900}, {0}, 100000, "XVID: CBR or 2PASS2 mode bitrate (Kilo)", "d"},
  {"xvid-quant", "XVID_QUANT", HAS_ARG, {(void*)&vc_xvid_param.quant}, {5}, {1}, 31, "XVID: QUANT mode quant", "d"},
//  {"xvid-motion-search", "XVID_MOTION_SEARCH", HAS_ARG, {(void*)&vc_xvid_param.motion_search}, {5}, {0}, 6, "Motion search precision", "m"},
  {"xvid-quality", "XVID_QUALITY", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.quality}, {5}, {0}, 6, "XVID: encode quality", "d"},
  {"xvid-quant-type", "XVID_QUANT_TYPE", HAS_ARG|OPT_SEL, {(void*)&vc_xvid_param.quant_type}, {VC_XVID_QUANT_TYPE_H263}, {(int)quant_types}, sizeof(quant_types)/sizeof(SEL_COMPO), "XVID: Quantization type", "sel"},

  {"xvid-rc-reaction-delay-factor", "XVID_RC_REACTION_DELAY_FACTOR", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.rc_reaction_delay_factor}, {0}, {0}, INT_MAX, "XVID: encode param rc_reaction_delay_factor", "d"},
  {"xvid-rc-averaging-period", "XVID_RC_AVERAGING_PERIOD", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.rc_averaging_period}, {0}, {0}, INT_MAX, "XVID: encode param rc_averaging_period", "d"},
  {"xvid-rc-buffer", "XVID_RC_BUFFER", HAS_ARG, {(void*)&vc_xvid_param.rc_buffer}, {0}, {0}, INT_MAX, "XVID: encode param rc_buffer", "d"},

  {"xvid-min-iquant", "XVID_MIN_IQUANT", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.min_iquant}, {2}, {1}, 31, "XVID: Min I-frame quantizer", "d"},
  {"xvid-max-iquant", "XVID_MAX_IQUANT", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.max_iquant}, {15}, {1}, 31, "XVID: Max I-frame quantizer", "d"},
  {"xvid-min-pquant", "XVID_MIN_PQUANT", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.min_pquant}, {2}, {1}, 31, "XVID: Min P-frame quantizer", "d"},
  {"xvid-max-pquant", "XVID_MAX_PQUANT", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.max_pquant}, {15}, {1}, 31, "XVID: Max P-frame quantizer", "d"},

  {"xvid-min-key-interval", "XVID_MIN_KEY_INTERVAL", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.min_key_interval}, {1}, {0}, INT_MAX, "XVID: Minimum I-frame interval", "d"},
  {"xvid-max-key-interval", "XVID_MAX_KEY_INTERVAL", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.max_key_interval}, {0}, {0}, INT_MAX, "XVID: Maximum I-frame interval", "d"},

  {"xvid-intra-only", "XVID_INTRA_ONLY", OPT_BOOL, {(void*)&vc_xvid_param.intra_only}, {0}, {0}, 1, "XVID: intra frame only", NULL },

//  {"xvid-me-method", "XVID_ME_METHOD", HAS_ARG|OPT_SEL, {(void*)&vc_xvid_param.me_method}, {0}, {(int)me_methods}, sizeof(me_methods)/sizeof(SEL_COMPO), "XVID motion estimation algorithm specified", "sel"},

  {"xvid-interlacing", "XVID_INTERLACING", OPT_BOOL, {(void*)&vc_xvid_param.interlacing}, {0}, {0}, 1, "XVID: Enable/Disable interlacing", NULL},
  {"xvid-adaptivequant", "XVID_ADAPTIVEQUANT", OPT_BOOL, {(void*)&vc_xvid_param.adaptivequant}, {0}, {0}, 1, "XVID: Enable/Disable xvid encode flag XVID_ADAPTIVEQUANT", NULL},
  {"xvid-halfpel", "XVID_HALFPEL", HAS_ARG|OPT_SEL, {(void*)&vc_xvid_param.halfpel}, {SWITCH_DEFAULT}, {(int)switch_sel}, sizeof(switch_sel)/sizeof(SEL_COMPO), "XVID: Enable/Disable xvid encode flag XVID_HALFPEL", "sel" },
  {"xvid-inter4v", "XVID_INTER4V", HAS_ARG|OPT_SEL, {(void*)&vc_xvid_param.inter4v}, {SWITCH_DEFAULT}, {(int)switch_sel}, sizeof(switch_sel)/sizeof(SEL_COMPO), "XVID: Enable/Disable xvid encode flag XVID_INTER4V", "sel" },
//  {"xvid-greyscale", "XVID_GREYSCALE", OPT_BOOL, {(void*)&vc_xvid_param.greyscale}, {0}, {0}, 1, "XVID: Enable/Disable xvid encode flag XVID_GREYSCALE", NULL},

  {"xvid-hintedme", "XVID_HINTEDME", OPT_BOOL, {(void*)&vc_xvid_param.hintedme}, {1}, {0}, 1, "XVID: Set/Get hint in 2PASS encoding mode", NULL},
  {"xvid-hintfile", "XVID_HINTFILE", HAS_ARG|OPT_STR, {(void*)&vc_xvid_param.hintfile}, {(int)"xvid.hints"}, {0}, 2048, "XVID: hint log file name", "s"},

#if 0
  {"xvid-quant-matrix-intra", "XVID_QUANT_MATRIX_INTRA", HAS_ARG|OPT_FUNC, {(void*)opt_quant_matrix_intra}, {0}, {0}, 0, "XVID custom quant matrix intra", "d"},
  {"xvid-quant-matrix-intrer", "XVID_QUANT_MATRIX_INTRA", HAS_ARG|OPT_FUNC, {(void*)opt_quant_matrix_intrer}, {0}, {0}, 0, "XVID custom quant matrix intrer", "d"},
#endif

  {"xvid-passfile", "XVID_PASSFILE", HAS_ARG|OPT_STR, {(void*)&vc_xvid_param.passfile}, {(int)"xvid.stats"}, {0}, 2048, "XVID: two pass log file name", "s"},
//  {"xvid-desired-bitrate", "XVID_DESIRED_BITRATE", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.desired_bitrate}, {0}, {0}, INT_MAX, "XVID: target bitrate in '2PASS2' mode", "d"},
//  {"xvid-desired-size", "XVID_DESIRED_SIZE", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.desired_size}, {0}, {0}, INT_MAX, "XVID: target size in '2PASS2' mode (only video track)", "d"},
  {"xvid-keyframe-boost", "XVID_KEYFRAME_BOOST", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.keyframe_boost}, {0}, {0}, INT_MAX, "XVID: kerframe_boost '2PASS2' mode", "d"},
  {"xvid-kftreshold", "XVID_KFTRESHOLD", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.kftreshold}, {10}, {0}, INT_MAX, "XVID: kftreshold '2PASS2' mode", "d"},
  {"xvid-kfreduction", "XVID_KFREDUCTION", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.kfreduction}, {30}, {0}, INT_MAX, "XVID: kftreduction '2PASS2' mode", "d"},

  {"xvid-curve-compression-high", "XVID_CURVE_COMPRESSION_HIGH", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.curve_compression_high}, {25}, {0}, INT_MAX, "XVID: curve_compression_heigh '2PASS2' mode", "d"},
  {"xvid-curve-compression-low", "XVID_CURVE_COMPRESSION_LOW", HAS_ARG|OPT_INT, {(void*)&vc_xvid_param.curve_compression_low}, {10}, {0}, INT_MAX, "XVID: curve_compression_low'2PASS2' mode", "d"},
};

int xvid_param_num = sizeof(xvid_param) / sizeof(OptionDef);

static struct CODEC_FOURCC {
  uint32_t codec;
  const char *fourcc_str;
} codec_fourccs[] = {
  { VCODEC_XVID, "XVID" },
  { VCODEC_XVID, "DIVX" },
  { VCODEC_XVID, "DX50" },
};
static int codec_fourcc_num = sizeof(codec_fourccs)/sizeof(struct CODEC_FOURCC);

static struct CODEC_CAP {
  uint32_t codec;
  uint32_t codec_cap;
} codec_caps[] = {
  { VCODEC_XVID, VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
};
static int codec_cap_num = sizeof(codec_caps)/sizeof(struct CODEC_CAP);

uint32_t
vc_xvid_codec_cap(uint32_t codec, uint32_t cap_flag)
{
  int i;

  for (i = 0; i < codec_cap_num; i++) {
    if (codec == codec_caps[i].codec)
      return (cap_flag & codec_caps[i].codec_cap);
  }
  return VCODEC_CAP_NONE;
}

const char*
vc_xvid_codec_to_fourcc_str(uint32_t codec)
{
  int i;

  for (i = 0; i < codec_fourcc_num; i++) {
    if (codec == codec_fourccs[i].codec)
      return codec_fourccs[i].fourcc_str;
  }
  return NULL;
}

uint32_t
vc_xvid_fourcc_str_to_codec(const char *fourcc_str)
{
  int i;

  if (!fourcc_str)
    return VCODEC_UNKNOWN;

  for (i = 0; i < codec_fourcc_num; i++) {
    if (!strcmp(fourcc_str, codec_fourccs[i].fourcc_str))
      return codec_fourccs[i].codec;
  }
  return VCODEC_UNKNOWN;
}


static int const motion_presets[7] = {
        0,                                                        /* Q 0 */
        PMV_EARLYSTOP16,                                          /* Q 1 */
        PMV_EARLYSTOP16,                                          /* Q 2 */
        PMV_EARLYSTOP16 | PMV_HALFPELREFINE16,                    /* Q 3 */
        PMV_EARLYSTOP16 | PMV_HALFPELREFINE16,                    /* Q 4 */
        PMV_EARLYSTOP16 | PMV_HALFPELREFINE16 | PMV_EARLYSTOP8 |  /* Q 5 */
        PMV_HALFPELREFINE8,
        PMV_EARLYSTOP16 | PMV_HALFPELREFINE16 | PMV_EXTSEARCH16 | /* Q 6 */
        PMV_USESQUARES16 | PMV_EARLYSTOP8 | PMV_HALFPELREFINE8
};

static int const general_presets[7] = {
        XVID_H263QUANT,                               /* Q 0 */
        XVID_MPEGQUANT,                               /* Q 1 */
        XVID_H263QUANT,                               /* Q 2 */
        XVID_H263QUANT | XVID_HALFPEL,                /* Q 3 */
        XVID_H263QUANT | XVID_HALFPEL | XVID_INTER4V, /* Q 4 */
        XVID_H263QUANT | XVID_HALFPEL | XVID_INTER4V, /* Q 5 */
        XVID_H263QUANT | XVID_HALFPEL | XVID_INTER4V  /* Q 6 */
};

static XVID_INIT_PARAM xinit;

static XVID_DEC_PARAM xdecparam;
static XVID_DEC_FRAME xdecframe;


static VC_XVID_CONF dec_conf;
static VC_XVID_CONF enc_conf;
static VC_XVID_VBR_CONF vbr_conf;
static vbr_control_t    vbr_ctl;

int
vc_xvid_init(void)
{
  if (!vc_xvid_init_called) {
#ifdef ARCH_IA64
    xinit.cpu_flags = XVID_CPU_FORCE | XVID_CPU_IA64;
#else
    xinit.cpu_flags = 0;
#endif /* ARCH_IA64 */
    xvid_init(NULL, 0, &xinit, NULL);
  }
  if (xinit.api_version != API_VERSION) {
    vc_xvid_init_called = 0;
    return -1;
  } else
    vc_xvid_init_called = 1;
  return xinit.cpu_flags;
}

const char *
vc_xvid_ret_code_to_string (int ret_code)
{
  switch (ret_code) {
    case XVID_ERR_FAIL:
      return "XVID_ERR_FAIL";
    case XVID_ERR_OK:
      return "XVID_ERR_OK";
    case XVID_ERR_MEMORY:
      return "XVID_ERR_MEMORY";
    case XVID_ERR_FORMAT:
      return "XVID_ERR_FORMAT";
  }
  return "UNKNOWN";
}

static int
csp_to_xvid_csp (int csp)
{
  int xcsp;

  switch (csp) {
    case CSP_RGB32:
      xcsp = XVID_CSP_RGB32;
      break;
    case CSP_RGB24:
      xcsp = XVID_CSP_RGB24;
      break;
    case CSP_RGB565:
      xcsp = XVID_CSP_RGB565;
      break;
    case CSP_RGB555:
      xcsp = XVID_CSP_RGB555;
      break;
    case CSP_YV12:
      xcsp = XVID_CSP_YV12;
      break;
    case CSP_YUV420: case CSP_YUYV: case CSP_YUY2:
      xcsp = XVID_CSP_YUY2;
      break;
    case CSP_UYVY:
      xcsp = XVID_CSP_UYVY;
      break;
    case CSP_YVYU:
      xcsp = XVID_CSP_YVYU;
      break;
    case CSP_YUV420P: case CSP_I420: case CSP_IYUV:
      xcsp = XVID_CSP_I420;
      break;
//    case CSP_YV12_INTERLACE:
//      xcsp = XVID_CSP_YV12;
      break;
    default:
      xcsp = XVID_CSP_NULL;
  }
  return xcsp;
}

/* encode functions */

int
vc_xvid_encode_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    case CSP_UNKNOWN:
      ret = CSP_YV12;
      break;
    case CSP_RGB32: case CSP_RGB24: case CSP_RGB565: case CSP_RGB555:
    case CSP_YUV422: case CSP_YUYV: case CSP_YUY2:
    case CSP_UYVY:
    case CSP_YVYU:
    case CSP_YUV420P: case CSP_I420: case CSP_IYUV:
    case CSP_YV12:
      ret = csp;
      break;
//    case CSP_YV12_INTERLACE:
//      ret = csp;
//      break;
    default:
      ret = CSP_UNKNOWN;
      break;
  }
  return ret;
}

int
vc_xvid_encode (unsigned char *pic_data, unsigned char *stream_buf,
    int stream_buf_size, unsigned int *flag)
{
  int ret;
  XVID_ENC_FRAME xencframe;
  XVID_ENC_STATS xencstats;

  xencframe.general = vbr_conf.gen_flag;
  xencframe.motion  = vbr_conf.mot_flag;
  xencframe.bitstream = stream_buf;
  xencframe.length = -1;
  xencframe.image = pic_data;
  xencframe.colorspace = enc_conf.xcsp;
  xencframe.quant_intra_matrix = vbr_conf.quant_intra_matrix;
  xencframe.quant_inter_matrix = vbr_conf.quant_inter_matrix;

  xencframe.quant = vbrGetQuant(&vbr_ctl);
  xencframe.intra = vbrGetIntra(&vbr_ctl);

  // modulated quantizer type
  // from MPlayer-0.9r5 ve_xvid.c:408
  if (vbr_conf.mod_quant) {
    xencframe.general |= (xencframe.quant < 4) ? XVID_MPEGQUANT:XVID_H263QUANT;
    xencframe.general &= (xencframe.quant < 4) ?~XVID_MPEGQUANT:~XVID_H263QUANT;
  }

//  xencframe.general &= ~(XVID_HINTEDME_GET|XVID_HINTEDME_SET);
  if (vbr_conf.hint_fp) {
    if (vbr_ctl.mode == VBR_MODE_2PASS_1) {
      xencframe.general |= XVID_HINTEDME_GET;
      xencframe.hint.rawhints = 0;
      xencframe.hint.hintstream = vbr_conf.hint_buf;
      xencframe.hint.hintlength = 0;
    } else if (vbr_ctl.mode == VBR_MODE_2PASS_2) {
      size_t read;
      int blocksize = 0;
      read = fread(&blocksize, sizeof(int), 1, vbr_conf.hint_fp);
      if (read == 1)
        read = fread(vbr_conf.hint_buf, blocksize, 1, vbr_conf.hint_fp);
      if (read == 1) {
        xencframe.general |= XVID_HINTEDME_SET;
      } else {
        fprintf(stderr, "vc_xvid_encode: %s read error.\n", vbr_conf.hint_fname);
        xencframe.general &= ~XVID_HINTEDME_SET;
      }
      xencframe.hint.rawhints = 0;
      xencframe.hint.hintstream = vbr_conf.hint_buf;
      xencframe.hint.hintlength = blocksize;
    }
  }

  if (*flag & VCODEC_FORCE_INTRA)
    xencframe.intra = 1;
  else if (*flag & VCODEC_FORCE_INTER)
    xencframe.intra = 0;
  else if (vbr_conf.intra_only &&
      (vbr_conf.mode==VBR_MODE_1PASS || vbr_conf.mode==VBR_MODE_FIXED_QUANT))
    xencframe.intra = 1;

  ret = xvid_encore(enc_conf.handle, XVID_ENC_ENCODE, &xencframe, &xencstats);
  if (ret != XVID_ERR_OK) {
    fprintf (stderr, "vc_xvid_encode: xvid_encore failed, %s\n",
	vc_xvid_ret_code_to_string(ret));
    return VCODEC_FAIL;
  }

  if (xencframe.length >= stream_buf_size) {
    fprintf (stderr, "vc_xvid_encode: stream_buf overflow\n");
    return VCODEC_FAIL;
  }

  vbrUpdate(&vbr_ctl, xencstats.quant, xencframe.intra,
      xencstats.hlength, xencframe.length,
      xencstats.kblks, xencstats.mblks, xencstats.ublks);

  if (xencframe.general & XVID_HINTEDME_GET) {
    size_t ret;
    ret = fwrite(&xencframe.hint.hintlength,
                 sizeof(xencframe.hint.hintlength), 1, vbr_conf.hint_fp);
    if (ret == 1)
      ret = fwrite(xencframe.hint.hintstream,
	           xencframe.hint.hintlength, 1, vbr_conf.hint_fp);
    if (ret != 1)
      fprintf(stderr,"vc_xvid_encode: %s write error.\n",vbr_conf.hint_fname);
  }

  if (xencframe.intra) {
    *flag = VCODEC_IS_INTRA;
  } else {
    *flag = VCODEC_IS_INTER;
  }

  return xencframe.length;
}

int
vc_xvid_encode_quit (void)
{
  vbrFinish(&vbr_ctl);
  if (enc_conf.handle) {
    xvid_encore(enc_conf.handle, XVID_ENC_DESTROY, 0, 0);
    enc_conf.handle = NULL;
  }
  if (vbr_conf.hint_fp) {
    fclose(vbr_conf.hint_fp);
    vbr_conf.hint_fp = NULL;
  }
  if (vbr_conf.hint_buf) {
    free(vbr_conf.hint_buf);
    vbr_conf.hint_buf = NULL;
  }
  return VCODEC_OK;
}

int
vc_xvid_encode_config (void)
{
  int ret, xcsp, max_key_interval, bitrate;
  double fps;
  XVID_ENC_PARAM xencparam;

  if (enc_conf.handle)
    vc_xvid_encode_quit();

  xcsp = csp_to_xvid_csp(enc_conf.in_csp);
  if (xcsp == XVID_CSP_NULL) {
    fprintf(stderr, "vc_xvid_encode_config: in_csp invalid, %s\n", csp_to_str(enc_conf.in_csp));
    return VCODEC_FAIL;
  }
//  if (xcsp == XVID_CSP_RGB24)
//    xcsp |= XVID_CSP_VFLIP;

  enc_conf.xcsp = xcsp;

  memset(&xencparam, 0, sizeof(xencparam));

  if (enc_conf.fps == 0)
    fps = 25.0;
  else
    fps = enc_conf.fps;

  if (vc_xvid_param.max_key_interval == 0)
    max_key_interval = (int)(fps*10+0.5);
  else
    max_key_interval = vc_xvid_param.max_key_interval;

  bitrate = vc_xvid_param.bitrate * 1000;

  vbr_conf.intra_only = vc_xvid_param.intra_only;

  xencparam.width = enc_conf.width;
  xencparam.height = enc_conf.height;
#define FRAMERATE_INCR 1001
  if ((fps - (int)fps) < 1e-5) {
    xencparam.fincr = 1;
    xencparam.fbase = (int)fps;
    fps = (double)xencparam.fbase;
  } else {
    xencparam.fincr = FRAMERATE_INCR;
    xencparam.fbase = (int)(FRAMERATE_INCR * fps + 0.5);
    fps = (double)xencparam.fbase / (double)xencparam.fincr;
  }

  xencparam.rc_reaction_delay_factor = vc_xvid_param.rc_reaction_delay_factor;
  xencparam.rc_averaging_period = vc_xvid_param.rc_averaging_period;
  xencparam.rc_buffer = vc_xvid_param.rc_buffer;
  xencparam.max_quantizer = vc_xvid_param.max_iquant;
  xencparam.min_quantizer = vc_xvid_param.min_iquant;
  xencparam.max_key_interval = max_key_interval;

  vbrSetDefaults(&vbr_ctl);

  if (vc_xvid_param.mode == VC_XVID_MODE_CBR) {
    vbr_ctl.mode = VBR_MODE_1PASS;
    xencparam.rc_bitrate = vc_xvid_param.bitrate * 1000;
  } else if (vc_xvid_param.mode == VC_XVID_MODE_VBR_QUANT) {
    vbr_ctl.mode = VBR_MODE_FIXED_QUANT;
    xencparam.rc_bitrate = 0;
    vbr_ctl.fixed_quant = vc_xvid_param.quant;
  } else if (vc_xvid_param.mode == VC_XVID_MODE_2PASS_1) {
    vbr_ctl.mode = VBR_MODE_2PASS_1;
  } else if (vc_xvid_param.mode == VC_XVID_MODE_2PASS_2) {
    vbr_ctl.mode = VBR_MODE_2PASS_2;
  }

  vbr_conf.mode = vbr_ctl.mode;
  vbr_ctl.fps = fps;
//  vbr_ctl.debug = vc_xvid_param.debug;
  vbr_ctl.debug = 0;
  vbr_ctl.filename = vc_xvid_param.passfile;

  // VBR_MODE_2PASS_2 param
  vbr_ctl.desired_bitrate = vc_xvid_param.bitrate * 1000;
  // VBR_MODE_2PASS_2 keyframe param
  vbr_ctl.keyframe_boost = vc_xvid_param.keyframe_boost;
  vbr_ctl.kftreshold = vc_xvid_param.kftreshold;
  vbr_ctl.kfreduction = vc_xvid_param.kfreduction;
  vbr_ctl.min_key_interval = vc_xvid_param.min_key_interval;
  vbr_ctl.max_key_interval = max_key_interval;
  vbr_ctl.max_iquant = vc_xvid_param.max_iquant;
  vbr_ctl.min_iquant = vc_xvid_param.min_iquant;
  vbr_ctl.max_pquant = vc_xvid_param.max_pquant;
  vbr_ctl.min_pquant = vc_xvid_param.min_pquant;
  // VBR_MODE_2PASS_2 normal curve
//  vbr_ctl.curve_compression_high = vc_xvid_param.curve_compression_high;
//  vbr_ctl.curve_compression_low  = vc_xvid_param.curve_compression_low;
  // VBR_MODE_2PASS_2 alternate curve
#if 0
  vbr_ctl.use_alt_curve = vc_xvid_param.use_alt_curve;
  vbr_ctl.alt_curve_type = vc_xvid_param.alt_curve_type;
  vbr_ctl.alt_curve_low_dist = vc_xvid_param.alt_curve_low_dist;
  vbr_ctl.alt_curve_high_dist = vc_xvid_param.alt_curve_high_dist;
  vbr_ctl.alt_curve_min_rel_qual = vc_xvid_param.alt_curve_min_rel_qual;
  vbr_ctl.alt_curve_use_auto = vc_xvid_param.alt_curve_use_auto;
  vbr_ctl.alt_curve_auto_str = vc_xvid_param.alt_curve_auto_str;
  vbr_ctl.alt_curve_use_auto_bonus_bias = vc_xvid_param.alt_curve_use_auto_bonus_bias;
  vbr_ctl.alt_curve_bonus_bias = vc_xvid_param.alt_curve_bonus_bias;
  vbr_ctl.bitrate_payback_method = vc_xvid_param.bitrate_payback_method;
  vbr_ctl.bitrate_payback_delay = vc_xvid_param.bitrate_payback_delay;
  vbr_ctl.twopass_max_bitrate = vc_xvid_param.twopass_max_bitrate;
  vbr_ctl.twopass_max_overflow_improvement = vc_xvid_param.twopass_max_overflow_improvement;
  vbr_ctl.twopass_max_overflow_degradation = vc_xvid_param.twopass_max_overflow_degradation;
#endif

  vbr_conf.gen_flag = general_presets[vc_xvid_param.quality];
  vbr_conf.mot_flag = motion_presets[vc_xvid_param.quality];

  vbr_conf.mod_quant = 0;
  if (vc_xvid_param.quant_type != VC_XVID_QUANT_TYPE_DEFAULT)
    vbr_conf.gen_flag &= ~(XVID_H263QUANT|XVID_MPEGQUANT|XVID_CUSTOM_QMATRIX);

  if (vc_xvid_param.quant_type == VC_XVID_QUANT_TYPE_H263) {
    vbr_conf.gen_flag |= XVID_H263QUANT;
  } else if (vc_xvid_param.quant_type == VC_XVID_QUANT_TYPE_MPEG) {
    vbr_conf.gen_flag |= XVID_MPEGQUANT;
  } else if (vc_xvid_param.quant_type == VC_XVID_QUANT_TYPE_MOD) {
    if (vbr_conf.mode == VBR_MODE_2PASS_2) {
      vbr_conf.mod_quant = 1;
      vbr_conf.gen_flag &= ~(XVID_MPEGQUANT|XVID_H263QUANT);
    } else {
      fprintf(stderr, "vc_xvid_encode_config: quant_type 'MODULATED' can be used only in '2PASS2' mode, quant_type uses 'H263'.\n");
      vbr_conf.gen_flag |= XVID_H263QUANT;
    }
  } else if (vc_xvid_param.quant_type == VC_XVID_QUANT_TYPE_CUSTOM) {
    if (!vbr_conf.quant_intra_matrix || !vbr_conf.quant_inter_matrix) {
      fprintf(stderr, "vc_xvid_encode_config: cannot use custom quantization matrix, quant_type uses 'H263'.\n");
      vbr_conf.gen_flag |= XVID_H263QUANT;
    } else {
      vbr_conf.gen_flag |= XVID_CUSTOM_QMATRIX;
      vbr_conf.quant_intra_matrix = vbr_conf.quant_intra_matrix;
      vbr_conf.quant_inter_matrix = vbr_conf.quant_inter_matrix;
    }
  }

  if (vc_xvid_param.interlacing)
    vbr_conf.gen_flag |= XVID_INTERLACING;
  if (vc_xvid_param.adaptivequant)
    vbr_conf.gen_flag |= XVID_ADAPTIVEQUANT;
  if (vc_xvid_param.halfpel == SWITCH_ON)
    vbr_conf.gen_flag |= XVID_HALFPEL;
  else if (vc_xvid_param.halfpel == SWITCH_OFF)
    vbr_conf.gen_flag &= ~XVID_HALFPEL;
  if (vc_xvid_param.inter4v == SWITCH_ON)
    vbr_conf.gen_flag |= XVID_INTER4V;
  else if (vc_xvid_param.inter4v == SWITCH_OFF)
    vbr_conf.gen_flag &= ~XVID_INTER4V;
#if 0
  if (vc_xvid_param.greyscale)
    vbr_conf.gen_flag |= XVID_GREYSCALE;
#endif

#if 0
  vbr_conf.gen_flag &= ~(XVID_ME_ZERO|XVID_ME_LOGARITHMIC|XVID_ME_FULLSEARCH|XVID_ME_PVMFAST|XVID_ME_EPZS);
  vbr_conf.gen_flag |= vc_xvid_param.me_method;
#endif

  if (vc_xvid_param.hintedme &&
      (vbr_ctl.mode == VBR_MODE_2PASS_1 || vbr_ctl.mode == VBR_MODE_2PASS_2)) {
    vbr_conf.hint_buf = malloc(100000);
    if (!vbr_conf.hint_buf) {
      fprintf(stderr, "vc_xvid_encode_init: memory allocate error.\n");
      vc_xvid_encode_quit();
      return VCODEC_FAIL;
    }
    vbr_conf.hint_fname = vc_xvid_param.hintfile;
    vbr_conf.hint_fp = fopen(vbr_conf.hint_fname,
	                     vbr_ctl.mode == VBR_MODE_2PASS_1 ? "w" : "r");
    if (!vbr_conf.hint_fp) {
      fprintf(stderr, "vc_xvid_encode_init: %s open error.\n",
	  vbr_conf.hint_fname);
      vc_xvid_encode_quit();
      return VCODEC_FAIL;
    }
  } else {
    vbr_conf.gen_flag &= ~(XVID_HINTEDME_SET|XVID_HINTEDME_GET);
  }

  if (vbrInit(&vbr_ctl) < 0) {
    fprintf (stderr, "vc_xvid_encode_config: vbrInit failed.\n");
    return VCODEC_FAIL;
  }

  ret = xvid_encore(NULL, XVID_ENC_CREATE, &xencparam, NULL);
  if (ret != 0) {
    fprintf (stderr, "vc_xvid_encode_config: encore failed %s.\n",
       	vc_xvid_ret_code_to_string(ret));
    return VCODEC_FAIL;
  }
  enc_conf.handle = xencparam.handle;

  return VCODEC_OK;
}

int
vc_xvid_encode_init (VCODEC *vcodec)
{
  static int first = 1;
  int csp, ret, need_conf = 0;

  if (!vc_xvid_init_called)
    vc_xvid_init();

  if (vcodec->codec != VCODEC_XVID) {
    fprintf(stderr, "vc_xvid_encode_init: invalid codec.\n");
    return VCODEC_FAIL;
  }

  if (first) {
    enc_conf.handle = NULL;
    enc_conf.width = 0;
    enc_conf.height = 0;
    enc_conf.fps = 0.0;
    enc_conf.in_csp = CSP_UNKNOWN;
    enc_conf.out_csp = CSP_UNKNOWN;
    vbr_conf.hint_fname = NULL;
    vbr_conf.hint_fp = NULL;
    vbr_conf.hint_buf = NULL;
    vbr_conf.quant_intra_matrix = NULL;
    vbr_conf.quant_inter_matrix = NULL;
    first = 0;
  }

  csp = vc_xvid_encode_csp_cap(vcodec->in_csp);
  if (vcodec->in_csp == CSP_UNKNOWN || csp == CSP_UNKNOWN) {
    fprintf(stderr, "vc_xvid_encode_init: invalid in_csp.\n");
    return VCODEC_FAIL;
  }
  vcodec->out_csp = CSP_COMPRESSED;

  if (enc_conf.handle == NULL)
    need_conf = 1;
  else if (enc_conf.width != vcodec->width)
    need_conf = 1;
  else if (enc_conf.height != vcodec->height)
    need_conf = 1;
  else if (enc_conf.fps != vcodec->fps)
    need_conf = 1;
  else if (enc_conf.in_csp != csp)
    need_conf = 1;

  if (need_conf == 0) {
    return VCODEC_OK;
  }

  enc_conf.in_csp = csp;
  enc_conf.out_csp = CSP_COMPRESSED;
  enc_conf.width = vcodec->width;
  enc_conf.height = vcodec->height;
  enc_conf.fps = vcodec->fps;
  ret = vc_xvid_encode_config();

  return ret;
}

void
vc_xvid_encode_print_param(void)
{
  SEL_COMPO *sel;
  int i;

  fprintf (stdout, "XVID ENCODE PARAMETER\n");
  sel = modes;
  for (i=0; (uint32_t)i < sizeof(modes)/sizeof(SEL_COMPO); i++, sel++) {
    if (sel->id == vc_xvid_param.mode) {
      fprintf (stdout, "mode:                    %s\n", sel->name);
      break;
    }
  }
  switch (vc_xvid_param.mode) {
    case VC_XVID_MODE_CBR:
    case VC_XVID_MODE_2PASS_2:
      fprintf (stdout, "bitrate:                 %d\n", vc_xvid_param.bitrate);
      break;
    case VC_XVID_MODE_VBR_QUANT:
    case VC_XVID_MODE_2PASS_1:
      fprintf (stdout, "quant:                   %d\n", vc_xvid_param.quant);
      break;
  }
  fprintf (stdout, "quality:                 %d\n",vc_xvid_param.quality);
  fprintf (stdout, "dimmension:              %dx%d\n",
      enc_conf.width, enc_conf.height);
  fprintf (stdout, "fps:                     %f\n", enc_conf.fps);
  fprintf (stdout, "max_iquant:              %d\n", vc_xvid_param.max_iquant);
  fprintf (stdout, "min_iquant:              %d\n", vc_xvid_param.min_iquant);
  fprintf (stdout, "max_pquant:              %d\n", vc_xvid_param.max_pquant);
  fprintf (stdout, "min_pquant:              %d\n", vc_xvid_param.min_pquant);
  fprintf (stdout, "min_key_interval:        %d\n",vc_xvid_param.min_key_interval);
  fprintf (stdout, "max_key_interval:        %d\n",vc_xvid_param.max_key_interval);
#if 0
  for (i=0, sel=quant_type; i < sizeof(quant_type)/sizeof(SEL_COMPO);i++,sel++){
    if (vc_xvid_conf.quant_type == sel->id) {
      fprintf (stdout, "quant_type:              %s\n", sel->name);
      break;
    }
  }
#endif
  fprintf (stdout, "general flags:           ");
  if (vbr_conf.gen_flag & XVID_CUSTOM_QMATRIX)
    fprintf (stdout, "CUSTOM_QMATRIX");
  if (vbr_conf.gen_flag & XVID_H263QUANT)
    fprintf (stdout, "H263QUANT");
  if (vbr_conf.gen_flag & XVID_MPEGQUANT)
    fprintf (stdout, "MPEGQUANT");
  if (vbr_conf.gen_flag & XVID_HALFPEL)
    fprintf (stdout, " | HALFPEL");
  if (vbr_conf.gen_flag & XVID_ADAPTIVEQUANT)
    fprintf (stdout, " | ADAPTIVEQUANT");
  if (vbr_conf.gen_flag & XVID_INTERLACING)
    fprintf (stdout, " | INTERLACING");
  if (vbr_conf.gen_flag & XVID_INTER4V)
    fprintf (stdout, " | INTER4V");
  if (vbr_conf.gen_flag & XVID_HINTEDME_GET)
    fprintf (stdout, " | HINTEDME_GET");
  if (vbr_conf.gen_flag & XVID_HINTEDME_SET)
    fprintf (stdout, " | HINTEDME_SET");
  if (vbr_conf.gen_flag & XVID_ME_PMVFAST)
    fprintf (stdout, " | ME_PMVFAST");
  if (vbr_conf.gen_flag & XVID_ME_EPZS)
    fprintf (stdout, " | ME_EPZS");
  if (vbr_conf.gen_flag & XVID_GREYSCALE)
    fprintf (stdout, " | GREYSCALE");
  fprintf (stdout, "\n");

  fflush (stdout);
}

/* decode functions */

int
vc_xvid_decode_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    case CSP_UNKNOWN:
      ret = CSP_YV12;
      break;
    case CSP_RGB32:
    case CSP_RGB24:
    case CSP_RGB565:
    case CSP_RGB555:
    case CSP_YUV422:
    case CSP_YUYV:
    case CSP_YUY2:
    case CSP_UYVY:
    case CSP_YVYU:
    case CSP_YUV420P:
    case CSP_I420:
    case CSP_IYUV:
    case CSP_YV12:
      ret = csp;
      break;
    default:
      ret = CSP_UNKNOWN;
      break;
  }
  return ret;
}

int
vc_xvid_decode (unsigned char *pic_data, unsigned char *stream,
    int stream_length, unsigned int *flag)
{
  int ret;

  xdecframe.bitstream = stream;
  xdecframe.length = stream_length;
  xdecframe.image = pic_data;
  if (*flag & VCODEC_NULL_DECODE)
    xdecframe.colorspace = XVID_CSP_NULL;
  else
    xdecframe.colorspace = dec_conf.out_csp;
#if 0
  xdecframe.colorspace = dec_conf.out_csp;
  xdecframe.stride = width;
#endif

  ret = xvid_decore(dec_conf.handle, XVID_DEC_DECODE, &xdecframe, NULL);
  if (ret != XVID_ERR_OK) {
    fprintf (stderr, "vc_xvid_decode() decore fail %s\n",
	vc_xvid_ret_code_to_string(ret));
    return VCODEC_FAIL;
  }

  return xdecframe.length;
}

int
vc_xvid_decode_quit (void)
{
  if (dec_conf.handle) {
    xvid_decore(dec_conf.handle, XVID_DEC_DESTROY, 0, 0);
    dec_conf.handle = NULL;
  }
  return VCODEC_OK;
}

int
vc_xvid_decode_config (void)
{
  int ret;

  if (dec_conf.handle) {
    xvid_decore(dec_conf.handle, XVID_DEC_DESTROY, NULL, NULL);
    dec_conf.handle = NULL;
  }

  memset(&xdecparam, 0, sizeof(xdecparam));
  memset(&xdecframe, 0, sizeof(xdecframe));

  xdecparam.width = dec_conf.width;
  xdecparam.height = dec_conf.height;
  xdecframe.colorspace = dec_conf.out_csp;
  xdecframe.stride = dec_conf.width;

  ret = xvid_decore(NULL, XVID_DEC_CREATE, &xdecparam, NULL);
  if (ret != 0) {
    fprintf (stderr, "vc_xvid_decode_config(): decore fail %s.\n",
       	vc_xvid_ret_code_to_string(ret));
    return VCODEC_FAIL;
  }
  dec_conf.handle = xdecparam.handle;

  return VCODEC_OK;
}

int
vc_xvid_decode_init (VCODEC *vcodec)
{
  static int first = 1;
  int csp, ret, need_conf = 0;

  if (!vc_xvid_init_called)
    vc_xvid_init();

  if (vcodec->codec != VCODEC_XVID) {
    fprintf(stderr, "vc_xvid_decode_init: invalid codec.\n");
    return VCODEC_FAIL;
  }

  if (first) {
    dec_conf.handle = NULL;
    dec_conf.width = 0;
    dec_conf.height = 0;
    dec_conf.in_csp = CSP_UNKNOWN;
    dec_conf.out_csp = CSP_UNKNOWN;
    first = 0;
  }

  csp = vc_xvid_decode_csp_cap(vcodec->out_csp);
  if (vcodec->out_csp == CSP_UNKNOWN || csp == CSP_UNKNOWN) {
    fprintf(stderr, "vc_xvid_decode_init: invalid out_csp.\n");
    return VCODEC_FAIL;
  }
  vcodec->in_csp = CSP_XVID;

  csp = csp_to_xvid_csp(csp);

  if (dec_conf.handle == NULL)
    need_conf = 1;
  if (dec_conf.width != vcodec->width)
    need_conf = 1;
  else if (dec_conf.height != vcodec->height)
    need_conf = 1;
  else if (dec_conf.out_csp != csp)
    need_conf = 1;

  if (need_conf == 0) {
    return VCODEC_OK;
  }

  dec_conf.out_csp = csp;
  dec_conf.width = vcodec->width;
  dec_conf.height = vcodec->height;
  ret = vc_xvid_decode_config();

  return ret;
}

void
vc_xvid_decode_print_param(void)
{
  fprintf (stdout, "XVID DECODE PARAMETER\n");
  fprintf (stdout, "dimmension:              %dx%d\n",
      xdecparam.width, xdecparam.height);
  fflush (stdout);
}

/* vcodec functions struct */

VCODEC_FUNCS vc_xvid_funcs = {
  vc_xvid_encode_init,
  vc_xvid_encode_quit,
  vc_xvid_encode_csp_cap,
  vc_xvid_encode,
  vc_xvid_encode_print_param,

  vc_xvid_decode_init,
  vc_xvid_decode_quit,
  vc_xvid_decode_csp_cap,
  vc_xvid_decode,
  vc_xvid_decode_print_param,

  vc_xvid_codec_to_fourcc_str,
  vc_xvid_fourcc_str_to_codec,
  vc_xvid_codec_cap,
};

#endif /* HAVE_LIBXVIDCORE */

