/* vc_libavcodec.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifdef HAVE_LIBAVCODEC


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h>
#include <limits.h>

#include "util.h"
#include "vcodec.h"
#include "parseopt.h"
#include "csp.h"
#include "avcodec.h"
#include "vc_libavcodec.h"

static int vc_libavcodec_init_called = 0;

static VC_LIBAVCODEC_PARAM vc_lavc_param;

int
opt_qblur(const char *arg)
{
  vc_lavc_param.qblur = atof(arg);
  return 0;
}
int
opt_qcomp(const char *arg)
{
  vc_lavc_param.qcomp= atof(arg);
  return 0;
}
int
opt_rc_init_cplx(const char *arg)
{
  vc_lavc_param.rc_init_cplx = atof(arg);
  return 0;
}
int
opt_i_qfactor(const char *arg)
{
  vc_lavc_param.i_qfactor= atof(arg);
  return 0;
}
int
opt_b_qfactor(const char *arg)
{
  vc_lavc_param.b_qfactor= atof(arg);
  return 0;
}
int
opt_b_qoffset(const char *arg)
{
  vc_lavc_param.b_qoffset = atof(arg);
  return 0;
}
int
opt_i_qoffset(const char *arg)
{
  vc_lavc_param.i_qoffset = atof(arg);
  return 0;
}

static SEL_COMPO me_methods[] = {
  { "DEFAULT", VLAVC_ME_DEF },
  { "ZERO",    VLAVC_ME_ZERO },
  { "FULL",    VLAVC_ME_FULL },
  { "LOG",     VLAVC_ME_LOG },
  { "PHODS",   VLAVC_ME_PHODS },
  { "EPZS",    VLAVC_ME_EPZS },
  { "X1",      VLAVC_ME_X1 },
};

OptionDef vlavc_param[] = {
  {"vlavc-bitrate", "VLAVC_BITRATE", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.bitrate}, {900}, {0}, INT_MAX, "libavcodec video set bitrate (Kilo)", "d"},
  {"vlavc-gop-size", "VLAVC_GOP_SIZE", HAS_ARG, {(void*)&vc_lavc_param.gop_size}, {0}, {0}, 10000, "libavcodec video set the group of picture size", "d"},
  {"vlavc-intra", "VLAVC_INTRA", OPT_BOOL, {(void*)&vc_lavc_param.intra_only}, {0}, {0}, 1, "libavcodec video use only intra frames", NULL},
//  {"vlavc-pass", "VLAVC_PASS", HAS_ARG, {(void*)&vc_lavc_param.do_pass}, {0}, {0}, 2, "libavcodec video select the pass number", "d"},
//  {"vlavc-passlogfile", "VLAVC_PASSLOGFILE", HAS_ARG, {(void*)&vc_lavc_param.passlogfile}, {(int)"vlavc2pass"}, {0}, 1024, "libavcodec video select two pass log file name", "s"},
  {"vlavc-qscale", "VLAVC_QSCALE", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.qscale}, {0}, {0}, 31, "libavcodec video use fixed video quantizer scale (VBR)", "d"},
  {"vlavc-qmin", "VLAVC_QMIN", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.qmin}, {2}, {0}, 31, "libavcodec video min quantiser scale (VBR)", "d"},
  {"vlavc-qmax", "VLAVC_QMAX", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.qmax}, {31}, {0}, 31, "libavcodec video max quantiser scale (VBR)", "d"},
  {"vlavc-mbqmin", "VLAVC_MB_QMIN", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.mb_qmin}, {2}, {0}, 31, "libavcodec video min macroblock quantizer scale (VBR)", "d"},
  {"vlavc-mbqmax", "VLAVC_MB_QMAX", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.mb_qmax}, {31}, {0}, 31, "libavcodec video max macroblock quantizer scale (VBR)", "d"},
  {"vlavc-qdiff", "VLAVC_QDIFF", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.qdiff}, {3}, {0}, 31, "libavcodec video max difference between the quantizer scale (VBR)", "d"},
  {"vlavc-qblur", "VLAVC_QBLUR", HAS_ARG|OPT_FUNC, {(void*)&opt_qblur}, {(int)"0.5"}, {0}, 0, "libavcodec video quantiser scale blur (VBR)", "f"},
  {"vlavc-qcomp", "VLAVC_QCOMP", HAS_ARG|OPT_FUNC, {(void*)&opt_qcomp}, {(int)"0.5"}, {0}, 0, "libavcodec video quantizer scale compression (VBR)", "f"},
  {"vlavc-rc-init-cplx", "VLAVC_RC_INIT_CPLX", HAS_ARG|OPT_FUNC, {(void*)&opt_rc_init_cplx}, {(int)"0"}, {0}, 0, "libavcodec video initail complexity for 1-pass encoding", "f"},
  {"vlavc-b-qfactor", "VLAVC_B_QFACTOR", HAS_ARG|OPT_FUNC, {(void*)&opt_b_qfactor}, {(int)"1.25"}, {0}, 0, "libavcodec video qp factor between p and b frames", "f"},
  {"vlavc-i-qfactor", "VLAVC_I_QFACTOR", HAS_ARG|OPT_FUNC, {(void*)&opt_i_qfactor}, {(int)"-0.8"}, {0}, 0, "libavcodec video qp factor between p and i frames", "f"},
  {"vlavc-b-qoffset", "VLAVC_B_QOFFSET", HAS_ARG|OPT_FUNC, {(void*)&opt_b_qoffset}, {(int)"1.25"}, {0}, 0, "libavcodec video qp offset between p and b frames", "f"},
  {"vlavc-i-qoffset", "VLAVC_I_QOFFSET", HAS_ARG|OPT_FUNC, {(void*)&opt_i_qoffset}, {(int)"0.0"}, {0}, 0, "libavcodec video qp offset between p and i frames", "f"},
  {"vlavc-rc-eq", "VLAVC_RC_EQ", HAS_ARG|OPT_STR, {(void*)&vc_lavc_param.rc_eq}, {(int)"tex^qComp"}, {0}, 2048, "libavcodec video rc_eq", "s"},
//  {"vlavc-rc-override", "VLAVC_RC_OVERRIDE", HAS_ARG|OPT_STR, {(void*)&vc_lavc_param.rc_override_string}, {(int)NULL}, {0}, 2048, "libavcodec video qualities for specific intervals", "s"},
  {"vlavc-bt", "VLAVC_BT", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.bitrate_tolerance}, {20000}, {0}, INT_MAX, "libavcodec video set bitrate tolerance (Kilo)", "d"},
  {"vlavc-maxrate", "VLAVC_MAXRATE", HAS_ARG, {(void*)&vc_lavc_param.rc_max_rate}, {0}, {0}, INT_MAX, "libavcodec video set max bitrate tolerance (Kilo)", "d"},
  {"vlavc-minrate", "VLAVC_MINRATE", HAS_ARG, {(void*)&vc_lavc_param.rc_min_rate}, {0}, {0}, INT_MAX, "libavcodec video set min bitrate tolerance (Kilo)", "d"},
  {"vlavc-buffer-size", "VLAVC_RC_BUFFER_SIZE", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.rc_buffer_size}, {0}, {0}, INT_MAX, "libavcodec video set ratecontrol buffer size (Kilo)", "d"},
  {"vlavc-me", "VLAVC_ME", HAS_ARG|OPT_SEL, {(void*)&vc_lavc_param.me_method}, {VLAVC_ME_DEF}, {(int)me_methods}, sizeof(me_methods)/sizeof(SEL_COMPO), "libavcodec video set motion estimation method", "sel"},
  {"vlavc-dct-algo", "VLAVC_DCT_ALGO", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.dct_algo}, {0}, {0}, INT_MAX, "libavcodec video set dct algo", "d"},
  {"vlavc-idct-algo", "VLAVC_IDCT_ALGO", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.idct_algo}, {0}, {0}, INT_MAX, "libavcodec video set idct algo", "d"},
  {"vlavc-er", "VLAVC_ER", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.error_resilience}, {2}, {0}, INT_MAX, "libavcodec video set error resilience", "d"},
  {"vlavc-ec", "VLAVC_EC", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.error_concealment}, {3}, {0}, INT_MAX, "libavcodec video set error concealment", "d"},
  {"vlavc-bf", "VLAVC_BF", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.b_frames}, {0}, {0}, FF_MAX_B_FRAMES, "libavcodec video use 'frames' B frames (only MPEG-4)", "d"},
  {"vlavc-hq", "VLAVC_HQ", OPT_BOOL, {(void*)&vc_lavc_param.use_hq}, {0}, {0}, 1, "libavcodec video avtivate high quality settings", NULL},
  {"vlavc-4mv", "VLAVC_4MV", OPT_BOOL, {(void*)&vc_lavc_param.use_4mv}, {0}, {0}, 1, "libavcodec video use four motion vector by macroblock (only MPEG-4)", NULL},
  {"vlavc-part", "VLAVC_PART", OPT_BOOL, {(void*)&vc_lavc_param.use_part}, {0}, {0}, 1, "libavcodec video use data partitioning (only MPEG-4)", NULL},
  {"vlavc-bug", "VLAVC_BUG", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.workaround_bugs}, {FF_BUG_AUTODETECT}, {0}, INT_MAX, "libavcodec video workaround not auto detected encoder bugs", "d"},
//  {"vlavc-ps", "VLAVC_PS", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.packet_size}, {0}, {0}, INT_MAX, "libavcodec video packet size (in bits)", "d"},
  {"vlavc-strict", "VLAVC_STRICT", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.strict}, {0}, {0}, INT_MAX, "libavcodec video strictness", "d"},
//  {"vlavc-sameq", "VLAVC_SAMEQ", OPT_BOOL, {(void*)&vc_lavc_param.same_quality}, {0}, {0}, 1, "libavcodec video use same quality as source (implies VBR)", NULL},
  {"vlavc-debug", "VLAVC_DEBUG", HAS_ARG|OPT_INT, {(void*)&vc_lavc_param.debug}, {0}, {0}, INT_MAX, "libavcodec video print specific debug info", "d" },

//  {"vlavc-psnr", "VLAVC_PSNR", OPT_BOOL, {(void*)&vc_lavc_param.do_psnr}, {0}, {0}, 1, "libavcodec video calculate PSNR of compressed frames", NULL },

//  {"vlavc-rc-buffer-aggressivity", "VLAVC_RC_BUFFER_AGGRESSIVITY", HAS_ARG|OPT_FUNC, {(void*)&opt_rc_buffer_aggressivity}, {(int)"1.0"}, {0}, 0, "libavcodec video rc_buffer_aggressivity", "f"},
};

int vlavc_param_num = sizeof(vlavc_param) / sizeof(OptionDef);

static VC_LIBAVCODEC_CONF enc_conf;
static VC_LIBAVCODEC_CONF dec_conf;

typedef struct {
  uint32_t codec;
  enum CodecID id;
  const char *fourcc_str;
  uint32_t codec_cap;
} CODECS;

static CODECS codecs[] = {
  { VCODEC_MPEG1, CODEC_ID_MPEG1VIDEO, "mpg1", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_H263, CODEC_ID_H263, "H263", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_H263P, CODEC_ID_H263P, "H263", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_RV10, CODEC_ID_RV10, "rv10", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_MJPEG, CODEC_ID_MJPEG, "MJPG", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_MPEG4, CODEC_ID_MPEG4, "MP4S", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_MSMPEG4V1, CODEC_ID_MSMPEG4V1, "MPG4", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_MSMPEG4V2, CODEC_ID_MSMPEG4V2, "MP42", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_MSMPEG4V3, CODEC_ID_MSMPEG4V3, "MP43", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_WMV1, CODEC_ID_WMV1, "WMV1", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_WMV2, CODEC_ID_WMV2, "WMV2", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_HUFFYUV, CODEC_ID_HUFFYUV, "HFYU", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
  { VCODEC_UNKNOWN, CODEC_ID_NONE, "NONE", VCODEC_CAP_ENCODE|VCODEC_CAP_DECODE },
};

static int codecs_num = sizeof(codecs) / sizeof(CODECS);


uint32_t
vc_libavcodec_codec_cap(uint32_t codec, uint32_t cap_flag)
{
  int i;

  for (i = 0; i < codecs_num; i++) {
    if (codec == codecs[i].codec)
      return (cap_flag & codecs[i].codec_cap);
  }
  return VCODEC_CAP_NONE;
}

const char*
vc_libavcodec_codec_to_fourcc_str(uint32_t codec)
{
  int i;

  for (i = 0; i < codecs_num; i++) {
    if (codec == codecs[i].codec)
      return codecs[i].fourcc_str;
  }
  return NULL;
}

uint32_t
vc_libavcodec_fourcc_str_to_codec(const char *fourcc_str)
{
  int i;

  if (!fourcc_str)
    return VCODEC_UNKNOWN;

  for (i = 0; i < codecs_num; i++) {
    if (!strcmp(fourcc_str, codecs[i].fourcc_str))
      return codecs[i].codec;
  }
  return VCODEC_UNKNOWN;
}


int
vc_libavcodec_init(void)
{
  if (!vc_libavcodec_init_called) {
    avcodec_init();
    avcodec_register_all();
  }
  vc_libavcodec_init_called = 1;
  return VCODEC_OK;
}

static enum CodecID
vc_libavcodec_vcodec_to_id(uint32_t vcodec)
{
  int i;

  for (i = 0; i < codecs_num; i++) {
    if (vcodec == codecs[i].codec) {
      return codecs[i].id;
    }
  }
  return CODEC_ID_NONE;
}

static struct CSP_PIX {
  int csp;
  enum PixelFormat fmt;
} csp_pix[] = {
  { CSP_RGB32, PIX_FMT_RGBA32 },
  { CSP_RGB24, PIX_FMT_RGB24 },
  { CSP_RGB565, PIX_FMT_RGB565 },
  { CSP_RGB555, PIX_FMT_RGB555 },
  { CSP_YUV422, PIX_FMT_YUV422 },
  { CSP_YUV422P, PIX_FMT_YUV422P },
  { CSP_YUV420P, PIX_FMT_YUV420P },
  { CSP_YUV411P, PIX_FMT_YUV411P },
};
static int csp_pix_num = sizeof(csp_pix) / sizeof(struct CSP_PIX);

static enum PixelFormat
csp_to_pix_fmt(int csp)
{
  int i;
  for (i = 0; i < csp_pix_num; i++) {
    if (csp_pix[i].csp == csp)
      return csp_pix[i].fmt;
  }
  return PIX_FMT_ANY;
}

#if 0
static int
pix_fmt_to_csp(enum PixelFormat fmt)
{
  int i;
  for (i = 0; i < csp_pix_num; i++) {
    if (csp_pix[i].fmt == fmt)
      return csp_pix[i].csp;
  }
  return CSP_UNKNOWN;
}
#endif

/* encode functions */

int
vc_libavcodec_encode_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    case CSP_UNKNOWN:
      ret = CSP_YUV420P;
      break;
    case CSP_YUV420P:
    case CSP_I420:
    case CSP_IYUV:
    case CSP_YV12:
      ret = csp;
      break;
    default:
      ret = CSP_UNKNOWN;
      break;
  }
  return ret;
}

int
vc_libavcodec_encode (unsigned char *pic_data, unsigned char *stream_buf,
    int stream_buf_size, unsigned int *flag)
{
  int out_size;

  enc_conf.picture->data[0] = pic_data;
  enc_conf.picture->data[1] = pic_data + enc_conf.u_offset;
  enc_conf.picture->data[2] = pic_data + enc_conf.v_offset;
  enc_conf.picture->linesize[0] = enc_conf.width;
  enc_conf.picture->linesize[1] = enc_conf.uv_width;
  enc_conf.picture->linesize[2] = enc_conf.uv_width;
  if (enc_conf.quality)
    enc_conf.picture->quality = enc_conf.quality;
  enc_conf.context->pix_fmt = enc_conf.pix_fmt;

  out_size = avcodec_encode_video(enc_conf.context, stream_buf, stream_buf_size,
      enc_conf.picture);
  if (out_size <= 0) {
    fprintf (stderr, "vc_libavcodec_encode() avcodec_encode_video fail. %d\n",out_size);
    return VCODEC_FAIL;
  }

  if (enc_conf.context->coded_frame->key_frame)
    *flag = VCODEC_IS_INTRA;
  else
    *flag = VCODEC_IS_INTER;

  if (enc_conf.logfile && enc_conf.context->stats_out) {
    fprintf(enc_conf.logfile, "%s", enc_conf.context->stats_out);
  }

  return out_size;
}

int
vc_libavcodec_encode_quit (void)
{
  if (enc_conf.logfile) {
    fclose(enc_conf.logfile);
    enc_conf.logfile = NULL;
  }
  if (enc_conf.context) {
    if (enc_conf.context->stats_in)
      free(enc_conf.context->stats_in);
    avcodec_close(enc_conf.context);
    free(enc_conf.context);
  }
  if (enc_conf.picture)
    free(enc_conf.picture);
  enc_conf.context = NULL;
  enc_conf.picture = NULL;
  enc_conf.avcodec = NULL;
  enc_conf.codec_id = CODEC_ID_NONE;
  enc_conf.codec = VCODEC_UNKNOWN;
  return VCODEC_OK;
}

int
vc_libavcodec_encode_config (void)
{
  AVCodecContext *ctx;
//  char *p;
//  int i;

  if (enc_conf.context) {
    vc_libavcodec_encode_quit();
  }

  enc_conf.avcodec = avcodec_find_encoder(enc_conf.codec_id);
  if (!enc_conf.avcodec) {
    return VCODEC_FAIL;
  }
  ctx = avcodec_alloc_context();
  enc_conf.context = ctx;
  enc_conf.picture = avcodec_alloc_frame();
  if (!enc_conf.context || !enc_conf.picture) {
    fprintf(stderr, "vc_libavcodec_encode_config: memory allocate error.\n");
    vc_libavcodec_encode_quit();
    return VCODEC_FAIL;
  }

  if (enc_conf.in_csp == CSP_YV12) {
    enc_conf.v_offset = enc_conf.width * enc_conf.height;
    enc_conf.u_offset = (enc_conf.width * enc_conf.height) / 4 * 5;
    enc_conf.uv_width = enc_conf.width / 2;
  } else {
    enc_conf.u_offset = enc_conf.width * enc_conf.height;
    enc_conf.v_offset = (enc_conf.width * enc_conf.height) / 4 * 5;
    enc_conf.uv_width = enc_conf.width / 2;
  }
  enc_conf.data_size = enc_conf.width * enc_conf.height * 12 / 8;
  enc_conf.pix_fmt = csp_to_pix_fmt(CSP_YUV420P);

  ctx->codec_id = enc_conf.codec_id;
  ctx->bit_rate = vc_lavc_param.bitrate * 1000;
  ctx->bit_rate_tolerance = vc_lavc_param.bitrate_tolerance * 1000;
  ctx->width = enc_conf.width;
  ctx->height = enc_conf.height;
  ctx->frame_rate = enc_conf.fps * FRAME_RATE_BASE;

  ctx->pix_fmt = enc_conf.pix_fmt;

  if (vc_lavc_param.intra_only)
    ctx->gop_size = 0;
  else if (vc_lavc_param.gop_size == 0)
    ctx->gop_size = enc_conf.fps * 10;
  else
    ctx->gop_size = vc_lavc_param.gop_size;

//  if (vc_lavc_param.qscale || vc_lavc_param.same_quality) {
  if (vc_lavc_param.qscale) {
    ctx->flags |= CODEC_FLAG_QSCALE;
    enc_conf.quality = vc_lavc_param.qscale;
  } else {
    enc_conf.quality = 0;
  }

  if (vc_lavc_param.use_hq)
    ctx->flags |= CODEC_FLAG_HQ;
  if (vc_lavc_param.use_4mv) {
    ctx->flags |= CODEC_FLAG_HQ;
    ctx->flags |= CODEC_FLAG_4MV;
  }

  if (vc_lavc_param.use_part)
    ctx->flags |= CODEC_FLAG_PART;

  if (vc_lavc_param.b_frames) {
    if (enc_conf.codec_id != CODEC_ID_MPEG4) {
      fprintf(stderr, "vc_libavcodec_encode_config: B frames encoding only supported by MPEG-4, ignore b frames parameter.\n");
    }
    ctx->max_b_frames = vc_lavc_param.b_frames;
    ctx->b_frame_strategy = 0;
    ctx->b_quant_factor = 2.0;
  }

  ctx->qmin = vc_lavc_param.qmin;
  ctx->qmax = vc_lavc_param.qmax;
  ctx->mb_qmin = vc_lavc_param.mb_qmin;
  ctx->mb_qmax = vc_lavc_param.mb_qmax;
  ctx->max_qdiff = vc_lavc_param.qdiff;
  ctx->qblur = vc_lavc_param.qblur;
  ctx->qcompress = vc_lavc_param.qcomp;
  ctx->rc_eq = vc_lavc_param.rc_eq;
  ctx->debug = vc_lavc_param.debug;

#if 0
  p = vc_lavc_param.rc_override_string;
  for (i = 0; p; i++) {
    int start, end, q;
    int e = sscanf(p, "%d,%d,%d", &start, &end, &q);
    if (e != 3) {
      fprintf(stderr, "vc_libavcodec_encode_config: error parsing rc_override, argument = \"%s\"\n", vc_lavc_param.rc_override_string);
      vc_libavcodec_encode_quit();
      return VCODEC_FAIL;
    }
    ctx->rc_override =
      realloc(ctx->rc_override, sizeof(RcOverride)*(i+1));
    ctx->rc_override[i].start_frame = start;
    ctx->rc_override[i].end_frame = end;
    if (q>0) {
      ctx->rc_override[i].qscale = q;
      ctx->rc_override[i].quality_factor = 1.0;
    } else {
      ctx->rc_override[i].qscale = 0;
      ctx->rc_override[i].quality_factor = -q/100.0;
    }
    p = strchr(p, '/');
    if (p) p++;
  }
  ctx->rc_override_count = i;
#endif

  ctx->rc_max_rate = vc_lavc_param.rc_max_rate * 1000;
  ctx->rc_min_rate = vc_lavc_param.rc_min_rate * 1000;
  ctx->rc_buffer_size = vc_lavc_param.rc_buffer_size * 1000;
  ctx->rc_buffer_aggressivity = vc_lavc_param.rc_buffer_size;
  ctx->rc_initial_cplx = vc_lavc_param.rc_init_cplx;
  ctx->i_quant_factor = vc_lavc_param.i_qfactor;
  ctx->b_quant_factor = vc_lavc_param.b_qfactor;
  ctx->i_quant_offset = vc_lavc_param.i_qoffset;
  ctx->b_quant_offset = vc_lavc_param.b_qoffset;
  ctx->dct_algo = vc_lavc_param.dct_algo;
  ctx->idct_algo = vc_lavc_param.idct_algo;
  ctx->strict_std_compliance = vc_lavc_param.strict;
  if (vc_lavc_param.packet_size) {
    ctx->rtp_mode = 1;
    ctx->rtp_payload_size = vc_lavc_param.packet_size;
  }

//  if (vc_lavc_param.do_psnr)
//    ctx->flags |= CODEC_FLAG_PSNR;

  ctx->me_method = vc_lavc_param.me_method;

  if (vc_lavc_param.do_pass) {
    char logfilename[1024];
    FILE *f;
    int size;
    char *logbuffer;
    snprintf(logfilename, sizeof(logfilename), "%s.log",
	vc_lavc_param.passlogfile);
    if (vc_lavc_param.do_pass == 1) {
      ctx->flags |= CODEC_FLAG_PASS1;
      f = fopen(logfilename, "w");
      if (!f) {
	perror(logfilename);
	vc_libavcodec_encode_quit();
	return VCODEC_FAIL;
      }
      enc_conf.logfile = f;
    } else {
      ctx->flags |= CODEC_FLAG_PASS2;
      f = fopen(logfilename, "r");
      if (!f) {
	perror(logfilename);
	fprintf(stderr, "vc_libavcodec_encode_config: log file open failed.\n");
	vc_libavcodec_encode_quit();
	return VCODEC_FAIL;
      }
      fseek(f, 0, SEEK_END);
      size = ftell(f);
      fseek(f, 0, SEEK_SET);
      logbuffer = av_malloc(size + 1);
      if (!logbuffer) {
	fprintf(stderr,"vc_libavcodec_encode_config: memory allocate error.\n");
	fclose(f);
	vc_libavcodec_encode_quit();
	return VCODEC_FAIL;
      }
      fread(logbuffer, 1, size, f);
      fclose(f);
      logbuffer[size] = '\0';
      ctx->stats_in = logbuffer;
      enc_conf.logfile = NULL;
    }
  }

  if (avcodec_open(ctx, enc_conf.avcodec) < 0) {
    fprintf(stderr, "vc_libavcodec_encode_config: avcodec_open failed.\n");
    vc_libavcodec_encode_quit();
    return VCODEC_FAIL;
  }

  return VCODEC_OK;
}

int
vc_libavcodec_encode_init (VCODEC *vcodec)
{
  static int first = 1;
  int csp, ret, need_conf = 0;
  enum CodecID id;

  if (first) {
    enc_conf.width = 0;
    enc_conf.height = 0;
    enc_conf.fps = 0.0;
    enc_conf.in_csp = CSP_UNKNOWN;
    enc_conf.out_csp = CSP_UNKNOWN;
    enc_conf.context  = NULL;
    enc_conf.avcodec    = NULL;
    enc_conf.picture  = NULL;
    enc_conf.codec_id = CODEC_ID_NONE;
    enc_conf.codec = VCODEC_UNKNOWN;
    first = 0;
  }

  csp = vc_libavcodec_encode_csp_cap(vcodec->in_csp);
  if (vcodec->in_csp == CSP_UNKNOWN || csp == CSP_UNKNOWN) {
    fprintf(stderr, "vc_libavcodec_encode_init: invalid in_csp.\n");
    return VCODEC_FAIL;
  }
  vcodec->out_csp = CSP_COMPRESSED;

  if (enc_conf.context == NULL)
    need_conf = 1;
  else if (enc_conf.codec != vcodec->codec)
    need_conf = 1;
  else if (enc_conf.width != vcodec->width)
    need_conf = 1;
  else if (enc_conf.height != vcodec->height)
    need_conf = 1;
  else if (enc_conf.fps != vcodec->fps)
    need_conf = 1;
  else if (enc_conf.in_csp != csp)
    need_conf = 1;

  if (need_conf == 0) {
    return VCODEC_OK;
  }

  if (!vc_libavcodec_init_called) {
    vc_libavcodec_init();
  }

  id = vc_libavcodec_vcodec_to_id(vcodec->codec);
  if (id == CODEC_ID_NONE) {
    fprintf(stderr, "vc_libavcodec_encode_init: cann't fine codec.\n");
    return VCODEC_FAIL;
  }

  enc_conf.codec = vcodec->codec;
  enc_conf.in_csp = csp;
  enc_conf.out_csp = CSP_COMPRESSED;
  enc_conf.width = vcodec->width;
  enc_conf.height = vcodec->height;
  enc_conf.fps = vcodec->fps;
  enc_conf.codec_id = id;

  ret = vc_libavcodec_encode_config();

  return ret;
}

void
vc_libavcodec_encode_print_param(void)
{
  SEL_COMPO *sel;
  int i;
  const char *on = "ON", *off = "OFF";

  fprintf(stdout, "LIBAVCODEC VIDEO ENCODE PARAMETER\n");
  fprintf(stdout, "fourcc:                  %s\n", vc_libavcodec_codec_to_fourcc_str(enc_conf.codec));
  fprintf(stdout, "dimension:               %dx%d\n", enc_conf.width, enc_conf.height);
  fprintf(stdout, "csp:                     %s\n", csp_to_str(enc_conf.in_csp));
  fprintf(stdout, "bitrate:                 %d\n", vc_lavc_param.bitrate);
  fprintf(stdout, "bitrate_tolerance:       %d\n", vc_lavc_param.bitrate_tolerance);
  fprintf(stdout, "intra_only:              %s\n", vc_lavc_param.intra_only ? on:off);
  fprintf(stdout, "gop_size:                %d\n", vc_lavc_param.gop_size);
  fprintf(stdout, "qscale:                  %d\n", vc_lavc_param.qscale);
  fprintf(stdout, "qmin:                    %d\n", vc_lavc_param.qmin);
  fprintf(stdout, "qmax:                    %d\n", vc_lavc_param.qmax);
  fprintf(stdout, "mbqmin:                  %d\n", vc_lavc_param.mb_qmin);
  fprintf(stdout, "mbqmax:                  %d\n", vc_lavc_param.mb_qmax);
  fprintf(stdout, "max_rate:                %d\n", vc_lavc_param.rc_max_rate);
  fprintf(stdout, "min_rate:                %d\n", vc_lavc_param.rc_min_rate);
  fprintf(stdout, "buffer_size:             %d\n", vc_lavc_param.rc_buffer_size);
  fprintf(stdout, "rc_eq:                   %s\n", vc_lavc_param.rc_eq);
  fprintf(stdout, "rc_init_cplx:            %f\n", vc_lavc_param.rc_init_cplx);
  fprintf(stdout, "qdiff:                   %d\n", vc_lavc_param.qdiff);
  fprintf(stdout, "qblur:                   %f\n", vc_lavc_param.qblur);
  fprintf(stdout, "qcomp:                   %f\n", vc_lavc_param.qcomp);
  fprintf(stdout, "use_hq:                  %s\n", vc_lavc_param.use_hq ? on:off);
  fprintf(stdout, "use_4mv:                 %s\n", vc_lavc_param.use_4mv ? on:off);
  fprintf(stdout, "use_part:                %s\n", vc_lavc_param.use_part ? on:off);
  fprintf(stdout, "b_frames:                %d\n", vc_lavc_param.b_frames);
  fprintf(stdout, "b_qfactor:               %f\n", vc_lavc_param.b_qfactor);
  fprintf(stdout, "i_qfactor:               %f\n", vc_lavc_param.i_qfactor);
  fprintf(stdout, "b_qoffset:               %f\n", vc_lavc_param.b_qoffset);
  fprintf(stdout, "i_qoffset:               %f\n", vc_lavc_param.i_qoffset);
  sel = me_methods;
  for (i=0; i < sizeof(me_methods)/sizeof(SEL_COMPO); i++, sel++) {
    if (sel->id == vc_lavc_param.me_method) {
      fprintf (stderr, "me_methods:             %s\n", sel->name);
      break;
    }
  }
  fprintf(stdout, "pass:                    %d\n", vc_lavc_param.do_pass);
  fprintf(stdout, "passlogfile:             %s\n", vc_lavc_param.passlogfile);
  fprintf(stdout, "dct_algo:                %d\n", vc_lavc_param.dct_algo);
  fprintf(stdout, "idct_algo:               %d\n", vc_lavc_param.idct_algo);
  fprintf(stdout, "strict:                  %d\n", vc_lavc_param.strict);
  fprintf(stdout, "debug:                   %d\n", vc_lavc_param.debug);
//  fprintf(stdout, "packet_size:             %d\n", vc_lavc_param.packet_size);
  fflush (stdout);
}

/* decode functions */

int
vc_libavcodec_decode_csp_cap(int csp)
{
  int ret;
  switch (csp) {
    case CSP_UNKNOWN:
      ret = CSP_YUV420P;
      break;
//    case CSP_RGB32:
//    case CSP_RGB24:
//    case CSP_RGB565:
//    case CSP_RGB555:
//    case CSP_YUV422:
//    case CSP_YUYV:
//    case CSP_YUY2:
//    case CSP_UYVY:
//    case CSP_YVYU:
    case CSP_YUV420P:
    case CSP_I420:
    case CSP_IYUV:
    case CSP_YV12:
      ret = csp;
      break;
    default:
      ret = CSP_UNKNOWN;
      break;
  }
  return ret;
}

int
vc_libavcodec_decode (unsigned char *pic_data, unsigned char *stream,
    int stream_length, unsigned int *flag)
{
  int size, len, got_pic;
  unsigned char *s;

  dec_conf.context->pix_fmt = dec_conf.pix_fmt;

  size = stream_length;
  s = stream;
  while (size > 0) {
    len = avcodec_decode_video(dec_conf.context, dec_conf.picture, &got_pic,
	s, size);
    if (len < 0) {
      fprintf(stderr, "vc_libavcodec_decode: decode error.\n");
      exit(1);
    }
    if (got_pic) {
      int y_size = dec_conf.width * dec_conf.height;
      int uv_size = y_size / 4;
      memcpy(pic_data, dec_conf.picture->data[0], y_size);
      memcpy(pic_data+dec_conf.u_offset, dec_conf.picture->data[1], uv_size);
      memcpy(pic_data+dec_conf.v_offset, dec_conf.picture->data[2], uv_size);
      return dec_conf.data_size;
    }
    size -= len;
    s += len;
  }

  return size;
}

int
vc_libavcodec_decode_quit (void)
{
  if (dec_conf.context) {
    avcodec_close(dec_conf.context);
    free(dec_conf.context);
  }
  if (dec_conf.picture)
    free(dec_conf.picture);
  dec_conf.context = NULL;
  dec_conf.picture = NULL;
  dec_conf.avcodec = NULL;
  dec_conf.codec_id = CODEC_ID_NONE;
  dec_conf.codec = VCODEC_UNKNOWN;
  return VCODEC_OK;
}

int
vc_libavcodec_decode_config (void)
{

  AVCodecContext *ctx;
//  char *p;
//  int i;

  if (dec_conf.context) {
    vc_libavcodec_decode_quit();
  }

  dec_conf.avcodec = avcodec_find_decoder(dec_conf.codec_id);
  if (!dec_conf.avcodec) {
    return VCODEC_FAIL;
  }
  ctx = avcodec_alloc_context();
  dec_conf.context = ctx;
  dec_conf.picture = avcodec_alloc_frame();
  if (!dec_conf.context || !dec_conf.picture) {
    fprintf(stderr, "vc_libavcodec_decode_config: memory allocate error.\n");
    vc_libavcodec_decode_quit();
    return VCODEC_FAIL;
  }

  if (dec_conf.out_csp == CSP_YV12) {
    dec_conf.v_offset = dec_conf.width * dec_conf.height;
    dec_conf.u_offset = (dec_conf.width * dec_conf.height) / 4 * 5;
    dec_conf.uv_width = dec_conf.width / 2;
  } else {
    dec_conf.u_offset = dec_conf.width * dec_conf.height;
    dec_conf.v_offset = (dec_conf.width * dec_conf.height) / 4 * 5;
    dec_conf.uv_width = dec_conf.width / 2;
  }
  dec_conf.data_size = dec_conf.width * dec_conf.height * 12 / 8;
  dec_conf.pix_fmt = csp_to_pix_fmt(CSP_YUV420P);

  ctx->codec_id = dec_conf.codec_id;
  ctx->width = dec_conf.width;
  ctx->height = dec_conf.height;
  ctx->frame_rate = dec_conf.fps * FRAME_RATE_BASE;
  ctx->pix_fmt = dec_conf.pix_fmt;

  ctx->workaround_bugs = vc_lavc_param.workaround_bugs;
  ctx->error_resilience = vc_lavc_param.error_resilience;
  ctx->error_concealment = vc_lavc_param.error_concealment;
  ctx->dct_algo = vc_lavc_param.dct_algo;
  ctx->idct_algo = vc_lavc_param.idct_algo;
  ctx->debug = vc_lavc_param.debug;

#if 0
  if (dec_conf.avcodec->capabilities & CODEC_CAP_TRUNCATED)
    dec_conf.context->flags |= CODEC_FLAG_TRUNCATED;
#endif
  if (dec_conf.codec_id == CODEC_ID_MPEG1VIDEO)
    dec_conf.context->flags |= CODEC_FLAG_TRUNCATED;

  if (avcodec_open(ctx, dec_conf.avcodec) < 0) {
    fprintf(stderr, "vc_libavcodec_decode_config: avcodec_open failed.\n");
    vc_libavcodec_decode_quit();
    return VCODEC_FAIL;
  }

  return VCODEC_OK;
}

int
vc_libavcodec_decode_init (VCODEC *vcodec)
{
  static int first = 1;
  int csp, ret, need_conf = 0;
  enum CodecID id;

  if (first) {
    dec_conf.width = 0;
    dec_conf.height = 0;
    dec_conf.fps = 0.0;
    dec_conf.in_csp = CSP_UNKNOWN;
    dec_conf.out_csp = CSP_UNKNOWN;
    dec_conf.context  = NULL;
    dec_conf.avcodec    = NULL;
    dec_conf.picture  = NULL;
    dec_conf.codec_id = CODEC_ID_NONE;
    dec_conf.codec = VCODEC_UNKNOWN;
    first = 0;
  }

  csp = vc_libavcodec_decode_csp_cap(vcodec->out_csp);
  if (vcodec->out_csp == CSP_UNKNOWN || csp == CSP_UNKNOWN) {
    fprintf(stderr, "vc_libavcodec_decode_init: invalid out_csp.\n");
    return VCODEC_FAIL;
  }
  vcodec->in_csp = CSP_COMPRESSED;

  if (dec_conf.context == NULL)
    need_conf = 1;
  else if (dec_conf.codec != vcodec->codec)
    need_conf = 1;
  else if (dec_conf.width != vcodec->width)
    need_conf = 1;
  else if (dec_conf.height != vcodec->height)
    need_conf = 1;
  else if (dec_conf.fps != vcodec->fps)
    need_conf = 1;
  else if (dec_conf.out_csp != csp)
    need_conf = 1;

  if (need_conf == 0) {
    return VCODEC_OK;
  }

  if (!vc_libavcodec_init_called) {
    vc_libavcodec_init();
  }

  id = vc_libavcodec_vcodec_to_id(vcodec->codec);
  if (id == CODEC_ID_NONE) {
    fprintf(stderr, "vc_libavcodec_decode_init: cannot find codec.\n");
    return VCODEC_FAIL;
  }

  dec_conf.codec = vcodec->codec;
  dec_conf.in_csp = CSP_COMPRESSED;
  dec_conf.out_csp = csp;
  dec_conf.width = vcodec->width;
  dec_conf.height = vcodec->height;
  dec_conf.fps = vcodec->fps;
  dec_conf.codec_id = id;

  ret = vc_libavcodec_decode_config();

  return ret;
}

void
vc_libavcodec_decode_print_param(void)
{
  fprintf(stdout, "LIBAVCODEC DECODE PARAMETER\n");
  fprintf(stdout, "fourcc:                  %s\n", vc_libavcodec_codec_to_fourcc_str(dec_conf.codec));
  fprintf(stdout, "dimmension:              %dx%d\n", dec_conf.width, dec_conf.height);
  fprintf(stdout, "csp:                     %s\n",csp_to_str(dec_conf.out_csp));
  fprintf(stdout, "workaround_bugs:         %d\n", vc_lavc_param.workaround_bugs);
  fprintf(stdout, "error_resilience:        %d\n", vc_lavc_param.error_resilience);
  fprintf(stdout, "error_concealment:       %d\n", vc_lavc_param.error_concealment);
  fprintf(stdout, "dct_algo:                %d\n", vc_lavc_param.dct_algo);
  fprintf(stdout, "idct_algo:               %d\n", vc_lavc_param.idct_algo);
  fprintf(stdout, "debug:                   %d\n", vc_lavc_param.debug);
  fflush (stdout);
}

/* vcodec functions struct */

VCODEC_FUNCS vc_libavcodec_funcs = {
  vc_libavcodec_encode_init,
  vc_libavcodec_encode_quit,
  vc_libavcodec_encode_csp_cap,
  vc_libavcodec_encode,
  vc_libavcodec_encode_print_param,

  vc_libavcodec_decode_init,
  vc_libavcodec_decode_quit,
  vc_libavcodec_decode_csp_cap,
  vc_libavcodec_decode,
  vc_libavcodec_decode_print_param,

  vc_libavcodec_codec_to_fourcc_str,
  vc_libavcodec_fourcc_str_to_codec,
  vc_libavcodec_codec_cap,
};


#endif /* HAVE_LIBAVCODEC */

