/* v4l.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <string.h>
#include <linux/videodev.h>

#include "csp.h"
#include "v4l.h"

static V4L_PARAM v4l_param;

#define DEFAULT_DEVICE_NAME "/dev/video0"
static char *default_device_name = DEFAULT_DEVICE_NAME;

int
v4l_capture_cap(void)
{
  if (v4l_param.fd == -1)
    return V4L_FAIL;
  if (!(v4l_param.type & VID_TYPE_CAPTURE)) {
    fprintf (stderr, "v4l_grab_set_mmap: device cannot capture.\n");
    return V4L_FAIL;
  }

  return V4L_OK;
}

static int
norm_to_v4l_norm(int n)
{
  int norm;
  switch (n) {
    case NORM_PAL:
      norm = VIDEO_MODE_PAL;
      break;
    case NORM_NTSC:
      norm = VIDEO_MODE_NTSC;
      break;
    case NORM_SECAM:
      norm = VIDEO_MODE_SECAM;
      break;
    case NORM_AUTO:
      norm = VIDEO_MODE_AUTO;
      break;
    default:
      return V4L_FAIL;
  }
  return norm;
}

int
v4l_set_tuner (int tuner)
{
  struct video_tuner vid_tun;
  int freq;

  if (v4l_param.fd == -1)
    return V4L_FAIL;
  
  if (!(v4l_param.channel_flags & VIDEO_VC_TUNER) || !v4l_param.channel_tuners){
    v4l_param.tuner = -1;
    return V4L_FAIL;
  }

#if 0
  if (tuner == v4l_param.tuner)
    return V4L_OK;
#endif

  if (tuner < 0)
    tuner = 0;
  else if (tuner >= v4l_param.channel_tuners)
    tuner = v4l_param.channel_tuners-1;

  vid_tun.tuner = tuner;
  if (ioctl (v4l_param.fd, VIDIOCGTUNER, &vid_tun) < 0) {
    perror ("v4l_set_tuner: VIDIOCGTUNER error");
    return V4L_FAIL;
  }
  vid_tun.tuner = tuner;
  vid_tun.mode = v4l_param.norm;
  vid_tun.signal = 65535;
  if (ioctl (v4l_param.fd, VIDIOCSTUNER, &vid_tun) < 0) {
    perror ("v4l_set_tuner: VIDIOCSTUNER error");
    return V4L_FAIL;
  }
  if (ioctl (v4l_param.fd, VIDIOCGTUNER, &vid_tun) < 0) {
    perror ("v4l_set_tuner: VIDIOCGTUNER error");
    return V4L_FAIL;
  }
  v4l_param.tuner = vid_tun.tuner;
  strncpy(v4l_param.tuner_name, vid_tun.name, 32);
  v4l_param.rangelow = vid_tun.rangelow;
  v4l_param.rangehigh = vid_tun.rangehigh;
  v4l_param.tuner_flags = vid_tun.flags;
  v4l_param.tuner_mode = vid_tun.mode;
  v4l_param.tuner_signal = vid_tun.signal;

  if (v4l_param.tuner_mode != v4l_param.norm) {
    fprintf(stderr, "v4l_set_tuner: norm failed.\n");
  }

  if (ioctl (v4l_param.fd, VIDIOCGFREQ, &freq) < 0) {
    perror ("v4l_get_freq: VIDIOCGFREQ error");
    return V4L_FAIL;
  }
  v4l_param.freq = freq;

  return V4L_OK;
}

int
v4l_get_tuner (void)
{
  if (v4l_param.fd == -1 ||
      !(v4l_param.channel_flags & VIDEO_VC_TUNER) || !v4l_param.channel_tuners)
    return V4L_FAIL;
  return v4l_param.tuner;
}

int
v4l_set_source_channel (int source_channel)
{
  struct video_channel vid_chan;
  int tuner;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

#if 0
  if (source_channel == v4l_param.source_channel &&
      v4l_param.norm == v4l_param.channel_norm &&
      (v4l_param.channel_flags & VIDEO_VC_TUNER &&
      v4l_param.norm == v4l_param.tuner_mode))
    return V4L_OK;
#endif

  if (source_channel < 0)
    source_channel = 0;
  if (source_channel >= v4l_param.channels)
    source_channel = v4l_param.channels-1;

  vid_chan.channel = source_channel;
  if (ioctl (v4l_param.fd, VIDIOCGCHAN, &vid_chan) < 0) {
    perror ("v4l_set_source_channel: VIDIOCGCHAN error");
    return V4L_FAIL;
  }
  if (v4l_param.norm == -1)
    v4l_param.norm = vid_chan.norm;
  else
    vid_chan.norm = v4l_param.norm;
  vid_chan.channel = source_channel;
  if (ioctl (v4l_param.fd, VIDIOCSCHAN, &vid_chan) < 0) {
    perror ("v4l_set_source_channel(): VIDIOCSCHAN error");
    return V4L_FAIL;
  }
  if (ioctl (v4l_param.fd, VIDIOCGCHAN, &vid_chan) < 0) {
    perror ("v4l_set_source_channel: VIDIOCGCHAN error");
    return V4L_FAIL;
  }
  v4l_param.source_channel = vid_chan.channel;
  strncpy(v4l_param.channel_name, vid_chan.name, 32);
  v4l_param.channel_tuners = vid_chan.tuners;
  v4l_param.channel_flags = vid_chan.flags;
  v4l_param.channel_type = vid_chan.type;
  v4l_param.channel_norm = vid_chan.norm;

  if (v4l_param.norm != v4l_param.channel_norm) {
    fprintf(stderr, "v4l_set_source_channel: norm failed.\n");
    v4l_param.norm = v4l_param.channel_norm;
  }

  tuner = v4l_param.tuner;
  v4l_param.tuner = -1;
  if (v4l_param.channel_flags & VIDEO_VC_TUNER) {
    if (tuner == -1)
      tuner = 0;
    v4l_set_tuner(tuner);
  }

  return V4L_OK;
}

int
v4l_get_source_channel (void)
{
  if (v4l_param.fd == -1)
    return V4L_FAIL;
  return v4l_param.source_channel;
}

int
v4l_get_norm (void)
{
  if (v4l_param.fd == -1)
    return V4L_FAIL;

  switch (v4l_param.norm) {
    case VIDEO_MODE_PAL:
      return NORM_PAL;
    case VIDEO_MODE_NTSC:
      return NORM_NTSC;
    case VIDEO_MODE_SECAM:
      return NORM_SECAM;
    case VIDEO_MODE_AUTO:
      return NORM_AUTO;
  }
  return NORM_UNKNOWN;
}

int
v4l_set_norm (int n)
{
  int norm;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

  norm = norm_to_v4l_norm(n);
  if (norm == V4L_FAIL) {
    fprintf(stderr, "v4l_set_norm: ERROR unknown norm.\n");
    return V4L_FAIL;
  }

  v4l_param.norm = norm;
  if (norm != v4l_param.channel_norm ||
     (v4l_param.channel_flags&VIDEO_VC_TUNER && v4l_param.tuner_mode!=norm)){
    if (v4l_set_source_channel(v4l_param.source_channel) == V4L_FAIL)
      return V4L_FAIL;
  }

  return V4L_OK;
}

int
v4l_set_freq (int freq)
{
  int f;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

  if (!(v4l_param.channel_flags & VIDEO_VC_TUNER)) {
    return V4L_FAIL;
  }

  f = freq;
  if (ioctl (v4l_param.fd, VIDIOCSFREQ, &f) < 0) {
    perror ("v4l_set_freq() VIDIOCSFREQ error");
    return V4L_FAIL;
  }
  if (ioctl (v4l_param.fd, VIDIOCGFREQ, &f) < 0) {
    perror ("v4l_set_freq: VIDIOCGFREQ error");
    return V4L_FAIL;
  }
  if (f != freq)
    return V4L_FAIL;
  v4l_param.freq = freq;
  return V4L_OK;
}

int
v4l_get_freq (void)
{
  int freq;

  if (v4l_param.fd == -1)
    return V4L_FAIL;
  if (!(v4l_param.channel_flags & VIDEO_VC_TUNER)) {
    return V4L_FAIL;
  }
  if (ioctl (v4l_param.fd, VIDIOCGFREQ, &freq) < 0) {
    perror ("v4l_get_freq: VIDIOCGFREQ error");
    return V4L_FAIL;
  }
  return freq;
}

typedef struct CSP_PALETTE {
  int csp;
  int palette;
} CSP_PALETTE;

static CSP_PALETTE csp_palette[] = {
//  { CSP_GREY, VIDEO_PALETTE_GREY },
//  { CSP_HI240, VIDEO_PALETTE_HI240 },
  { CSP_RGB565, VIDEO_PALETTE_RGB565 },
  { CSP_RGB555, VIDEO_PALETTE_RGB555 },
  { CSP_RGB24, VIDEO_PALETTE_RGB24 },
  { CSP_RGB32, VIDEO_PALETTE_RGB32 },
  { CSP_YUV422, VIDEO_PALETTE_YUV422 },
  { CSP_YUYV, VIDEO_PALETTE_YUYV },
  { CSP_UYVY, VIDEO_PALETTE_UYVY },
  { CSP_YUV420, VIDEO_PALETTE_YUV420 },
  { CSP_YUV411, VIDEO_PALETTE_YUV411 },
//  { CSP_RAW, VIDEO_PALETTE_RAW },
  { CSP_YUV422P, VIDEO_PALETTE_YUV422P },
  { CSP_YUV411P, VIDEO_PALETTE_YUV411P },
  { CSP_YUV420P, VIDEO_PALETTE_YUV420P },
  { CSP_YUV410P, VIDEO_PALETTE_YUV410P },

  { CSP_I420, VIDEO_PALETTE_YUV420P },
};
static int csp_palette_num = sizeof(csp_palette)/sizeof(CSP_PALETTE);

static int
csp_to_palette(int csp)
{
  int i, palette;

  for (i = 0; i < csp_palette_num; i++) {
    if (csp_palette[i].csp == csp)
      break;
  }
  if (i == csp_palette_num)
    palette = -1;
  else
    palette = csp_palette[i].palette;
  return palette;
}

static int
palette_to_csp(int palette)
{
  int i, csp;

  for (i = 0; i < csp_palette_num; i++) {
    if (csp_palette[i].palette == palette)
      break;
  }
  if (i == csp_palette_num)
    csp = CSP_UNKNOWN;
  else
    csp = csp_palette[i].csp;
  return csp;
}

int
v4l_set_csp(int csp)
{
  struct video_picture vid_pic;
  int palette;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

  palette = csp_to_palette(csp);
  if (palette == -1) {
    fprintf(stderr, "v4l_set_csp: unsupported csp.\n");
    return V4L_FAIL;
  } else if (palette == v4l_param.palette) {
    return V4L_OK;
  }

  if (ioctl(v4l_param.fd, VIDIOCGPICT, &vid_pic) < 0) {
    perror ("v4l_set_csp: VIDIOCGPICT error");
    return V4L_FAIL;
  }
  if (vid_pic.palette == palette)
    return V4L_OK;

  vid_pic.palette = palette;
  if (ioctl(v4l_param.fd, VIDIOCSPICT, &vid_pic) < 0) {
    perror ("v4l_set_csp: VIDIOCSPICT error");
    return V4L_FAIL;
  }
  if (ioctl(v4l_param.fd, VIDIOCGPICT, &vid_pic) < 0) {
    perror ("v4l_set_csp: VIDIOCGPICT error");
    return V4L_FAIL;
  }
  if (vid_pic.palette != palette) {
    fprintf(stderr, "v4l_set_csp: unsupported csp.\n");
    return V4L_FAIL;
  }
  v4l_param.palette = palette;

  return V4L_OK;
}

int
v4l_get_csp(void)
{
  if (v4l_param.fd == -1)
    return V4L_FAIL;
  return palette_to_csp(v4l_param.palette);
}

int
v4l_set_audio(int audio, int on)
{
  struct video_audio vid_aud;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

  if (v4l_param.audios <= 0 || !(v4l_param.channel_flags & VIDEO_VC_AUDIO)) {
    v4l_param.audio = -1;
    return V4L_FAIL;
  }

  if (audio < 0)
    audio = 0;
  else if (audio >= v4l_param.audios)
    audio = v4l_param.audios - 1;

  vid_aud.audio = audio;
  if (ioctl (v4l_param.fd, VIDIOCGAUDIO, &vid_aud) < 0) {
    perror ("v4l_set_audio: VIDIOCGAUDIO error");
    return V4L_FAIL;
  }
  if (on) {
    vid_aud.flags &= ~VIDEO_AUDIO_MUTE;
  } else {
    if (vid_aud.flags & VIDEO_AUDIO_MUTABLE)
      vid_aud.flags |= VIDEO_AUDIO_MUTE;
  }
  if (ioctl(v4l_param.fd, VIDIOCSAUDIO, &vid_aud) < 0) {
    perror ("v4l_set_audio: VIDIOCSAUDIO error");
    return V4L_FAIL;
  }
  if (ioctl (v4l_param.fd, VIDIOCGAUDIO, &vid_aud) < 0) {
    perror ("v4l_set_audio: VIDIOCGAUDIO error");
    return V4L_FAIL;
  }
  v4l_param.audio = vid_aud.audio;
  v4l_param.audio_volume = vid_aud.volume;
  v4l_param.audio_bass = vid_aud.bass;
  v4l_param.audio_treble = vid_aud.treble;
  v4l_param.audio_flags = vid_aud.flags;
  strncpy(v4l_param.audio_name, vid_aud.name, 16);
  v4l_param.audio_mode = vid_aud.mode;
  v4l_param.audio_balance = vid_aud.balance;
  v4l_param.audio_step = vid_aud.step;
  return V4L_OK;
}

int
v4l_set_window_size (int w, int h)
{
  struct video_window vid_win;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

  if (ioctl(v4l_param.fd, VIDIOCGWIN, &vid_win) < 0) {
    perror ("v4l_window_size: VIDIOCGWIN error");
    return V4L_FAIL;
  }

  if (w > 0)
    vid_win.width = w;
  if (h > 0)
    vid_win.height = h;

  vid_win.x         = 0;
  vid_win.y         = 0;
//  vid_win.flags     = VIDEO_WINDOW_INTERLACE;
  vid_win.flags     = 0;
  vid_win.clips     = NULL;
  vid_win.clipcount = 0;
  if (ioctl(v4l_param.fd, VIDIOCSWIN, &vid_win) < 0) {
    perror ("v4l_set_window_size: VIDIOCSWIN error");
    return V4L_FAIL;
  }
  if (ioctl (v4l_param.fd, VIDIOCGWIN, &vid_win) < 0) {
    perror ("v4l_set_window_size: VIDIOCGWIN error");
    return V4L_FAIL;
  }
  v4l_param.x = vid_win.x;
  v4l_param.y = vid_win.y;
  v4l_param.width = vid_win.width;
  v4l_param.height = vid_win.height;
  v4l_param.chromakey = vid_win.chromakey;
  v4l_param.win_flags = vid_win.flags;

  return V4L_OK;
}

int
v4l_get_window_size (int *width, int *height)
{
  if (v4l_param.fd == -1) {
    *width = *height = 0;
    return V4L_FAIL;
  }
  *width  = v4l_param.width;
  *height = v4l_param.height;
  return V4L_OK;
}

int
v4l_color(int brightness, int hue, int colour, int contrast, int whiteness)
{
  struct video_picture vid_pic;
  if (v4l_param.fd == -1)
    return V4L_FAIL;

  if (ioctl(v4l_param.fd, VIDIOCGPICT, &vid_pic) < 0) {
    perror ("v4l_set_color: VIDIOCGPICT error");
    return V4L_FAIL;
  }
  if (brightness >= 0)
    vid_pic.brightness = brightness;
  if (hue >= 0)
    vid_pic.hue        = hue;
  if (colour >= 0)
    vid_pic.colour     = colour;
  if (contrast >= 0)
    vid_pic.contrast   = contrast;
  if (whiteness >= 0)
    vid_pic.whiteness  = whiteness;
  if (ioctl(v4l_param.fd, VIDIOCSPICT, &vid_pic) < 0) {
    perror ("v4l_set_color: VIDIOCSPICT error");
    return V4L_FAIL;
  }
  if (ioctl(v4l_param.fd, VIDIOCGPICT, &vid_pic) < 0) {
    perror ("v4l_set_color: VIDIOCGPICT error");
    return V4L_FAIL;
  }
  v4l_param.brightness = vid_pic.brightness;
  v4l_param.hue        = vid_pic.hue;
  v4l_param.colour     = vid_pic.colour;
  v4l_param.contrast   = vid_pic.contrast;
  v4l_param.whiteness  = vid_pic.whiteness;
  v4l_param.depth      = vid_pic.depth;
  v4l_param.palette    = vid_pic.palette;

  if (brightness >= 0 && vid_pic.brightness != brightness)
    return V4L_FAIL;
  else if (hue >= 0 && vid_pic.hue != hue)
    return V4L_FAIL;
  else if (colour >= 0 && vid_pic.colour != colour)
    return V4L_FAIL;
  else if (contrast >= 0 && vid_pic.contrast != contrast)
    return V4L_FAIL;
  else if (whiteness >= 0 && vid_pic.whiteness  != whiteness)
    return V4L_FAIL;

  return V4L_OK;
}

int
v4l_set_color (float percent, int element)
{
  float max;
  int val;
  int brightness, hue, colour, contrast, whiteness;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

  if (percent < 0)
    return V4L_OK;

  max = (float)65535;
  if (percent > 100.0)
    percent = 100.0;
  val = percent / 100.0 * max;
  if (val < 0)
    val = 0;
  else if (val > 65535)
    val = 65535;

  brightness = hue = colour = contrast = whiteness = -1;
  switch (element) {
    case V4L_BRIGHTNESS:
      brightness = val;
      break;
    case V4L_HUE:
      hue = val;
      break;
    case V4L_COLOUR:
      colour = val;
      break;
    case V4L_CONTRAST:
      contrast = val;
      break;
    case V4L_WHITENESS:
      whiteness = val;
      break;
  }

  if (v4l_color(brightness, hue, colour, contrast, whiteness) == V4L_FAIL)
      return V4L_FAIL;

  return V4L_OK;
}

int
v4l_set_brightness(float percent)
{
  return v4l_set_color(percent, V4L_BRIGHTNESS);
}

int
v4l_set_hue(float percent)
{
  return v4l_set_color(percent, V4L_HUE);
}

int
v4l_set_colour(float percent)
{
  return v4l_set_color(percent, V4L_COLOUR);
}

int
v4l_set_contrast(float percent)
{
  return v4l_set_color(percent, V4L_CONTRAST);
}

int
v4l_set_whiteness(float percent)
{
  return v4l_set_color(percent, V4L_WHITENESS);
}

float
v4l_get_color (int element)
{
  float max;
  float percent;
  float val = 0;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

  switch (element) {
    case V4L_BRIGHTNESS:
      val = (float)v4l_param.brightness;
      break;
    case V4L_HUE:
      val = (float)v4l_param.hue;
      break;
    case V4L_COLOUR:
      val = (float)v4l_param.colour;
      break;
    case V4L_CONTRAST:
      val = (float)v4l_param.contrast;
      break;
    case V4L_WHITENESS:
      val = (float)v4l_param.whiteness;
      break;
  }
  max = (float)65535;
  percent = val / max * 100.0;
  return percent;
}

float
v4l_get_brightness(void)
{
  return v4l_get_color(V4L_BRIGHTNESS);
}

float
v4l_get_hue(void)
{
  return v4l_get_color(V4L_HUE);
}

float
v4l_get_colour(void)
{
  return v4l_get_color(V4L_COLOUR);
}

float
v4l_get_contrast(void)
{
  return v4l_get_color(V4L_CONTRAST);
}

float
v4l_get_whiteness(void)
{
  return v4l_get_color(V4L_WHITENESS);
}

int
v4l_overlay_on (void)
{
  int one = 1;

  if (v4l_param.fd == -1)
    return V4L_FAIL;
  if ((v4l_param.type & VID_TYPE_CAPTURE) &&
      (v4l_param.type & VID_TYPE_OVERLAY)) {
    if (ioctl (v4l_param.fd, VIDIOCCAPTURE, &one) < 0) {
      perror ("v4l_overlay_on: VIDIOCCAPTURE error");
      return V4L_FAIL;
    }
  } else {
    return V4L_FAIL;
  }
  return V4L_OK;
}

int
v4l_overlay_off (void)
{
  int zero = 0;

  if (v4l_param.fd == -1)
    return V4L_FAIL;
  if ((v4l_param.type & VID_TYPE_CAPTURE) &&
      (v4l_param.type & VID_TYPE_OVERLAY)) {
    if (ioctl(v4l_param.fd, VIDIOCCAPTURE, &zero) < 0) {
      perror ("v4l_overlay_off: VIDIOCCAPTURE error");
      return V4L_FAIL;
    }
  } else {
    return V4L_FAIL;
  }
  return V4L_OK;
}

static int
v4l_grab_set_mmap (void)
{
  struct video_mbuf vid_mbuf;
  int i;

  if (v4l_param.fd == -1)
    return V4L_FAIL;

  if (v4l_param.offsets) {
    fprintf(stderr, "v4l_grab_set_mmap: offsets pointer remains.\n");
    free(v4l_param.offsets);
    v4l_param.offsets = NULL;
  }
  if (v4l_param.map_data) {
    fprintf(stderr, "v4l_grab_set_mmap: map_data pointer remains.\n");
    munmap(v4l_param.map_data, v4l_param.map_size);
    v4l_param.map_data = NULL;
  }

  if (ioctl(v4l_param.fd, VIDIOCGMBUF, &vid_mbuf) < 0) {
    perror ("v4l_grab_set_mmap: VIDIOCGMBUF error");
    return V4L_FAIL;
  }
  v4l_param.frames = vid_mbuf.frames;
  v4l_param.offsets = (int*) calloc (sizeof(int*), vid_mbuf.frames);
  for (i = 0; i < vid_mbuf.frames; i++)
    v4l_param.offsets[i] = vid_mbuf.offsets[i];
  v4l_param.map_size = vid_mbuf.size;
  v4l_param.map_data = (char*) mmap (0, v4l_param.map_size,
      PROT_READ|PROT_WRITE, MAP_SHARED, v4l_param.fd, 0);
  if (v4l_param.map_data == MAP_FAILED) {
    perror ("v4l_grab_set_mmap: mmap error");
    if (v4l_param.offsets) {
      free(v4l_param.offsets);
      v4l_param.offsets = NULL;
    }
    v4l_param.map_data = NULL;
    return V4L_FAIL;
  }
  return V4L_OK;
}

int
v4l_grab_pic_start (void)
{
  int i;

//  v4l_overlay_on ();
  if (v4l_param.fd == -1)
    return V4L_FAIL;

  if (!(v4l_param.type & VID_TYPE_CAPTURE)) {
    fprintf (stderr, "v4l_grab_set_mmap: device cannot capture.\n");
    return V4L_FAIL;
  }

  if (v4l_param.map_data != NULL || v4l_param.offsets != NULL) {
    fprintf(stderr, "v4l_grab_pic_start: map_data pointer remains.\n");
    return V4L_FAIL;
  }

  if (v4l_grab_set_mmap() == V4L_FAIL)
    return V4L_FAIL;

  v4l_param.vid_mmap.width  = v4l_param.width;
  v4l_param.vid_mmap.height = v4l_param.height;
  v4l_param.vid_mmap.format = v4l_param.palette;
  for (i = 0; i < v4l_param.frames; i++) {
    v4l_param.vid_mmap.frame = i;
    if (ioctl(v4l_param.fd, VIDIOCMCAPTURE, &v4l_param.vid_mmap) < 0) {
      perror("v4l_grab_pic_start: VIDIOCMCAPTURE error");
      if (v4l_param.map_data) {
        munmap (v4l_param.map_data, v4l_param.map_size);
        v4l_param.map_data = NULL;
      }
      if (v4l_param.offsets) {
        free (v4l_param.offsets);
        v4l_param.offsets = NULL;
      }
      return V4L_FAIL;
    }
  }
  v4l_param.vid_mmap.frame = 0;
  return V4L_OK;
}

char*
v4l_grab_pic_sync (void)
{
  if (ioctl(v4l_param.fd, VIDIOCSYNC, &v4l_param.vid_mmap.frame) < 0) {
    /*
    perror ("v4l_grab_sync() VIDIOCSYNC error");
    fprintf (stderr, "v4l_grab_sync(): cur_frame %d, frames %d\n",
	cur_frame, frames);
    */
    return NULL;
  }
  return &v4l_param.map_data[v4l_param.offsets[v4l_param.vid_mmap.frame]];
}

int
v4l_grab_pic_capture (void)
{
  if (ioctl(v4l_param.fd, VIDIOCMCAPTURE, &v4l_param.vid_mmap) < 0) {
    perror("v4l_grab_cap: VIDIOCMCAPTURE error");
    return V4L_FAIL;
  }
  if (++v4l_param.vid_mmap.frame >= (uint32_t)v4l_param.frames)
    v4l_param.vid_mmap.frame = 0;
  return V4L_OK;
}

int
v4l_grab_pic_stop (void)
{
  int i;

  if (v4l_param.map_data == NULL && v4l_param.offsets == NULL)
    return V4L_OK;

  for (i = 0; i < v4l_param.frames; i++) {
    if (ioctl(v4l_param.fd, VIDIOCSYNC, &v4l_param.vid_mmap.frame) < 0) {
      perror ("v4l_grab_pic_stop: VIDIOCSYNC error");
//      fprintf (stderr, "v4l_grab_pic_stop: cur_frame %d, frames %d\n",
//	cur_frame, frames);
      break;
    }
    if (++v4l_param.vid_mmap.frame >= (uint32_t)v4l_param.frames)
      v4l_param.vid_mmap.frame = 0;
  }

  if (v4l_param.map_data) {
    munmap (v4l_param.map_data, v4l_param.map_size);
    v4l_param.map_data = NULL;
  }
  if (v4l_param.offsets) {
    free (v4l_param.offsets);
    v4l_param.offsets = NULL;
  }
//  v4l_overlay_off ();
  return V4L_OK;
}

int
v4l_open (const char *device_name)
{
  const char *name;
  struct video_capability vid_cap;
  int brightness, hue, colour, contrast, whiteness;
  int width, height;
static int first = 1;

  if (first) {
    v4l_param.fd = -1;
    v4l_param.map_size = 0;
    v4l_param.map_data = NULL;
    v4l_param.offsets = NULL;
    first = 0;
  }

  if (v4l_param.fd != -1) {
    return V4L_OK;
  }

  if (device_name == NULL) {
    name = default_device_name;
  } else {
    name = device_name;
  }

  if ((v4l_param.fd = open(name, O_RDWR)) < 0) {
    fprintf (stderr, "v4l_open: %s open error", name);
    perror (" ");
    v4l_param.fd = -1;
    return V4L_FAIL;
  }

  if (ioctl (v4l_param.fd, VIDIOCGCAP, &vid_cap) < 0) {
    perror ("v4l_open: VIDIOCGCAP error");
    v4l_close ();
    return V4L_FAIL;
  }

  v4l_param.device_name = name;
  v4l_param.type = vid_cap.type;
  v4l_param.channels = vid_cap.channels;
  v4l_param.audios = vid_cap.audios;
  v4l_param.maxwidth = vid_cap.maxwidth;
  v4l_param.maxheight = vid_cap.maxheight;
  v4l_param.minwidth = vid_cap.minwidth;
  v4l_param.minheight = vid_cap.minheight;

  v4l_set_source_channel(v4l_param.source_channel);

  brightness = hue = colour = contrast = whiteness = -1;
  v4l_color(brightness, hue, colour, contrast, whiteness);

  width = height = 0;
  v4l_set_window_size(width, height);

  v4l_set_audio(v4l_param.audio, 1);

  return V4L_OK;
}

int
v4l_close (void)
{
  if (v4l_param.map_data) {
    v4l_grab_pic_stop();
    v4l_param.map_data = NULL;
  }
  if (v4l_param.offsets) {
    v4l_param.offsets = NULL;
  }

  if (v4l_param.fd == -1)
    return V4L_OK;

  v4l_set_audio (v4l_param.audio, 0);

  if (v4l_param.fd != -1 && (close(v4l_param.fd)) < 0) {
    perror ("v4l_close: Device close error");
    v4l_param.fd = -1;
    return V4L_FAIL;
  }
  v4l_param.fd = -1;
  v4l_param.device_name = NULL;

  return V4L_OK;
}

int
v4l_init (const char *device_name, int source_channel, int tuner, int norm, int audio)
{
static int first = 1;

  if (first) {
    memset(&v4l_param, 0, sizeof(v4l_param));
    v4l_param.fd = -1;
    v4l_param.source_channel = -1;
    v4l_param.tuner = -1;
    v4l_param.audio = -1;
    v4l_param.norm = -1;
    v4l_param.width = 0;
    v4l_param.height = 0;
    v4l_param.offsets = NULL;
    v4l_param.map_data = NULL;
    first = 0;
  }

  if (v4l_param.fd != -1)
    return V4L_OK;

  v4l_param.source_channel = source_channel;
  v4l_param.tuner = tuner;
  v4l_param.norm = norm;
  v4l_param.audio = audio;
  if (v4l_open(device_name) == V4L_FAIL) {
    return V4L_FAIL;
  }

  return V4L_OK;
}

int
v4l_quit (void)
{
  return v4l_close ();
}

const char*
norm_to_str(int norm)
{
  const char *n;

  switch (norm) {
    case VIDEO_MODE_PAL:
      n = "PAL";
      break;
    case VIDEO_MODE_NTSC:
      n = "NTSC";
      break;
    case VIDEO_MODE_SECAM:
      n = "SECAM";
      break;
    case VIDEO_MODE_AUTO:
      n = "AUTO";
      break;
    default:
      n = "UNKNOWN";
      break;
  }
  return n;
}

void
v4l_print_param(void)
{
  fprintf(stdout, "VIDEO DEVICE:            %s\n", v4l_param.device_name);
  fprintf(stdout, "video device has %d source, %d audios.\n", v4l_param.channels, v4l_param.audios);
  fprintf(stdout, "VIDEO DEVICE SOURCE:     %d [%s]\n", v4l_param.source_channel, v4l_param.channel_name);
  if (v4l_param.channel_flags & VIDEO_VC_TUNER) {
    fprintf(stdout, "source %d has %d tuners.\n", v4l_param.source_channel, v4l_param.channel_tuners);
    fprintf(stdout, "VIDEO DEVICE TUNER:      %d [%s]\n", v4l_param.tuner, v4l_param.tuner_name);
  } else {
    fprintf(stdout, "source %d has no tuners.\n", v4l_param.source_channel);
  }
  if (v4l_param.channel_flags & VIDEO_VC_AUDIO) {
    fprintf(stdout, "source %d has audio.\n", v4l_param.source_channel);
    fprintf(stdout, "VIDEO DEVICE AUDIO:      %d [%s]\n", v4l_param.audio, v4l_param.audio_name);
  } else {
    fprintf(stdout, "source %d has no audio.\n", v4l_param.source_channel);
  }
  fprintf(stdout, "VIDEO DEVICE NORM:       %s\n", norm_to_str(v4l_param.norm));
  fprintf(stdout, "VIDEO DEVICE BRIGHTNESS: %d%%\n", (int)v4l_get_brightness());
  fprintf(stdout, "VIDEO DEVICE HUE:        %d%%\n", (int)v4l_get_hue());
  fprintf(stdout, "VIDEO DEVICE COLOR:      %d%%\n", (int)v4l_get_colour());
  fprintf(stdout, "VIDEO DEVICE CONTRAST:   %d%%\n", (int)v4l_get_contrast());
  if (v4l_param.channel_flags & VIDEO_VC_TUNER) {
    fprintf(stdout, "VIDEO DEVICE FREQ:       %d\n",v4l_param.freq * 1000 / 16);
  }

  fflush(stdout);
}

