/* parseopt.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <ctype.h>
#include <pwd.h>

#include "util.h"
#include "parseopt.h"

static PARSEOPT_DEFAULT_FUNC default_func = NULL;

static OptionDefs *options;

static void **alloc_mems = NULL;
static int alloc_mems_num = 0;
static int alloc_mems_max = 0;

#define ALLOC_MEMS_SIZE 64
static char *
str_regist(const char *arg, int str_max)
{
  char *str;
  alloc_mems_num++;
  if (alloc_mems == NULL || alloc_mems_num >= alloc_mems_max ||
      alloc_mems_num < (alloc_mems_max - ALLOC_MEMS_SIZE)) {
    if ((alloc_mems_num > 0) &&
	(alloc_mems_num < (alloc_mems_max - ALLOC_MEMS_SIZE))) {
      alloc_mems_max -= ALLOC_MEMS_SIZE;
    } else {
      alloc_mems_max += ALLOC_MEMS_SIZE;
    }
    alloc_mems = (void**)realloc(alloc_mems, alloc_mems_max * sizeof(void*));
    if (!alloc_mems)
      mem_alloc_fail("parseopt str_regist", IMM_EXIT);
  }
  alloc_mems_num--;
  if (str_max <= 0)
    str_max = MAX_STRING_LENGTH;
  str = (char*)strndup(arg, str_max);
  alloc_mems[alloc_mems_num] = str;
  alloc_mems_num++;
  return str;
}

static int
str_free(const char *str)
{
  int i;
  void **p = alloc_mems;
  if (!alloc_mems || alloc_mems_num <= 0)
    return 0;
  for (i = 0; i < alloc_mems_num; i++) {
    if (p[i] == str) {
      free(p[i]);
      alloc_mems_num--;
      p[i] = p[alloc_mems_num];
      p[alloc_mems_num] = NULL;
      return OK;
    }
  }
  return FAIL;
}

void
parseopt_set_option_defs (OptionDefs *ops)
{
  options = ops;
}

void
parseopt_set_default_func (PARSEOPT_DEFAULT_FUNC func)
{
  default_func = func;
}

OptionDef*
parseopt_get_optiondef(const char *name)
{
  OptionDefs *ods;
  OptionDef *od;
  int i;

  if (name[0] == '-')
    name++;

  ods = options;
  while (ods->opts != NULL) {
    od = ods->opts;
    for (i = 0; i < *ods->opts_num; i++, od++) {
      if (od->name && (strcmp(name, od->name) == 0)) {
        return od;
      }
    }
    ods++;
  }
  return NULL;
}

OptionDef*
parseopt_get_optiondef_alt(const char *alt_name)
{
  OptionDefs *ods;
  OptionDef *od;
  int i;

  ods = options;
  while (ods->opts != NULL) {
    od = ods->opts;
    for (i = 0; i < *ods->opts_num; i++, od++) {
      if (od->alt_name && (strcasecmp(alt_name, od->alt_name) == 0)) {
        return od;
      }
    }
    ods++;
  }
  return NULL;
}

int
parseopt_get_default(const char *name, int *flags_ret)
{
  OptionDef *po;
  int ret;

  po = parseopt_get_optiondef(name);
  if (po) {
    if (flags_ret)
      *flags_ret = po->flags;
    ret = po->def.val;
  } else {
    if (flags_ret)
      *flags_ret = FAIL;
    ret = 0;
  }
  return ret;
}

int
parseopt_get_default_alt(const char *alt_name, int *flags_ret)
{
  OptionDef *po;
  int ret;

  po = parseopt_get_optiondef_alt(alt_name);
  if (po) {
    *flags_ret = po->flags;
    ret = po->def.val;
  } else {
    *flags_ret = FAIL;
    ret = 0;
  }
  return ret;
}

int
parseopt_set_default (const char *name, int *flags_ret)
{
  OptionDefs *opo;
  OptionDef *po;
  int i;
  int all_opts = 0;
  int count = 0;

  if (strcmp(name, "all_options") == 0)
    all_opts = 1;

  opo = options;
  while (opo->opts != NULL) {
    po = opo->opts;
    for (i = 0; i < *opo->opts_num; i++, po++) {
      if ( all_opts ||
	  (po->name && strcmp(name, po->name) == 0) ||
          (po->alt_name && strcasecmp(name, po->alt_name) == 0)) {
        if ((po->flags & OPT_INT) || (po->flags & OPT_BOOL) ||
            (po->flags & OPT_SEL)) {
          *po->v.int_arg = po->def.val;
        } else if (po->flags & OPT_STR) {
          *po->v.str_arg = po->def.str;
        } else if ((po->flags & HAS_ARG) && !(po->flags & OPT_FUNC)) {
          *po->v.int_arg = po->def.val;
        } else if (po->flags & OPT_FUNC) {
          if ((po->flags & HAS_ARG) && (po->def.str != NULL))
	    po->v.func_arg(po->def.str);
	  else if (po->def.val)
	    po->v.func_arg();
	}
        if (po->flags & OPT_KILO)
          *po->v.int_arg = *po->v.int_arg * 1000;

        if (!all_opts) {
  	  if (flags_ret)
	    *flags_ret = po->flags;
	  return po->def.val;
        }
	count++;
      }
    }
    opo++;
  }
  if (flags_ret)
    *flags_ret = 0;
  return count;
}

void
parseopt_quit(void)
{
  int i;
  for (i = 0; i < alloc_mems_num; i++) {
    free(alloc_mems[i]);
  }
  alloc_mems_num = 0;
  free(alloc_mems);
  alloc_mems = NULL;
}

int
parseopt_set_value(OptionDef *po, const char *arg)
{
  SEL_COMPO *sel;
  int i, error;

  if (po->flags & HAS_ARG) {
    if (arg == NULL || arg[0] == '\0')
      return FAIL;
  }

  if (po->flags & OPT_STR) {
    str_free(*po->v.str_arg);
    *po->v.str_arg = str_regist(arg, po->max_val);
  } else if (po->flags & OPT_BOOL) {
    *po->v.int_arg = !po->def.val;
  } else if (po->flags & OPT_FUNC) {
    if (po->flags & HAS_ARG)
      error = po->v.func_arg(arg);
    else
      error = po->v.func_arg();
    if (error != OK) {
      fprintf(stderr, "command line error: -%s %s\n", po->name, arg);
      return FAIL;
    }
  } else if (po->flags & OPT_SEL) {
    sel = po->min_sel.sel;
    for (i=0; i < po->max_val; i++, sel++) {
      if (strcasecmp (sel->name, arg) == 0) {
        *po->v.int_arg = sel->id;
        break;
      }
    }
  } else if (po->flags == 0) {
    po->v.func_arg ();
  } else if (po->flags & HAS_ARG) {
    int tval;
    tval = atoi(arg);
    if (tval < po->min_sel.min_val) {
      fprintf(stderr, "parseopt: -%s augumenat %d too small, minimal value is %d\n", po->name, tval, po->min_sel.min_val);
      return FAIL;
      //po->def.val = po->min_sel.min_val;
    } else if (tval > po->max_val) {
      fprintf(stderr, "parseopt: -%s augumenat %d too big, maximal value is %d\n", po->name, tval, po->max_val);
      return FAIL;
      //po->def.val = po->max_val;
    } else {
      if (po->flags & OPT_KILO)
        tval *= 1000;
      *po->v.int_arg = tval;
    }
  }
  return OK;
}

int
parseopt_set_command(const char *com, const char *arg)
{
  OptionDef *po = NULL;

  if (com[0] == '-')
    po = parseopt_get_optiondef(com+1);
  else
    po = parseopt_get_optiondef(com);
  if (!po)
    po = parseopt_get_optiondef_alt(com);
  if (!po) {
    fprintf(stderr, "parseopt: %s not found.\n", com);
    return FAIL;
  }

  return parseopt_set_value(po, arg);
}
 
int
parseopt_cmd(int argc, char **argv)
{
  int optindex;
  OptionDef *po = NULL;
  int ret_flag, error;
  const char *opt, *arg;

  parseopt_set_default("all_options", &ret_flag);

  optindex = 1;
  while (optindex < argc) {
    opt = argv[optindex++];
    if (opt[0] == '-' && opt[1] != '\0') {
      po = parseopt_get_optiondef(opt+1);
      if (!po) {
        fprintf(stderr, "cannot find option: %s\n", opt);
	exit(1);
      }
      if (po->flags & HAS_ARG)
	arg = argv[optindex++];
      else
	arg = NULL;
      if (!(po->flags & OPT_BEFORE))
	continue;
      if (parseopt_set_value(po, arg) != OK) {
        fprintf(stderr, "command line error: %s %s\n", opt, arg);
	exit(1);
      }
    }
  }

  optindex = 1;
  while (optindex < argc) {
    opt = argv[optindex++];
    if (opt[0] == '-' && opt[1] != '\0') {
      po = parseopt_get_optiondef(opt+1);
      if (!po) {
        fprintf(stderr, "cannot find option: %s\n", opt);
	exit(1);
      }
      if (po->flags & HAS_ARG)
	arg = argv[optindex++];
      else
	arg = NULL;
      if (po->flags & OPT_BEFORE)
	continue;
      if (parseopt_set_value(po, arg) != OK) {
        fprintf(stderr, "command line error: %s %s\n", opt, arg);
	exit(1);
      }
    } else {
      if (default_func == NULL) {
        fprintf(stderr,"parseopt_cmd: cannot parse default argument, %s.\n", opt);
      } else {
        error = default_func(opt);
        if (error) {
          fprintf(stderr, "command line error: %s\n", opt);
          exit (1);
        }
      }
    }
  }
  return OK;
}

#define skip_space(p) while(*p==' '||*p=='\t')p++
#define skip_space_rev(p) while(*p==' '||*p=='\t')p--

int 
parse_line(char *line, char *subj, char *pred)
{
  char *lp, *sp, *ep;
  int len;

  lp = line;
  skip_space(lp);
  if (*lp == '#' || *lp == '\n' || *lp == '\0') {
    subj[0] = '#'; subj[1] = '\0'; pred[0] = '\0';
    return OK;
  }
  if (*lp == '[') {
    lp++;
    skip_space(lp);
    sp = lp;
    for (; *lp != ']' && *lp != '\n' && *lp != '\0'; lp++) ;
    if (*lp != ']') {
      subj[0] = '\0'; pred[0] = '\0';
      return FAIL;
    }
    ep = lp;
    lp++;
    skip_space(lp);
    if (*lp != '#' && *lp != '\n' && *lp != '\0') {
      subj[0] = '\0'; pred[0] = '\0';
      return FAIL;
    }
    ep--;
    skip_space_rev(ep);
    ep++;
    *ep = '\0';
    len = strlen(sp);
    if (len <= 0) {
      subj[0] = '\0'; pred[0] = '\0';
      return FAIL;
    }
    subj[0] = '[';
    subj[1] = '\0';
    strcpy(pred, sp);
    return OK;
  } else {
    int c;
    sp = lp;
    for (; *lp != ' ' && *lp != '\t' && *lp != ':' && *lp != '=' &&
       	   *lp != '#' && *lp != '\n' && *lp != '\0'; lp++) ;
    c = *lp;
    *lp = '\0';
    len = strlen(sp);
    if (len <= 0) {
      subj[0] = '\0'; pred[0] = '\0';
      return FAIL;
    }
    strcpy(subj, sp);
    if (c == '#' || c == '\n' || c == '\0') {
      pred[0] = '\0';
      return OK;
      //return FAIL;
    }
    lp++;
    for (; *lp == ' ' || *lp == '\t' || *lp == '=' || *lp == ':'; lp++);
    if (*lp == '\'' || *lp == '"') {
      c = *lp;
      sp = ++lp;
      for (; *lp != c && *lp != '\n' && *lp != '\0'; lp++) ;
      if (*lp != c) {
        subj[0] = '\0'; pred[0] = '\0';
        return FAIL;
      }
      *lp = '\0';
      c = *(++lp);
    } else {
      sp = lp;
      for (; *lp != ' ' && *lp != '\t' && *lp != '\'' && *lp != '"' &&
	     *lp != '#' && *lp != '\n' && *lp != '\0'; lp++) ;
      if (*lp == '\'' || *lp == '"') {
        subj[0] = '\0'; pred[0] = '\0';
	return FAIL;
      }
      c = *lp;
      *lp = '\0';
    }
    if (c != '#' && c != '\n' && c != '\0') {
      lp++;
      skip_space(lp);
      if (*lp != '#' && *lp != '\n' && *lp != '\0') {
        subj[0] = '\0'; pred[0] = '\0';
	return FAIL;
      }
    }
    len = strlen(sp);
    if (len == 0) {
//      subj[0] = '\0'; pred[0] = '\0';
//      return FAIL;
      pred[0] = '\0';
      return OK;
    }
    strcpy(pred, sp);
  }
  return OK;
}

int
parseopt_change_default(OptionDef *po, const char *arg)
{
  int i;
  SEL_COMPO *sel;

  if (po->flags == 0) {
    return OK;
  } else if (po->flags & OPT_STR) {
    if (arg[0] == '\0') {
      fprintf(stderr, "parseopt: error %s or %s, argument %s.\n",po->name,po->alt_name,arg);
      return FAIL;
    }
    str_free(po->def.str);
    po->def.str = str_regist(arg, po->max_val);
  } else if (po->flags & OPT_BOOL) {
    if (arg[0] == '\0') {
      if (!po->def.val)
        po->def.val = 1;
      else
        po->def.val = 0;
    } else {
      if (strcasecmp ("ON", arg) == 0)
        po->def.val = 1;
      else
        po->def.val = 0;
    }
  } else if (po->flags & OPT_FUNC) {
    if (po->flags & HAS_ARG) {
      if (po->v.func_arg(arg) != OK) {
        fprintf(stderr, "parseopt: error %s or %s, argument %s.\n",po->name,po->alt_name,arg);
	return FAIL;
      }
      str_free(po->def.str);
      po->def.str = str_regist(arg, po->max_val);
    } else {
      po->def.val = 1;
    }
  } else if (po->flags & OPT_SEL) {
    sel = po->min_sel.sel;
    for (i=0; i < po->max_val; i++, sel++) {
      if (strcasecmp (sel->name, arg) == 0) {
        po->def.val = sel->id;
        break;
      }
    }
    if (i == po->max_val) {
      fprintf(stderr, "parseopt: error %s or %s.\n",po->name,po->alt_name);
      return FAIL;
    }
  } else {
    int tval;
    tval = atoi(arg);
    if (tval < po->min_sel.min_val) {
      fprintf(stderr, "parseopt: %s augumenat %d too small, minimal value is %d\n", po->name, tval, po->min_sel.min_val);
      return FAIL;
      //po->def.val = po->min_sel.min_val;
    } else if (tval > po->max_val) {
      fprintf(stderr, "parseopt: %s augumenat %d too big, maximal value is %d\n", po->name, tval, po->max_val);
      return FAIL;
      //po->def.val = po->max_val;
    } else {
      po->def.val = tval;
    }
  }
  return OK;
}

int
parseopt_defaults(const char* fname)
{
  FILE *fp;
  char line_buf[4096];
  char com_buf[4096];
  char arg_buf[4096];
  int line_num = 0;
  OptionDef *po;

  if ((fp = xfopen (fname, "r")) == NULL) {
    fprintf(stderr, "parseopt_defaults: cannot open %s.\n", fname);
    return FAIL;
  }

  while (fgets(line_buf, sizeof(line_buf), fp) != NULL) {
    line_num++;
    if (parse_line(line_buf, com_buf, arg_buf) != OK) {
      fprintf(stderr,"parseopt_defaults: error: %s, line %d.\n",fname,line_num);
      return FAIL;
    }

    if (com_buf[0] == '#' || com_buf[0] == ']')
      continue;

    if (com_buf[0] == '-')
      po = parseopt_get_optiondef(com_buf+1);
    else
      po = parseopt_get_optiondef(com_buf);
    if (!po)
      po = parseopt_get_optiondef_alt(com_buf);
    if (!po) {
      fprintf(stderr, "parseopt: %s not found.\n", com_buf);
      return FAIL;
    }

    if (parseopt_change_default(po, arg_buf) != OK) {
      fprintf(stderr,"parseopt_defaults: error: %s, line %d.\n",fname,line_num);
      return FAIL;
    }
  }

  fclose (fp);

  return OK;
}

#if 0
int parseopt_cmd(int argc, char **argv)
{
    int optindex;
    char *opt = NULL, *arg;
    OptionDef *po = NULL;
    OptionDefs *opo;
    SEL_COMPO *sel;
    int i;
    int error = 0;

    parseopt_set_default("all_options", &i);

    optindex = 1;
    while (optindex < argc) {
        opt = argv[optindex++];
        opo = options;
        
        if (opt[0] == '-' && opt[1] != '\0') {
	  while (opo->opts != NULL) {
            po = opo->opts;
	    for (i = 0; i < *opo->opts_num; i++) {
              if ((po->name != NULL) && !strcasecmp(opt + 1, po->name))
                break;
	      po++;
            }
            if (i < *opo->opts_num) {
	      break;
            }
	    opo++;
	  }
          if (opo->opts == NULL) {
            fprintf(stderr, "%s: unrecognized option '%s'\n", argv[0], opt);
            exit(1);
          }
          arg = NULL;
          if (po->flags & HAS_ARG) {
            arg = argv[optindex++];
	    if (arg == NULL) {
	      fprintf (stderr, "option %s required argument.\n", opt);
	      exit (1);
	    }
	  }
          if (po->flags & OPT_STR) {
	    str_free(*po->v.str_arg);
            *po->v.str_arg = str_regist(arg, po->max_val);
          } else if (po->flags & OPT_BOOL) {
            *po->v.int_arg = !po->def.val;
          } else if (po->flags & OPT_FUNC) {
	    if (po->flags & HAS_ARG)
              error = po->v.func_arg(arg);
	    else
              error = po->v.func_arg();
	    if (error) {
	      fprintf(stderr, "command line error: -%s\n", po->name);
	      exit (1);
	    }
	  } else if (po->flags & OPT_SEL) {
	    sel = po->min_sel.sel;
	    for (i=0; i < po->max_val; i++, sel++) {
	      if (strcasecmp (sel->name, arg) == 0) {
	        *po->v.int_arg = sel->id;
	        break;
	      }
	    }
	  } else if (po->flags == 0) {
	    po->v.func_arg ();
          } else if (po->flags & HAS_ARG) {
	    int tval;
            tval = atoi(arg);
	    if (tval < po->min_sel.min_val) {
	      fprintf(stderr, "parseopt_cmd: -%s augumenat %d under flow, minimal value is %d\n", po->name, tval, po->min_sel.min_val);
	      exit(1);
              //po->def.val = po->min_sel.min_val;
	    } else if (tval > po->max_val) {
	      fprintf(stderr, "parseopt_cmd: -%s augumenat %d over flow, maximal value is %d\n", po->name, tval, po->max_val);
	      exit(1);
              //po->def.val = po->max_val;
	    } else
            if (po->flags & OPT_KILO)
	      tval *= 1000;
            *po->v.int_arg = tval;
          }
        } else {
	  if (default_func == NULL) {
	    fprintf(stderr, "parseopt_cmd: cannot parse default argument, %s.\n", opt);
	  }
	  else {
	    error = default_func(opt);
	    if (error) {
	      fprintf(stderr, "command line error: %s\n", opt);
	      exit (1);
	    }
	  }
        }
    }

    return 0;
}

static char*
expand_file_name(const char* path)
{
    char buf[1024];
    char* bufp = buf;
    const char* p = path;
    char *home;
    struct passwd *pw;
    char user[64];
    char* userp;
    const char* q;
    char *result;

    if (p[0] == '~') {
        if (p[1] == '/') {
            home = getenv("HOME");
            if (home && getuid() != 0) {
                strcpy(bufp, home);
                bufp += strlen(home);
                p++;
            }
            else {
                uid_t uid = getuid();
                pw = (struct passwd *)getpwuid(uid);
                if (pw) {
                    strcpy(bufp, pw->pw_dir);
                    bufp += strlen(pw->pw_dir);
                    p++;
                }
                else {
                    *bufp++ = *p++;
                }
            }
        }
        else {
            userp = user;
            for (q = p + 1; *q && (*q != '/'); q++)
                *userp++ = *q;
            *userp = '\0';
            pw = (struct passwd *)getpwnam(user);
            if (pw) {
                strcpy(bufp, pw->pw_dir);
                bufp += strlen(pw->pw_dir);
                p = q;
            }
            else {
                *bufp++ = *p++;
            }
        }
    }

    while (*p)
        *bufp++ = *p++;
    *bufp = '\0';
    result = (char*) malloc (strlen(buf) + 1);
    strcpy(result, buf);
    return result;
}

int
parseopt_rc (const char* fname)
{
  FILE *fp;
  char *val, *p;
  char *line_buf;
  char buf[4096];
  char *expand_file = NULL;
  int i;
  int error = 0;

  char *arg = NULL;
  OptionDef *po = NULL;
  OptionDefs *opo;
  SEL_COMPO *sel;

//  printf ("parseopt_rc: %s\n", fname);
  if ((fp = fopen (fname, "r")) == NULL) {
    strcpy (buf, "~/");
    strcpy (buf+2, fname);
    expand_file = expand_file_name (buf);
    fp = fopen (expand_file, "r");
    if (fp == NULL) {
      if (expand_file)
        free (expand_file);
      return 0;
    }
  }

  while (fgets(buf, sizeof(buf), fp) != NULL) {
    line_buf = buf;
    while (*line_buf == ' ' || *line_buf == '\t')
      line_buf++;
    if ((line_buf[0] == '#') || (line_buf[0] == '\n'))
      continue;
    val = NULL;
    for (p = line_buf; *p; p++) {
      if (*p == ' ' || *p == '\t') {
	*p++ = '\0';
	while (*p == ' ' || *p == '\t')
	  p++;
	if (*p == '"') {
	  val = ++p;
	  p = strchr (val, '"');
	  if (p)
	    *p = '\0';
	  else {
            fprintf(stderr, "%s: unrecognized parameter '%s'\n",fname,line_buf);
	    exit (1);
	  }
	} else if (*p == '\'') {
	  val = ++p;
	  p = strchr (val, '\'');
	  if (p)
	    *p = '\0';
	  else {
            fprintf(stderr, "%s: unrecognized parameter '%s'\n",fname,line_buf);
	    exit (1);
	  }
	} else if (*p != '\0' && *p != '\n' && *p != '\r') {
	  val = p;
	  p = strchr (val, '\n');
	  if (p)
	    *p = '\0';
	  p = strchr (val, '\r');
	  if (p)
	    *p = '\0';
	  p = strchr (val, '#');
	  if (p)
	    *p = '\0';
	  p = strchr (val, '\t');
	  if (p)
	    *p = '\0';
	  p = strchr (val, ' ');
	  if (p)
	    *p = '\0';
	  break;
	}
      }
    }
    if (!val)
      continue;

    opo = options;
    while (opo->opts != NULL) {
      po = opo->opts;
      for (i = 0; i < *opo->opts_num; i++) {
        if ((po->alt_name != NULL) && !strcasecmp(line_buf, po->alt_name))
          break;
        po++;
      }
      if (i < *opo->opts_num) {
        break;
      }
      opo++;
    }
    if (opo->opts == NULL) {
      fprintf(stderr, "%s: unrecognized option '%s'\n", expand_file, line_buf);
      exit(1);
    }
    arg = val;
    if (po->flags == 0) {
      continue;
    } else if (po->flags & OPT_STR) {
      str_free(po->def.str);
      po->def.str = str_regist(arg, po->max_val);
    } else if (po->flags & OPT_BOOL) {
      if (strcasecmp ("ON", arg) == 0)
        po->def.val = 1;
      else
        po->def.val = 0;
    } else if (po->flags & OPT_FUNC) {
      if (po->flags & HAS_ARG) {
        error = po->v.func_arg(arg);
        str_free(po->def.str);
        po->def.str = str_regist(arg, po->max_val);
      } else {
        error = po->v.func_arg();
	po->def.val = 1;
      }
      if (error) {
        fprintf(stderr, "rc_parseopt: %s error\n", po->alt_name);
        exit (1);
      }
    } else if (po->flags & OPT_SEL) {
      sel = po->min_sel.sel;
      for (i=0; i < po->max_val; i++, sel++) {
        if (strcasecmp (sel->name, arg) == 0) {
          po->def.val = sel->id;
          break;
        }
      }
    } else {
      int tval;
      tval = atoi(arg);
      if (tval < po->min_sel.min_val) {
        fprintf(stderr, "parseopt_cmd: %s augumenat %d too small, minimal value is %d\n", po->name, tval, po->min_sel.min_val);
	exit(1);
        //po->def.val = po->min_sel.min_val;
      } else if (tval > po->max_val) {
        fprintf(stderr, "parseopt_cmd: %s augumenat %d too big, maximal value is %d\n", po->name, tval, po->max_val);
	exit(1);
        //po->def.val = po->max_val;
      } else
        po->def.val = tval;
    }
  }
  if (expand_file)
    free (expand_file);

  fclose (fp);

  return 1;
}
#endif

void parseopt_print_cmd_opt(void)
{
    OptionDef *po;
    OptionDefs *opo;
    int i=0, j, k;
    SEL_COMPO *sel = NULL;

    if (options == NULL) {
      fprintf(stderr, "parseopt_print_cmd_opt: ERROR OptionDefs == NULL\n");
      return;
    }

    printf("\nOptions are:\n");

    opo = options;
    for (j = 0; opo[j].opts != NULL; j++) {
      po = opo[j].opts;
      if (opo[j].name != NULL)
	printf ("\n%s\n", opo[j].name);
      for(i = 0; i < *opo[j].opts_num; i++, po++) {
        char buf[64];
	char argbuf[128];
	if (po->name == NULL)
	  continue;
	if (po->flags & HAS_ARG) {
	  if (po->flags & OPT_STR) {
	    if (po->def.str)
	      sprintf(argbuf, "%s", po->def.str);
	    else
	      sprintf(argbuf, "%s", po->argname);
	  } else if (po->flags & OPT_SEL) {
	    int k = po->max_val;
	    for (k = 0; k < po->max_val; k++) {
	      if (po->min_sel.sel[k].id == po->def.val) {
	        sprintf(argbuf, "%s", po->min_sel.sel[k].name);
	        break;
	      }
	    }
	    if (k == po->max_val)
	      sprintf(argbuf, "%s", po->argname);
	  } else if (po->flags & OPT_FUNC) {
	    if ((po->flags & HAS_ARG) && po->def.str)
	      sprintf(argbuf, "%s", po->def.str);
	    else
	      sprintf(argbuf, "%s", po->argname);
	  } else if (po->flags & HAS_ARG) {
	    if (po->min_sel.min_val <= po->def.val && po->max_val>=po->def.val)
	      sprintf(argbuf, "%d", po->def.val);
	    else
	      sprintf(argbuf, "%s", po->argname);
	  }
	} else {
	  sprintf(argbuf, "  ");
	}
	sprintf(buf, "-%-17s %s", po->name, argbuf);

        printf("%-31s %s", buf, po->help);

	if (po->flags & HAS_ARG || po->flags & OPT_BOOL) {
	  printf("  (default: ");
	  if (po->flags & OPT_STR) {
	    if (po->def.str)
	      printf("%s)", po->def.str);
	    else
	      printf("NOT SET)");
//	    printf("%-26s   argument string\n","\n");
	  } else if (po->flags & OPT_BOOL) {
	    printf("%s)", (po->def.val) ? "on" : "off");
//	    printf("\n");
	  } else if (po->flags & OPT_SEL) {
	    int k = po->max_val;
	    for (k = 0; k < po->max_val; k++) {
	      if (po->min_sel.sel[k].id == po->def.val) {
	        printf("%s)", po->min_sel.sel[k].name);
	        break;
	      }
	    }
	    if (k == po->max_val)
	       printf("NOT SET)");
//	    printf("%-26s   argument selection\n","\n");
	  } else if (po->flags & OPT_FUNC) {
	    if ((po->flags & HAS_ARG) && po->def.str)
	      printf("%s)", po->def.str);
	    else
	      printf("NOT SET)");
//	    printf("%-26s   argument string\n","\n");
	  } else if (po->flags & HAS_ARG) {
	    if (po->min_sel.min_val <= po->def.val && po->max_val>=po->def.val)
	      printf("%d)", po->def.val);
	    else
	      printf("NOT SET)");
//	    printf("%-26s   argument integer\n","\n");
	  }
	}

	if (po->flags & OPT_SEL) {
	  sel = po->min_sel.sel;
	  k = 0;
	  printf ("%-31s  [%s", "\n",sel[k].name);
	  for (k++; k < po->max_val; k++) {
	    printf (",%s", sel[k].name);
	  }
	  printf ("]");
	}
	printf ("\n");
      }
    }

    printf ("\n");
}

void parseopt_print_rc_opt(void)
{
    OptionDef *po;
    OptionDefs *opo;
    int i=0, j, k;
    SEL_COMPO *sel = NULL;
    char *on = "ON";
    char *off = "OFF";

    if (options == NULL) {
      fprintf(stderr, "parseopt_print_cmd_opt: ERROR OptionDefs == NULL\n");
      return;
    }

    printf("\n# rcfile options:\n\n");

    opo = options;
    for (j = 0; opo[j].opts != NULL; j++) {
      po = opo[j].opts;
      if (opo[j].name != NULL)
	printf ("\n# %s\n\n", opo[j].name);
      //printf ("%d:opts_num %d\n", j, *opo[j].opts_num);
      for(i = 0; i < *opo[j].opts_num; i++, po++) {
        char buf[4096];
	if (po->alt_name == NULL)
	  continue;

	printf("# %s", po->help);

	if (po->flags & OPT_SEL) {
	  sel = po->min_sel.sel;
	  k = 0;
	  printf (" [%s", sel[k].name);
	  for (k++; k < po->max_val; k++) {
	    printf (",%s", sel[k].name);
	  }
	  printf ("]");
	}
	printf("\n");
	if (po->name != NULL) {
	  printf ("# command line: -%s", po->name);
	}
	//if (po->argname != NULL) {
	//  printf (" %s", po->name, po->argname);
	//}
	printf("\n");

	if (po->flags & OPT_FUNC)
	  sprintf(buf, "#%-46s %s", po->alt_name, "NOT_SET");
	else if (po->flags & OPT_BOOL)
	  sprintf(buf, "%-46s %s", po->alt_name, (po->def.val) ? on : off);
	else if (po->flags & OPT_STR)
	  sprintf(buf, "%-46s %s", po->alt_name, po->def.str);
	else if (po->flags & OPT_INT)
	  sprintf(buf, "%-46s %d", po->alt_name, po->def.val);
	else if (po->flags & OPT_SEL) {
	  for (k = 0; k < po->max_val; k++) {
	    if (po->def.val == po->min_sel.sel[k].id)
	      break;
	  }
	  if (k != 100)
	    sprintf(buf, "%-46s %s", po->alt_name, po->min_sel.sel[k].name);
	  else
	    sprintf(buf, "#%-46s %s", po->alt_name, "NOT_SET");
	}
	else if (po->flags & HAS_ARG)
	  sprintf(buf, "%-46s %d", po->alt_name, po->def.val);

	printf ("%s\n\n", buf);
      }
    }

    printf ("\n");
}

