/* ntvrec.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/select.h>
#include <termios.h>
#include <libgen.h>
#include <time.h>

#include "ntvrec.h"
#include "util.h"
#include "parseopt.h"
#include "term_util.h"
#include "mgr.h"
#include "ffmt.h"

static void show_help(void);
static int opt_start_time(const char *arg);
static int opt_end_time(const char *arg);
static int opt_recording_time(const char *arg);

OptionDef main_param[] = {
    { "h", NULL, 0, {(void*)show_help}, {0}, {0}, 0, "show help" , NULL },
    { "t", NULL, HAS_ARG|OPT_FUNC, {(void*)opt_recording_time}, {0}, {0}, 0, "set recording time", "time"},
    { "start-time", "START_TIME", HAS_ARG|OPT_FUNC, {(void*)opt_start_time}, {0}, {0}, 0, "set recording start time", "time"},
    { "end-time", "END_TIME", HAS_ARG|OPT_FUNC, {(void*)opt_end_time}, {0}, {0}, 0, "set recording end time", "time"},
};

int main_param_num = sizeof(main_param) / sizeof(OptionDef);

extern OptionDef ffmt_param[];
extern int ffmt_param_num;
extern OptionDef mgr_param[];
extern int mgr_param_num;
extern OptionDef audio_param[];
extern int audio_param_num;
extern OptionDef vdev_param[];
extern int vdev_param_num;
#ifdef HAVE_LIBXVIDCORE
extern OptionDef xvid_param[];
extern int xvid_param_num;
#endif /* HAVE_LIBXVIDCORE */
#ifdef HAVE_LIBMP3LAME
extern OptionDef mp3_param[];
extern int mp3_param_num;
#endif /* HAVE_LIBMP3LAME */
#ifdef HAVE_LIBAVCODEC
extern OptionDef vlavc_param[];
extern int vlavc_param_num;
#endif /* HAVE_LIBAVCODEC */

OptionDefs options[] = {
  {main_param,   &main_param_num, "MAIN OPTIONS"},
  {mgr_param, &mgr_param_num, "VIDEO OPTIONS"},
  {audio_param, &audio_param_num, "AUDIO OPTIONS"},
  {vdev_param, &vdev_param_num, "VIDEO DEVICE OPTIONS"},
  {ffmt_param, &ffmt_param_num, "FILE FORMAT OPTIONS"},

#ifdef HAVE_LIBXVIDCORE
  {xvid_param, &xvid_param_num, "XVID OPTIONS"},
#endif /* HAVE_LIBXVIDCORE */
#ifdef HAVE_LIBMP3LAME
  {mp3_param, &mp3_param_num, "LAME OPTIONS"},
#endif /* HAVE_LIBMP3LAME */
#ifdef HAVE_LIBAVCODEC
  {vlavc_param, &vlavc_param_num, "LIBAVCODEC VIDEO OPTIONS"},
#endif /* HAVE_LIBAVCODEC */
  {NULL, NULL, NULL },
};


static const char *prog_name;
static unsigned int recording_time = 0;
static time_t recording_start_time = 0;
static time_t recording_end_time = 0;

int
opt_recording_time(const char *arg)
{
  char *endp = NULL;
  const char *sp = NULL;
  int val, ch;
  int hour, minute, second;
  int v[3];
  int v_num = 3;
  int v_pos = 0;
  int ch_seq = 0;
  const char *opt_name = "-recording_time";
  const char *err_mes = "argument is HH:MM:SS or HHhMMmSSs. [ HH = hour, MM = minute, SS = second ]";

  sp = arg;

  hour = minute = second = 0;

  for (sp = arg; *sp != '\0'; sp++) {
    if (*sp == 'h' || *sp == 'm' || *sp == 's') {
      ch_seq = 1;
      break;
    }
  }

  sp = arg;
  while (*sp != '\0') {
    val = strtol(sp, &endp, 10);
    if (sp == endp) {
      if (!ch_seq) {
        if (v_pos < v_num && *sp == ':') {
	  v[v_pos] = 0;
	  v_pos++;
          if (v_pos < v_num && *(sp+1) == '\0') {
	    v[v_pos] = 0;
	    v_pos++;
          }
        }
        if (v_pos == v_num)
	  break;
      }
      sp++;
      continue;
    }
    if (endp != NULL)
      ch = *endp;
    else {
      fprintf(stderr, "opt_recording_time: endp == NULL\n");
      exit(1);
    }

    if (ch_seq) {
      switch (ch) {
        case 'h':
	  hour = val;
	  break;
        case 'm':
	  minute = val;
	  break;
        case 's':
        case '\0':
	  second = val;
	  break;
	case ':':
	  fprintf(stderr, "invalid separator '%c': %s %s\n", ch, opt_name, arg);
	  fprintf(stderr, "%s\n", err_mes);
	  return -1;
	  break;
        default:
	  fprintf(stderr, "unknown charactor '%c': %s %s\n", ch, opt_name, arg);
	  fprintf(stderr, "%s\n", err_mes);
	  return -1;
	  break;
      }
    } else {
      if (*endp != ':' && *endp != '\0') {
	fprintf(stderr, "invalid separator '%c': %s %s\n", ch, opt_name, arg);
	fprintf(stderr, "%s\n", err_mes);
	return -1;
      }
      if (v_pos < v_num) {
	v[v_pos] = val;
	v_pos++;
      }
      if (v_pos < v_num && *endp == ':' && *(endp+1) == '\0') {
	v[v_pos] = 0;
	v_pos++;
      }
      if (v_pos == v_num)
	break;
    }
    if (!endp || *endp == '\0')
      break;
    else
      sp = endp + 1;
  }

  if (!ch_seq) {
    if (v_pos > 0)
      second = v[--v_pos];
    if (v_pos > 0)
      minute = v[--v_pos];
    if (v_pos > 0)
      hour   = v[--v_pos];
  }

  if (hour < 0 || minute < 0 || second < 0) {
    fprintf(stderr, "negative time: %s %s\n", opt_name, arg);
    fprintf(stderr, "%s\n", err_mes);
    return -1;
  }

#ifdef DEBUG
  printf("arg: %s\n", arg);
  printf("hour %d: minute %d: second %d\n", hour, minute,second);
#endif /* DEBUG */

  recording_time = hour * 60 * 60 + minute * 60 + second;

  return 0;
}

time_t
calc_calender_time(const char *arg)
{
  char *endp = NULL;
  const char *sp = NULL;
  int val, ch;
  int second, minute, hour, day, month, year;
  time_t cal_time, cur_time;
  struct tm *tm_ret, cur_tm, cal_tm;

  if (arg == NULL)
    return 0;

  year = month = day = hour = minute = second = -1;

  sp = arg;
  while (*sp != '\0') {
    val = strtol(sp, &endp, 10);
    if (sp == endp) {
      sp++;
      continue;
    }
    if (endp != NULL)
      ch = *endp;
    else {
      fprintf(stderr, "opt_start_time: endp == NULL\n");
      exit(1);
    }

    switch (ch) {
      case 'y':
	if (val < 100)
	  val += 2000;
        val -= 1900;
	year = val;
	break;
      case 'M':
	month = val - 1;
	break;
      case 'd':
	day = val;
	break;
      case 'h':
	hour = val;
	break;
      case 'm':
	minute = val;
	break;
      case 's':
      case '\0':
	second = val;
	break;
      default:
	fprintf(stderr, "get_start_time: unknown charactor %c.\n", ch);
	exit(1);
	break;
    }
    if (!endp || *endp == '\0') {
      sp = endp;
      break;
    } else
      sp = endp + 1;
  }

  cur_time = time(NULL);
  if (cur_time < 0) {
    perror("get_start_time");
    exit(1);
  }
  tm_ret = localtime(&cur_time);
  cur_tm = *tm_ret;
  cal_tm = *tm_ret;

  if (second == -1) {
    if (year != -1 || month != -1 || day != -1 || hour != -1 || minute != -1)
      second = 0;
  }
  if (minute == -1) {
    if (year != -1 || month != -1 || day != -1 || hour != -1)
      minute = 0;
  }
  if (hour == -1) {
    if (year != -1 || month != -1 || day != -1)
      hour = 0;
  }
  if (day == -1) {
    if (year != -1 || month != -1)
      day = 1;
  }
  if (month == -1) {
    if (year != -1)
      month = 0;
  }

  if (second != -1)
    cal_tm.tm_sec = second;
  if (minute != -1)
    cal_tm.tm_min = minute;
  if (hour   != -1)
    cal_tm.tm_hour = hour;
  if (day    != -1)
    cal_tm.tm_mday = day;
  if (month  != -1)
    cal_tm.tm_mon  = month;
  if (year   != -1)
    cal_tm.tm_year = year;

  if (cal_tm.tm_year < cur_tm.tm_year) {
    if (year != -1)
      goto error_exit;
    else
      cal_tm.tm_year = cur_tm.tm_year;
  }

  if (cal_tm.tm_year == cur_tm.tm_year) {
    if (cal_tm.tm_mon < cur_tm.tm_mon) {
      if (year != -1)
	goto error_exit;
      else
        cal_tm.tm_year = cur_tm.tm_year + 1;
    } else if (cal_tm.tm_mon == cur_tm.tm_mon) {
      if (cal_tm.tm_mday < cur_tm.tm_mday) {
        if (year != -1 || month != -1)
	  goto error_exit;
	else
          cal_tm.tm_mon = cur_tm.tm_mon + 1;
      } else if (cal_tm.tm_mday == cur_tm.tm_mday) {
        if (cal_tm.tm_hour < cur_tm.tm_hour) {
          if (year != -1 || month != -1 || day != -1)
	    goto error_exit;
	  else
            cal_tm.tm_mday = cur_tm.tm_mday + 1;
        } else if (cal_tm.tm_hour == cur_tm.tm_hour) {
          if (cal_tm.tm_min < cur_tm.tm_min) {
            if (year != -1 || month != -1 || day != -1)
	      goto error_exit;
	    else
              cal_tm.tm_mday = cur_tm.tm_mday + 1;
	  } else if (cal_tm.tm_min == cur_tm.tm_min) {
            if (cal_tm.tm_sec < cur_tm.tm_sec) {
              if (year != -1 || month != -1 || day != -1)
	        goto error_exit;
	      else
                cal_tm.tm_mday = cur_tm.tm_mday + 1;
	    }
	  }
	}
      }
    }
  }

  cal_time = mktime(&cal_tm);

#ifdef DEBUG
  printf("arg: %s\n", arg);
  printf("year %d: month %d: day %d: hour %d: minute %d: second %d\n",
      year, month, day, hour, minute,second);
  printf("cal_time: %s\n", ctime(&cal_time));
#endif /* DEBUG */

  return cal_time;

error_exit:
  fprintf(stderr, "invalid argument: -start_time %s\n", arg);
  exit(1);
  return -1;
}

int
opt_start_time(const char *arg)
{
  time_t start_time;

  start_time = calc_calender_time(arg);
  if (start_time < 0)
    return -1;
  recording_start_time = start_time;
  return 0;
}

int
opt_end_time(const char *arg)
{
  time_t end_time;

  end_time = calc_calender_time(arg);
  if (end_time < 0)
    return -1;
  recording_end_time = end_time;
  return 0;
}

void
show_help(void)
{
  const char *ver = VERSION;
  printf("\n"
         "%s version %s Copyright (c) 2003 nejik\n"
	 "\n"
	 "usage: %s [OPTIONS] [output file name]\n"
	 , prog_name, ver, prog_name);
  parseopt_print_cmd_opt();
  exit(0);
}

static int normal_quit = 0;

void
prog_quit(void)
{
  if (normal_quit)
    mgr_quit();
  else {
    fprintf(stderr, "prog_quit: error quit !!!\n");
    mgr_error_quit();
  }
  term_quit();
}

int
main (int argc, char **argv)
{
  int ch;
  int ret;
  int rec_start = 0;
  u_time start_time = 0;
  u_time rec_end_time = 0;
  u_time cur_time = 0;
  u_time next_time = 0;
  u_time interval = 1000000LL;
  char *mes_pause = "Pause:";
  char *mes_start = "Rec:";
  char *mes_stop = "STOP:";
  char *mes_pre = mes_stop;
  char buf[1024];

  normal_quit = 0;

  prog_name = basename (argv[0]);
  parseopt_set_default_func (mgr_set_output_file_name);
  parseopt_set_option_defs (options);
  //parseopt_print_rc_opt();
  //exit(0);

  parseopt_defaults(RC_FILE_NAME);
  parseopt_cmd(argc, argv);

  atexit(prog_quit);

  if (recording_end_time == 0 && recording_time > 0) {
    int hour, minute, second;
    second = recording_time % 60;
    minute = (recording_time / 60) % 60;
    hour = recording_time / 60 / 60;
    fprintf(stdout, "Recording time: ");
    if (hour > 0)
      fprintf(stdout, "%d hour", hour);
    if (minute > 0) {
      if (hour > 0)
        fprintf(stdout, ", ");
      fprintf(stdout, "%d minute", minute);
    }
    if (second > 0) {
      if (hour > 0 || minute > 0)
        fprintf(stdout, ", ");
      fprintf(stdout, "%d second", second);
    }
    fprintf(stdout, "\n");
  }

  if (recording_end_time > 0) {
    if (recording_start_time >= recording_end_time) {
      fprintf(stderr, "end_time is earlier than start_time.\n");
      exit(1);
    }
    if (recording_time > 0) {
      fprintf(stderr, "-t option, record time disabled.\n");
      recording_time = 0;
    }
    fprintf(stdout, "End time: %s", ctime(&recording_end_time));
    rec_end_time = (u_time)recording_end_time * 1000000;
  }

  if (recording_start_time > 0) {
    int hour, minute, second;
    time_t cur_cal;
    int sleep_time;

    if (mgr_init() != MGR_OK) {
      exit(1);
    }
    mgr_print_param();
    if (mgr_quit() != MGR_OK) {
      exit(1);
    }

    fprintf(stdout, "Start time: %s", ctime(&recording_start_time));
    if (recording_end_time > 0)
      fprintf(stdout, "End time: %s", ctime(&recording_end_time));
    else
      fprintf(stdout, "End time: Not specified");

    cur_cal = time(NULL);
    sleep_time = recording_start_time - cur_cal;
    second = sleep_time % 60;
    minute = (sleep_time / 60) % 60;
    hour = sleep_time / 60 / 60;
    fprintf(stdout, "Sleeping : ");
    if (hour > 0)
      fprintf(stdout, "%d hour", hour);
    if (minute > 0) {
      if (hour > 0)
        fprintf(stdout, ", ");
      fprintf(stdout, "%d minute", minute);
    }
    if (second > 0) {
      if (hour > 0 || minute > 0)
        fprintf(stdout, ", ");
      fprintf(stdout, "%d second", second);
    }
    fprintf(stdout, "\n");

    sleep(sleep_time);
  }

  if (mgr_init() != MGR_OK) {
    exit(1);
  }

  mgr_print_param();

  //if (ntvrec_end_time == 0)
    term_init();

  cur_time = get_cur_u_time();
  next_time = cur_time + interval;

  ret = MGR_OK;
  printf ("start\n");
  if (recording_time > 0 || rec_end_time > 0 || recording_start_time > 0) {
    int hour, minute, second;
    if (recording_time > 0) {
      second = recording_time % 60;
      minute = (recording_time / 60) % 60;
      hour = recording_time / 60 / 60;
      fprintf(stdout, "Recording time: ");
      if (hour > 0)
        fprintf(stdout, "%d hour", hour);
      if (minute > 0) {
        if (hour > 0)
          fprintf(stdout, ", ");
        fprintf(stdout, "%d minute", minute);
      }
      if (second > 0) {
        if (hour > 0 || minute > 0)
          fprintf(stdout, ", ");
        fprintf(stdout, "%d second", second);
      }
      fprintf(stdout, "\n");
    } else {
      time_t ct = time(NULL);
      if (recording_start_time > 0)
        fprintf(stdout, "Start time: %s", ctime(&recording_start_time));
      else
        fprintf(stdout, "Start time: %s", ctime(&ct));
      fprintf(stdout,   "End time:   %s", ctime(&recording_end_time));
    }

    ret = mgr_state_start();
    rec_start = 1;
    start_time = get_cur_u_time();
    cur_time = start_time;
    next_time = cur_time + interval;
    mes_pre = mes_start;
  }

  printf ("\n");
  printf ("Press 's' to start\n");
  printf ("Press 'p' to pause or restart\n");
  printf ("Press 't' to stop\n");
  printf ("Press 'q' to quit\n");
  printf ("\n");

  while (1) {
    ch = read_key ();
    if (ch == 'q' || ch == 'Q') {
      break;
    } else if (ch == 'p' || ch == 'P') {
      ret = mgr_state_toggle_pause();
      rec_start = !rec_start;
      mes_pre = rec_start ? mes_start : mes_pause;
    } else if (ch == 's' || ch == 'S') {
      ret = mgr_state_start();
      rec_start = 1;
      start_time = get_cur_u_time();
      cur_time = start_time;
      next_time = cur_time + interval;
      mes_pre = mes_start;
    } else if (ch == 't' || ch == 'T') {
      ret = mgr_state_stop();
      rec_start = 0;
      mes_pre = mes_stop;
    } else
      usleep (1000);
    if (ret == MGR_FAIL)
      break;

    cur_time = get_cur_u_time();
    if (rec_end_time && cur_time > rec_end_time) {
      fprintf(stdout, "Record end.\n");
      break;
    } else if(recording_time && recording_time<(cur_time - start_time)/1000000){
      fprintf(stdout, "Record end.\n");
      break;
    }
    if (cur_time > next_time) {
      double bitrate = ffmt_get_bitrate();
      long long data_size = ffmt_get_total_data_size();
      double record_time = ffmt_get_record_time();
      int video_frames = ffmt_get_video_frame_count();
      double fps;

      bitrate = bitrate / 1000.0;
      record_time = record_time / 1000000.0;
      if (record_time == 0)
	fps = 0;
      else
	fps = (double)video_frames / record_time;
      sprintf (buf,
         "%s frame=%5d fps=%3.2f size=%8lldkB time=%0.1f bitrate=%6.1fkbits/s",
         mes_pre, video_frames, fps, data_size / 1024, record_time, bitrate);
      fprintf (stderr, "%s \r", buf);
      fflush (stderr);
      while (next_time < cur_time)
        next_time = next_time + interval;
    }
  }

  mgr_state_quit();
  mgr_quit();
  //if (ntvrec_end_time == 0)
    term_quit ();

  normal_quit = 1;

  printf ("\nquit\n");

  return 0;
}

