/* xwin.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <inttypes.h>
#include <sys/mman.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <sys/shm.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/extensions/XShm.h>
#include <X11/extensions/Xv.h>
#include <X11/extensions/Xvlib.h>
#define FCC_YV12 0x32315659

#include "xwin.h"

#include "yuv2rgb.h"


extern const char *prog_name;

static const char *display_name = NULL;

static int xwin_vo_type = XWIN_VO_XSHM;

static Display *dpy = NULL;
static int scrn;
static Window win;
static GC gc;

static XImage *shm_image;
static XImage *x11_image;
static XvImage *xv_image;
static XvPortID port;

static XShmSegmentInfo shm_info;
static int image_size;
static int win_width, win_height;
static int d_width, d_height;
static Atom wm_protocols;
static Atom wm_delete_window;
static XVisualInfo xvinfo;

static int x_window = 0;

static int shm_flag;

static int offset24;
static int rgb_stride;
static int y_stride;
static int uv_stride;
static int u_offset;
static int v_offset;
static int disp_depth;
static int image_depth;
static int bytes_per_line;
static int mode_rgb;


int
xwin_set_display (const char *dsp_name)
{
  display_name = dsp_name;
  return 0;
}

int
xwin_set_vo_type (const char *vo_name, int vo_type)
{
  if (!vo_name) {
    if ((vo_type != XWIN_VO_X11) && 
	(vo_type != XWIN_VO_XSHM) &&
	(vo_type != XWIN_VO_XV)) {
      fprintf(stderr, "xwin_set_vo_type: unknown vo_type %d\n", vo_type);
      return -1;
    }
    xwin_vo_type = vo_type;
    return 0;
  }
  if (strcasecmp(vo_name, "X11") == 0)
    xwin_vo_type = XWIN_VO_X11;
  else if (strcasecmp(vo_name, "XSHM") == 0)
    xwin_vo_type = XWIN_VO_XSHM;
  else if (strcasecmp(vo_name, "XV") == 0)
    xwin_vo_type = XWIN_VO_XV;
  else {
    fprintf(stderr, "xwin_set_vo_type: unknown vo_type %s\n", vo_name);
    return -1;
  }
  return 0;
}

static int haderror;
static int (*origerrorhandler)(Display *, XErrorEvent *);

static int
shmerrorhandler (Display *d, XErrorEvent *e)
{
  haderror++;
  if (e->error_code == BadAccess) {
    fprintf (stderr, "failed to attach shared memory\n");
    return 0;
  } else
    return (*origerrorhandler)(d, e);
}

static int
create_shm (XShmSegmentInfo *info, int size)
{
  info->shmid = shmget (IPC_PRIVATE, size, IPC_CREAT|0777);
  if (info->shmid == -1) {
    fprintf (stderr, "create_shm(): cannot create shared memory\n");
    return -1;
  } 
  info->shmaddr = (char*) shmat (info->shmid, 0, 0);
  if (info->shmaddr == ((char*)-1)) {
    perror ("shmat");
    shmctl (info->shmid, IPC_RMID, 0);
    return -1;
  } 
  info->readOnly = True;
  XSync (dpy, True);
  haderror = False;
  origerrorhandler = XSetErrorHandler(shmerrorhandler);
  XShmAttach (dpy, info);
  XSync (dpy, True);
  XSetErrorHandler (origerrorhandler);
  if (haderror) {
    if (shmdt (info->shmaddr) == -1)
      perror ("shmdt:");
    if (shmctl (info->shmid, IPC_RMID, 0) == -1)
      perror ("shmctl rmid:");
    return -1;
  }
  return 0;
}

static int xv_check_yv12 (XvPortID xport)
{
    XvImageFormatValues * formatValues;
    int formats;
    int i;

    formatValues = XvListImageFormats (dpy, xport, &formats);
    for (i = 0; i < formats; i++)
        if ((formatValues[i].id == FCC_YV12) &&
            (! (strcmp (formatValues[i].guid, "YV12")))) {
            XFree (formatValues);
            return 0;
        }
    XFree (formatValues);
    return 1;
}
static int xv_check_extension (Display *dpy, Window win)
{
    unsigned int version;
    unsigned int release;
    unsigned int dummy;
    int adaptors;
    int i;
    unsigned long j;
    XvAdaptorInfo * adaptorInfo;

    if ((XvQueryExtension (dpy, &version, &release,
                           &dummy, &dummy, &dummy) != Success) ||
        (version < 2) || ((version == 2) && (release < 2))) {
        fprintf (stderr, "No xv extension\n");
        return 1;
    }

    XvQueryAdaptors (dpy, win, &adaptors,
                     &adaptorInfo);

    for (i = 0; i < adaptors; i++)
        if (adaptorInfo[i].type & XvImageMask)
            for (j = 0; j < adaptorInfo[i].num_ports; j++)
                if ((! (xv_check_yv12 (adaptorInfo[i].base_id + j))) &&
                    (XvGrabPort (dpy, adaptorInfo[i].base_id + j,
                                 0) == Success)) {
                    port = adaptorInfo[i].base_id + j;
                    XvFreeAdaptorInfo (adaptorInfo);
                    return 0;
                }

    XvFreeAdaptorInfo (adaptorInfo);
    fprintf (stderr, "Cannot find xv port\n");
    return 1;
}

int
open_x (int width, int height)
{
  unsigned int mask;
  int major, minor;
  Bool pixmaps;
  XVisualInfo visualTemplate;
  XVisualInfo *XvisualInfoTable;
  XVisualInfo *XvisualInfo;
  int number;
  int i;
  XSetWindowAttributes attr;
  XGCValues gcValues;

  XSizeHints size_hints;
  const char *title;

  int blue_mask = 0;
  int bits_per_pixel = 0;

  dpy = XOpenDisplay (display_name);
  if (!dpy) {
    fprintf (stderr, "open_x(): Can not open display\n");
    return -1;
  }
  scrn = DefaultScreen (dpy);
  gc = DefaultGC (dpy, scrn);

  visualTemplate.class = TrueColor;
  visualTemplate.screen = scrn;
  XvisualInfoTable = XGetVisualInfo (dpy, VisualScreenMask | VisualClassMask,
      &visualTemplate, &number);
  if (XvisualInfoTable == NULL) {
    fprintf (stderr, "open_x(): No truecolor visual\n");
    XCloseDisplay (dpy);
    dpy = NULL;
    return -1;
  }

  XvisualInfo = XvisualInfoTable;
  for (i = 0; i < number; i++)
    if (XvisualInfoTable[i].depth > XvisualInfo->depth)
      XvisualInfo = XvisualInfoTable + i;

  xvinfo = *XvisualInfo;
  XFree (XvisualInfoTable);

  attr.background_pixmap = None;
  attr.backing_store = NotUseful;
  attr.border_pixel = 0;
  attr.event_mask = 0;
  attr.colormap = DefaultColormap (dpy, scrn);
  //attr.colormap = XCreateColormap (dpy, RootWindow(dpy, xvinfo.screen),
  //    xvinfo.visual, AllocNone);

  win = XCreateWindow (dpy, DefaultRootWindow(dpy), 0, 0, width, height, 0,
      xvinfo.depth, InputOutput, xvinfo.visual,
      (CWBackPixmap | CWBackingStore | CWBorderPixel |
       CWEventMask | CWColormap), &attr);

  mask = KeyPressMask|ExposureMask|StructureNotifyMask;
  XSelectInput (dpy, win, mask);

  gc = XCreateGC (dpy, win, 0, &gcValues);

  xv_image = NULL;
  shm_image = NULL;
  x11_image = NULL;
  if (xwin_vo_type == XWIN_VO_XV) {
    if (xv_check_extension (dpy, win)) {
      fprintf (stderr, "open_x(): not support xv extension\n");
      //return -1;
      xwin_vo_type = XWIN_VO_XSHM;
    }
    else {
      xv_image = XvShmCreateImage (dpy, port, FCC_YV12, NULL,
	          width, height, &shm_info);
      if (xv_image == NULL) {
        fprintf (stderr, "open_x(): Cannot create xv image\n");
	xwin_vo_type = XWIN_VO_XSHM;
      } else {
        image_size = xv_image->data_size;
printf("open_x: xvimage->data_size %d, width %d, height %d, yv12 size %d\n", xv_image->data_size, width, height, width * height * 12 / 8);
      }
    }
  }

  if (xwin_vo_type == XWIN_VO_XSHM) {
    if ((XShmQueryVersion (dpy, &major, &minor, &pixmaps) == 0) ||
        (major < 1) || ((major == 1) && (minor < 1))) {
      fprintf (stderr, "open_x(): No shm extension\n");
      xwin_vo_type = XWIN_VO_X11;
    } else {
      shm_image = XShmCreateImage (dpy, xvinfo.visual, xvinfo.depth, ZPixmap,
	  NULL, &shm_info, width, height);
      if (shm_image == NULL) {
        fprintf (stderr, "open_x(): Cannot create x shm image\n");
	xwin_vo_type = XWIN_VO_X11;
      } else {
        image_size = shm_image->bytes_per_line * shm_image->height;
      }
    }
  }

  shm_flag = 0;
  if ((xwin_vo_type == XWIN_VO_XV) || (xwin_vo_type == XWIN_VO_XSHM)) {
    if (create_shm(&shm_info, image_size) < 0) {
      fprintf (stderr, "open_x(): cannot create shared memory\n");
      if (xv_image) {
	XFree (xv_image);
	xv_image = XvCreateImage (dpy, port, FCC_YV12, NULL, width, height);
	if (xv_image == NULL) {
          fprintf (stderr, "open_x(): cannot create xv image.\n");
	  xwin_vo_type = XWIN_VO_X11;
	} else {
	  image_size = xv_image->data_size;
	  xv_image->data = (char*) malloc (image_size);
	}
      }
      else if (shm_image) {
        XDestroyImage (shm_image);
	shm_image = NULL;
        xwin_vo_type = XWIN_VO_X11;
      }
    } else {
      shm_flag = 1;
      if (xwin_vo_type == XWIN_VO_XV) {
        xv_image->data = shm_info.shmaddr;
        xv_image->obdata = (char*) &shm_info;
      } else {
        shm_image->data = shm_info.shmaddr;
        shm_image->obdata = (char*) &shm_info;
        bits_per_pixel = shm_image->bits_per_pixel;
        blue_mask = shm_image->blue_mask;
        bytes_per_line = shm_image->bytes_per_line;
      }
    }
  }

  if (xwin_vo_type == XWIN_VO_X11) {
    x11_image = XCreateImage (dpy, xvinfo.visual, xvinfo.depth, ZPixmap, 0,
	NULL, width, height, 8, 0);
    bits_per_pixel = x11_image->bits_per_pixel;
    blue_mask = x11_image->blue_mask;
    bytes_per_line = x11_image->bytes_per_line;
    image_size = bytes_per_line * height;
    x11_image->data = (char*) malloc (bytes_per_line * height);
  }

  disp_depth = (xvinfo.depth == 24) ? bits_per_pixel : xvinfo.depth;
  mode_rgb = (blue_mask & 0x01);

  win_width = width;
  win_height = height;
  d_width = width;
  d_height = height;

  wm_protocols = XInternAtom (dpy, "WM_PROTOCOLS", False);
  wm_delete_window = XInternAtom (dpy, "WM_DELETE_WINDOW", False);
  (void) XSetWMProtocols (dpy, win, &wm_delete_window, 1);

  title = prog_name;
  XmbSetWMProperties (dpy, win, title, title,
      NULL, 0, NULL, NULL, NULL);

  if (!xv_image) {
    size_hints.flags = PMinSize | PMaxSize;
    size_hints.min_width = width;
    size_hints.min_height = height;
    size_hints.max_width = width;
    size_hints.max_height = height;
    XSetWMNormalHints (dpy, win, &size_hints);
  }

  XMapWindow (dpy, win);

  return 1;
}

int
close_x (void)
{
  if (dpy == NULL)
    return 0;

  if (shm_flag) {
    XShmDetach (dpy, &shm_info);
    XSync (dpy, False);
    if (shmdt (shm_info.shmaddr) == -1)
      perror ("shmdt:");
    if (shmctl (shm_info.shmid, IPC_RMID, 0) == -1)
      perror ("shmctl rmid:");
  }

  if (xv_image) {
    XvUngrabPort (dpy, port, 0);
    if (!shm_flag)
      free (xv_image->data);
    XFree (xv_image);
    xv_image = NULL;
  }
  else if (shm_image) {
    XDestroyImage (shm_image);
    shm_image = NULL;
  }
  else if (x11_image) {
    XDestroyImage (x11_image);
    x11_image = NULL;
  }

  XFreeGC (dpy, gc);
  XDestroyWindow (dpy, win);
  XCloseDisplay (dpy);
  dpy = NULL;
  return 1;
}

int
x_get_disp_depth (void)
{
  return disp_depth;
}

int
x_dispatch_event (void)
{
  XEvent ev;
  KeySym keysym;
  int ret = -1;

  while (XPending(dpy)) {
    XNextEvent (dpy, &ev);
    switch (ev.type) {
      case KeyPress:
        keysym = XKeycodeToKeysym (dpy, ev.xkey.keycode, 0);
	switch (keysym) {
	  case XK_q:
	    ret = 'q';
	    return ret;
	    break;
	  case XK_k:
	    ret = 'k';
	    return ret;
            break;
	  case XK_j:
	    ret = 'j';
	    return ret;
            break;
	  case XK_l:
	    ret = 'l';
	    return ret;
            break;
	  case XK_h:
	    ret = 'h';
	    return ret;
            break;
	  case XK_semicolon:
	    ret = ';';
	    return ret;
            break;
	  case XK_g:
	    ret = 'g';
	    return ret;
            break;
	  case XK_m:
	    ret = 'm';
	    return ret;
            break;
	  case XK_n:
	    ret = 'n';
	    return ret;
            break;
	  default:
	    ret = -1;
	    return ret;
	    break;
	}
	break;
      case Expose:
	if (xv_image) {
	  if (shm_flag)
	    XvShmPutImage (dpy, port, win, gc, xv_image,
	        0, 0, win_width, win_height, 0, 0, d_width, d_height, True);
	  else
	    XvPutImage (dpy, port, win, gc, xv_image,
		0, 0, win_width, win_height, 0, 0, d_width, d_height);
	}
	else if (shm_image)
          XShmPutImage(dpy, win, gc, shm_image, 0, 0, 0, 0, win_width, win_height, True);
	ret = 1;
	break;
      case ClientMessage:
        if (ev.xclient.message_type == wm_protocols &&
            ev.xclient.data.l[0] == (int)wm_delete_window)
          return ((int)'q');
        break;
      case ConfigureNotify:
	if (xv_image) {
	  d_width = ev.xconfigure.width;
	  d_height = ev.xconfigure.height;
	  printf("window width %d, window height %d\n", d_width, d_height);
	}
	ret = 2;
        break;
    }
  }
  return ret;
}

static void
conv_24to32 (int width, int height, char* dest, char *src)
{
  int *dp;
  int offset = offset24;

  dp = (int*)(dest + (width * height * sizeof(int)));
  dp--;
  offset -= 3;
  for (; dp >= ((int*)dest); dp--) {
    *dp = *((int*)(src + offset)) & 0xffffff;
    offset -= 3;
  }
}

static void
conv_24to16 (int width, int height, char* dest, char *src)
{
  uint16_t *dp;
  const uint8_t *sp;
  unsigned int b, g, r;
  int i;
  int len = width * height;

  dp = (uint16_t*)dest;
  sp = (const uint8_t*)src;
  for (i = 0; i < len; i++, dp++) {
    b = *sp++;
    g = *sp++;
    r = *sp++;
    *dp = (b>>3) | ((g&0xfc)<<3) | ((r&0xf8)<<8);
  }
}

static void
conv_24to15 (int width, int height, char* dest, char *src)
{
  uint16_t *dp;
  const uint8_t *sp;
  unsigned int b, g, r;
  int i;
  int size = width * height;

  dp = (uint16_t*)dest;
  sp = (const uint8_t*)src;
  for (i = 0; i < size; i++, dp++) {
    b = *sp++;
    g = *sp++;
    r = *sp++;
    *dp = (b>>3) | ((g&0xf8)<<2) | ((r&0xf8)<<7);
  }
}

int
x_put_pic (int width, int height, void *data)
{
  char *buf = (char*) data;
  char *out_buf = NULL;

  if (xv_image) {
    memcpy (xv_image->data, data, image_size);
    if (shm_flag)
      XvShmPutImage (dpy, port, win, gc, xv_image,
          0, 0, win_width, win_height, 0, 0, d_width, d_height, True);
    else
      XvPutImage (dpy, port, win, gc, xv_image,
	  0, 0, win_width, win_height, 0, 0, d_width, d_height);
    return 0;
  }

  if (shm_image)
    out_buf = shm_image->data;
  else
    out_buf = x11_image->data;

  if (image_depth == 12) {
    yuv2rgb (out_buf, buf, buf+u_offset, buf+v_offset,
        width, height, rgb_stride, y_stride, uv_stride);
  } else if (image_depth == 24) {
    if (disp_depth == 32)
      conv_24to32 (width, height, out_buf, buf);
    else if (disp_depth == 24)
      memcpy (out_buf, buf, width * height * 3);
    else if (disp_depth == 16)
      conv_24to16 (width, height, out_buf, buf);
    else if (disp_depth == 15)
      conv_24to15 (width, height, out_buf, buf);
  } else if (image_depth == 0) {
    memcpy (out_buf, buf, width * height * ((disp_depth+7) / 8));
  } else {
    fprintf (stderr, "xwin: not support palette\n");
    return -1;
  }

  if (shm_image) {
    //printf("x_put_pic: XSHM out\n");
    XShmPutImage (dpy, win, gc, shm_image, 0, 0, 0, 0, width, height, True);
  }
  else {
    //printf("x_put_pic: X11 out\n");
    XPutImage (dpy, win, gc, x11_image, 0, 0, 0, 0, width, height);
  }
//  XSync (dpy, True);

  return 0;
}

int
xwin_quit (void)
{
  if (!x_window)
    return 0;

  x_window = 0;

  close_x ();
  return 0;
}

int
xwin_init (int width, int height, int in_depth)
{
  char *vo_name;

  if (x_window)
    return 0;

  x_window = 1;

  if (in_depth != 12 && xwin_vo_type == XWIN_VO_XV) {
    fprintf(stderr, "xwin_init: XV extension doesn't support RGB24 output, force XSHM.\n");
    xwin_vo_type = XWIN_VO_XSHM;
  }

  if (open_x (width, height) < 0) {
    x_window = 0;
    return -1;
  }

  if (in_depth == 12 && xwin_vo_type != XWIN_VO_XV) {
    yuv2rgb_init (disp_depth, (mode_rgb) ? MODE_RGB : MODE_BGR);
    v_offset = width * height;
    u_offset = v_offset + (width * height / 4);
    rgb_stride = bytes_per_line;
    y_stride = width;
    uv_stride = width / 2;
    image_depth = 12;
  } else if (in_depth == 24) {
    offset24 = width * height * (24/8);
    image_depth = 24;
  } else if (in_depth == 0) {
    image_depth = 0;
  }

  switch (xwin_vo_type) {
    case XWIN_VO_XV:
      vo_name = "XV";
      break;
    case XWIN_VO_X11:
      vo_name = "X11";
      break;
    case XWIN_VO_XSHM:
      vo_name = "XSHM";
      break;
    default:
      vo_name = "UNKNOWN";
  }
  printf ("xwin_init: vo %s, width %d, height %d\n", vo_name, width, height);

  //printf ("xwin_init: done\n");
  return 0;
}

