/* audio.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdint.h>

#include <avilib.h>
#include <lame.h>

#include "ifile.h"
#include "ofile.h"
#include "parseopt.h"
#include "acodec.h"
#include "audio.h"
#include "util.h"
#include "resample.h"

#define debug printf

static int audio_disable = 0;
static int audio_in_file_index = 0;
static int in_acodec = ACODEC_UNKNOWN;
static int in_format = 0;
static int in_channels = 0;
static int in_rate = 0;
static int in_bytespspl = 0;
static int in_bits = 0;
static int in_bitrate = 0;

static int out_acodec = ACODEC_UNKNOWN;
static int out_format = 0;
static int out_channels = 0;
static int out_rate = 0;
static int out_bytespspl = 0;
static int out_audio_bytes;
static int out_bits = 0;
//static int out_bitrate = 0;

static int last_in_rate = 0;
static int resample_flag = 0;
static int resample_in_size = 0;
static int resample_out_size = 0;

static double in_fps;
static double out_fps;
static double samples_per_frame = 0;
static int max_rate = 0;

static unsigned char *read_buf = NULL;
static int read_buf_size = 0;
static short *pcm_left = NULL;
static short *pcm_right = NULL;
static int pcm_buf_size = 0;
static int pcm_buf_spl = 0;
static int pcm_buf_pos = 0;
static unsigned char *stream_buf = NULL;
static int stream_buf_pos = 0;
static int stream_cur_pos = 0;
static int stream_buf_size = 0;

static int frame_size = 0;

static CUT_BLOCK *cut_samples = NULL;
static int cut_samples_num = 0;

static int in_sample_position = 0;
static int out_sample_position = 0;
static int out_samples = 0;

static IN_FILE *in_file = NULL;
static OUT_FILE *out_file = NULL;

static SEL_COMPO acodecs[] = {
  { "PCM", ACODEC_PCM },
  { "MP3", ACODEC_MP3 },
};

OptionDef audio_param[] = {
  {"acodec", "AUDIO_CODEC", HAS_ARG|OPT_SEL, {(void*)&out_acodec}, {ACODEC_UNKNOWN}, {(int)acodecs}, sizeof(acodecs)/sizeof(SEL_COMPO), "audio codec", "sel"},
  {"audio-rate", "AUDIO_RATE", HAS_ARG|OPT_INT, {(void*)&out_rate}, {0}, {0}, 96000, "audio rate", "d"},
  {"audio-channels", "AUDIO_CHANNELS", HAS_ARG|OPT_INT, {(void*)&out_channels}, {0}, {1}, 2, "audio channels (1 or 2)", "d"},
};

int audio_param_num = sizeof(audio_param) / sizeof(OptionDef);

#define READ_BUF_SIZE (1152*4)
#define OUT_BUF_SIZE (READ_BUF_SIZE*24)

int
check_audio(IN_FILE *i_file)
{
  short *l, *r;
  int num_samples;

  if (audio_disable) {
    fprintf(stderr, "check_audio: %s audio disable.\n", i_file->name);
    return -1;
  }

  if (i_file->audio_samples == 0) {
    fprintf(stderr, "%s doesn't have audio track.\n", i_file->name);
    i_file->acodec = ACODEC_UNKNOWN;
    audio_disable = 1;
    return -1;
  }

  in_format = i_file->format;
  in_acodec = i_file->acodec;
  in_bits = i_file->bits;
  in_channels = i_file->channels;
  in_rate = i_file->rate;
  in_bytespspl = i_file->bytespspl;
  in_bitrate = i_file->bitrate;

  if (acodec_decode_set_codec(in_acodec) != ACODEC_OK) {
    i_file->acodec = ACODEC_UNKNOWN;
    i_file->audio_samples = 0;
    i_file->audio_bytes = 0;
    fprintf(stderr, "check_audio: cannot find audio codec.\n");
    audio_disable = 1;
    return -1;
  }

  if (acodec_get_cap(in_acodec, ACODEC_CAP_ANALYSIS) != ACODEC_CAP_NONE) {
    int ret, size;
    int rate, channels, bits;
    uint8_t rbuf[64*1024];

    ret = ACODEC_NEED_MORE_DATA;
    while (ret == ACODEC_NEED_MORE_DATA) {
      size = read_audio_data(i_file, rbuf, &num_samples);
      ret = acodec_decode_analyze_stream(rbuf, size, &rate, &channels, &bits);
      if (ret == ACODEC_FAIL) {
        audio_disable = 1;
        fprintf(stderr, "check_audio: acodec_decode_analyze_stream() error.\n");
        return -1;
      }
    }

    if (in_channels != channels) {
      fprintf(stderr, "check_audio: stream channels doesn't suit, %d, %d.\n", channels, in_channels);
      i_file->channels = in_channels = channels;
    }
    if (in_rate != rate) {
      fprintf(stderr, "check_audio: stream rate doesn't suit, %d, %d.\n", rate, in_rate);
      i_file->rate = in_rate = rate;
    }
    if (in_bits != bits) {
      fprintf(stderr, "check_audio: stream bits doesn't suit, %d, %d.\n", bits, in_bits);
      i_file->bits = in_bits = bits;
    }
    if (in_bytespspl != (((in_bits+7)/8)*in_channels)) {
      i_file->bytespspl = in_bytespspl = ((in_bits+7)/8)*in_channels;
    }

    i_file->bitrate = in_bitrate = acodec_decode_get_bitrate();

    set_audio_position_start(i_file);
  }

  if (read_buf == NULL ||
      read_buf_size < (in_rate * in_bytespspl * 2)) {
    read_buf_size = in_rate * in_bytespspl * 2;
    read_buf = (unsigned char*) malloc(read_buf_size);
    if (!read_buf) {
      fprintf(stderr, "check_audio: memory allocate error.\n");
      exit(1);
    }
  }

  if (!pcm_left || !pcm_right ||
      pcm_buf_size < (in_rate * ((in_bits+7)/8)*2)) {
    pcm_buf_size = in_rate * ((in_bits+7)/8) * 2;
    l = (short*)realloc(pcm_left, pcm_buf_size);
    r = (short*)realloc(pcm_right, pcm_buf_size);
    if (!l || !r) {
      printf("check_audio: pmc_buf_size %d\n", pcm_buf_size);
      fprintf(stderr, "check_audio: memory allocate error.\n");
      exit(1);
    }
    pcm_left = l;
    pcm_right = r;
    pcm_buf_spl = pcm_buf_size / ((in_bits+7)/8);
  }

  if (!last_in_rate) {
    last_in_rate = in_rate;
  } else if (in_rate != last_in_rate) {
    resample_flag = 1;
  }
  last_in_rate = in_rate;

  acodec_decode_quit();

  max_rate = (max_rate >= in_rate) ? max_rate : in_rate;

  return 0;
}

int
audio_init(void)
{
  int i;
  int rate, channels;
//  int bits;

  out_file = NULL;
  if (audio_disable)
    return -1;

  out_file = get_audio_out_file();
  if (out_file == NULL)
    return -1;

  if (out_acodec == ACODEC_UNKNOWN)
    out_acodec = out_file->acodec;
  else
    out_file->acodec = out_acodec;

  if (out_acodec == ACODEC_UNKNOWN) {
    out_file = NULL;
    return -1;
  }

  if (out_rate != 0)
    out_file->rate = out_rate;
  else
    out_rate = out_file->rate;

  if (out_rate != last_in_rate)
    resample_flag = 1;

  if (out_channels != 0)
    out_file->channels = out_channels;
  else
    out_channels = out_file->channels;

  if (out_file->file_format == FILE_FORMAT_AVI) {
    if (out_file->acodec == ACODEC_PCM || out_file->acodec == ACODEC_PCM_BE)
      out_file->acodec = ACODEC_PCM_LE;
  } else if (out_file->file_format == FILE_FORMAT_MOV) {
    if (out_file->acodec == ACODEC_PCM || out_file->acodec == ACODEC_PCM_LE)
      out_file->acodec = ACODEC_PCM_BE;
  }
  out_acodec = out_file->acodec;

  if (acodec_encode_set_codec(out_file->acodec) != ACODEC_OK) {
    fprintf(stderr, "audio_init: acodec_encode_set_codec failed\n");
    audio_disable = 1;
    return -1;
  }

  if ((rate = acodec_encode_rate_cap(out_rate)) != out_rate) {
    fprintf(stderr, "audio_init: output rate, %d, %d.\n", rate, out_rate);
    out_file->rate = out_rate = rate;
  }
  if ((channels = acodec_encode_channels_cap(out_channels)) != out_channels) {
    fprintf(stderr, "audio_init: output channels, %d, %d.\n", channels, out_channels);
    out_file->channels = out_channels = channels;
  }
#if 0
  if ((bits = acodec_encode_bits_cap(out_bits)) != out_bits) {
    fprintf(stderr, "audio_init: output bits, %d, %d.\n", bits, out_bits);
    out_file->bits = out_bits = bits;
  }
#else
  out_bits = acodec_encode_bits_cap(out_bits);
#endif

  if (acodec_encode_init(out_file->acodec, out_rate, out_channels, out_bits) != ACODEC_OK) {
    fprintf(stderr, "audio_init: acodec_encode_init failed\n");
    audio_disable = 1;
    return -1;
  }

  /* for mp3lame codec's rate */
  if (out_acodec == ACODEC_MP3) {
    out_rate = acodec_encode_get_out_rate();
    if (out_rate != last_in_rate) {
      resample_flag = 1;
    }
    if (out_rate != out_file->rate) {
      acodec_encode_quit();
      if (acodec_encode_init(out_file->acodec, out_rate, out_channels, out_bits) != ACODEC_OK) {
        fprintf(stderr, "audio_init: acodec_encode_init failed\n");
        audio_disable = 1;
        return -1;
      }
    }
    out_file->rate = out_rate;
  }

  switch (out_acodec) {
    case ACODEC_PCM_LE:
      out_file->format = out_format = 0x1;
      break;
    case ACODEC_MP3:
      out_file->format = out_format = 0x55;
      break;
  }
  out_file->frame_size = acodec_encode_get_frame_size();
  frame_size = out_file->frame_size;
  out_file->bitrate = acodec_encode_get_bitrate();
  out_bytespspl = ((out_bits+7)/8) * out_channels;
  out_file->bytespspl = out_bytespspl;
  out_file->audio_bytes = out_audio_bytes = 0;

  out_fps = out_file->fps;

  if (!read_buf || read_buf_size < (out_rate * out_bytespspl * 2) ||
       read_buf_size < (max_rate * out_bytespspl * 2)) {
    read_buf_size = out_rate * out_bytespspl * 2;
    read_buf = (unsigned char*) realloc(read_buf, read_buf_size);
    if (!read_buf) {
      fprintf(stderr, "audio_init: memory allocate error.\n");
      exit(1);
    }
  }

  if (!pcm_left || !pcm_right ||
      pcm_buf_size < (out_rate * out_bytespspl / out_channels * 2) ||
      pcm_buf_size < (max_rate * out_bytespspl / out_channels * 2)) {
    pcm_buf_size = out_rate * out_bytespspl / out_channels * 2;
    pcm_buf_spl = pcm_buf_size / (out_bytespspl / out_channels);
    pcm_left = (short*) realloc (pcm_left, pcm_buf_size);
    pcm_right = (short*) realloc (pcm_right, pcm_buf_size);
    if (!pcm_left || !pcm_right) {
      fprintf(stderr, "audio_init: memory allocate error.\n");
      exit(1);
    }
  }
  pcm_buf_pos = 0;

  if (!stream_buf || stream_buf_size < (out_rate * out_bytespspl * 2) ||
      stream_buf_size < (max_rate * out_bytespspl * 2)) {
    stream_buf_size = out_rate * out_bytespspl * 2;
    stream_buf = (unsigned char*) realloc(stream_buf, stream_buf_size);
    if (!stream_buf) {
      fprintf(stderr, "audio_init: memory allocate error.\n");
      exit(1);
    }
  }
  stream_buf_pos = 0;
  stream_cur_pos = 0;

  samples_per_frame = (double)out_file->rate / out_file->fps;

  out_sample_position = 0;
  out_samples = 0;

  ofile_set_audio (out_file);

  for (i = 0; (uint32_t)i < sizeof(acodecs)/sizeof(SEL_COMPO); i++) {
    if (acodecs[i].id == out_file->acodec)
      printf ("acodec:                  %s\n", acodecs[i].name);
  }
  printf ("channels:                %d\n", out_file->channels);
  printf ("rate:                    %d\n", out_file->rate);
  printf ("bits:                    %d\n", out_file->bits);
  printf ("format:                  0x%X\n", out_file->format);
  printf ("bitrate:                 %d\n", out_file->bitrate);
  if (out_file->acodec == ACODEC_MP3) {
    ;
  }

  return 0;
}

int
audio_config(void)
{
  int i;

  if (audio_disable)
    return -1;

  if (in_file->rate != out_file->rate) {
    if (resample_flag == 0) {
      fprintf(stderr, "audio_config: resample_flag isn't set, in_rate %d, out_rate %d\n", in_rate, out_rate);
    }
    resample_flag = 1;
  }

  set_audio_position_start(in_file);
  in_format = in_file->format;
  in_acodec = in_file->acodec;
  in_bits = in_file->bits;
  in_channels = in_file->channels;
  in_rate = in_file->rate;
  in_bytespspl = in_file->bytespspl;
  in_bitrate = in_file->bitrate;

  if (in_file->cut_blocks_num > 0) {
    double fps = in_file->fps;
    double rate = in_file->rate;
    CUT_BLOCK *cut_blocks = in_file->cut_blocks;
    if (in_file->cut_blocks_num > cut_samples_num) {
      CUT_BLOCK *tmp_p;
      tmp_p = (CUT_BLOCK*) realloc(cut_samples,
	  sizeof(CUT_BLOCK) * in_file->cut_blocks_num);
      if (tmp_p == NULL)
	mem_alloc_fail("audio_config: realloc error.\n", 1);
      cut_samples = tmp_p;
    }
    cut_samples_num = in_file->cut_blocks_num;
    for (i = 0; i < cut_samples_num; i++) {
      cut_samples[i].start = (int)((double)cut_blocks[i].start * rate / fps);
      if (cut_blocks[i].end >= in_file->actual_frames)
#define MAX_INT (2147483647)
        cut_samples[i].end = MAX_INT;
      else
        cut_samples[i].end = (int)((double)(cut_blocks[i].end+1) * rate / fps);
    }
  } else {
    cut_samples_num = 0;
  }

  in_fps = in_file->fps;

  if (resample_flag) {
    if (resample_init(in_rate, out_rate, in_channels, 0, 0) < 0) {
      fprintf(stderr, "audio_config: resample_init failed.\n");
      exit(1);
    }
    resample_in_size = resample_get_input_sample_size();
    resample_out_size = resample_get_output_sample_size();
  }

  in_sample_position = 0;
  out_sample_position = out_sample_position - out_samples;
  out_samples = 0;

  if (acodec_decode_set_codec(in_acodec) != ACODEC_OK) {
    in_file->audio_samples = 0;
    in_file->audio_bytes = 0;
    audio_disable = 1;
    fprintf(stderr, "audio_config: cannot find audio codec.\n");
    return -1;
  }

  if (acodec_decode_init(in_acodec,in_rate,in_channels,in_bits) != ACODEC_OK) {
    in_file->audio_samples = 0;
    in_file->audio_bytes = 0;
    fprintf(stderr, "audio_config: cannot find audio codec.\n");
    audio_disable = 1;
    return -1;
  }

  return 0;
}

int
audio_set_in_file(int index)
{
  int ret;

  if (audio_disable)
    return 0;

  if (out_file == NULL)
    return 0;

  in_file = set_in_file_audio(index);
  ret = audio_config();
  if (ret < 0) {
    quit_iofile();
    exit(1);
  }
  audio_in_file_index = index;
  return ret;
}

void
audio_unset_in_file(int index)
{
  if (audio_disable)
    return;

  if (out_file == NULL)
    return;

  acodec_decode_quit();
  unset_in_file_audio(index);
  audio_in_file_index = -1;
}

int
read_audio_sample(short *pcm_l, short *pcm_r, int buf_spl_size)
{
  int read_size;
  int num_spl;
  int i;
//  int spls;
  void *pcm[2];

  num_spl = buf_spl_size;
  read_size = read_audio_data(in_file, read_buf, &num_spl);
  if (read_size <= 0) {
    for (i = 0; i < buf_spl_size; i++)
      pcm_l[i] = pcm_r[i] = 0;
    return -1;
  }

  pcm[0] = pcm_l;
  pcm[1] = pcm_r;
  num_spl = acodec_decode(pcm, read_buf, read_size);
#if 0
  spls = acodec_decode(pcm, read_buf, read_size);
  if (spls != num_spl) {
    fprintf(stderr, "read_audio_sample: decode sample doesn't suit, %d, %d.\n", spls, num_spl);
    num_spl = spls;
  }
#endif

//  if (num_spl >= buf_spl_size) {
  if (num_spl > buf_spl_size) {
    fprintf (stderr, "read_audio_sample: pcm_buf buffer over flow. pos %d, size %d\n", num_spl, buf_spl_size);
    exit(1);
  }

  return num_spl;
}

int
audio_cut_samples(short *pcm_l, short *pcm_r, int num_spl, int start_pos)
{
  int i;
  int end_pos = start_pos + num_spl;
  int cut_spl = 0;
  int total_cut_spl = 0;
  int start_spl, end_spl;
  int cut_flag;

  for (i = 0; i < cut_samples_num; i++) {
    cut_flag = 0;
    if (cut_samples[i].start <= start_pos &&
	cut_samples[i].end > start_pos) {
      cut_flag = 1;
    }
    if (cut_samples[i].start < end_pos &&
	cut_samples[i].end >= end_pos) {
      cut_flag = 1;
    }
    if (cut_samples[i].start >= start_pos &&
	cut_samples[i].end <= end_pos) {
      cut_flag = 1;
    }

    if (cut_flag) {
      if (cut_samples[i].end == MAX_INT && start_pos >= cut_samples[i].start)
	return -1;

      start_spl = cut_samples[i].start - start_pos;
      if (start_spl <= 0)
        start_spl = 0;
      else {
        pcm_l += start_spl;
        pcm_r += start_spl;
        start_pos += start_spl;
        num_spl -= start_spl;
      }
      end_spl = end_pos - cut_samples[i].end;
      if (end_spl < 0)
        end_spl = 0;
      cut_spl = num_spl - end_spl;
      for (i = 0; i < (num_spl - cut_spl); i++) {
	pcm_l[i] = pcm_l[i+cut_spl];
	pcm_r[i] = pcm_r[i+cut_spl];
      }
      start_pos += cut_spl;
      num_spl -= cut_spl;
      total_cut_spl += cut_spl;
      if (num_spl <= 0)
	break;
    }
  }

  return total_cut_spl;
}

int
read_sample(int num_samples)
{
  short *pcm_l = pcm_left;
  short *pcm_r = pcm_right;
  int buf_spl_size;
  int num_spl;
  int cut_spl;

#if 0
  if (out_acodec == ACODEC_MP3) {
    num_samples += frame_size-1;
    num_samples -= num_samples % frame_size;
  }
#endif

  while (pcm_buf_pos < num_samples) {
    if (resample_flag) {
      short pcm_l_in[resample_in_size];
      short pcm_r_in[resample_in_size];
      short *pcm_in[2];
      short *pcm_out[2];

      buf_spl_size = resample_in_size;
      num_spl = read_audio_sample(pcm_l_in, pcm_r_in, resample_in_size);
      if (num_spl < 0) {
        buf_spl_size = pcm_buf_spl - pcm_buf_pos;
        pcm_out[0] = &pcm_left[pcm_buf_pos];
        pcm_out[1] = &pcm_right[pcm_buf_pos];
        num_spl = resample_drain(pcm_out, buf_spl_size);
	if (num_spl > 0)
          pcm_buf_pos += num_spl;
        break;
      }

      cut_spl = audio_cut_samples(pcm_l_in, pcm_r_in, num_spl, in_sample_position);
      in_sample_position += num_spl;
      if (cut_spl < 0) {
        buf_spl_size = pcm_buf_spl - pcm_buf_pos;
        pcm_out[0] = &pcm_left[pcm_buf_pos];
        pcm_out[1] = &pcm_right[pcm_buf_pos];
        num_spl = resample_drain(pcm_out, buf_spl_size);
	if (num_spl > 0)
          pcm_buf_pos += num_spl;
        break;
      }

      num_spl -= cut_spl;

      pcm_in[0] = pcm_l_in;
      pcm_in[1] = pcm_r_in;
      buf_spl_size = pcm_buf_spl - pcm_buf_pos;
      if (buf_spl_size < resample_out_size) {
	fprintf(stderr, "read_sample: resample out_buf_size too small.\n");
	fprintf(stderr, "read_sample: resample_out_size %d, buf_spl_size %d.\n", resample_out_size, buf_spl_size);
      }
      pcm_out[0] = &pcm_left[pcm_buf_pos];
      pcm_out[1] = &pcm_right[pcm_buf_pos];
      num_spl = resample(pcm_in, num_spl, pcm_out, buf_spl_size);
      if (num_spl > 0)
        pcm_buf_pos += num_spl;
    }
    else {
      pcm_l = &pcm_left[pcm_buf_pos];
      pcm_r = &pcm_right[pcm_buf_pos];
      buf_spl_size = pcm_buf_spl - pcm_buf_pos;
      num_spl = read_audio_sample(pcm_l, pcm_r, buf_spl_size);
      if (num_spl < 0)
        break;

      cut_spl = audio_cut_samples(pcm_l, pcm_r, num_spl, in_sample_position);
      if (cut_spl < 0)
        break;

      pcm_buf_pos += (num_spl - cut_spl);

      in_sample_position += num_spl;
    }

    if (pcm_buf_pos > pcm_buf_spl) {
      fprintf(stderr, "read_sample: buf_size too small. pcm_buf_pos %d, pcm_buf_spl %d\n", pcm_buf_pos, pcm_buf_spl);
      exit(1);
    }
  }

  return pcm_buf_pos;
}

int
channel_conv(short *pcm_l, short *pcm_r, int num_samples)
{
  int i;

  if (in_channels == 1) {
    memcpy (pcm_r, pcm_l, num_samples * sizeof(short));
  } else if (in_channels == 2) {
    for (i = 0; i < num_samples; i++) {
      pcm_l[i] = (short)((int)((int)pcm_l[i] + (int)pcm_r[i]) >> 1);
    }
  }
  return num_samples;
}

int
audio_put_samples(int num_samples)
{
  int stream_size;
  int num_spl;
  int i;
  void *pcm[2];
  int put_samples;

  if (out_acodec == ACODEC_PCM ||
      out_acodec == ACODEC_PCM_BE ||
      out_acodec == ACODEC_PCM_LE) {
    int spls;

    if (pcm_buf_pos < num_samples) {
      read_sample(num_samples);
    }
    num_spl = pcm_buf_pos;
    if (num_samples > num_spl) {
      fprintf(stderr, "audio_put_sample: num_samples %d, num_spl %d, %d\n", num_samples, num_spl, num_samples - num_spl);
      num_samples = num_spl;
    }

    if (in_channels != out_channels) {
      channel_conv(pcm_left, pcm_right, num_samples);
    }

    pcm[0] = pcm_left;
    pcm[1] = pcm_right;
    spls = num_samples;
    stream_size = acodec_encode(pcm, &spls, stream_buf, stream_buf_size);

    write_audio_data(out_file, stream_buf, stream_size, spls);
    if ((pcm_buf_pos-num_samples) < 0) {
      fprintf(stderr, "\nput_samples: pcm_buf_pos invalid %d, num_samples %d, %d.\n", pcm_buf_pos, num_samples, pcm_buf_pos - num_samples);
      exit(1);
    }
    memmove(pcm_left, &pcm_left[num_samples], (pcm_buf_pos - num_samples) * sizeof(short));
    if (in_channels == 2)
      memmove(pcm_right, &pcm_right[num_samples], (pcm_buf_pos - num_samples) * sizeof(short));
    pcm_buf_pos -= num_samples;
    out_sample_position += num_samples;
  } else if (out_acodec == ACODEC_MP3) {
    int size;
    int stm_pos = stream_buf_pos;
    int cur_pos = stream_cur_pos;
    int spls;
    put_samples = 0;
    num_spl = 0;
    stream_size = 0;

    while (num_spl < num_samples) {
      spls = (num_samples - num_spl);
      spls = (spls < frame_size) ? spls : frame_size;
      if (pcm_buf_pos < spls)
        read_sample(spls);

      if (pcm_buf_pos < frame_size) {
//        fprintf(stderr, "\naudio_put_samples: pcm_buf_pos %d < frame_size %d, %d.\n", pcm_buf_pos, frame_size, pcm_buf_pos - frame_size);
        break;
      }

      if (in_channels != out_channels)
        channel_conv(pcm_left, pcm_right, frame_size);

      pcm[0] = pcm_left;
      pcm[1] = pcm_right;
      spls = frame_size;
      size = acodec_encode(pcm, &spls,
           (unsigned char*)(&stream_buf[stm_pos]), stream_buf_size-stm_pos);
      if (size < 0) {
        fprintf(stderr, "audio_put_samples: mp3 encode error %d.\n", size);
        exit (1);
      } else if (size > 0) {
        if (out_file->file_format == FILE_FORMAT_AVI) {
	  stm_pos += size;
	  put_samples += spls;
        } else if (out_file->file_format == FILE_FORMAT_MOV) {
          write_audio_data(out_file, &stream_buf[stm_pos], size, spls);
        } else {
          fprintf(stderr, "audio_put_samples: invalid file format.\n");
	}
	stream_size += size;
      }

      num_spl += frame_size;
      if ((pcm_buf_pos-frame_size) < 0) {
        fprintf(stderr, "\naudio_put_samples: pcm_buf_pos invalid %d, num_samples %d, %d.\n", pcm_buf_pos, num_samples, pcm_buf_pos - frame_size);
        //exit(1);
        pcm_buf_pos = 0;
      } else {
        for (i = 0; i < pcm_buf_pos-frame_size; i++)
          pcm_left[i] = pcm_left[i+frame_size];
        if (in_channels == 2)
          for (i = 0; i < pcm_buf_pos-frame_size; i++)
	    pcm_right[i] = pcm_right[i+frame_size];
        pcm_buf_pos -= frame_size;
      }
    }
    if (out_file->file_format == FILE_FORMAT_AVI && stm_pos > 0) {
      write_audio_data(out_file, stream_buf, stm_pos, put_samples);
      stream_size = stm_pos;
      stm_pos = 0;
    }
    stream_buf_pos = stm_pos;
    stream_cur_pos = cur_pos;
    out_sample_position += num_spl;
  } else {
    stream_size = 0;
  }
  out_file->audio_bytes += stream_size;

  return stream_size;
}

int
audio_frame_process(int frm_pos)
{
  int spl_pos;
  int num_spl;
  int data_size;

  if (audio_disable)
    return 0;

  if (out_file == NULL)
    return 0;
  if (in_file->audio_samples == 0)
    return 0;

  frm_pos++;
  spl_pos = (int)(samples_per_frame * (double)frm_pos + 0.5);

  num_spl = spl_pos - out_sample_position;
  if (num_spl <= 0) {
//    fprintf(stderr, "audio_frame_process: num_spl %d, pos %d, spl_pos %d, out_sample_position %d.\n", num_spl, frm_pos, spl_pos, out_sample_position);
//    num_spl = 0;
    return 0;
  }
  out_samples = spl_pos;
  data_size = audio_put_samples(num_spl);
//  printf("num_spl %d, data_size %d\n", num_spl, data_size);

  return data_size;
}

int
audio_flush (void)
{
  int stream_size = 0;
  int num_spl;
  void *pcm[2];

  if (audio_disable)
    return 0;

  if (out_file == NULL)
    return 0;

  if (stream_buf_pos > 0) {
    fprintf(stderr, "audio_flush: stream_buf_pos > 0, %d.\n", stream_buf_pos);
  }

  if (out_acodec == ACODEC_PCM ||
      out_acodec == ACODEC_PCM_LE ||
      out_acodec == ACODEC_PCM_BE) {
    int spls;
    stream_size = 0;
    if (pcm_buf_pos > 0) {
      num_spl = pcm_buf_pos;
      if (in_channels != out_channels)
        channel_conv(pcm_left, pcm_right, num_spl);

      pcm[0] = pcm_left;
      pcm[1] = pcm_right;
      spls = num_spl;
      stream_size = acodec_encode(pcm, &spls, stream_buf, stream_buf_size);
      write_audio_data(out_file, stream_buf, stream_size, spls);
      out_sample_position += num_spl;
    }
  } else if (out_acodec == ACODEC_MP3) {
    int stm_pos = stream_buf_pos;
    int size;
    int i, num_spl;
    int num_samples = 0;
    stream_size = 0;

    while (pcm_buf_pos > 0) {
      pcm[0] = pcm_left;
      pcm[1] = pcm_right;
      num_spl = frame_size;
      size = acodec_encode(pcm, &num_spl,
           (unsigned char*)(&stream_buf[stm_pos]), stream_buf_size-stm_pos);
      if (size < 0) {
        fprintf(stderr, "audio_put_samples: mp3 encode error %d.\n", size);
        exit (1);
      } else if (size > 0) {
	if (stm_pos > 0) {
          write_audio_data(out_file, stream_buf, stm_pos+ size, num_spl);
          out_file->audio_bytes += stm_pos + size;
          stream_size += stm_pos + size;
	  stm_pos = 0;
	} else {
          write_audio_data(out_file, &stream_buf[stm_pos], size, num_spl);
          out_file->audio_bytes += size;
          stream_size += size;
	}
      }
      for (i = 0; i < pcm_buf_pos-frame_size; i++)
        pcm_left[i] = pcm_left[i+frame_size];
      if (out_channels == 2)
        for (i = 0; i < pcm_buf_pos-frame_size; i++)
	  pcm_right[i] = pcm_right[i+frame_size];
      pcm_buf_pos -= frame_size;
    }

    while ((size = acodec_encode_flush(&num_spl, 
        (unsigned char*)(&stream_buf[stm_pos]), stream_buf_size-stm_pos)) > 0) {
      if (out_file->file_format == FILE_FORMAT_MOV) {
        write_audio_data(out_file, &stream_buf[stm_pos], size, num_spl);
        out_file->audio_bytes += size;
	stream_size += size;
      } else {
        stm_pos += size;
	num_samples += num_spl;
      }
    }

    if (out_file->file_format == FILE_FORMAT_AVI) {
      write_audio_data(out_file, stream_buf, stm_pos, num_samples);
      out_file->audio_bytes += stm_pos;
      stream_size += stm_pos;
    }
  }

  return stream_size;
}

void
audio_quit(void)
{
  if (audio_disable)
    return;

  if (out_file == NULL)
    return;

  acodec_encode_quit();
  if (pcm_right)
    free (pcm_right);
  if (pcm_left)
    free(pcm_left);
  if (read_buf)
    free(read_buf);
  if (stream_buf)
    free(stream_buf);
  pcm_left = NULL;
  pcm_right = NULL;
  read_buf = NULL;
  stream_buf = NULL;
  pcm_buf_size = read_buf_size = stream_buf_size = 0;

  if (cut_samples) {
    free(cut_samples);
    cut_samples = NULL;
    cut_samples_num = 0;
  }

  if (resample_flag)
    resample_quit();

  quit_iofile();
  in_file = NULL;
  out_file = NULL;
}

