/* video_desc.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "ctab.h"
#include "video_desc.h"

void
video_desc_init(video_desc_t *video_desc)
{
  int i;
  sample_desc_init(&video_desc->sample_desc);
  video_desc->version = 0;
  video_desc->revision_level = 0;
  video_desc->vendor = 0;
  video_desc->temporal_quality = 100;
  video_desc->spatial_quality = 258;
  video_desc->width = 0;
  video_desc->height = 0;
  video_desc->horizontal_resolution = qtime_float_to_fixed32(72.0);
  video_desc->vertical_resolution = qtime_float_to_fixed32(72.0);
  video_desc->data_size = 0;
  video_desc->frame_count = 1;
  for (i = 0; i < 32; i++)
    video_desc->compressor_name[i] = 0;
  video_desc->depth = 24;
  video_desc->color_table_id = -1;

  video_desc->ctab = NULL;
  video_desc->ext_flag = 0;
  video_desc->gama = 0;
  video_desc->fiel[0] = 0;
  video_desc->fiel[1] = 0;
  video_desc->mjqt = NULL;
  video_desc->mjht = NULL;
}

video_desc_t*
video_desc_new(void)
{
  video_desc_t *video_desc;
  video_desc = (video_desc_t*)qtime_malloc(sizeof(video_desc_t));
  if (!video_desc) return NULL;
  video_desc_init(video_desc);
  return video_desc;
}

void
video_desc_clean(video_desc_t *video_desc)
{
  if (video_desc->ctab)
    ctab_delete(video_desc->ctab);
  if (video_desc->mjqt)
    qtime_free(video_desc->mjqt);
  if (video_desc->mjht)
    qtime_free(video_desc->mjht);
  video_desc_init(video_desc);
}

void
video_desc_delete(video_desc_t *video_desc)
{
  if (video_desc->ctab)
    ctab_delete(video_desc->ctab);
  if (video_desc->mjqt)
    qtime_free(video_desc->mjqt);
  if (video_desc->mjht)
    qtime_free(video_desc->mjht);
  qtime_free(video_desc);
}

video_desc_t*
video_desc_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, video_desc_t *video_desc_ptr)
{
  atom_head_t subatom;
  video_desc_t *video_desc;
  ctab_t *ctab;

  if (video_desc_ptr) {
    video_desc = video_desc_ptr;
    video_desc_clean(video_desc);
  } else {
    if ((video_desc = video_desc_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  video_desc->sample_desc.size = atom_head->size;
  video_desc->sample_desc.data_format = atom_head->type;
  qtime_io_read(qtio, video_desc->sample_desc.reserved, 6);
  qtime_io_read16(qtio, &video_desc->sample_desc.data_reference_id);

  qtime_io_read16(qtio, &video_desc->version);
  qtime_io_read16(qtio, &video_desc->revision_level);
  qtime_io_read(qtio, &video_desc->vendor, 4);
  qtime_io_read32(qtio, &video_desc->temporal_quality);
  qtime_io_read32(qtio, &video_desc->spatial_quality);
  qtime_io_read16(qtio, &video_desc->width);
  qtime_io_read16(qtio, &video_desc->height);
  qtime_io_read32(qtio, &video_desc->horizontal_resolution);
  qtime_io_read32(qtio, &video_desc->vertical_resolution);
  qtime_io_read32(qtio, &video_desc->data_size);
  qtime_io_read16(qtio, &video_desc->frame_count);
  qtime_io_read(qtio, video_desc->compressor_name, 32);
  qtime_io_read16(qtio, &video_desc->depth);
  qtime_io_read16(qtio, &video_desc->color_table_id);

  if (atom_head->end_offset - 8 <= qtime_io_get_offset(qtio))
    return video_desc;

  if (video_desc->version != 0) {
//    char fcc[5];
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: unsupported video description version '%d'.\n", video_desc->version);
    fprintf(stderr, "QTIME_ERROR: extended size %d.\n", (int)(atom_head->size - SAMPLE_DESC_HEAD_SIZE - VIDEO_DESC_PROP_SIZE));
#if 0
    atom_head_init(&subatom);
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
    qtime_type_to_str(subatom.type, fcc);
    fprintf(stderr, "ERR: type = '%4s'.\n", fcc);
    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
    return video_desc;
#endif
  }

  atom_head_init(&subatom);
  subatom.media_type = atom_head->media_type;

  ctab = NULL;
  while (atom_head->end_offset > subatom.end_offset+7) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    switch (subatom.type) {
      case QTIME_TYPE_CTAB:
	if (video_desc->ctab) {
	  qtime_error_debug_info(QTIME_ERROR_TOO_MANY_ATOM);
          qtime_error_too_many_atom(QTIME_TYPE_VDSC, subatom.type);
	} else {
	  if ((ctab = ctab_read_atom(qtio, &subatom, ctab)) == NULL) {
            qtime_error_debug_info(subatom.error_code);
            atom_head->error_code = subatom.error_code;
	    goto fail;
	  }
	}
	break;
      case QTIME_TYPE_GAMA:
	qtime_io_read32(qtio, &video_desc->gama);
	video_desc->ext_flag |= VIDEO_DESC_EXT_GAMA;
	break;
      case QTIME_TYPE_FIEL:
	qtime_io_read(qtio, video_desc->fiel, 2);
	video_desc->ext_flag |= VIDEO_DESC_EXT_FIEL;
	break;
      case QTIME_TYPE_MJQT:
	video_desc->mjqt = (uint8_t*)qtime_malloc(subatom.body_size);
	if (!video_desc->mjqt) {
	  qtime_error_debug_info(QTIME_ERROR_MEMORY);
          atom_head->error_code = QTIME_ERROR_MEMORY;
	  goto fail;
	}
        if (qtime_io_read(qtio, video_desc->mjqt, subatom.body_size) !=
	    subatom.body_size) {
	  qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
          atom_head->error_code = QTIME_ERROR_ATOM_READ;
	  goto fail;
        }
	video_desc->ext_flag |= VIDEO_DESC_EXT_MJQT;
	break;
      case QTIME_TYPE_MJHT:
	video_desc->mjht = (uint8_t*)qtime_malloc(subatom.body_size);
	if (!video_desc->mjht) {
	  qtime_error_debug_info(QTIME_ERROR_MEMORY);
          atom_head->error_code = QTIME_ERROR_MEMORY;
	  goto fail;
	}
        if (qtime_io_read(qtio, video_desc->mjht, subatom.body_size) !=
	    subatom.body_size) {
	  qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
          atom_head->error_code = QTIME_ERROR_ATOM_READ;
	  goto fail;
        }
	video_desc->ext_flag |= VIDEO_DESC_EXT_MJHT;
	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_VDSC, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  return video_desc;

fail:
  if (video_desc_ptr)
    video_desc_clean(video_desc);
  else
    video_desc_delete(video_desc);
  qtime_error_atom_read(QTIME_TYPE_VDSC);
  return NULL;
}

video_desc_t*
video_desc_create(video_desc_t *video_desc_ptr)
{
  video_desc_t *video_desc;

  if (video_desc_ptr) {
    video_desc = video_desc_ptr;
    video_desc_clean(video_desc);
  } else {
    if ((video_desc = video_desc_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  return video_desc;
}

int64_t
video_desc_calc_size(video_desc_t *video_desc)
{
  int64_t size;

  size = SAMPLE_DESC_HEAD_SIZE + VIDEO_DESC_PROP_SIZE;

  if (video_desc->color_table_id == 0 && video_desc->ctab)
    size += ctab_calc_size(video_desc->ctab);

  if (video_desc->ext_flag & VIDEO_DESC_EXT_GAMA)
    size += 8 + 4;
  if (video_desc->ext_flag & VIDEO_DESC_EXT_FIEL)
    size += 8 + 2;

#if 0
  if (video_desc->ext_flag & VIDEO_DESC_EXT_MJQT)
    size += 8 + ;
  if (video_desc->ext_flag & VIDEO_DESC_EXT_MJHT)
    size += 8 + ;
#endif

  video_desc->sample_desc.size = size;

  return size;
}

int
video_desc_write_atom(qtime_io_t *qtio, video_desc_t *video_desc)
{
  atom_head_t atom_head;
  atom_head_t subatom;

  if (video_desc == NULL)
    return QTIME_ERROR;

  if (video_desc->version != 0) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: unsupported video description version '%d'.\n", video_desc->version);
  }

  atom_head_init(&atom_head);
  atom_head.size = video_desc->sample_desc.size;
  atom_head.type = video_desc->sample_desc.data_format;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_VDSC);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, video_desc->sample_desc.reserved, 6);
  qtime_io_write16(qtio, &video_desc->sample_desc.data_reference_id);

  qtime_io_write16(qtio, &video_desc->version);
  qtime_io_write16(qtio, &video_desc->revision_level);
  qtime_io_write(qtio, &video_desc->vendor, 4);
  qtime_io_write32(qtio, &video_desc->temporal_quality);
  qtime_io_write32(qtio, &video_desc->spatial_quality);
  qtime_io_write16(qtio, &video_desc->width);
  qtime_io_write16(qtio, &video_desc->height);
  qtime_io_write32(qtio, &video_desc->horizontal_resolution);
  qtime_io_write32(qtio, &video_desc->vertical_resolution);
  qtime_io_write32(qtio, &video_desc->data_size);
  qtime_io_write16(qtio, &video_desc->frame_count);
  qtime_io_write(qtio, video_desc->compressor_name, 32);
  qtime_io_write16(qtio, &video_desc->depth);
  qtime_io_write16(qtio, &video_desc->color_table_id);

  if (video_desc->color_table_id == 0 && video_desc->ctab)
    ctab_write_atom(qtio, video_desc->ctab);

  if (video_desc->ext_flag & VIDEO_DESC_EXT_GAMA) {
    atom_head_init(&subatom);
    subatom.size = 8+4;
    subatom.type = QTIME_TYPE_GAMA;
    if (atom_write_header(qtio, &subatom) < 0) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
      qtime_error_atom_write(QTIME_TYPE_VDSC);
      return QTIME_ERROR_ATOM_WRITE;
    }
    qtime_io_write32(qtio, &video_desc->gama);
    if (atom_write_footer(qtio, &subatom) < 0) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
      qtime_error_atom_write(QTIME_TYPE_VDSC);
      return QTIME_ERROR_ATOM_WRITE;
    }
  }

  if (video_desc->ext_flag & VIDEO_DESC_EXT_FIEL) {
    atom_head_init(&subatom);
    subatom.size = 8+2;
    subatom.type = QTIME_TYPE_FIEL;
    if (atom_write_header(qtio, &subatom) < 0) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
      qtime_error_atom_write(QTIME_TYPE_VDSC);
      return QTIME_ERROR_ATOM_WRITE;
    }
    qtime_io_write(qtio, video_desc->fiel, 2);
    if (atom_write_footer(qtio, &subatom) < 0) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
      qtime_error_atom_write(QTIME_TYPE_VDSC);
      return QTIME_ERROR_ATOM_WRITE;
    }
  }

  if (video_desc->ext_flag & VIDEO_DESC_EXT_MJQT) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stdout, "QTIME_ERROR: 'mjqt' is not implement.\n");
  }
  if (video_desc->ext_flag & VIDEO_DESC_EXT_MJHT) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stdout, "QTIME_ERROR: 'mjht' is not implement.\n");
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_VDSC);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
video_desc_dump(const char *parent_types, video_desc_t *video_desc)
{
  int len = strlen(parent_types);
  uint8_t types[len+12];
//  char name[video_desc->compressor_name[0]+1];
  char name[32];
  char vendor[5];
  char data_format[5];

  qtime_type_to_str(video_desc->sample_desc.data_format, data_format);
  qtime_type_to_str(video_desc->vendor, vendor);
  sprintf(types, "%s.%s", parent_types, "VDSC");

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)video_desc->sample_desc.size);
  fprintf(stdout, "%s: data format  '%.4s'\n", types, data_format);
  fprintf(stdout, "%s: data reference id %d\n", types, video_desc->sample_desc.data_reference_id);

  fprintf(stdout, "%s: version      %d\n", types, video_desc->version);
  fprintf(stdout, "%s: vendor       %.4s\n", types, vendor);
  fprintf(stdout, "%s: temporal quality %d\n", types, video_desc->temporal_quality);
  fprintf(stdout, "%s: spatial quality %d\n", types, video_desc->spatial_quality);
  fprintf(stdout, "%s: width        %d\n", types, video_desc->width);
  fprintf(stdout, "%s: height       %d\n", types, video_desc->height);
  fprintf(stdout, "%s: horizontal resolution %f\n", types, qtime_fixed32_to_float(video_desc->horizontal_resolution));
  fprintf(stdout, "%s: vertical resolution %f\n", types, qtime_fixed32_to_float(video_desc->vertical_resolution));
  fprintf(stdout, "%s: data size    %d\n", types, video_desc->data_size);
  fprintf(stdout, "%s: frame count  %d\n", types, video_desc->frame_count);
  memcpy(name, video_desc->compressor_name+1, video_desc->compressor_name[0]);
  name[video_desc->compressor_name[0]] = '\0';
  fprintf(stdout, "%s: compressor name '%s'\n", types, name);
  fprintf(stdout, "%s: depth        %d\n", types, video_desc->depth);
  fprintf(stdout, "%s: color table id %d\n", types, video_desc->color_table_id);
  if (video_desc->ctab)
    ctab_dump(types, video_desc->ctab);
  if (video_desc->ext_flag & VIDEO_DESC_EXT_GAMA)
    fprintf(stdout, "%s: gama         %f\n", types, qtime_fixed32_to_float(video_desc->gama));
  if (video_desc->ext_flag & VIDEO_DESC_EXT_FIEL)
    fprintf(stdout, "%s: fiel         %d,%d\n", types, video_desc->fiel[0], video_desc->fiel[1]);
  if (video_desc->ext_flag & VIDEO_DESC_EXT_MJQT)
    fprintf(stdout, "%s: mjqt\n", types);
  if (video_desc->ext_flag & VIDEO_DESC_EXT_MJHT)
    fprintf(stdout, "%s: mjht\n", types);
}

