/* udta.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "udta.h"
#include "moov.h"


/* udta_text */

void
udta_text_init(udta_text_t *udta_text)
{
  udta_text->size = 0;
  udta_text->type = 0;
  udta_text->number_of_text = 0;
  udta_text->text_list = NULL;
}

void
udta_text_clean(udta_text_t *udta_text)
{
  int i;

  for (i = 0; i < udta_text->number_of_text; i++)
    qtime_free(udta_text->text_list[i]);
  if (udta_text->text_list)
    qtime_free(udta_text->text_list);
  udta_text_init(udta_text);
}

udta_text_t*
udta_text_new(void)
{
  udta_text_t *udta_text;
  udta_text = (udta_text_t*)qtime_malloc(sizeof(udta_text_t));
  if (!udta_text)
    return NULL;
  udta_text_init(udta_text);
  return udta_text;
}

void
udta_text_delete(udta_text_t *udta_text)
{
  int i;
  if (!udta_text) return;
  for (i = 0; i < udta_text->number_of_text; i++)
    qtime_free(udta_text->text_list[i]);
  if (udta_text->text_list)
    qtime_free(udta_text->text_list);
  qtime_free(udta_text);
}

udta_text_t*
udta_text_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, udta_text_t *udtx_ptr)
{
  int64_t offset;
  udta_text_t *udtx;
 
  if (udtx_ptr) {
    udtx = udtx_ptr;
    udta_text_clean(udtx);
  } else {
    if ((udtx = udta_text_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  udtx->size = atom_head->size;
  udtx->type = atom_head->type;

  offset = 0;

  while (atom_head->end_offset > offset + 4) {
    uint16_t size;
    uint16_t lang;
    text_t *text;
    text_t **text_list;

    qtime_io_read16(qtio, &size);
    qtime_io_read16(qtio, &lang);
    text = (text_t*)qtime_malloc(size+1);
    if (!text) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      goto fail;
    }
    text->size = size;
    text->language_code = lang;
    qtime_io_read(qtio, &text->text[0], size-4);
    text->text[size-4] = '\0';

    text_list = (text_t**)qtime_realloc(udtx->text_list, sizeof(text_t*) * (udtx->number_of_text+1));
    if (!text_list) {
      qtime_free(text);
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      goto fail;
    }
    text_list[udtx->number_of_text] = text;
    udtx->text_list = text_list;
    udtx->number_of_text++;

    offset = qtime_io_get_offset(qtio);
  }

#ifndef NDEBUG
  if (atom_head->end_offset != offset) {
    fprintf(stderr, "udta_text_read_atom: end_offset doesn't suit, %lld, %lld.\n", atom_head->end_offset, offset);
  }
#endif /* DEBUG */

  return udtx;

fail:
  if (udtx_ptr)
    udta_text_clean(udtx);
  else
    udta_text_delete(udtx);
  qtime_error_atom_read(atom_head->type);
  return NULL;
}

udta_text_t*
udta_text_create(udta_text_t *udtx_ptr)
{
  udta_text_t *udtx;
  if (udtx_ptr) {
    udtx = udtx_ptr;
    udta_text_clean(udtx);
  } else {
    if ((udtx = udta_text_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return udtx;
}

int64_t
udta_text_calc_size(udta_text_t *udtx)
{
  int64_t size = 0;
  int i;

  if (udtx->number_of_text <= 0)
    return 0;

  for (i = 0; i < udtx->number_of_text; i++)
    size += udtx->text_list[i]->size;

  size += 8;
  udtx->size = size;

  return size;
}

int
udta_text_write_atom(qtime_io_t *qtio, udta_text_t *udta_text)
{
  int i;
  atom_head_t atom_head;

  if (udta_text->size <= 0)
    return QTIME_OK;

  if (udta_text->number_of_text <= 0)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = udta_text->size;
  atom_head.type = udta_text->type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(udta_text->type);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < udta_text->number_of_text; i++) {
    int size;
    size = udta_text->text_list[i]->size;
    qtime_io_write16(qtio, &udta_text->text_list[i]->size);
    qtime_io_write16(qtio, &udta_text->text_list[i]->language_code);
    qtime_io_write(qtio, &udta_text->text_list[i]->text[0], size-4);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(udta_text->type);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
udta_text_dump(const char *parent_types, udta_text_t *udtx)
{
  int len = strlen(parent_types);
  uint8_t type_str[5];
  uint8_t types[len+6];
  int i;

  qtime_type_to_str(udtx->type, type_str);
  sprintf(types, "%s.%.4s", parent_types, type_str);
  fprintf(stdout, "%s: size              %lld\n", types, (int64_t)udtx->size);
  for (i = 0; i < udtx->number_of_text; i++) {
    fprintf(stdout, "%s: size              %d\n", types, udtx->text_list[i]->size);
    fprintf(stdout, "%s: language code     %d\n", types, udtx->text_list[i]->language_code);
    fprintf(stdout, "%s: text '%s'\n", types, &udtx->text_list[i]->text[0]);
  }
}

int
udta_text_set_text(udta_text_t *udtx, char *text, int lang_code)
{
  text_t **list;
  int num_list;
  int text_length;
  text_t *tx;
  int i;
  int exist;
 
  if (!text)
    return QTIME_OK;

  text_length = strlen(text);
  if (text_length <= 0)
    return QTIME_OK;
  if (text_length > 255-4) {
    qtime_error_debug_info(QTIME_ERROR);
    return QTIME_ERROR;
  }

  tx = NULL;
  exist = -1;
  for (i = 0; i < udtx->number_of_text; i++) {
    if (udtx->text_list[i]->language_code == lang_code) {
      tx = udtx->text_list[i];
      exist = i;
      break;
    }
  }

  tx = (text_t*)qtime_realloc(tx, text_length+4+1);
  if (!tx) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    return QTIME_ERROR_MEMORY;
  }

  tx->size = text_length + 4;
  tx->language_code = lang_code;
  memcpy(&tx->text[0], text, text_length);
  tx->text[text_length] = '\0';

  if (exist > -1) {
    udtx->text_list[exist] = tx;
    return QTIME_OK;
  }

  list = udtx->text_list;
  num_list = udtx->number_of_text;

  list = (text_t**)qtime_realloc(list, sizeof(text_t*)*(num_list+1));
  if (!list) {
    qtime_free(tx);
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    return QTIME_ERROR_MEMORY;
  }

  list[num_list] = tx;
  udtx->text_list = list;
  num_list++;
  udtx->number_of_text = num_list;

  return QTIME_OK;
}

const char*
udta_text_get_text(udta_text_t *udtx, int lang_code)
{
  int i;

  for (i = 0; i < udtx->number_of_text; i++)
    if (udtx->text_list[i]->language_code == lang_code)
      return (const char*)&(udtx->text_list[i]->text[0]);
  return NULL;
}

/* udta_ctrl */

void
udta_ctrl_init(udta_ctrl_t *udta_ctrl)
{
  memset(udta_ctrl, 0, sizeof(udta_ctrl_t));
}

void
udta_ctrl_clean(udta_ctrl_t *udta_ctrl)
{
  if (udta_ctrl->type == QTIME_TYPE_NAME)
    if (udta_ctrl->data.name)
      qtime_free(udta_ctrl->data.name);

  if (udta_ctrl->type == QTIME_TYPE_PTV)
    if (udta_ctrl->data.ptv)
      qtime_free(udta_ctrl->data.ptv);

  udta_ctrl_init(udta_ctrl);
}

udta_ctrl_t*
udta_ctrl_new(void)
{
  udta_ctrl_t *udta_ctrl;
  udta_ctrl = (udta_ctrl_t*)qtime_malloc(sizeof(udta_ctrl_t));
  if (!udta_ctrl)
    return NULL;
  udta_ctrl_init(udta_ctrl);
  return udta_ctrl;
}

void
udta_ctrl_delete(udta_ctrl_t *udta_ctrl)
{
  if (udta_ctrl->type == QTIME_TYPE_NAME)
    if (udta_ctrl->data.name)
      qtime_free(udta_ctrl->data.name);

  if (udta_ctrl->type == QTIME_TYPE_PTV)
    if (udta_ctrl->data.ptv)
      qtime_free(udta_ctrl->data.ptv);

  qtime_free(udta_ctrl);
}

udta_ctrl_t*
udta_ctrl_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, udta_ctrl_t *udct_ptr)
{
  udta_ctrl_t *udct;

  if (udct_ptr) {
    udct = udct_ptr;
    udta_ctrl_clean(udct);
  } else {
    if ((udct = udta_ctrl_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  udct->size = atom_head->size;
  udct->type = atom_head->type;

  switch (atom_head->type) {
    case QTIME_TYPE_WLOC:
      qtime_io_read16(qtio, &udct->data.wloc[0]);
      qtime_io_read16(qtio, &udct->data.wloc[1]);
      break;
    case QTIME_TYPE_LOOP:
      qtime_io_read32(qtio, &udct->data.loop);
      break;
    case QTIME_TYPE_SELO:
#ifndef NDEBUG
      qtime_error_debug_info(QTIME_ERROR);
      fprintf(stderr, "'SelO' atom size %d\n", (int)atom_head->body_size);
#endif /* DEBUG */
      break;
    case QTIME_TYPE_ALLF:
#ifndef NDEBUG
      qtime_error_debug_info(QTIME_ERROR);
      fprintf(stderr, "'AllF' size %d\n", (int)atom_head->body_size);
#endif /* DEBUG */
      break;
    case QTIME_TYPE_PTV:
      udct->data.ptv = (ptv_t*) qtime_malloc(sizeof(ptv_t));
      if (!udct->data.ptv) {
        qtime_error_debug_info(QTIME_ERROR_MEMORY);
	atom_head->error_code = QTIME_ERROR_MEMORY;
	goto fail;
      }
      qtime_io_read16(qtio, &udct->data.ptv->display_size);
      qtime_io_read16(qtio, &udct->data.ptv->reserved1);
      qtime_io_read16(qtio, &udct->data.ptv->reserved2);
      qtime_io_read(qtio, &udct->data.ptv->slide_show, 1);
      qtime_io_read(qtio, &udct->data.ptv->play_on_open, 1);
      break;
    case QTIME_TYPE_NAME:
      udct->data.name = (char*)qtime_malloc(atom_head->body_size + 1);
      if (!udct->data.name) {
        qtime_error_debug_info(QTIME_ERROR_MEMORY);
	atom_head->error_code = QTIME_ERROR_MEMORY;
	goto fail;
      }
      qtime_io_read(qtio, &udct->data.name, atom_head->body_size);
      udct->data.name[atom_head->body_size] = '\0';
      break;
    default:
      qtime_error_unknown_atom(QTIME_TYPE_UDTA, atom_head->type, (int32_t)atom_head->size);
      break;
  }

#ifndef NDEBUG
  if (atom_head->end_offset != qtime_io_get_offset(qtio)) {
    fprintf(stderr, "'%.4s' end_offset doesn't suit, %lld, %lld.\n", (char*)&atom_head->type, atom_head->end_offset, qtime_io_get_offset(qtio));
  }
#endif

  return udct;

fail:
  if (udct_ptr)
    udta_ctrl_clean(udct);
  else
    udta_ctrl_delete(udct);
  qtime_error_atom_read(atom_head->type);
  return NULL;
}

int64_t
udta_ctrl_calc_size(udta_ctrl_t *udct)
{
  int64_t size = 0;

  switch (udct->type) {
    case QTIME_TYPE_WLOC:
      size = 2 * 2;
      break;
    case QTIME_TYPE_LOOP:
      size = 4;
      break;
    case QTIME_TYPE_SELO:
      return 0;
      break;
    case QTIME_TYPE_ALLF:
      return 0;
      break;
    case QTIME_TYPE_NAME:
      size = strlen(udct->data.name);
      break;
    case QTIME_TYPE_PTV:
      size = 2 * 3 + 2;
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
      qtime_error_illegal_atom(QTIME_TYPE_UDTA, udct->type);
      break;
  }
  if (size <= 0) return 0;
  size += 8;
  udct->size = size;

  return size;
}

udta_ctrl_t*
udta_ctrl_create(udta_ctrl_t *udct_ptr)
{
  udta_ctrl_t *udct;
  if (udct_ptr) {
    udct = udct_ptr;
    udta_ctrl_clean(udct);
  } else {
    if ((udct = udta_ctrl_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return udct;
}

int
udta_ctrl_write_atom(qtime_io_t *qtio, udta_ctrl_t *udct)
{
  atom_head_t atom_head;

  if (udct->size <= 8)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = udct->size;
  atom_head.type = udct->type;

  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(udct->type);
    return QTIME_ERROR_ATOM_WRITE;
  }

  switch (udct->type) {
    case QTIME_TYPE_WLOC:
      qtime_io_write16(qtio, &udct->data.wloc[0]);
      qtime_io_write16(qtio, &udct->data.wloc[1]);
      break;
    case QTIME_TYPE_LOOP:
      qtime_io_write32(qtio, &udct->data.loop);
      break;
    case QTIME_TYPE_SELO:
      break;
    case QTIME_TYPE_ALLF:
      break;
    case QTIME_TYPE_PTV:
      qtime_io_write16(qtio, &udct->data.ptv->display_size);
      qtime_io_write16(qtio, &udct->data.ptv->reserved1);
      qtime_io_write16(qtio, &udct->data.ptv->reserved2);
      qtime_io_write(qtio, &udct->data.ptv->slide_show, 1);
      qtime_io_write(qtio, &udct->data.ptv->play_on_open, 1);
      break;
    case QTIME_TYPE_NAME:
      qtime_io_write(qtio, &udct->data.name, strlen(udct->data.name));
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
      qtime_error_illegal_atom(QTIME_TYPE_UDTA, udct->type);
      break;
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(udct->type);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
udta_ctrl_dump(const char *parent_types, udta_ctrl_t *udct)
{
  char fcc[5];
  int len = strlen(parent_types);
  uint8_t types[len+6];

  qtime_type_to_str(udct->type, fcc);
  sprintf(types, "%s.%.4s", parent_types, fcc);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)udct->size);
  switch (udct->type) {
    case QTIME_TYPE_WLOC:
      fprintf(stdout, "%s: x,y          %d,%d\n", types, udct->data.wloc[0], udct->data.wloc[1]);
      break;
    case QTIME_TYPE_LOOP:
      fprintf(stdout, "%s: loop         %d\n", types, udct->data.loop);
      break;
    case QTIME_TYPE_SELO:
      break;
    case QTIME_TYPE_ALLF:
      break;
    case QTIME_TYPE_PTV:
      fprintf(stdout, "%s: display size %d\n", types, udct->data.ptv->display_size);
      fprintf(stdout, "%s: slide show   %d\n", types, udct->data.ptv->slide_show);
      fprintf(stdout, "%s: play on open %d\n", types, udct->data.ptv->play_on_open);
      break;
    case QTIME_TYPE_NAME:
      fprintf(stdout, "%s: name         '%s'\n", types, udct->data.name);
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
      qtime_error_illegal_atom(QTIME_TYPE_UDTA, udct->type);
      break;
  }
}

/* udta */

void
udta_init(udta_t *udta)
{
  atom_init(&udta->atom);
  udta->atom.type = QTIME_TYPE_UDTA;
  udta->number_of_udta_text = 0;
  udta->udta_text = NULL;
  udta->number_of_udta_ctrl = 0;
  udta->udta_ctrl = NULL;
//  udta->hnti = NULL;
//  udta->hinf = NULL;
}

void
udta_clean(udta_t *udta)
{
  int i;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_UDTA, udta->atom.type)) {
    return;
  }
#endif
  if (udta->udta_text || udta->number_of_udta_text > 0) {
    for (i = 0; i < udta->number_of_udta_text; i++)
      udta_text_delete(udta->udta_text[i]);
    qtime_free(udta->udta_text);
  }
  udta_init(udta);
}

udta_t*
udta_new(void)
{
  udta_t *udta;
  udta = (udta_t*)qtime_malloc(sizeof(udta_t));
  if (!udta)
    return NULL;
  udta_init(udta);
  return udta;
}

void
udta_delete(udta_t *udta)
{
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_UDTA, udta->atom.type)) {
    return;
  }
#endif
  udta_clean(udta);
  qtime_free(udta);
}

static int
add_udta_text(udta_t *udta, udta_text_t *udtx)
{
  udta_text_t **list;
  int num;

  num = udta->number_of_udta_text;
  list = udta->udta_text;
  list = (udta_text_t**)qtime_realloc(list, sizeof(udta_text_t*) * (num+1));
  if (!list) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    return QTIME_ERROR_MEMORY;
  }
  list[num] = udtx;
  udta->udta_text = list;
  num++;
  udta->number_of_udta_text = num;
  return QTIME_OK;
}

static int
add_udta_ctrl(udta_t *udta, udta_ctrl_t *udct)
{
  udta_ctrl_t **list;
  int num;

  num = udta->number_of_udta_ctrl;
  list = udta->udta_ctrl;
  list = (udta_ctrl_t**)qtime_realloc(list, sizeof(udta_ctrl_t*) * (num+1));
  if (!list) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    return QTIME_ERROR_MEMORY;
  }
  list[num] = udct;
  udta->udta_ctrl = list;
  num++;
  udta->number_of_udta_ctrl = num;
  return QTIME_OK;
}

udta_t*
udta_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, udta_t *udta_ptr)
{
  atom_head_t subatom;
  udta_t *udta;
  udta_text_t *udtx;
  udta_ctrl_t *udct;
  int udct_read = 0;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_UDTA, atom_head->type)) {
    atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
    return NULL;
  }
#endif

  if (udta_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_UDTA, udta_ptr->atom.type)) {
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
    }
#endif
    udta = udta_ptr;
    udta_clean(udta);
  } else {
    if ((udta = udta_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  udta->atom.size = atom_head->size;
  udta->atom.type = QTIME_TYPE_UDTA;
  udta->atom.parent = atom_head->parent;

  if (atom_head->body_size < 8)
    return udta;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)udta;

  udtx = NULL;
  udct = NULL;

  while (atom_head->end_offset > (subatom.end_offset+4)) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    udct_read = 0;

    if (atom_head->parent->type == QTIME_TYPE_MOOV) {
      switch (subatom.type) {
        case QTIME_TYPE_WLOC:
        case QTIME_TYPE_LOOP:
        case QTIME_TYPE_SELO:
        case QTIME_TYPE_ALLF:
	  if ((udct = udta_ctrl_read_atom(qtio, &subatom, udct)) == NULL) {
            qtime_error_debug_info(subatom.error_code);
            atom_head->error_code = subatom.error_code;
	    goto fail;
	  }
	  if ((atom_head->error_code = add_udta_ctrl(udta, udct)) < 0) {
            qtime_error_debug_info(atom_head->error_code);
	    goto fail;
	  }
	  udct = NULL;
          udct_read = 1;
	  break;
      }
    }

    if (!udct_read && (subatom.type & INTER_TEXT_MASK) == (uint32_t)INTER_TEXT_TYPE) {
      if ((udtx = udta_text_read_atom(qtio, &subatom, udtx)) == NULL) {
        qtime_error_debug_info(subatom.error_code);
        atom_head->error_code = subatom.error_code;
	goto fail;
      }
      if ((atom_head->error_code = add_udta_text(udta, udtx)) < 0) {
        qtime_error_debug_info(atom_head->error_code);
        goto fail;
      }
      udtx = NULL;
    } else if (!udct_read) {
      switch (subatom.type) {
        case QTIME_TYPE_NAME:
	  if ((udct = udta_ctrl_read_atom(qtio, &subatom, udct)) == NULL) {
            qtime_error_debug_info(subatom.error_code);
            atom_head->error_code = subatom.error_code;
	    goto fail;
	  }
	  if ((atom_head->error_code = add_udta_ctrl(udta, udct)) < 0) {
            qtime_error_debug_info(atom_head->error_code);
	    goto fail;
	  }
	  udct = NULL;
	  break;
#if 0
        case QTIME_TYPE_HNTI:
	  break;
        case QTIME_TYPE_HINF:
	  break;
#endif
        default:
          qtime_error_unknown_atom(QTIME_TYPE_UDTA, subatom.type, (int32_t)subatom.size);
	  break;
      }
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  return udta;

fail:
  if (udta_ptr)
    udta_clean(udta);
  else
    udta_delete(udta);
  qtime_error_atom_read(QTIME_TYPE_UDTA);
  return NULL;
}

udta_t*
udta_create(udta_t *udta_ptr)
{
  udta_t *udta;

  if (udta_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_UDTA, udta_ptr->atom.type)) {
      return NULL;
    }
#endif
    udta = udta_ptr;
    udta_clean(udta);
  } else {
    if ((udta = udta_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return udta;
}

int64_t
udta_calc_size(udta_t *udta)
{
  int64_t size = 0;
  int i;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_UDTA, udta->atom.type)) {
    return 0;
  }
#endif

  if (udta->number_of_udta_text <= 0 && udta->number_of_udta_ctrl <= 0)
    return 0;

  for (i = 0; i < udta->number_of_udta_text; i++)
    size += udta_text_calc_size(udta->udta_text[i]);
  for (i = 0; i < udta->number_of_udta_ctrl; i++)
    size += udta_ctrl_calc_size(udta->udta_ctrl[i]);
#if 0
  if (udta->hnti)
    size += hnti_calc_size(udta->hnti);
  if (udta->hinf)
    size += hinf_calc_size(udta->hinf);
#endif
  size += 8;
  udta->atom.size = size;

  return size;
}

int
udta_write_atom(qtime_io_t *qtio, udta_t *udta)
{
  int i;
  atom_head_t atom_head;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_UDTA, udta->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif

  if (udta->atom.number_of_childs <= 0)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = udta->atom.size;
  atom_head.type = udta->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_UDTA);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < udta->number_of_udta_ctrl; i++) {
    if (udta->udta_ctrl[i]->type == QTIME_TYPE_NAME)
      udta_ctrl_write_atom(qtio, udta->udta_ctrl[i]);
    else if (udta->atom.parent->type == QTIME_TYPE_MOOV)
      udta_ctrl_write_atom(qtio, udta->udta_ctrl[i]);
  }

  for (i = 0; i < udta->number_of_udta_text; i++)
    udta_text_write_atom(qtio, udta->udta_text[i]);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_UDTA);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
udta_dump(const char *parent_types, udta_t *udta)
{
  int i;
  int len = strlen(parent_types);
  uint8_t type_str[5];
  uint8_t types[len+6];

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_UDTA, udta->atom.type)) {
    return;
  }
#endif
  qtime_type_to_str(udta->atom.type, type_str);
  sprintf(types, "%s.%.4s", parent_types, type_str);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)udta->atom.size);
  for (i = 0; i < udta->number_of_udta_ctrl; i++)
    udta_ctrl_dump(types, udta->udta_ctrl[i]);
  for (i = 0; i < udta->number_of_udta_text; i++)
    udta_text_dump(types, udta->udta_text[i]);
}

const char*
udta_get_text(udta_t *udta, uint32_t type, int lang_code)
{
  int i;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_UDTA, udta->atom.type)) {
    return NULL;
  }
#endif
  if (type == QTIME_TYPE_NAME) {
    for (i = 0; i < udta->number_of_udta_ctrl; i++)
      if (udta->udta_ctrl[i]->type == QTIME_TYPE_NAME)
	return udta->udta_ctrl[i]->data.name;
  }
  for (i = 0; i < udta->number_of_udta_text; i++)
    if (udta->udta_text[i]->type == type)
      return udta_text_get_text(udta->udta_text[i], lang_code);
  return NULL;
}

int
udta_set_text(udta_t *udta, uint32_t type, int lang_code, char *text)
{
  int i;
  udta_text_t *udtx = NULL;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_UDTA, udta->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif
  if (type == QTIME_TYPE_NAME) {
    int len;
    char *cp;
    udta_ctrl_t *udct = NULL;
    for (i = 0; i < udta->number_of_udta_ctrl; i++)
      if (udta->udta_ctrl[i]->type == QTIME_TYPE_NAME)
	udct = udta->udta_ctrl[i];
    if (!udct) {
      udct = udta_ctrl_new();
      if (!udct) {
	qtime_error_debug_info(QTIME_ERROR_MEMORY);
        return QTIME_ERROR_MEMORY;
      }
      udct->type = QTIME_TYPE_NAME;
    }
    len = strlen(text);
    cp = udct->data.name;
    if ((cp = (char*)qtime_realloc(cp, len+1)) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return QTIME_ERROR_MEMORY;
    }
    udct->data.name = cp;
    memcpy(udct->data.name, text, len);
    udct->data.name[len] = '\0';
    return QTIME_OK;
  }
  for (i = 0; i < udta->number_of_udta_text; i++)
    if (udta->udta_text[i]->type == type)
      udtx = udta->udta_text[i];
  if (!udtx) {
    if ((udtx = udta_text_new()) == NULL)
      return QTIME_ERROR_MEMORY;
    udtx->type = type;
  }
  return udta_text_set_text(udtx, text, lang_code);
}

