/* tref.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "tref.h"

/* track reference type*/

void
tref_type_init(tref_type_t *tref_type)
{
  tref_type->size = 0;
  tref_type->type = 0;
  tref_type->track_id_num= 0;
  tref_type->track_id = NULL;
}

void
tref_type_clean(tref_type_t *tref_type)
{
  if (tref_type->track_id)
    qtime_free(tref_type->track_id);
  tref_type_init(tref_type);
}

tref_type_t*
tref_type_new(void)
{
  tref_type_t *tref_type;
  tref_type = (tref_type_t*)qtime_malloc(sizeof(tref_type_t));
  if (!tref_type) return NULL;
  tref_type_init(tref_type);
  return tref_type;
}

void
tref_type_delete(tref_type_t *tref_type)
{
  if (!tref_type) return;

  if (tref_type->track_id) {
    qtime_free(tref_type->track_id);
  }
  qtime_free(tref_type);
}

tref_type_t*
tref_type_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, tref_type_t* tref_type_ptr)
{
  tref_type_t *tref_type;
  int i = 0;
  int id_num;
  int32_t *ids = NULL;

  switch (atom_head->type) {
    case QTIME_TYPE_TMCD:
    case QTIME_TYPE_CHAP:
    case QTIME_TYPE_SYNC:
    case QTIME_TYPE_SCPT:
    case QTIME_TYPE_SSRC:
    case QTIME_TYPE_HINT:
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
      qtime_error_illegal_atom(QTIME_TYPE_TREF, atom_head->type);
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
      break;
  }

  if (tref_type_ptr) {
    tref_type = tref_type_ptr;
    tref_type_clean(tref_type);
  } else {
    if ((tref_type = tref_type_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  tref_type->size = atom_head->size;
  tref_type->type = atom_head->type;

  id_num = atom_head->body_size / sizeof(int32_t);

  if (id_num <= 0) {
    return tref_type;
  }

  ids = tref_type->track_id;
  if ((ids = (int32_t*)qtime_realloc(ids, id_num * sizeof(int32_t))) == NULL) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    atom_head->error_code = QTIME_ERROR_MEMORY;
    goto fail;
  }
  tref_type->track_id_num = id_num;
  tref_type->track_id  = ids;

  for (i = 0; i < id_num; i++) {
    qtime_io_read32(qtio, &ids[i]);
  }

  return tref_type;

fail:
  if (tref_type_ptr)
    tref_type_clean(tref_type);
  else
    tref_type_delete(tref_type);
  qtime_error_atom_read(atom_head->type);
  return NULL;
}

tref_type_t*
tref_type_create(tref_type_t *tref_type_ptr)
{
  tref_type_t *tref_type;

  if (tref_type_ptr) {
    tref_type = tref_type_ptr;
    tref_type_clean(tref_type);
  } else {
    if ((tref_type = tref_type_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return tref_type;
}

int
tref_type_valid(tref_type_t *tref_type)
{
  if (tref_type->track_id && tref_type->track_id_num > 0) {
    return 1;
  }
  return 0;
}

int64_t
tref_type_calc_size(tref_type_t *tref_type)
{
  int64_t size = 0;
  if (!tref_type_valid(tref_type))
    return 0;
  size = tref_type->track_id_num * sizeof(int32_t);
  size += 8;
  tref_type->size = size;
  return size;
}

int
tref_type_write_atom(qtime_io_t *qtio, tref_type_t *tref_type)
{
  int i;
  atom_head_t atom_head;

  if (!tref_type_valid(tref_type))
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = tref_type->size;
  atom_head.type = tref_type->type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    goto fail;
  }

  for (i = 0; i < tref_type->track_id_num; i++)
    qtime_io_write32(qtio, &tref_type->track_id[i]);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    goto fail;
  }

fail:
  qtime_error_atom_write(tref_type->type);
  return QTIME_ERROR_ATOM_WRITE;
}

void
tref_type_dump(const char *parent_types, tref_type_t *tref_type)
{
  int i;
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];

  qtime_type_to_str(tref_type->type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  fprintf(stdout, "%s: size              %lld\n", types, (int64_t)tref_type->size);
  fprintf(stdout, "%s: track id          ", types);
  for (i = 0; i < tref_type->track_id_num; i++) {
    if (i == 0)
      fprintf(stdout, "%d", tref_type->track_id[i]);
    else
      fprintf(stdout, ",%d", tref_type->track_id[i]);
  }
  if (i == 0)
    fprintf(stdout, "none");
  fprintf(stdout, "\n");
}

/* tref */

void
tref_init(tref_t *tref)
{
  atom_init(&tref->atom);
  tref->atom.type = QTIME_TYPE_TREF;
  tref->tref_type_num = 0;
  tref->tref_type = NULL;
}

tref_t*
tref_new(void)
{
  tref_t *tref;
  tref = (tref_t*)qtime_malloc(sizeof(tref_t));
  if (!tref) return NULL;
  tref_init(tref);
  return tref;
}

void
tref_clean(tref_t *tref)
{
  int i;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_TREF, tref->atom.type)) {
    return;
  }
#endif
  if (tref->tref_type) {
    for (i = 0; i < tref->tref_type_num; i++)
      tref_type_delete(tref->tref_type[i]);
    qtime_free(tref->tref_type);
  }
  tref_init(tref);
}

void
tref_delete(tref_t *tref)
{
  int i;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_TREF, tref->atom.type)) {
    return;
  }
#endif
  if (tref->tref_type) {
    for (i = 0; i < tref->tref_type_num; i++)
      tref_type_delete(tref->tref_type[i]);
    qtime_free(tref->tref_type);
  }
  qtime_free(tref);
}

int
tref_add_list(tref_t *tref, tref_type_t *tref_type)
{
  int type_num = 0;
  tref_type_t **types = NULL;

  type_num = tref->tref_type_num;
  types = tref->tref_type;
  types =(tref_type_t**)qtime_realloc(types,sizeof(tref_type_t*)*(type_num+1));
  if (!types) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    return QTIME_ERROR_MEMORY;
  }
  types[type_num] = tref_type;
  type_num++;
  tref->tref_type_num = type_num;
  tref->tref_type = types;
  return QTIME_OK;
}

tref_t*
tref_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, tref_t *tref_ptr)
{
  atom_head_t subatom;
  tref_t *tref;
  tref_type_t *tref_type;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_TREF, atom_head->type)) {
    atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
    return NULL;
  }
#endif

  if (tref_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_TREF, tref_ptr->atom.type)) {
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
    }
#endif
    tref = tref_ptr;
    tref_clean(tref);
  } else {
    if ((tref = tref_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  tref->atom.size = atom_head->size;
  tref->atom.type = QTIME_TYPE_TREF;
  tref->atom.parent = atom_head->parent;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)tref;

  tref_type = NULL;

  while (atom_head->end_offset > (subatom.end_offset)) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    switch (subatom.type) {
      case QTIME_TYPE_TMCD:
      case QTIME_TYPE_CHAP:
      case QTIME_TYPE_SYNC:
      case QTIME_TYPE_SCPT:
      case QTIME_TYPE_SSRC:
      case QTIME_TYPE_HINT:
	if ((tref_type = tref_type_read_atom(qtio, &subatom, tref_type)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	if ((atom_head->error_code = tref_add_list(tref, tref_type)) < 0) {
	  qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
	  goto fail;
	}
	tref_type = NULL;
	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_TREF, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  return tref;

fail:
  if (tref_ptr)
    tref_clean(tref);
  else
    tref_delete(tref);
  qtime_error_atom_read(QTIME_TYPE_TREF);
  return NULL;
}

tref_t*
tref_create(tref_t *tref_ptr)
{
  tref_t *tref;

  if (tref_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_TREF, tref_ptr->atom.type)) {
      return NULL;
    }
#endif
    tref = tref_ptr;
    tref_clean(tref);
  } else {
    if ((tref = tref_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return tref;
}

int64_t
tref_calc_size(tref_t *tref)
{
  int64_t size = 0;
  int i;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_TREF, tref->atom.type)) {
    return 0;
  }
#endif

  if (tref->tref_type_num <= 0)
    return 0;
  for (i = 0; i < tref->tref_type_num; i++)
    size += tref_type_calc_size(tref->tref_type[i]);
  size += 8;
  tref->atom.size = size;
  return size;
}

int
tref_write_atom(qtime_io_t *qtio, tref_t *tref)
{
  int i;
  atom_head_t atom_head;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_TREF, tref->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif

  if (tref->tref_type_num <= 0)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = tref->atom.size;
  atom_head.type = tref->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_TREF);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < tref->tref_type_num; i++) {
    tref_type_write_atom(qtio, tref->tref_type[i]);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_TREF);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
tref_dump(const char *parent_types, tref_t *tref)
{
  int i;
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];

  qtime_type_to_str(tref->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)tref->atom.size);
  for (i = 0; i < tref->tref_type_num; i++)
    tref_type_dump(types, tref->tref_type[i]);
}

