/* stss.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_sample_info.h"
#include "atom.h"
#include "stss.h"

void
stss_init(stss_t *stss)
{
  atom_init(&stss->atom);
  stss->atom.type = QTIME_TYPE_STSS;
  stss->version = 0;
  qtime_flags_set(stss->flags, 0);
  stss->number_of_entries = 0;
  stss->table_max = 0;
  stss->table = NULL;

  stss->idx = 0;
  stss->all_key = 0;
}

void
stss_clean(stss_t *stss)
{
  qtime_error_type_check(QTIME_TYPE_STSS, stss->atom.type)
  if (stss->table)
    qtime_free(stss->table);
  stss_init(stss);
}

stss_t*
stss_new(void)
{
  stss_t *stss;
  stss = (stss_t*)qtime_malloc(sizeof(stss_t));
  if (!stss) return NULL;
  stss_init(stss);
  return stss;
}

void
stss_delete(stss_t *stss)
{
  qtime_error_type_check(QTIME_TYPE_STSS, stss->atom.type)
  if (stss->table)
    qtime_free(stss->table);
  qtime_free(stss);
}

int
stss_valid(stss_t *stss)
{
  if (!stss || stss->number_of_entries <= 0)
    return 0;
  return 1;
}

stss_t*
stss_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, stss_t *stss_ptr)
{
  int table_byte_size;
  int i, table_num;
  int32_t *table;
  stss_t *stss;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_STSS, atom_head->type, NULL)
  if (stss_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STSS, stss_ptr->atom.type, NULL)
    stss = stss_ptr;
    stss_clean(stss);
  } else {
    if ((stss = stss_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  stss->atom.size = atom_head->size;
  stss->atom.type = atom_head->type;
  stss->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &stss->version, 1);
  qtime_io_read(qtio, &stss->flags, 3);
  qtime_io_read32(qtio, &stss->number_of_entries);

#ifndef NDEBUG
  if (stss->number_of_entries < 0) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: stss_read_atom: error, number_of_entries value is negative, %d.\n", stss->number_of_entries);
  }
#endif

  table_num = stss->number_of_entries;
  table_byte_size = table_num * sizeof(int32_t);
  table = (int32_t*)qtime_malloc(table_byte_size);
  if (!table) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    atom_head->error_code = QTIME_ERROR_MEMORY;
    goto fail;
  }
  stss->table = table;
  stss->table_max = table_num;

  for (i = 0; i < table_num; i++, table++) {
    qtime_io_read32(qtio, table);
  }

  return stss;

fail:
  if (stss_ptr)
    stss_clean(stss);
  else
    stss_delete(stss);
  qtime_error_atom_read(QTIME_TYPE_STSS);
  return NULL;
}

stss_t*
stss_create(stss_t *stss_ptr)
{
  stss_t *stss;

  if (stss_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STSS, stss_ptr->atom.type, NULL)
    stss = stss_ptr;
    stss_clean(stss);
  } else {
    if ((stss = stss_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return stss;
}

int64_t
stss_calc_size(stss_t *stss)
{
  int64_t size;

  qtime_error_type_check_v(QTIME_TYPE_STSS, stss->atom.type, 0)

  if (stss->number_of_entries <= 0)
    return 0;

  size = (int64_t)(8 + STSS_PROP_SIZE) +
          (int64_t)stss->number_of_entries * (int64_t)sizeof(uint32_t);

  if (size & SIZE64MASK)
    size += 8;

  stss->atom.size = size;

  return size;
}

int
stss_write_atom(qtime_io_t *qtio, stss_t *stss)
{
  int i;
  int table_num;
  uint32_t *table;
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_STSS, stss->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  if (stss->number_of_entries <= 0)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = stss->atom.size;
  atom_head.type = stss->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STSS);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &stss->version, 1);
  qtime_io_write(qtio, &stss->flags, 3);
  qtime_io_write32(qtio, &stss->number_of_entries);

  table_num = stss->number_of_entries;
  table = stss->table;

  for (i = 0; i < table_num; i++, table++) {
    qtime_io_write32(qtio, table);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STSS);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
stss_dump(const char *parent_types, stss_t *stss)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  if (!stss_valid(stss))
    return;

  qtime_type_to_str(stss->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(stss->flags, &flags);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)stss->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, stss->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: number of entries %d\n", types, stss->number_of_entries);

#if 0
  {
    int i;
    for (i = 0; i < stss->number_of_entries; i++)
    fprintf(stdout, "%s: table[%d] = %u\n", types, i, stss->table[i]);
  }
#endif
}

int32_t
stss_add_keyframe(stss_t *stss, uint32_t keyframe_count)
{
  uint32_t *table;

#ifndef NDEBUG
  if ((uint32_t)stss->number_of_entries > (MAX_INT32-(4096*2)) ||
      stss->all_key > (MAX_INT32-(4096*2))) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stss' atom, entries = %d, all_key = %u.\n", stss->number_of_entries, stss->all_key);
  }
#endif

  if (stss->number_of_entries == 0) {
    if (keyframe_count == stss->all_key) {
      stss->all_key++;
      return stss->number_of_entries;
    }
  }

  if (stss->all_key) {
    uint32_t i, table_max = stss->all_key + 1;
    table = (uint32_t*)qtime_realloc(stss->table, sizeof(uint32_t) * table_max);
    if (!table)
      return QTIME_ERROR_MEMORY;
    for (i = 0; i < stss->all_key; i++)
      table[i] = i+1;
    stss->table_max = table_max;
    stss->table = table;
    stss->number_of_entries = stss->all_key;
    stss->all_key = 0;
  }

  if ((stss->number_of_entries+1) > stss->table_max) {
    int table_max = stss->table_max + 4096;
    table = (uint32_t*)qtime_realloc(stss->table, sizeof(uint32_t) * table_max);
    if (!table)
      return QTIME_ERROR_MEMORY;
    stss->table_max = table_max;
    stss->table = table;
  }
  stss->table[stss->number_of_entries] = keyframe_count + 1;
  stss->number_of_entries++;
  return stss->number_of_entries;
}

#if 0
int32_t
stss_set_keyframe(stss_t *stss, int32_t number, uint32_t keyframe_number)
{
//  if (number < 0 || number >= stss->number_of_entries) {
  if (number >= stss->number_of_entries) {
    fprintf(stderr, "stss_set: invalid entry number, %d, %d\n", number, stss->number_of_entries);
    return QTIME_ERROR;
  }
  stss->table[number] = keyframe_number;
  return number;
}

uint32_t
stss_get_keyframe(stss_t *stss, int32_t index)
{
#ifndef NDEBUG
  if (index < 0 || index >= stss->number_of_entries) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: requested index is over the maximum entry in 'stss' atom, index = %d, entries = %d.\n", index, stss->number_of_entries);
  }
#endif
  if (index < 0)
    return stss->table[0];
  else if (index >= stss->number_of_entries)
    return stss->table[stss->number_of_entries-1];
  return stss->table[index];
}
#endif

int32_t
stss_is_keyframe(stss_t *stss, uint32_t count)
{
  int32_t idx, max;
  uint32_t *table;

  if (!stss || stss->number_of_entries <= 0)
    return 1;

  // 0 origin -> 1 origin
  count++;

  idx = stss->idx;
  table = stss->table;
  max = stss->number_of_entries-1;

  if (count < table[idx]) {
    for (; idx > 0; idx--)
      if (count >= table[idx])
	break;
  } else {
//    for (; idx < stss->number_of_entries-1; idx++)
    for (; idx < max; idx++)
      if (count < table[idx+1])
	break;
  }
  stss->idx = idx;

  return (table[idx] == count);
}

uint32_t
stss_get_prev_keyframe(stss_t *stss, uint32_t count)
{
  int32_t idx, max;
  uint32_t *table;

  if (!stss || stss->number_of_entries <= 0)
    return 1;

  count++;
  idx = stss->idx;
  table = stss->table;
  max = stss->number_of_entries-1;

  if (count < table[idx]) {
    for (; idx > 0; idx--)
      if (count >= table[idx])
	break;
  } else {
    for (; idx < max; idx++)
      if (count < table[idx+1])
	break;
  }
  stss->idx = idx;

  if (idx == 0)
    return table[idx];
//  else if (count == table[idx])
//    return table[idx-1];
  
  return table[idx];
}

uint32_t
stss_get_next_keyframe(stss_t *stss, uint32_t count)
{
  int32_t idx, max;
  uint32_t *table;

  if (!stss || stss->number_of_entries <= 0)
    return 1;

  count++;
  idx = stss->idx;
  table = stss->table;
  max = stss->number_of_entries-1;

  if (count < table[idx]) {
    for (; idx > 0; idx--)
      if (count >= table[idx])
	break;
  } else {
    for (; idx < max; idx++)
      if (count < table[idx+1])
	break;
  }
  stss->idx = idx;

  if (count == table[idx] || idx == max)
//  if (idx == max)
    return table[idx];
  
  return table[idx+1];
}

int
stss_get_sample_info(stss_t *stss, qtime_sample_info_t *spinfo)
{
  if (stss_is_keyframe(stss, spinfo->count))
    spinfo->flag |= QTIME_SAMPLE_INFO_FLAG_KEYFRAME;
  else
    spinfo->flag &= ~QTIME_SAMPLE_INFO_FLAG_KEYFRAME;
  return QTIME_OK;
}

int
stss_add_sample_info(stss_t *stss, qtime_sample_info_t *spinfo)
{
  if (spinfo->flag & QTIME_SAMPLE_INFO_FLAG_KEYFRAME)
    stss_add_keyframe(stss, spinfo->count);
  return QTIME_OK;
}

