/* stsd.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "qtime_sample_info.h"
#include "atom.h"
#include "stsd.h"
#include "sample_desc.h"
#include "video_desc.h"
#include "sound_desc.h"

void
stsd_init(stsd_t *stsd)
{
  atom_init(&stsd->atom);
  stsd->atom.type = QTIME_TYPE_STSD;
  stsd->version = 0;
  qtime_flags_set(stsd->flags, 0);
  stsd->number_of_entries = 0;
  stsd->table = NULL;
  stsd->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
  stsd->start_offset = 0;
}

void
stsd_clean(stsd_t *stsd)
{
  int i;
  qtime_error_type_check(QTIME_TYPE_STSD, stsd->atom.type)
  for (i = 0; i < stsd->number_of_entries; i++) {
    switch (stsd->media_type) {
      case QTIME_MEDIA_TYPE_VIDEO:
        video_desc_delete((video_desc_t*)stsd->table[i]);
        break;
      case QTIME_MEDIA_TYPE_SOUND:
	sound_desc_delete((sound_desc_t*)stsd->table[i]);
        break;
      default:
        qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
        qtime_error_invalid_media(QTIME_TYPE_STSD, stsd->media_type, stsd->media_type);
        break;
    }
  }
  if (stsd->table)
    qtime_free(stsd->table);
  stsd_init(stsd);
}

stsd_t*
stsd_new(void)
{
  stsd_t *stsd;
  stsd = (stsd_t*)qtime_malloc(sizeof(stsd_t));
  if (!stsd) return NULL;
  stsd_init(stsd);
  return stsd;
}

void
stsd_delete(stsd_t *stsd)
{
  qtime_error_type_check(QTIME_TYPE_STSD, stsd->atom.type)
  stsd_clean(stsd);
  qtime_free(stsd);
}

stsd_t*
stsd_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, stsd_t *stsd_ptr)
{
  uint32_t media_type;
  stsd_t *stsd;
  atom_head_t subatom;
  int sample_desc_index, num_ent;
  sample_desc_t *sample_desc;
  video_desc_t *video_desc;
  sound_desc_t *sound_desc;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_STSD, atom_head->type, NULL)
  if (stsd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STSD, stsd_ptr->atom.type, NULL)
    stsd = stsd_ptr;
    stsd_clean(stsd);
  } else {
    if ((stsd = stsd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  stsd->atom.size = atom_head->size;
  stsd->atom.type = QTIME_TYPE_STSD;
  stsd->atom.parent = atom_head->parent;
  stsd->media_type = atom_head->media_type;

  if (stsd->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    stsd->start_offset = atom_head->start_offset;
    return stsd;
  } else {
    stsd->start_offset = 0;
  }

  qtime_io_read(qtio, &stsd->version, 1);
  qtime_io_read(qtio, &stsd->flags, 3);
  qtime_io_read32(qtio, &num_ent);

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)stsd;
  subatom.media_type = atom_head->media_type;

  media_type = atom_head->media_type;
  sample_desc_index = 0;
  video_desc = NULL;
  sound_desc = NULL;

  while (atom_head->end_offset >= (subatom.end_offset+8)) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    sample_desc = NULL;
    switch (media_type) {
      case QTIME_MEDIA_TYPE_VIDEO:
	if ((video_desc = video_desc_read_atom(qtio, &subatom, video_desc)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	sample_desc = (sample_desc_t*)video_desc;
	video_desc = NULL;
	break;
      case QTIME_MEDIA_TYPE_SOUND:
	if ((sound_desc = sound_desc_read_atom(qtio, &subatom, sound_desc)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	sample_desc = (sample_desc_t*)sound_desc;
	sound_desc = NULL;
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
	qtime_error_invalid_media(QTIME_TYPE_STSD, atom_head->media_type, atom_head->media_type);
	break;
    }

    if (sample_desc) {
      if ((atom_head->error_code = stsd_add_sample_desc(stsd, sample_desc)) < 0){
	qtime_error_debug_info(atom_head->error_code);
	goto fail;
      }
      sample_desc_index++;
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

#ifndef NDEBUG
  if (num_ent != stsd->number_of_entries) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: 'stsd' atom's number_of_entries doesn't suit, %d, %d\n", stsd->number_of_entries, num_ent);
  }
#endif

  return stsd;

fail:
  if (stsd_ptr)
    stsd_clean(stsd);
  else
    stsd_delete(stsd);
  qtime_error_atom_read(QTIME_TYPE_STSD);
  return NULL;
}

int
stsd_read_atom_post(qtime_io_t *qtio, stsd_t *stsd)
{
  atom_head_t atom_head;

  if (stsd->start_offset == 0)
    return QTIME_OK;

#ifndef NDEBUG
  if (stsd->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: cannot specified 'stsd' atom's media type.\n");
  }
#endif
  qtime_io_set_offset(qtio, stsd->start_offset);
  atom_head_init(&atom_head);
  atom_read_header(qtio, &atom_head);
  atom_head.parent = stsd->atom.parent;
  atom_head.media_type = stsd->media_type;

  stsd_read_atom(qtio, &atom_head, stsd);

  atom_read_footer(qtio, &atom_head);

  return QTIME_OK;
}

stsd_t*
stsd_create(stsd_t *stsd_ptr)
{
  stsd_t *stsd;

  if (stsd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STSD, stsd_ptr->atom.type, NULL)
    stsd = stsd_ptr;
    stsd_clean(stsd);
  } else {
    if ((stsd = stsd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return stsd;
}

int64_t
stsd_calc_size(stsd_t *stsd)
{
  int i;
  int64_t size;

  qtime_error_type_check_v(QTIME_TYPE_STSD, stsd->atom.type, 0)
  size = STSD_PROP_SIZE;
  switch (stsd->media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      for (i = 0; i < stsd->number_of_entries; i++)
        size += video_desc_calc_size((video_desc_t*)stsd->table[i]);
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      for (i = 0; i < stsd->number_of_entries; i++)
	size += sound_desc_calc_size((sound_desc_t*)stsd->table[i]);
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
      qtime_error_invalid_media(QTIME_TYPE_STSD, stsd->media_type, stsd->media_type);
      break;
  }
  size += 8;
  stsd->atom.size = size;
  return size;
}

int
stsd_write_atom(qtime_io_t *qtio, stsd_t *stsd)
{
  atom_head_t atom_head;
  int i;

  qtime_error_type_check_v(QTIME_TYPE_STSD, stsd->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  atom_head_init(&atom_head);
  atom_head.size = stsd->atom.size;
  atom_head.type = stsd->atom.type;

  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STSD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &stsd->version, 1);
  qtime_io_write(qtio, &stsd->flags, 3);
  qtime_io_write32(qtio, &stsd->number_of_entries);

  switch (stsd->media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      for (i = 0; i < stsd->number_of_entries; i++)
        video_desc_write_atom(qtio, (video_desc_t*)stsd->table[i]);
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      for (i = 0; i < stsd->number_of_entries; i++)
	sound_desc_write_atom(qtio, (sound_desc_t*)stsd->table[i]);
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
      qtime_error_invalid_media(QTIME_TYPE_STSD, stsd->media_type, stsd->media_type);
      break;
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STSD);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
stsd_dump(const char *parent_types, stsd_t *stsd)
{
  int i, len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  qtime_type_to_str(stsd->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(stsd->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)stsd->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, stsd->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: number_of_entries %d\n", types, stsd->number_of_entries);
  switch (stsd->media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      for (i = 0; i < stsd->number_of_entries; i++)
	video_desc_dump(types, (video_desc_t*)stsd->table[i]);
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      for (i = 0; i < stsd->number_of_entries; i++)
	sound_desc_dump(types, (sound_desc_t*)stsd->table[i]);
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
      qtime_error_invalid_media(QTIME_TYPE_STSD, stsd->media_type, stsd->media_type);
      break;
  }
}

int
stsd_get_number_of_descriptions(stsd_t *stsd)
{
  return stsd->number_of_entries;
}

sample_desc_t*
stsd_get_sample_desc(stsd_t *stsd, int idx)
{
  sample_desc_t *sd;

  if (stsd->number_of_entries <= 0)
    return NULL;
  if (idx <= 0 || idx > stsd->number_of_entries)
    return NULL;
  sd = stsd->table[idx-1];

  return sd;
}

int
stsd_add_sample_desc(stsd_t *stsd, sample_desc_t *sample_desc)
{
  sample_desc_t **table;

  table = (sample_desc_t**)qtime_realloc(stsd->table, sizeof(sample_desc_t*) * (stsd->number_of_entries+1));
  if (!table) {
    return QTIME_ERROR_MEMORY;
  }
  stsd->table = table;
  stsd->table[stsd->number_of_entries] = sample_desc;
  stsd->number_of_entries++;
  return stsd->number_of_entries;
}

uint32_t
stsd_get_media_type(stsd_t *stsd)
{
  return stsd->media_type;
}

int
stsd_set_media_type(stsd_t *stsd, uint32_t media_type)
{
  qtime_error_media_check_v(QTIME_TYPE_STSD, stsd->media_type, media_type, QTIME_ERROR_INVALID_MEDIA);
  stsd->media_type = media_type;
  return QTIME_OK;
}

int
stsd_get_sample_info(stsd_t *stsd, qtime_sample_info_t *spinfo)
{
  sound_desc_t *sound_desc;

  if (spinfo->sample_description_id > stsd->number_of_entries) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: sample_description_id is over the maximum entry in 'stsd' atom, id %d, max entry %d.\n", spinfo->sample_description_id, stsd->number_of_entries);
    return QTIME_ERROR;
  }

  spinfo->sample_desc = stsd->table[spinfo->sample_description_id-1];
  spinfo->media_type = stsd->media_type;

  switch (stsd->media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      spinfo->bytes_per_sample = 0;
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      sound_desc = (sound_desc_t*)spinfo->sample_desc;
      if (sound_desc->version == 0) {
	spinfo->bytes_per_sample = sound_desc->number_of_channels *
	                           (sound_desc->sample_size/8);
#ifndef NDEBUG
        if (!spinfo->bytes_per_sample) {
          qtime_error_debug_info(QTIME_ERROR);
          fprintf(stderr, "QTIME_ERROR: qtime_sample_get_info: bytes_per_sample is '0'.\n");
        }
#endif
      } else {
	spinfo->bytes_per_sample = 0;
      }
      break;
    default:
      spinfo->bytes_per_sample = 0;
      break;
  }
  return QTIME_OK;
}

int
stsd_add_sample_info(stsd_t *stsd, qtime_sample_info_t *spinfo)
{
  sound_desc_t *sound_desc;

  if (spinfo->sample_description_id > stsd->number_of_entries) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: sample_description_id is over the maximum entry in 'stsd' atom, id %d, max entry %d.\n", spinfo->sample_description_id, stsd->number_of_entries);
    return QTIME_ERROR;
  }

  spinfo->sample_desc = stsd->table[spinfo->sample_description_id-1];
  spinfo->media_type = stsd->media_type;
//  spinfo->last_sample_description_id = spinfo->sample_description_id;

  switch (stsd->media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      spinfo->bytes_per_sample = 0;
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      sound_desc = (sound_desc_t*)spinfo->sample_desc;
      if (sound_desc->version == 0) {
	spinfo->bytes_per_sample = sound_desc->number_of_channels *
	                           (sound_desc->sample_size/8);
	if (!spinfo->bytes_per_sample)
	  spinfo->bytes_per_sample = 1;
      } else {
	spinfo->bytes_per_sample = 0;
      }
      break;
    default:
      spinfo->bytes_per_sample = 0;
      break;
  }
  return QTIME_OK;
}

