/* stsc.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_sample.h"
#include "atom.h"
#include "stsc.h"

#define MAX MAX_UINT32

void
stsc_init(stsc_t *stsc)
{
  atom_init(&stsc->atom);
  stsc->atom.type = QTIME_TYPE_STSC;
  stsc->version = 0;
  qtime_flags_set(stsc->flags, 0);
  stsc->number_of_entries = 0;
  stsc->table_max = 0;
  stsc->table = NULL;

  stsc->first_chunk = 0;
  stsc->cur_idx = 0;
  stsc->st_count = 0;
  stsc->st_chunk = 0;
  stsc->ed_count = 0;
  stsc->ed_chunk = 0;
}

void
stsc_clean(stsc_t *stsc)
{
  qtime_error_type_check(QTIME_TYPE_STSC, stsc->atom.type)
  if (stsc->table)
    qtime_free(stsc->table);
  stsc_init(stsc);
}

stsc_t*
stsc_new(void)
{
  stsc_t *stsc;
  stsc = (stsc_t*)qtime_malloc(sizeof(stsc_t));
  if (!stsc) return NULL;
  stsc_init(stsc);
  return stsc;
}

void
stsc_delete(stsc_t *stsc)
{
  qtime_error_type_check(QTIME_TYPE_STSC, stsc->atom.type)
  if (stsc->table)
    qtime_free(stsc->table);
  qtime_free(stsc);
}

int
stsc_valid(stsc_t *stsc)
{
  if (!stsc || stsc->number_of_entries <= 0)
    return 0;
  return 1;
}

stsc_t*
stsc_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, stsc_t *stsc_ptr)
{
  int table_byte_size;
  int i, table_num;
  sample_to_chunk_t *table;
  stsc_t *stsc;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_STSC, atom_head->type, NULL)
  if (stsc_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STSC, stsc_ptr->atom.type, NULL)
    stsc = stsc_ptr;
    stsc_clean(stsc);
  } else {
    if ((stsc = stsc_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  stsc->atom.size = atom_head->size;
  stsc->atom.type = atom_head->type;
  stsc->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &stsc->version, 1);
  qtime_io_read(qtio, &stsc->flags, 3);
  qtime_io_read32(qtio, &stsc->number_of_entries);

#ifndef NDEBUG
  if (stsc->number_of_entries <= 0) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: stsc_read_atom: number_of_entries invalid, %d\n", stsc->number_of_entries);
    goto fail;
  }
#endif

  table_num = stsc->number_of_entries;
  table_byte_size = table_num * sizeof(sample_to_chunk_t);
  table = (sample_to_chunk_t*)qtime_malloc(table_byte_size);
  if (!table) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    atom_head->error_code = QTIME_ERROR_MEMORY;
    goto fail;
  }
  stsc->table = table;
  stsc->table_max = table_num;

  for (i = 0; i < table_num; i++, table++) {
    qtime_io_read32(qtio, &table->first_chunk);
    qtime_io_read32(qtio, &table->samples_per_chunk);
    qtime_io_read32(qtio, &table->sample_description_id);
  }

  stsc_move_init(stsc);

  return stsc;

fail:
  if (stsc_ptr)
    stsc_clean(stsc);
  else
    stsc_delete(stsc);
  qtime_error_atom_read(QTIME_TYPE_STSC);
  return NULL;
}

stsc_t*
stsc_create(stsc_t *stsc_ptr)
{
  stsc_t *stsc;

  if (stsc_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STSC, stsc_ptr->atom.type, NULL)
    stsc = stsc_ptr;
    stsc_clean(stsc);
  } else {
    if ((stsc = stsc_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return stsc;
}

int64_t
stsc_calc_size(stsc_t *stsc)
{
  int64_t size;

  qtime_error_type_check_v(QTIME_TYPE_STSC, stsc->atom.type, 0)

  if (stsc->number_of_entries <= 0)
    return 0;

  stsc_add_chunk_finish(stsc);

  size = 8 + STSC_PROP_SIZE;
  size += stsc->number_of_entries * SAMPLE_TO_CHUNK_SIZE;

  if (size & SIZE64MASK)
    size += 8;

  stsc->atom.size = size;

  return size;
}

int
stsc_write_atom_pre(stsc_t *stsc)
{
  return stsc_add_chunk_finish(stsc);
}

int
stsc_write_atom(qtime_io_t *qtio, stsc_t *stsc)
{
  int i;
  int table_num;
  sample_to_chunk_t *table;
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_STSC, stsc->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  if (stsc->number_of_entries <= 0)
    return QTIME_OK;

  stsc_write_atom_pre(stsc);

  atom_head.size = stsc->atom.size;
  atom_head.type = stsc->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STSC);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &stsc->version, 1);
  qtime_io_write(qtio, &stsc->flags, 3);
  qtime_io_write32(qtio, &stsc->number_of_entries);

  table_num = stsc->number_of_entries;
  table = stsc->table;

  for (i = 0; i < table_num; i++, table++) {
    qtime_io_write32(qtio, &table->first_chunk);
    qtime_io_write32(qtio, &table->samples_per_chunk);
    qtime_io_write32(qtio, &table->sample_description_id);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STSC);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
stsc_dump(const char *parent_types, stsc_t *stsc)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  if (!stsc_valid(stsc))
    return;

  qtime_type_to_str(stsc->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(stsc->flags, &flags);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)stsc->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, stsc->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: number of entries %d\n", types, stsc->number_of_entries);

#if 0
  {
    int i;
    for (i = 0; i < stsc->number_of_entries; i++) {
      fprintf(stdout, "%s: table[%d] fist_chunk = %d, samples_per_chunk %d, sample_description_id %d\n", types, i, stsc->table[i].first_chunk, stsc->table[i].samples_per_chunk, stsc->table[i].sample_description_id);
    }
  }
#endif
}

int32_t
stsc_move_init(stsc_t *stsc)
{
  stsc->cur_idx = 0;
  stsc->st_count = 0;
  stsc->st_chunk = stsc->table[0].first_chunk;
  if (stsc->number_of_entries == 1) {
    stsc->ed_count = MAX;
    stsc->ed_chunk = MAX;
  } else {
    stsc->ed_count = (stsc->table[1].first_chunk - stsc->table[0].first_chunk) *
                     stsc->table[0].samples_per_chunk;
    stsc->ed_chunk = stsc->table[1].first_chunk;
  }
  return stsc->cur_idx;
}

int32_t
stsc_get_chunk(stsc_t *stsc, sc_info_t *si)
{
  int32_t idx;
  uint32_t ck, sk, ek, sc, ec;
  uint32_t count;
  sample_to_chunk_t *table;
 
  if (stsc->number_of_entries == 1) {
    si->sample_description_id = stsc->table[0].sample_description_id;
    si->samples_per_chunk = stsc->table[0].samples_per_chunk;
    ck = si->count / si->samples_per_chunk;
//    si->chunk = stsc->table[0].first_chunk + ck;
    si->chunk = 1 + ck;
    si->chunk_start_count = ck * si->samples_per_chunk;
    si->samples = si->chunk_start_count + si->samples_per_chunk - si->count;
    return si->chunk;
  }

  if (stsc->st_chunk == 0)
    stsc_move_init(stsc);

  count = si->count;
  table = stsc->table;

  idx = stsc->cur_idx;
  sc = stsc->st_count;
  sk = stsc->st_chunk;
  ec = stsc->ed_count;
  ek = stsc->ed_chunk;

  if (count >= ec) {
    if (idx < stsc->number_of_entries-1) {
      for (idx++; idx < stsc->number_of_entries-1; idx++) {
        sc = ec;
        sk = ek;
        ec = ec + (table[idx+1].first_chunk - table[idx].first_chunk) *
	           table[idx].samples_per_chunk;
        ek = table[idx+1].first_chunk;
        if (sc <= count && count < ec)
	  break;
      }
      if (idx >= stsc->number_of_entries-1) {
        idx = stsc->number_of_entries-1;
        sc = ec;
        sk = ek;
        ec = MAX;
        ek = MAX;
      }
#ifndef NDEBUG
        if (count < sc || count >= ec) {
	  qtime_error_debug_info(QTIME_ERROR);
	  fprintf(stderr, "stsc_move_count: chunk index invalid, count %u, sc %u, ec %u.\n", count, sc, ec);
        }
#endif
    }
  } else if (count < sc) {
    if (idx > 0) {
      for (idx--; idx >= 0 ; idx--) {
        ec = sc;
        ek = sk;
        sc = sc - (table[idx+1].first_chunk - table[idx].first_chunk) *
	           table[idx].samples_per_chunk;
        sk = table[idx].first_chunk;
        if (sc <= count && count < ec)
	  break;
      }
      if (idx < 0) {
        idx = 0;
        sc = 0;
        ec = (table[1].first_chunk - table[0].first_chunk) *
	      table[0].samples_per_chunk;
        sk = table[0].first_chunk;
        ek = table[1].first_chunk;
      }
#ifndef NDEBUG
        if (count < sc || count >= ec) {
	  qtime_error_debug_info(QTIME_ERROR);
	  fprintf(stderr, "stsc_move_count: chunk index invalid, count %u, sc %u, ec %u.\n", count, sc, ec);
        }
#endif
    }
  }

  stsc->cur_idx = idx;
  stsc->st_count = sc;
  stsc->ed_count = ec;
  stsc->st_chunk = sk;
  stsc->ed_chunk = ek;

  ck = (count - sc) / table[idx].samples_per_chunk;
  si->chunk = sk + ck;
  si->chunk_start_count = sc + ck * table[idx].samples_per_chunk;
  si->samples_per_chunk = table[idx].samples_per_chunk;
  si->sample_description_id = table[idx].sample_description_id;
  si->samples = si->chunk_start_count + si->samples_per_chunk - count;

  return si->chunk;
}

static int
stsc_table_resize(stsc_t *stsc, int32_t entry_num)
{
  sample_to_chunk_t *table;
  int32_t table_max;

  if (entry_num <= stsc->table_max)
    return QTIME_OK;

  if (stsc->table == NULL) {
    table_max = 2;
    table = (sample_to_chunk_t*)qtime_realloc(stsc->table,
                                     sizeof(sample_to_chunk_t) * table_max);
    if (!table)
      return QTIME_ERROR_MEMORY;
    stsc->first_chunk = 1;
    stsc->table_max = table_max;
    stsc->table = table;
    stsc->number_of_entries = 0;
    return QTIME_OK;
  } 

  table_max = stsc->table_max + 4096;
  table = (sample_to_chunk_t*)qtime_realloc(stsc->table,
           sizeof(sample_to_chunk_t) * table_max);
  if (!table)
    return QTIME_ERROR_MEMORY;
  stsc->table_max = table_max;
  stsc->table = table;
  return QTIME_OK;
}

int32_t
stsc_add_chunk(stsc_t *stsc, sc_info_t *si)
{
  int idx;
  sample_to_chunk_t *table;

#ifndef NDEBUG
  if (stsc->number_of_entries > (MAX_INT32-(4096*2))) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stsc' atom, entries = %d.\n", stsc->number_of_entries);
  }
#endif

  idx = stsc->number_of_entries;
  if (stsc->number_of_entries == 0 ||
      si->sample_description_id != stsc->table[idx-1].sample_description_id) {
    idx = stsc->number_of_entries;
    if (stsc_table_resize(stsc, stsc->number_of_entries+1) < 0) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return QTIME_ERROR_MEMORY;
    }
    stsc->table[idx].first_chunk = stsc->first_chunk;
    stsc->table[idx].samples_per_chunk = si->samples;
    stsc->table[idx].sample_description_id = si->sample_description_id;
    stsc->number_of_entries++;
    stsc->first_chunk++;
  } else if (!si->consecutive_data) {
    table = stsc->table;
    idx = stsc->number_of_entries-1;
    if (stsc->number_of_entries <= 1 ||
      table[idx].samples_per_chunk != table[idx-1].samples_per_chunk ||
      table[idx].sample_description_id != table[idx-1].sample_description_id) {
      if (stsc_table_resize(stsc, stsc->number_of_entries+1) < 0) {
        qtime_error_debug_info(QTIME_ERROR_MEMORY);
        return QTIME_ERROR_MEMORY;
      }
//      table = stsc->table;
      idx = stsc->number_of_entries;
      stsc->number_of_entries++;
    }
    stsc->table[idx].first_chunk = stsc->first_chunk;
    stsc->table[idx].samples_per_chunk = si->samples;
    stsc->table[idx].sample_description_id = si->sample_description_id;
    stsc->first_chunk++;
  } else {
    idx = stsc->number_of_entries-1;
    stsc->table[idx].samples_per_chunk += si->samples;
  }

  si->chunk = stsc->first_chunk-1;

  stsc->st_chunk = 0;

  return si->chunk;
}

int32_t
stsc_add_chunk_finish(stsc_t *stsc)
{
  int idx;
  sample_to_chunk_t *table;

  if (stsc->table == NULL) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "stsc_update_chunk_finish: table NULL.\n");
  } 

  idx = stsc->number_of_entries - 1;
  table = stsc->table;

  if (stsc->number_of_entries > 1 &&
      table[idx].samples_per_chunk == table[idx-1].samples_per_chunk &&
      table[idx].sample_description_id == table[idx-1].sample_description_id) {
    table[idx].first_chunk = 0;
    table[idx].samples_per_chunk = 0;
    table[idx].sample_description_id = 0;
    stsc->number_of_entries--;
  }

  return stsc->number_of_entries;
}

int32_t
stsc_get_sample_info(stsc_t *stsc, qtime_sample_info_t *spinfo)
{
  int32_t idx;
  uint32_t ck, sk, ek, sc, ec;
  uint32_t count;
  sample_to_chunk_t *table;
  uint32_t samples;
 
  spinfo->last_sample_description_id = spinfo->sample_description_id;
  spinfo->last_chunk = spinfo->chunk;

  count = spinfo->count;
  table = stsc->table;

  if (stsc->number_of_entries == 1) {
    spinfo->sample_description_id = table[0].sample_description_id;
    spinfo->samples_per_chunk = table[0].samples_per_chunk;
    ck = count / spinfo->samples_per_chunk;
//    spinfo->chunk = table[0].first_chunk + ck;
    spinfo->chunk = 1 + ck;
    spinfo->chunk_start_count = ck * spinfo->samples_per_chunk;
    samples = spinfo->chunk_start_count + spinfo->samples_per_chunk - count;
    spinfo->chunk_samples = samples;
    spinfo->samples = (spinfo->samples <= samples) ? spinfo->samples : samples;
    if (spinfo->last_chunk != spinfo->chunk)
      spinfo->last_count = spinfo->chunk_start_count;
    return spinfo->chunk;
  }


  if (stsc->st_chunk == 0)
    stsc_move_init(stsc);

  idx = stsc->cur_idx;
  sc = stsc->st_count;
  sk = stsc->st_chunk;
  ec = stsc->ed_count;
  ek = stsc->ed_chunk;

  if (count >= ec) {
    if (idx < stsc->number_of_entries-1) {
      for (idx++; idx < stsc->number_of_entries-1; idx++) {
        sc = ec;
        sk = ek;
        ec = ec + (table[idx+1].first_chunk - table[idx].first_chunk) *
	           table[idx].samples_per_chunk;
        ek = table[idx+1].first_chunk;
        if (sc <= count && count < ec)
	  break;
      }
      if (idx >= stsc->number_of_entries-1) {
        idx = stsc->number_of_entries-1;
        sc = ec;
        sk = ek;
        ec = MAX;
        ek = MAX;
      }
    }
  } else if (count < sc) {
    if (idx > 0) {
      for (idx--; idx >= 0 ; idx--) {
        ec = sc;
        ek = sk;
        sc = sc - (table[idx+1].first_chunk - table[idx].first_chunk) *
	           table[idx].samples_per_chunk;
        sk = table[idx].first_chunk;
        if (sc <= count && count < ec)
	  break;
      }
      if (idx <= 0) {
        idx = 0;
        sc = 0;
        ec = (table[1].first_chunk - table[0].first_chunk) *
	      table[0].samples_per_chunk;
        sk = table[0].first_chunk;
        ek = table[1].first_chunk;
      }
    }
  }

  stsc->cur_idx = idx;
  stsc->st_count = sc;
  stsc->ed_count = ec;
  stsc->st_chunk = sk;
  stsc->ed_chunk = ek;

  ck = (count - sc) / table[idx].samples_per_chunk;
  spinfo->chunk = sk + ck;
  spinfo->chunk_start_count = sc + ck * table[idx].samples_per_chunk;
  spinfo->samples_per_chunk = table[idx].samples_per_chunk;
  spinfo->sample_description_id = table[idx].sample_description_id;
  samples = spinfo->chunk_start_count + spinfo->samples_per_chunk - count;
  spinfo->chunk_samples = samples;
  spinfo->samples = (spinfo->samples <= samples) ? spinfo->samples : samples;
  if (spinfo->last_chunk != spinfo->chunk)
    spinfo->last_count = spinfo->chunk_start_count;

  return spinfo->chunk;
}

int32_t
stsc_add_sample_info(stsc_t *stsc, qtime_sample_info_t *spinfo)
{
  int32_t idx;
  sample_to_chunk_t *table;

#ifndef NDEBUG
  if (stsc->number_of_entries > (MAX_INT32-(4096*2))) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stsc' atom, entries = %d.\n", stsc->number_of_entries);
  }
#endif

  if (spinfo->last_sample_description_id != spinfo->sample_description_id) {
    idx = stsc->number_of_entries;
    if (stsc_table_resize(stsc, stsc->number_of_entries+1) < 0) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return QTIME_ERROR_MEMORY;
    }
    stsc->table[idx].first_chunk = stsc->first_chunk;
    stsc->table[idx].samples_per_chunk = spinfo->samples;
    stsc->table[idx].sample_description_id = spinfo->sample_description_id;
    stsc->number_of_entries++;
    stsc->first_chunk++;
  } else if (spinfo->last_data_offset != spinfo->data_start_offset) {
    table = stsc->table;
    idx = stsc->number_of_entries-1;
    if (stsc->number_of_entries <= 1 ||
      (table[idx].samples_per_chunk != table[idx-1].samples_per_chunk ||
      table[idx].sample_description_id != table[idx-1].sample_description_id)){
      if (stsc_table_resize(stsc, stsc->number_of_entries+1) < 0) {
        qtime_error_debug_info(QTIME_ERROR_MEMORY);
        return QTIME_ERROR_MEMORY;
      }
//      table = stsc->table;
      idx = stsc->number_of_entries;
      stsc->number_of_entries++;
    }
    stsc->table[idx].first_chunk = stsc->first_chunk;
    stsc->table[idx].samples_per_chunk = spinfo->samples;
    stsc->table[idx].sample_description_id = spinfo->sample_description_id;
    stsc->first_chunk++;
  } else {
    idx = stsc->number_of_entries-1;
    stsc->table[idx].samples_per_chunk += spinfo->samples;
  }

  spinfo->last_chunk = spinfo->chunk;
  spinfo->chunk = stsc->first_chunk-1;
  spinfo->last_data_offset = spinfo->data_end_offset;
  spinfo->last_sample_description_id = spinfo->sample_description_id;

  stsc->st_chunk = 0;

  return spinfo->chunk;
}

