/* stco.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_sample_info.h"
#include "atom.h"
#include "stco.h"

void
stco_init(stco_t *stco)
{
  atom_init(&stco->atom);
  stco->atom.type = QTIME_TYPE_STCO;
  stco->version = 0;
  qtime_flags_set(stco->flags, 0);
  stco->number_of_entries = 0;
  stco->table_max = 0;
  stco->num_ent32 = 0;
  stco->table32 = NULL;
  stco->num_ent64 = 0;
  stco->table64 = NULL;
  stco->use_co64 = 0;
}

void
stco_clean(stco_t *stco)
{
#ifndef NDEBUG
  uint32_t ui32;
  ui32 = (stco->atom.type == QTIME_TYPE_CO64) ? QTIME_TYPE_STCO : stco->atom.type;
#endif
  qtime_error_type_check(QTIME_TYPE_STCO, ui32)
  if (stco->table32)
    qtime_free(stco->table32);
  if (stco->table64)
    qtime_free(stco->table64);
  stco_init(stco);
}

stco_t*
stco_new(void)
{
  stco_t *stco;
  stco = (stco_t*)qtime_malloc(sizeof(stco_t));
  if (!stco) return NULL;
  stco_init(stco);
  return stco;
}

void
stco_delete(stco_t *stco)
{
#ifndef NDEBUG
  uint32_t ui32;
  ui32 = (stco->atom.type == QTIME_TYPE_CO64) ? QTIME_TYPE_STCO : stco->atom.type;
#endif
  qtime_error_type_check(QTIME_TYPE_STCO, ui32)
  if (stco->table32)
    qtime_free(stco->table32);
  if (stco->table64)
    qtime_free(stco->table64);
  qtime_free(stco);
}

int
stco_valid(stco_t *stco)
{
  if (stco->number_of_entries <= 0)
    return 0;
  return 1;
}

stco_t*
stco_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, stco_t *stco_ptr)
{
  int i;
  int table_num;
  int table_byte_size;
  int64_t  i64;
  uint32_t *table32;
  stco_t *stco;

#ifndef NDEBUG
  uint32_t ui32;
  ui32 = (atom_head->type == QTIME_TYPE_CO64) ? QTIME_TYPE_STCO : atom_head->type;
#endif

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_STCO, ui32, NULL)
  if (stco_ptr) {
#ifndef NDEBUG
    ui32 = (stco_ptr->atom.type == QTIME_TYPE_CO64) ? QTIME_TYPE_STCO : stco_ptr->atom.type;
#endif
    qtime_error_type_check_v(QTIME_TYPE_STCO, ui32, NULL)
    stco = stco_ptr;
    stco_clean(stco);
  } else {
    if ((stco = stco_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  stco->atom.size = atom_head->size;
  stco->atom.type = atom_head->type;
  stco->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &stco->version, 1);
  qtime_io_read(qtio, &stco->flags, 3);
  qtime_io_read32(qtio, &stco->number_of_entries);

#ifndef NDEBUG
  if (stco->number_of_entries < 0) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: stco_read_atom: error, number_of_entries value is negative, %d.\n", stco->number_of_entries);
  }
#endif

  if (atom_head->type == QTIME_TYPE_STCO) {
    table_num = stco->number_of_entries;
    table_byte_size = table_num * sizeof(uint32_t);
    table32 = (uint32_t*)qtime_malloc(table_byte_size);
    if (!table32) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      goto fail;
    }
    stco->table32 = table32;
    for (i = 0; i < table_num; i++)
      qtime_io_read32(qtio, &table32[i]);
    stco->num_ent32 = table_num;
    stco->table_max = table_num;
  } else {
    table_num = stco->number_of_entries;
    stco->number_of_entries = 0;
    for (i = 0; i < table_num; i++) {
      qtime_io_read64(qtio, &i64);
      stco_add_offset(stco, i64);
    }
  }

  return stco;

fail:
  if (stco_ptr)
    stco_clean(stco);
  else
    stco_delete(stco);
  qtime_error_atom_read(QTIME_TYPE_STCO);
  return NULL;
}

stco_t*
stco_create(stco_t *stco_ptr)
{
  stco_t *stco;

  if (stco_ptr) {
#ifndef NDEBUG
    uint32_t ui32;
    ui32 = (stco_ptr->atom.type == QTIME_TYPE_CO64) ? QTIME_TYPE_STCO : stco_ptr->atom.type;
#endif
    qtime_error_type_check_v(QTIME_TYPE_STCO, ui32, NULL)
    stco = stco_ptr;
    stco_clean(stco);
  } else {
    if ((stco = stco_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return stco;
}

int64_t
stco_calc_size(stco_t *stco)
{
  int64_t num_ent;
  int64_t bytes;
  int64_t size;

#ifndef NDEBUG
  uint32_t ui32;
  ui32 = (stco->atom.type == QTIME_TYPE_CO64) ? QTIME_TYPE_STCO : stco->atom.type;
#endif
  qtime_error_type_check_v(QTIME_TYPE_STCO, ui32, 0)

  if (stco->number_of_entries <= 0)
    return 0;

  num_ent = stco->number_of_entries;

  if (stco->use_co64) {
    bytes = sizeof(int64_t);
  } else {
    bytes = sizeof(uint32_t);
  }
  size = 8 + STCO_PROP_SIZE + num_ent * bytes;

  if (size & SIZE64MASK)
    size += 8;

  stco->atom.size = size;

  return size;
}

int
stco_write_atom(qtime_io_t *qtio, stco_t *stco)
{
  int i;
  int table_num;
  uint32_t *table32;
  int64_t  *table64;
  int64_t  i64;
  atom_head_t atom_head;

#ifndef NDEBUG
  { uint32_t ui32;
    ui32 = (stco->atom.type == QTIME_TYPE_CO64) ? QTIME_TYPE_STCO : stco->atom.type;
    qtime_error_type_check_v(QTIME_TYPE_STCO, ui32, QTIME_ERROR_ILLEGAL_ATOM)
  }
#endif

  if (stco->number_of_entries <= 0)
    return QTIME_OK;

  if (stco->use_co64)
    stco->atom.type = QTIME_TYPE_CO64;
  else
    stco->atom.type = QTIME_TYPE_STCO;

  atom_head.size = stco->atom.size;
  atom_head.type = stco->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STCO);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &stco->version, 1);
  qtime_io_write(qtio, &stco->flags, 3);
  qtime_io_write32(qtio, &stco->number_of_entries);

  if (stco->atom.type == QTIME_TYPE_STCO) {
    table_num = stco->number_of_entries;
    table32 = stco->table32;
    for (i = 0; i < table_num; i++)
      qtime_io_write32(qtio, &table32[i]);
  } else {
    table_num = stco->num_ent32;
    table32 = stco->table32;
    for (i = 0; i < table_num; i++) {
      i64 = table32[i];
      qtime_io_write64(qtio, &i64);
    }
    table_num = stco->num_ent64;
    table64 = stco->table64;
    for (i = 0; i < table_num; i++)
      qtime_io_write64(qtio, &table64[i]);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STCO);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
stco_dump(const char *parent_types, stco_t *stco)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  if (!stco_valid(stco))
    return;

  qtime_type_to_str(stco->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(stco->flags, &flags);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)stco->atom.size);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: version      %d\n", types, stco->version);
  fprintf(stdout, "%s: number of entries %d\n", types, stco->number_of_entries);

#if 0
  {
    int i;
    for (i = 0; i < stco->number_of_entries; i++) {
      fprintf(stdout, "%s: table[%d] offset = %lld\n", types, i, stco->table[i]);
//      fprintf(stdout, "%s: table[%d] offset = %lld\n", types, i, stco->table[i+1] - stco->table[i]);
    }
  }
#endif
}

#if 0
static int32_t
stco_table_resize(stco_t *stco, int32_t entry_num)
{
  int64_t *table;
  int32_t table_max;
  if (entry_num <= stco->table_max) {
    return QTIME_OK;
  }
  table_max = stco->table_max + 4096;
  table = (int64_t*)qtime_realloc(stco->table, sizeof(int64_t) * table_max);
  if (!table)
    return QTIME_ERROR_MEMORY;
  stco->table_max = table_max;
  stco->table = table;
  return QTIME_OK;
}
#endif

int32_t
stco_shift_offset(stco_t *stco, int64_t shift)
{
  int32_t num_ent, num_ent32, num_ent64;
  uint32_t *table32;
  int64_t  *table64;
  int32_t i;

  num_ent = stco->number_of_entries;
  num_ent32 = stco->num_ent32;
  num_ent64 = stco->num_ent64;
  table32 = stco->table32;
  table64 = stco->table64;
  stco->number_of_entries = 0;
  stco->num_ent32 = 0;
  stco->num_ent64 = 0;
  stco->table32 = NULL;
  stco->table64 = NULL;
  stco->use_co64 = 0;
  stco->table_max = 0;
  for (i = 0; i < num_ent32; i++)
    if (stco_add_offset(stco, ((int64_t)table32[i] + shift)) < 0)
      return QTIME_ERROR;
  qtime_free(table32);
  for (i = 0; i < num_ent64; i++)
    if (stco_add_offset(stco, table64[i] + shift) < 0)
      return QTIME_ERROR;
  qtime_free(table64);
#ifndef NDEBUG
  if (stco->number_of_entries != num_ent) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_entries doesn't suit, %d, %d.\n", num_ent, stco->number_of_entries);
  }
#endif
  return stco->number_of_entries;
}

int32_t
stco_add_offset(stco_t *stco, int64_t offset)
{
  int32_t  table_max;
  uint32_t *table32;
  int64_t  *table64;

#ifndef NDEBUG
  if (stco->number_of_entries > (MAX_INT32-(4096*2))) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stco' atom, entries = %d.\n", stco->number_of_entries);
  }
#endif
  if (stco->use_co64) {
    if ((stco->num_ent64+1) > stco->table_max) {
      table_max = stco->table_max + 4096;
      table64 = (int64_t*)qtime_realloc(stco->table64,
                                        sizeof(int64_t) * table_max);
      if (!table64) return QTIME_ERROR_MEMORY;
      stco->table_max = table_max;
      stco->table64 = table64;
    }
    stco->table64[stco->num_ent64] = offset;
    stco->num_ent64++;
    stco->number_of_entries++;
  } else {
    if (offset & SIZE64MASK) {
      table32 = (uint32_t*)qtime_realloc(stco->table32,
                                         sizeof(uint32_t) * stco->num_ent32);
      if (!table32) return QTIME_ERROR_MEMORY;
      stco->table32 = table32;
      table_max = 4096;
      table64 = (int64_t*)qtime_realloc(stco->table64,
                                        sizeof(int64_t) * table_max);
      if (!table64) return QTIME_ERROR_MEMORY;
      stco->table_max = table_max;
      stco->table64 = table64;
      stco->table64[stco->num_ent64] = offset;
      stco->num_ent64++;
      stco->number_of_entries++;
      stco->use_co64 = 1;
    } else {
      if ((stco->num_ent32+1) > stco->table_max) {
        table_max = stco->table_max + 4096;
        table32 = (uint32_t*)qtime_realloc(stco->table32,
                                           sizeof(uint32_t) * table_max);
        if (!table32) return QTIME_ERROR_MEMORY;
        stco->table_max = table_max;
        stco->table32 = table32;
      }
      stco->table32[stco->num_ent32] = offset;
      stco->num_ent32++;
      stco->number_of_entries++;
    }
  }
  return stco->number_of_entries;
}

#if 0
uint32_t
stco_set_offset(stco_t *stco, uint32_t count, int64_t offset)
{
  if (count >= stco->number_of_entries) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: requested count is over the maximum entry in 'stco' atom, count = %d, entries = %d.\n", count, stco->number_of_entries);
#endif
    return QTIME_ERROR;
  }
  stco->table[count] = offset;
  if (offset & SIZE64MASK)
    stco->use_co64 = 1;
  return count;
}
#endif

int64_t
stco_get_offset(stco_t *stco, uint32_t chunk)
{
  if ((chunk-1) >= (uint32_t)(stco->number_of_entries)) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: requested chunk is over the maximum entry in 'stco' atom, chunk = %u, entries = %d.\n", chunk, stco->number_of_entries);
#endif
    return QTIME_ERROR;
  }
  if (chunk <= (uint32_t)stco->num_ent32)
    return stco->table32[chunk-1];
  return stco->table64[chunk - stco->num_ent32 - 1];
}

int
stco_get_sample_info(stco_t *stco, qtime_sample_info_t *spinfo)
{
  if ((spinfo->chunk-1) >= stco->number_of_entries) {
#ifndef NDEBUG
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: requested chunk is over the maximum entry in 'stco' atom, chunk = %u, entries = %d.\n", spinfo->chunk , stco->number_of_entries);
#endif
    return QTIME_ERROR;
  }

  if (spinfo->last_chunk != spinfo->chunk) {
    if (spinfo->chunk <= stco->num_ent32)
      spinfo->last_data_offset = stco->table32[spinfo->chunk - 1];
    else
      spinfo->last_data_offset = stco->table64[spinfo->chunk-stco->num_ent32-1];
  } else {
    spinfo->last_data_offset = spinfo->data_end_offset;
  }

  return QTIME_OK;
}

int
stco_add_sample_info(stco_t *stco, qtime_sample_info_t *spinfo)
{
  int32_t  table_max;
  uint32_t *table32;
  int64_t  *table64;


#ifndef NDEBUG
  if ((spinfo->chunk-1) != (stco->number_of_entries) &&
      (spinfo->chunk-1) != (stco->number_of_entries-1)) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: chunk count is invalid in 'stco' atom, chunk = %u, entries = %d.\n", spinfo->chunk , stco->number_of_entries);
    return QTIME_ERROR;
  }
#endif
#ifndef NDEBUG
  if (stco->number_of_entries > (MAX_INT32-(4096*2))) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number_of_enries is reached max in 'stco' atom, entries = %d.\n", stco->number_of_entries);
  }
#endif
  if (spinfo->last_chunk != spinfo->chunk) {
    if (stco->use_co64) {
      if ((stco->num_ent64+1) > stco->table_max) {
        table_max = stco->table_max + 4096;
        table64 = (int64_t*)qtime_realloc(stco->table64,
                                          sizeof(int64_t) * table_max);
        if (!table64) return QTIME_ERROR_MEMORY;
        stco->table_max = table_max;
        stco->table64 = table64;
      }
      stco->table64[stco->num_ent64] = spinfo->data_start_offset;
      stco->num_ent64++;
      stco->number_of_entries++;
    } else {
      if (spinfo->data_start_offset & SIZE64MASK) {
        table32 = (uint32_t*)qtime_realloc(stco->table32,
                                           sizeof(uint32_t) * stco->num_ent32);
        if (!table32) return QTIME_ERROR_MEMORY;
        stco->table32 = table32;
        table_max = 4096;
        table64 = (int64_t*)qtime_realloc(stco->table64,
                                          sizeof(int64_t) * table_max);
        if (!table64) return QTIME_ERROR_MEMORY;
        stco->table_max = table_max;
        stco->table64 = table64;
        stco->table64[stco->num_ent64] = spinfo->data_start_offset;
        stco->num_ent64++;
        stco->number_of_entries++;
        stco->use_co64 = 1;
      } else {
        if ((stco->num_ent32+1) > stco->table_max) {
          table_max = stco->table_max + 4096;
          table32 = (uint32_t*)qtime_realloc(stco->table32,
                                             sizeof(uint32_t) * table_max);
          if (!table32) return QTIME_ERROR_MEMORY;
          stco->table_max = table_max;
          stco->table32 = table32;
        }
        stco->table32[stco->num_ent32] = spinfo->data_start_offset;
        stco->num_ent32++;
        stco->number_of_entries++;
      }
    }
  }

  return QTIME_OK;
}

