/* stbl.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "stbl.h"
#include "stsd.h"
#include "stts.h"
#include "stss.h"
#include "stsc.h"
#include "stsz.h"
#include "stco.h"
//#include "stsh.h"

void
stbl_init(stbl_t *stbl)
{
  atom_init(&stbl->atom);
  stbl->atom.type = QTIME_TYPE_STBL;
  stbl->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
  stsd_init(&stbl->stsd);
  stts_init(&stbl->stts);
  stss_init(&stbl->stss);
  stsc_init(&stbl->stsc);
  stsz_init(&stbl->stsz);
  stco_init(&stbl->stco);
//  stsh_init(&stbl->stsh);
}

void
stbl_clean(stbl_t *stbl)
{
  int i;
  atom_t *atom = &stbl->atom;

  qtime_error_type_check(QTIME_TYPE_STBL, stbl->atom.type)

  for (i = 0; i < atom->number_of_childs; i++) {
    if (atom->childs[i] == (atom_t*)&stbl->stsd) {
      stsd_clean((stsd_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&stbl->stts) {
      stts_clean((stts_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&stbl->stss) {
      stss_clean((stss_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&stbl->stsc) {
      stsc_clean((stsc_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&stbl->stsz) {
      stsz_clean((stsz_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&stbl->stco) {
      stco_clean((stco_t*)atom->childs[i]);
      continue;
    }

    switch (atom->childs[i]->type) {
      case QTIME_TYPE_STSD:
	stsd_delete((stsd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STTS:
	stts_delete((stts_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSS:
	stss_delete((stss_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSC:
	stsc_delete((stsc_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSZ:
	stsz_delete((stsz_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STCO:
      case QTIME_TYPE_CO64:
	stco_delete((stco_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_STSH:
	stsh_delete((stsh_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_STBL, atom->childs[i]->type);
	break;
    }
  }

  atom_remove_child_all(atom);
  atom_init(atom);
  stbl->atom.type = QTIME_TYPE_STBL;
  stbl->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
}

stbl_t*
stbl_new(void)
{
  stbl_t *stbl;
  stbl = (stbl_t*)qtime_malloc(sizeof(stbl_t));
  if (!stbl) return NULL;
  stbl_init(stbl);
  return stbl;
}

void
stbl_delete(stbl_t *stbl)
{
  qtime_error_type_check(QTIME_TYPE_STBL, stbl->atom.type)
  stbl_clean(stbl);
  qtime_free(stbl);
}

stbl_t*
stbl_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, stbl_t *stbl_ptr)
{
  atom_head_t subatom;
  atom_t *atom;
  int stsd_index, stts_index, stss_index, stsc_index, stsz_index, stco_index;
  stbl_t *stbl;
  stsd_t *stsd;
  stts_t *stts;
  stss_t *stss;
  stsc_t *stsc;
  stsz_t *stsz;
  stco_t *stco;
  uint32_t media_type;
  int64_t total_size;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_STBL, atom_head->type, NULL)
  if (stbl_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STBL, stbl_ptr->atom.type, NULL)
    stbl = stbl_ptr;
    stbl_clean(stbl);
  } else {
    if ((stbl = stbl_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  stbl->atom.size = atom_head->size;
  stbl->atom.type = QTIME_TYPE_STBL;
  stbl->atom.parent = atom_head->parent;
  stbl->media_type = atom_head->media_type;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)stbl;
  subatom.media_type = atom_head->media_type;

  media_type = atom_head->media_type;
  stsd_index = stts_index = stss_index = stsc_index
    = stsz_index = stco_index = 0;
  stsd = &stbl->stsd;
  stts = &stbl->stts;
  stss = &stbl->stss;
  stsc = &stbl->stsc;
  stsz = &stbl->stsz;
  stco = &stbl->stco;

  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    atom = NULL;
    switch (subatom.type) {
      case QTIME_TYPE_STSD:
	if ((stsd = stsd_read_atom(qtio, &subatom, stsd)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)stsd;
	atom->index = ++stsd_index;
	stsd = NULL;
	break;
      case QTIME_TYPE_STTS:
	if ((stts = stts_read_atom(qtio, &subatom, stts)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)stts;
	atom->index = ++stts_index;
	stts = NULL;
	break;
      case QTIME_TYPE_STSS:
	if ((stss = stss_read_atom(qtio, &subatom, stss)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)stss;
	atom->index = ++stss_index;
	stss = NULL;
	break;
      case QTIME_TYPE_STSC:
	if ((stsc = stsc_read_atom(qtio, &subatom, stsc)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)stsc;
	atom->index = ++stsc_index;
	stsc = NULL;
	break;
      case QTIME_TYPE_STSZ:
	if ((stsz = stsz_read_atom(qtio, &subatom, stsz)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)stsz;
	atom->index = ++stsz_index;
	stsz = NULL;
	break;
      case QTIME_TYPE_STCO:
      case QTIME_TYPE_CO64:
	if ((stco = stco_read_atom(qtio, &subatom, stco)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)stco;
	atom->index = ++stco_index;
	stco = NULL;
	break;
//      case QTIME_TYPE_STSH:
//	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_STBL, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom) {
      if ((atom_head->error_code = atom_add_child((atom_t*)stbl, atom)) < 0) {
	qtime_error_debug_info(atom_head->error_code);
	goto fail;
      }
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

#if 0
  stco = (stsc_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STCO);
  if (!stco)
    stco = (stsc_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_CO64);
  stsc = (stsc_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STSC);
  stsc->first_chunk = stco->number_of_entries;
#endif

  stsz = (stsz_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STSZ);
  if (!stsz) {
    total_size = stsz_get_total_size(stsz);
    if (total_size == 0) {
      stts = (stts_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STTS);
      if (!stts) {
	uint32_t max_count;
	max_count = stts_get_max_count(stts);
        total_size = stsz_get_sample_size(stsz);
        total_size *= max_count;
	stsz_set_total_size(stsz, total_size);
      }
    }
  }

  return stbl;

fail:
  if (stbl_ptr)
    stbl_clean(stbl);
  else
    stbl_delete(stbl);
  qtime_error_atom_read(QTIME_TYPE_STBL);
  return NULL;
}

stbl_t*
stbl_create(stbl_t *stbl_ptr)
{
  stbl_t *stbl;
  int error_code;

  if (stbl_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_STBL, stbl_ptr->atom.type, NULL)
    stbl = stbl_ptr;
    stbl_clean(stbl);
  } else {
    if ((stbl = stbl_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  stsd_create(&stbl->stsd);
  stts_create(&stbl->stts);
  stss_create(&stbl->stss);
  stsc_create(&stbl->stsc);
  stsz_create(&stbl->stsz);
  stco_create(&stbl->stco);

  if ((error_code = atom_add_child((atom_t*)stbl, (atom_t*)&stbl->stsd)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)stbl, (atom_t*)&stbl->stts)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)stbl, (atom_t*)&stbl->stss)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)stbl, (atom_t*)&stbl->stsc)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)stbl, (atom_t*)&stbl->stsz)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)stbl, (atom_t*)&stbl->stco)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }

  return stbl;

fail:
  if (stbl_ptr)
    stbl_clean(stbl);
  else
    stbl_delete(stbl);
  return NULL;
}

void
stbl_dump(const char *parent_types, stbl_t *stbl)
{
  int i, len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  atom_t *atom = &stbl->atom;

  qtime_type_to_str(stbl->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)stbl->atom.size);
  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_STSD:
	stsd_dump(types, (stsd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STTS:
	stts_dump(types, (stts_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSS:
	stss_dump(types, (stss_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSC:
	stsc_dump(types, (stsc_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSZ:
	stsz_dump(types, (stsz_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STCO:
      case QTIME_TYPE_CO64:
	stco_dump(types, (stco_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_STSH:
	stsh_dump(types, (stsh_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_STBL, atom->childs[i]->type);
	break;
    }
  }
}

int64_t
stbl_calc_size(stbl_t *stbl)
{
  int64_t size = 0;
  int i;
  atom_t *atom = &stbl->atom;

  qtime_error_type_check_v(QTIME_TYPE_STBL, stbl->atom.type, 0)

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_STSD:
	size += stsd_calc_size((stsd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STTS:
	size += stts_calc_size((stts_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSS:
	size += stss_calc_size((stss_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSC:
	size += stsc_calc_size((stsc_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSZ:
	size += stsz_calc_size((stsz_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STCO:
      case QTIME_TYPE_CO64:
	size += stco_calc_size((stco_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_STSH:
	size += stsh_calc_size((stsh_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_STBL, atom->childs[i]->type);
	break;
    }
  }

  if (size & SIZE64MASK)
    size += 4 + 4 + 8;
  else
    size += 4 + 4;
  stbl->atom.size = size;

  return size;
}

int
stbl_write_atom(qtime_io_t *qtio, stbl_t *stbl)
{
  atom_head_t atom_head;
  int i;
  atom_t *atom = &stbl->atom;

  qtime_error_type_check_v(QTIME_TYPE_STBL, stbl->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  atom_head_init(&atom_head);
  atom_head.size = stbl->atom.size;
  atom_head.type = stbl->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STBL);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_STSD:
	stsd_write_atom(qtio, (stsd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STTS:
	stts_write_atom(qtio, (stts_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSS:
	stss_write_atom(qtio, (stss_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSC:
	stsc_write_atom(qtio, (stsc_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STSZ:
	stsz_write_atom(qtio, (stsz_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STCO:
      case QTIME_TYPE_CO64:
	stco_write_atom(qtio, (stco_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_STSH:
	stsh_write_atom(qtio, (stsh_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_STBL, atom->childs[i]->type);
	break;
    }
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_STBL);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

int
stbl_set_media_type(stbl_t *stbl, uint32_t media_type)
{
  qtime_error_type_check_v(QTIME_TYPE_STBL, stbl->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  if (stbl->media_type != QTIME_MEDIA_TYPE_UNKNOWN &&
      stbl->media_type != media_type) {
    fprintf(stderr, "stbl_set_media_type: media_type doesn't suit.\n");
    fprintf(stderr, "stbl_set_media_type: '%.4s' '%.4s'.\n", (char*)&stbl->media_type, (char*)&media_type);
//    return QTIME_ERROR;
  }
  stbl->media_type = media_type;
  return QTIME_OK;
}

