/* sound_desc.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "sample_desc.h"
#include "sound_desc.h"

void
sound_desc_init(sound_desc_t *sound_desc)
{
  sample_desc_init(&sound_desc->sample_desc);
  sound_desc->version = 0;
  sound_desc->revision_level = 0;
  sound_desc->vendor = 0;
  sound_desc->number_of_channels = 0;
  sound_desc->sample_size = 0;
  sound_desc->compression_id = 0;
  sound_desc->packet_size = 0;
  sound_desc->sample_rate = 0;

  sound_desc->samples_per_packet = 0;
  sound_desc->bytes_per_packet = 0;
  sound_desc->bytes_per_frame = 0;
  sound_desc->bytes_per_sample = 0;
  return;
}

sound_desc_t*
sound_desc_new(void)
{
  sound_desc_t *sound_desc;
  sound_desc = (sound_desc_t*)qtime_malloc(sizeof(sound_desc_t));
  if (!sound_desc) return NULL;
  sound_desc_init(sound_desc);
  return sound_desc;
}

void
sound_desc_clean(sound_desc_t *sound_desc)
{
  sound_desc_init(sound_desc);
}

void
sound_desc_delete(sound_desc_t *sound_desc)
{
  qtime_free(sound_desc);
}

sound_desc_t*
sound_desc_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, sound_desc_t *sound_desc_ptr)
{
  sound_desc_t *sound_desc;
  atom_head_t subatom;

  if (sound_desc_ptr) {
    sound_desc = sound_desc_ptr;
    sound_desc_clean(sound_desc);
  } else {
    if ((sound_desc = sound_desc_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  sound_desc->sample_desc.size = atom_head->size;
  sound_desc->sample_desc.data_format = atom_head->type;
  qtime_io_read(qtio, sound_desc->sample_desc.reserved, 6);
  qtime_io_read16(qtio, &sound_desc->sample_desc.data_reference_id);

  qtime_io_read16(qtio, &sound_desc->version);
  qtime_io_read16(qtio, &sound_desc->revision_level);
  qtime_io_read32(qtio, &sound_desc->vendor);
  qtime_io_read16(qtio, &sound_desc->number_of_channels);
  qtime_io_read16(qtio, &sound_desc->sample_size);
  qtime_io_read16(qtio, &sound_desc->compression_id);
  qtime_io_read16(qtio, &sound_desc->packet_size);
  qtime_io_read32(qtio, &sound_desc->sample_rate);

  if (sound_desc->version == 1) {
    qtime_io_read32(qtio, &sound_desc->samples_per_packet);
    qtime_io_read32(qtio, &sound_desc->bytes_per_packet);
    qtime_io_read32(qtio, &sound_desc->bytes_per_frame);
    qtime_io_read32(qtio, &sound_desc->bytes_per_sample);
  }

  if (atom_head->end_offset - 8 <= qtime_io_get_offset(qtio))
    return sound_desc;

  atom_head_init(&subatom);

  while (atom_head->end_offset > subatom.end_offset+7) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    switch (subatom.type) {
      default:
        qtime_error_unknown_atom(QTIME_TYPE_SDSC, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  return sound_desc;

fail:
  if (sound_desc_ptr)
    sound_desc_clean(sound_desc);
  else
    sound_desc_delete(sound_desc);
  qtime_error_atom_read(QTIME_TYPE_SDSC);
  return NULL;
}

sound_desc_t*
sound_desc_create(sound_desc_t *sound_desc_ptr)
{
  sound_desc_t *sound_desc;

  if (sound_desc_ptr) {
    sound_desc = sound_desc_ptr;
    sound_desc_clean(sound_desc);
  } else {
    if ((sound_desc = sound_desc_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  return sound_desc;
}

int64_t
sound_desc_calc_size(sound_desc_t *sound_desc)
{
  int64_t size;

  size = SAMPLE_DESC_HEAD_SIZE + SOUND_DESC_PROP_SIZE;
  if (sound_desc->version == 1)
    size += SOUND_DESC_V1_PROP_SIZE;
  sound_desc->sample_desc.size = size;
  return size;
}

int
sound_desc_write_atom(qtime_io_t *qtio, sound_desc_t *sound_desc)
{
  atom_head_t atom_head;

  atom_head_init(&atom_head);
  atom_head.size = sound_desc->sample_desc.size;
  atom_head.type = sound_desc->sample_desc.data_format;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_SDSC);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, sound_desc->sample_desc.reserved, 6);
  qtime_io_write16(qtio, &sound_desc->sample_desc.data_reference_id);

  qtime_io_write16(qtio, &sound_desc->version);
  qtime_io_write16(qtio, &sound_desc->revision_level);
  qtime_io_write32(qtio, &sound_desc->vendor);
  qtime_io_write16(qtio, &sound_desc->number_of_channels);
  qtime_io_write16(qtio, &sound_desc->sample_size);
  qtime_io_write16(qtio, &sound_desc->compression_id);
  qtime_io_write16(qtio, &sound_desc->packet_size);
  qtime_io_write32(qtio, &sound_desc->sample_rate);

  if (sound_desc->version == 1) {
    qtime_io_write32(qtio, &sound_desc->samples_per_packet);
    qtime_io_write32(qtio, &sound_desc->bytes_per_packet);
    qtime_io_write32(qtio, &sound_desc->bytes_per_frame);
    qtime_io_write32(qtio, &sound_desc->bytes_per_sample);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_SDSC);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
sound_desc_dump(const char *parent_types, sound_desc_t *sound_desc)
{
  int len = strlen(parent_types);
  uint8_t types[len+12];
  double rate = sound_desc->sample_rate;
  char vendor[5];
  char data_format[5];
  rate /= 65536;

  qtime_type_to_str(sound_desc->vendor, vendor);
  qtime_type_to_str(sound_desc->sample_desc.data_format, data_format);
  sprintf(types, "%s.%s", parent_types, "SDSC");

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)sound_desc->sample_desc.size);
  fprintf(stdout, "%s: data format  '%.4s'\n", types, data_format);
  fprintf(stdout, "%s: data reference id %d\n", types, sound_desc->sample_desc.data_reference_id);

  fprintf(stdout, "%s: version      %d\n", types, sound_desc->version);
  fprintf(stdout, "%s: vendor       %.4s\n", types, vendor);
  fprintf(stdout, "%s: number of channels %d\n", types, sound_desc->number_of_channels);
  fprintf(stdout, "%s: sample size  %d\n", types, sound_desc->sample_size);
  fprintf(stdout, "%s: compression id %d\n", types, sound_desc->compression_id);
  fprintf(stdout, "%s: packet size  %d\n", types, sound_desc->packet_size);
  fprintf(stdout, "%s: sample rate  %f\n", types, rate);
  if (sound_desc->version == 1) {
    fprintf(stdout, "%s: samples per packet %d\n", types, sound_desc->samples_per_packet);
    fprintf(stdout, "%s: bytes per packet %d\n", types, sound_desc->bytes_per_packet);
    fprintf(stdout, "%s: bytes per frame  %d\n", types, sound_desc->bytes_per_frame);
    fprintf(stdout, "%s: bytes per sample %d\n", types, sound_desc->bytes_per_sample);
  }
}

