/* smhd.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "smhd.h"

void
smhd_init(smhd_t *smhd)
{
  atom_init(&smhd->atom);
  smhd->atom.type = QTIME_TYPE_SMHD;
  smhd->version = 0;
  qtime_flags_set(smhd->flags, 0);
  smhd->balance = 0;
  smhd->reserved = 0;
}

void
smhd_clean(smhd_t *smhd)
{
  qtime_error_type_check(QTIME_TYPE_SMHD, smhd->atom.type)
  smhd_init(smhd);
}

smhd_t*
smhd_new(void)
{
  smhd_t *smhd;
  smhd = (smhd_t*)qtime_malloc(sizeof(smhd_t));
  if (!smhd) return NULL;
  smhd_init(smhd);
  return smhd;
}

void
smhd_delete(smhd_t *smhd)
{
  qtime_error_type_check(QTIME_TYPE_SMHD, smhd->atom.type)
  qtime_free(smhd);
}

smhd_t*
smhd_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, smhd_t *smhd_ptr)
{
  smhd_t *smhd;

#ifndef NDEBUG
  if (atom_head->media_type != QTIME_MEDIA_TYPE_UNKNOWN &&
      atom_head->media_type != QTIME_MEDIA_TYPE_SOUND) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    qtime_error_invalid_media(QTIME_TYPE_SMHD, QTIME_MEDIA_TYPE_VIDEO, atom_head->media_type);
    return NULL;
  }
#endif

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_SMHD, atom_head->type, NULL)
  if (smhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_SMHD, smhd_ptr->atom.type, NULL)
    smhd = smhd_ptr;
    smhd_clean(smhd);
  } else {
    if ((smhd = smhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  smhd->atom.size = atom_head->size;
  smhd->atom.type = QTIME_TYPE_SMHD;
  smhd->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &smhd->version, 1);
  qtime_io_read(qtio, smhd->flags, 3);
  qtime_io_read16(qtio, &smhd->balance);
  qtime_io_read16(qtio, &smhd->reserved);

  atom_head->media_type = QTIME_MEDIA_TYPE_SOUND;

  return smhd;
}

smhd_t*
smhd_create(smhd_t *smhd_ptr)
{
  smhd_t *smhd;

  if (smhd_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_SMHD, smhd_ptr->atom.type, NULL)
    smhd = smhd_ptr;
    smhd_clean(smhd);
  } else {
    if ((smhd = smhd_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return smhd;
}

int64_t
smhd_calc_size(smhd_t *smhd)
{
  int64_t size = 0;
  qtime_error_type_check_v(QTIME_TYPE_SMHD, smhd->atom.type, 0)
  size = 8 + SMHD_PROP_SIZE;
  smhd->atom.size = size;
  return size;
}

int
smhd_write_atom(qtime_io_t *qtio, smhd_t *smhd)
{
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_SMHD, smhd->atom.type, QTIME_ERROR_ILLEGAL_ATOM);

  atom_head_init(&atom_head);
  atom_head.size = smhd->atom.size;
  atom_head.type = smhd->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_SMHD);
    return QTIME_ERROR_WRITE;
  }

  qtime_io_write(qtio, &smhd->version, 1);
  qtime_io_write(qtio, smhd->flags, 3);
  qtime_io_write16(qtio, &smhd->balance);
  qtime_io_write16(qtio, &smhd->reserved);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_SMHD);
    return QTIME_ERROR_WRITE;
  }

  return QTIME_OK;
}

void
smhd_dump(const char *parent_types, smhd_t *smhd)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  qtime_type_to_str(smhd->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(smhd->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)smhd->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, smhd->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: balance      %f\n", types, qtime_fixed16_to_float(smhd->balance));
//  fprintf(stdout, "%s: reserved     %d\n", types, smhd->graphics_mode);
}

