/* qtime_util.c */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <time.h>
#include <endian.h>
#include <ctype.h>

#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_be.h"

static int imm_exit = 1;

int
qtime_fps_to_time_scale_duration(double fps, uint32_t *time_scale, uint32_t *duration)
{
  uint32_t scale, dura;

  if (fps - (int)fps != 0) {
    scale = (int)(fps * 1000 + 0.5);
    dura = 1000;
  } else {
    scale = fps;
    dura = 1;
  }
  *time_scale = scale;
  *duration = dura;
#if 0
  int scale = 600;

  if (fps - (int)fps != 0)
    scale = (int)(fps * 1001 + 0.5);
  else if ((600 / fps) - (int)(600 / fps) != 0)
    scale = (int)(fps * 100 + 0.5);

  *time_scale = scale;
  *duration = (uint32_t)((double)scale / fps + 0.5);
#endif
  return QTIME_OK;
}

int
qtime_time_scale_duration_to_fps(uint32_t time_scale, uint32_t duration, double *fps)
{
  *fps = (double)time_scale / (double)duration;
  return QTIME_OK;
}

void
qtime_flags_set(uint8_t *flags, uint32_t f)
{
#ifndef WORDS_BIGENDIAN
  flags[0] = ((UC32*)&(f))->c[2];
  flags[1] = ((UC32*)&(f))->c[1];
  flags[2] = ((UC32*)&(f))->c[0];
#else
  flags[0] = ((UC32*)&(f))->c[1];
  flags[1] = ((UC32*)&(f))->c[2];
  flags[2] = ((UC32*)&(f))->c[3];
#endif
}

void
qtime_flags_get(uint8_t *flags, uint32_t *f)
{
#ifndef WORDS_BIGENDIAN
  ((UC32*)(f))->c[3] = 0;
  ((UC32*)(f))->c[2] = flags[0];
  ((UC32*)(f))->c[1] = flags[1];
  ((UC32*)(f))->c[0] = flags[2];
#else
  ((UC32*)(f))->c[0] = 0;
  ((UC32*)(f))->c[1] = flags[0];
  ((UC32*)(f))->c[2] = flags[1];
  ((UC32*)(f))->c[3] = flags[2];
#endif
}

int32_t
qtime_float_to_fixed32(float f)
{
  int32_t i;
  f *= 65536.;
  if (f > INT32_MAX)
    i = INT32_MAX;
  else if (f < INT32_MIN)
    i = INT32_MIN;
  else
    i = (int32_t)f;
  return i;
}

float
qtime_fixed32_to_float(int32_t i)
{
  float f;
  f = (float)i;
  f /= 65536.;
  return f;
}

int16_t
qtime_float_to_fixed16(float f)
{
  int i;
  f *= 256.;
  if (f > INT16_MAX)
    i = INT16_MAX;
  else if (f < INT16_MIN)
    i = INT16_MIN;
  else
    i = (int16_t)f;
  return (int16_t)i;
}

float
qtime_fixed16_to_float(int16_t i)
{
  float f;
  f = (float)i;
  f /= 256.;
  return f;
}

void*
qtime_malloc(int byte_size)
{
  void *p;

#ifndef NDEBUG
  if (byte_size < 0) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: memory allocate size value negative %d.\n",byte_size);
    if (imm_exit)
      exit(1);
  }
#endif

  p = (void*) malloc(byte_size);
  if (!p) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    qtime_error_memory(byte_size);
    if (imm_exit)
      exit(1);
  }
  return p;
}


void*
qtime_realloc(void *ptr, int byte_size)
{
  void *p;

#ifndef NDEBUG
  if (byte_size < 0) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: memory allocate size value negative %d.\n",byte_size);
    if (imm_exit)
      exit(1);
  }
#endif

  p = (void*) realloc(ptr, byte_size);
  if (!p) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    qtime_error_memory(byte_size);
    if (imm_exit)
      exit(1);
  }
  return p;
}

uint32_t
qtime_current_time(void)
{
  time_t t;
  t = time(NULL);
  return (t+(66*31536000)+1468800);
}

char*
qtime_ctime_r(uint32_t tim, char *buf)
{
  tim -= (66*31536000)+1468800;
  return ctime_r((time_t*)&tim, buf);
}

void
qtime_type_to_str(uint32_t type, char str[5])
{
  int i;
  QTIME_BE32(type, *str);
  for (i = 0; i < 4; i++)
    str[i] = (isprint(str[i])) ? str[i] : ' ';
  str[4] = '\0';
}

void
qtime_type_to_char(uint32_t type, char ch[4])
{
  QTIME_BE32(type, *((uint32_t*)ch));
}

void
qtime_char_to_type(char ch[4], uint32_t *type)
{
  QTIME_BE32(*((uint32_t*)ch), *type);
}

