/* qtime_sample.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_sample.h"
#include "qtime_media_type.h"
#include "qtime_sample_desc.h"
#include "atom.h"
#include "stbl.h"
#include "sample_desc.h"
#include "video_desc.h"
#include "sound_desc.h"


void
qtime_sample_info_init(qtime_sample_info_t *qtsp_info)
{
  qtsp_info->count = 0;
  qtsp_info->duration = 0;
  qtsp_info->flag = 0;
  qtsp_info->samples = 0;
  qtsp_info->sample_size = 0;
  qtsp_info->sample_duration = 0;
  qtsp_info->chunk = 0;
  qtsp_info->chunk_start_count = 0;
  qtsp_info->samples_per_chunk = 0;
  qtsp_info->sample_description_id = 0;
  qtsp_info->chunk_samples = 0;
  qtsp_info->last_chunk = 0;
  qtsp_info->last_sample_description_id = 0;
  qtsp_info->sample_desc = NULL;
  qtsp_info->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
  qtsp_info->bytes_per_sample = 0;
  qtsp_info->last_count = 0;
  qtsp_info->last_data_offset = -1;
  qtsp_info->data_start_offset = 0;
  qtsp_info->data_end_offset = 0;
}

void
qtime_sample_init(qtime_sample_t *qtsp)
{
  qtsp->stbl = NULL;
  qtsp->stsd = NULL;
  qtsp->stts = NULL;
  qtsp->stss = NULL;
  qtsp->stsc = NULL;
  qtsp->stsz = NULL;
  qtsp->stco = NULL;

  qtsp->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
}

void
qtime_sample_clean(qtime_sample_t *qtsp)
{
  qtime_sample_init(qtsp);
}

qtime_sample_t*
qtime_sample_new(void)
{
  qtime_sample_t *qtsp;
  qtsp = (qtime_sample_t*)qtime_malloc(sizeof(qtime_sample_t));
  if (!qtsp) return NULL;
  qtime_sample_init(qtsp);
  return qtsp;
}

void
qtime_sample_delete(qtime_sample_t *qtsp)
{
  qtime_free(qtsp);
}

qtime_sample_t*
qtime_sample_create(qtime_sample_t *qtsp_ptr, stbl_t *stbl)
{
  qtime_sample_t *qtsp;

  if (stbl == NULL) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: 'stbl' atom is NULL, cannot create qtime_sample.\n");
    return NULL;
  }

  if (qtsp_ptr) {
    qtsp = qtsp_ptr;
    qtime_sample_clean(qtsp);
  } else {
    if ((qtsp = qtime_sample_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  qtsp->stbl = stbl;
  qtsp->stsd = (stsd_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STSD);
  qtsp->stts = (stts_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STTS);
  qtsp->stss = (stss_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STSS);
  qtsp->stsc = (stsc_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STSC);
  qtsp->stsz = (stsz_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STSZ);
  qtsp->stco = (stco_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STCO);
  if (qtsp->stco == NULL)
    qtsp->stco = (stco_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_CO64);
  qtsp->media_type = stsd_get_media_type(qtsp->stsd);

  return qtsp;
}

uint32_t
qtime_sample_get_media_type(qtime_sample_t *qtsp)
{
  qtime_error_media_check_v(QTIME_TYPE_QTSP, qtsp->media_type, stsd_get_media_type(qtsp->stsd), QTIME_ERROR_INVALID_MEDIA)

  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN)
    qtsp->media_type = stsd_get_media_type(qtsp->stsd);
  return qtsp->media_type;
}

int
qtime_sample_set_media_type(qtime_sample_t *qtsp, uint32_t media_type)
{
  qtime_error_media_check_v(QTIME_TYPE_QTSP, qtsp->media_type, media_type, QTIME_ERROR_INVALID_MEDIA)
  qtsp->media_type = media_type;
  if (stbl_set_media_type(qtsp->stbl, media_type) != QTIME_OK)
    return QTIME_ERROR;
  if (stsd_set_media_type(qtsp->stsd, media_type) != QTIME_OK)
    return QTIME_ERROR;
  return QTIME_OK;
}

int64_t
qtime_sample_get_total_size(qtime_sample_t *qtsp)
{
  return stsz_get_total_size(qtsp->stsz);
}

uint32_t
qtime_sample_get_max_count(qtime_sample_t *qtsp)
{
  return stts_get_max_count(qtsp->stts);
}

uint32_t
qtime_sample_get_max_duration(qtime_sample_t *qtsp)
{
  return stts_get_max_duration(qtsp->stts);
}

uint32_t
qtime_sample_duration_to_count(qtime_sample_t *qtsp, uint32_t duration)
{
  uint32_t count;
  count = stts_get_count(qtsp->stts, duration);
  return count;
}

uint32_t
qtime_sample_count_to_duration(qtime_sample_t *qtsp, uint32_t count)
{
  uint32_t duration;
  duration = stts_get_duration(qtsp->stts, count);
  return duration;
}

sample_desc_t*
qtime_sample_get_sample_desc(qtime_sample_t *qtsp, int id)
{
  sample_desc_t *sample_desc;
  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: media type is unknown.\n");
  }
  sample_desc = stsd_get_sample_desc(qtsp->stsd, id);
  return sample_desc;
}

int32_t
qtime_sample_add_sample_desc(qtime_sample_t *qtsp, sample_desc_t *sample_desc)
{
  int32_t id;
#ifndef NDEBUG
  if (qtsp->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: media type is unknown.\n");
  }
  if (stsd_get_media_type(qtsp->stsd) == QTIME_MEDIA_TYPE_UNKNOWN) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: media type is unknown.\n");
  }
#endif
  id = stsd_add_sample_desc(qtsp->stsd, sample_desc);
  return id;
}

int
qtime_sample_set_constant_duration(qtime_sample_t *qtsp, uint32_t sample_duration)
{
  return stts_set_constant_duration(qtsp->stts, 0, sample_duration);
}

int
qtime_sample_is_keyframe(qtime_sample_t *qtsp, uint32_t count)
{
  if (!qtsp->stss)
    return 1;
  return stss_is_keyframe(qtsp->stss, count);
}

uint32_t
qtime_sample_next_keyframe(qtime_sample_t *qtsp, uint32_t count)
{
  if (!qtsp->stss)
    return count+1;
  return stss_get_next_keyframe(qtsp->stss, count);
}

uint32_t
qtime_sample_prev_keyframe(qtime_sample_t *qtsp, uint32_t count)
{
  if (!qtsp->stss)
    return count-1;
  return stss_get_prev_keyframe(qtsp->stss, count);
}

int
qtime_sample_get_sample_info(qtime_sample_t *qtsp, qtime_sample_info_t *spinfo)
{
  uint32_t count, max_count, end_count;
  sc_info_t si;
  int64_t offset;

  max_count = stts_get_max_count(qtsp->stts);
  count = spinfo->count;
  if (count >= max_count) {
    spinfo->count = max_count;
    spinfo->duration = stts_get_max_duration(qtsp->stts);
    spinfo->samples = 0;
    spinfo->sample_size = 0;
    spinfo->sample_duration = 0;
    spinfo->flag = QTIME_SAMPLE_INFO_FLAG_OVER_MAX;
    return QTIME_ERROR;
  }
  end_count = count + spinfo->samples;
  if (end_count > max_count) {
    spinfo->samples = max_count - count;
  }

  spinfo->last_sample_description_id = spinfo->sample_description_id;
  spinfo->last_chunk = spinfo->chunk;
  si.count = spinfo->count;
  stsc_get_chunk(qtsp->stsc, &si);
  spinfo->chunk = si.chunk;
  spinfo->chunk_start_count = si.chunk_start_count;
  spinfo->samples_per_chunk = si.samples_per_chunk;
  spinfo->sample_description_id = si.sample_description_id;
  spinfo->chunk_samples = si.samples;
  if (spinfo->samples > spinfo->chunk_samples)
    spinfo->samples = spinfo->chunk_samples;

  if (spinfo->last_chunk != spinfo->chunk) {
    spinfo->last_count = spinfo->chunk_start_count;
    spinfo->last_data_offset = stco_get_offset(qtsp->stco, spinfo->chunk);
  }

  spinfo->sample_desc = stsd_get_sample_desc(qtsp->stsd, spinfo->sample_description_id);
  spinfo->media_type = stsd_get_media_type(qtsp->stsd);
  switch (spinfo->media_type) {
    sound_desc_t *sound_desc;
    case QTIME_MEDIA_TYPE_VIDEO:
      spinfo->bytes_per_sample = 0;
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      sound_desc = (sound_desc_t*)spinfo->sample_desc;
      if (sound_desc->version == 0) {
        spinfo->bytes_per_sample = sound_desc->number_of_channels *
                                  (sound_desc->sample_size/8);
        if (!spinfo->bytes_per_sample) {
	  qtime_error_debug_info(QTIME_ERROR);
	  fprintf(stderr, "QTIME_ERROR: qtime_sample_get_info: bytes_per_sample is '0'.\n");
	}
      } else {
        spinfo->bytes_per_sample = 0;
      }
      break;
    default:
      spinfo->bytes_per_sample = 0;
      break;
  }
  if (spinfo->bytes_per_sample) {
    spinfo->samples = (spinfo->chunk_samples < spinfo->sample_duration) ?
                       spinfo->chunk_samples : spinfo->sample_duration;
  }

  offset = spinfo->last_data_offset;
  if (spinfo->bytes_per_sample) {
    if (spinfo->count > spinfo->last_count)
      offset += (spinfo->count - spinfo->last_count) * spinfo->bytes_per_sample;
    else if (spinfo->count < spinfo->last_count)
      offset -= (spinfo->last_count - spinfo->count) * spinfo->bytes_per_sample;
    spinfo->sample_size = spinfo->samples * spinfo->bytes_per_sample;
  } else {
    if (spinfo->count > spinfo->last_count)
      offset += stsz_get_sizes(qtsp->stsz, spinfo->last_count, spinfo->count - spinfo->last_count);
    else
      offset -= stsz_get_sizes(qtsp->stsz, spinfo->count, spinfo->last_count - spinfo->count);
    spinfo->sample_size = stsz_get_sizes(qtsp->stsz, spinfo->count, spinfo->samples);
  }
  spinfo->data_start_offset = offset;
  offset += spinfo->sample_size;
  spinfo->data_end_offset = offset;
  spinfo->last_data_offset = offset;

  if (stss_is_keyframe(qtsp->stss, spinfo->count))
    spinfo->flag |= QTIME_SAMPLE_INFO_FLAG_KEYFRAME;
  else
    spinfo->flag &= ~QTIME_SAMPLE_INFO_FLAG_KEYFRAME;

  spinfo->duration = stts_get_duration(qtsp->stts, spinfo->count);
  if (spinfo->bytes_per_sample) {
    spinfo->sample_duration = spinfo->samples;
  } else {
    spinfo->sample_duration =stts_get_sample_duration(qtsp->stts,spinfo->count);
  }

  spinfo->last_count = spinfo->count + spinfo->samples;

  return QTIME_OK;
}

#if 0
int
qtime_sample_get_sample_info(qtime_sample_t *qtsp, qtime_sample_info_t *spinfo)
{
  uint32_t count, max_count, end_count;

  max_count = stts_get_max_count(qtsp->stts);
  count = spinfo->count;
  if (count >= max_count) {
    spinfo->count = max_count;
    spinfo->duration = stts_get_max_duration(qtsp->stts);
    spinfo->samples = 0;
    spinfo->sample_size = 0;
    spinfo->sample_duration = 0;
    spinfo->flag = QTIME_SAMPLE_INFO_FLAG_OVER_MAX;
    return QTIME_ERROR;
  }
  end_count = count + spinfo->samples;
  if (end_count > max_count) {
    spinfo->samples = max_count - count;
  }

  stsc_get_sample_info(qtsp->stsc, spinfo);
  stco_get_sample_info(qtsp->stco, spinfo);
  stsd_get_sample_info(qtsp->stsd, spinfo);
  if (spinfo->bytes_per_sample) {
    spinfo->samples = (spinfo->chunk_samples < spinfo->sample_duration) ?
                       spinfo->chunk_samples : spinfo->sample_duration;
  }
  stsz_get_sample_info(qtsp->stsz, spinfo);
  stss_get_sample_info(qtsp->stss, spinfo);
  stts_get_sample_info(qtsp->stts, spinfo);

  return QTIME_OK;
}
#endif

int
qtime_sample_add_sample_info(qtime_sample_t *qtsp, qtime_sample_info_t *spinfo)
{
  sample_desc_t *sample_desc;
  sc_info_t si;

  sample_desc = stsd_get_sample_desc(qtsp->stsd, spinfo->sample_description_id);
  if (!spinfo->sample_desc)
    spinfo->sample_desc = sample_desc;
#if 0
#ifndef NDEBUG
  if (sample_desc != spinfo->sample_desc) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: qtime_sample_add_sample_info: sample_desc doesn't suit.\n");
  }
#endif
#endif
#ifndef NDEBUG
  { uint32_t media_type;
    media_type = stsd_get_media_type(qtsp->stsd);
    if (media_type != spinfo->media_type) {
      qtime_error_debug_info(QTIME_ERROR);
      fprintf(stderr, "QTIME_ERROR: qtime_sample_add_sample_info: media_type doesn't suit.\n");
    }
  }
#endif
  switch (spinfo->media_type) {
    sound_desc_t *sound_desc;
    case QTIME_MEDIA_TYPE_VIDEO:
      spinfo->bytes_per_sample = 0;
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      sound_desc = (sound_desc_t*)sample_desc;
      if (sound_desc->version == 0) {
        spinfo->bytes_per_sample = sound_desc->number_of_channels *
                                   (sound_desc->sample_size/8);
        if (!spinfo->bytes_per_sample)
          spinfo->bytes_per_sample = 1;
      } else {
       spinfo->bytes_per_sample = 0;
      }
      break;
    default:
      spinfo->bytes_per_sample = 0;
      break;
  }
#ifndef NDEBUG
  if (!spinfo->bytes_per_sample && spinfo->samples > 1) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: qtime_sample_add_sample_info: too many samples, %d.\n", spinfo->samples);
    return QTIME_ERROR;
  }
#endif
  if (spinfo->bytes_per_sample) {
    spinfo->samples = spinfo->sample_duration;
    spinfo->sample_duration = 1;
  }

  if (spinfo->last_data_offset == spinfo->data_start_offset)
    si.consecutive_data = 1;
  else
    si.consecutive_data = 0;
  spinfo->last_data_offset = spinfo->data_end_offset;
//  si.count = spinfo->count;
  si.sample_description_id = spinfo->sample_description_id;
  si.samples = spinfo->samples;
  stsc_add_chunk(qtsp->stsc, &si);
  spinfo->last_chunk = spinfo->chunk;
  spinfo->chunk = si.chunk;

  if (spinfo->last_chunk != spinfo->chunk)
    stco_add_offset(qtsp->stco, spinfo->data_start_offset);

  if (spinfo->bytes_per_sample)
    stsz_add_sizes(qtsp->stsz, 1, spinfo->samples);
  else
    stsz_add_size(qtsp->stsz, spinfo->sample_size);

  if (spinfo->flag & QTIME_SAMPLE_INFO_FLAG_KEYFRAME)
    stss_add_keyframe(qtsp->stss, spinfo->count);

  stts_add_sample_duration(qtsp->stts,spinfo->samples,spinfo->sample_duration);

  spinfo->last_count = spinfo->count;
  spinfo->count += spinfo->samples;

  return QTIME_OK;
}

#if 0
int
qtime_sample_add_sample_info(qtime_sample_t *qtsp, qtime_sample_info_t *spinfo)
{
  stsd_add_sample_info(qtsp->stsd, spinfo);
  if (!spinfo->bytes_per_sample && spinfo->samples > 1) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: qtime_sample_add_sample_info: too many samples, %d.\n", spinfo->samples);
    return QTIME_ERROR;
  }
  if (spinfo->bytes_per_sample) {
    spinfo->samples = spinfo->sample_duration;
    spinfo->sample_duration = 1;
  }
  stsc_add_sample_info(qtsp->stsc, spinfo);
  stco_add_sample_info(qtsp->stco, spinfo);
  stsz_add_sample_info(qtsp->stsz, spinfo);
  stss_add_sample_info(qtsp->stss, spinfo);
  stts_add_sample_info(qtsp->stts, spinfo);

  spinfo->last_count = spinfo->count;
  spinfo->count += spinfo->samples;

  return QTIME_OK;
}
#endif

