/* qtime_media.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media.h"
#include "qtime_media_type.h"
#include "qtime_sample.h"
#include "atom.h"
#include "mdia.h"
#include "dref.h"
#include "sample_desc.h"
#include "video_desc.h"
#include "sound_desc.h"


void
qtime_media_init(qtime_media_t *qtmd)
{
  qtmd->mdia = NULL;
  qtmd->mdhd = NULL;
  qtmd->hdlr = NULL;
  qtmd->minf = NULL;

  qtime_sample_init(&qtmd->qtsp);
  qtime_sample_info_init(&qtmd->spinfo_rd);
  qtime_sample_info_init(&qtmd->spinfo_wr);
  qtmd->moio = NULL;
  qtmd->media_type = QTIME_MEDIA_TYPE_UNKNOWN;

  qtmd->sample_duration = 0;
  qtmd->sample_description_id = 0;
  qtmd->sample_desc = NULL;
  qtmd->data_reference_id = 0;
}

void
qtime_media_clean(qtime_media_t *qtmd)
{
  qtime_sample_clean(&qtmd->qtsp);
  qtime_media_init(qtmd);
}

qtime_media_t*
qtime_media_new(void)
{
  qtime_media_t *qtmd;
  qtmd = (qtime_media_t*)qtime_malloc(sizeof(qtime_media_t));
  if (!qtmd) return NULL;
  qtime_media_init(qtmd);
  return qtmd;
}

void
qtime_media_delete(qtime_media_t *qtmd)
{
  qtime_media_clean(qtmd);
  qtime_free(qtmd);
}

qtime_media_t*
qtime_media_create(qtime_media_t *qtmd_ptr, mdia_t *mdia)
{
  qtime_media_t *qtmd;
  minf_t *minf;
  stbl_t *stbl;
  int desc_id;

  if (mdia == NULL) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: 'mdia' atom is NULL, cannot create qtime_media.\n");
    return NULL;
  }

  if (qtmd_ptr) {
    qtmd = qtmd_ptr;
    qtime_media_clean(qtmd);
  } else {
    if ((qtmd = qtime_media_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  qtmd->mdia = mdia;
  qtmd->mdhd = (mdhd_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_MDHD);
  qtmd->hdlr = (hdlr_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_HDLR);

  minf = (minf_t*)atom_find_child((atom_t*)mdia, QTIME_TYPE_MINF);
  qtmd->minf = minf;
  qtmd->minf_hdlr = (hdlr_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_HDLR);

  stbl = (stbl_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_STBL);
  qtime_sample_create(&qtmd->qtsp, stbl);

  qtmd->media_type = qtime_media_get_media_type(qtmd);

  if (qtmd->media_type != QTIME_MEDIA_TYPE_UNKNOWN) {
    desc_id = 1;
    qtmd->sample_desc = qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
    if (qtmd->sample_desc) {
      qtmd->sample_description_id = desc_id;
      qtmd->data_reference_id = qtmd->sample_desc->data_reference_id;
    }
  }

  return qtmd;
}

int
qtime_media_set_moov_io(qtime_media_t *qtmd, qtime_io_t *moio)
{
  qtmd->moio = moio;
//  qtmd->qtsp.moio = moio;
  return QTIME_OK;
}

qtime_sample_t*
qtime_media_get_qtime_sample(qtime_media_t* qtmd)
{
  return &qtmd->qtsp;
}

uint32_t
qtime_media_get_media_type(qtime_media_t *qtmd)
{
  if (qtmd->media_type != QTIME_MEDIA_TYPE_UNKNOWN)
    return qtmd->media_type;
  qtmd->media_type = mdia_get_media_type(qtmd->mdia);
  return qtmd->media_type;
}

int
qtime_media_set_media_type(qtime_media_t *qtmd, uint32_t media_type)
{
  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, media_type, QTIME_ERROR_INVALID_MEDIA)

  qtmd->media_type = media_type;

  if (mdia_set_media_type(qtmd->mdia, media_type) != QTIME_OK) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: mdia_set_media_type failed.\n");
    return QTIME_ERROR;
  }

  if (qtime_sample_set_media_type(&qtmd->qtsp, media_type) != QTIME_OK) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: qtime_sample_set_media_type failed.\n");
    return QTIME_ERROR;
  }

  return QTIME_OK;
}

int
qtime_media_set_data_reference(qtime_media_t *qtmd, uint8_t *name)
{
  int len = 0;
  dref_t *dref;

  hdlr_set_data_handler(qtmd->minf_hdlr, QTIME_TYPE_ALIS);

  if (name)
    len = strlen(name);
  else {
    name = "";
    len = 1;
  }
  dref = (dref_t*)atom_find_recursive((atom_t*)qtmd->minf, QTIME_TYPE_DREF);
  return dref_add_data_ref(dref, QTIME_TYPE_ALIS, DREF_FLAG_SELF_REFERENCE, name, len);
}

uint32_t
qtime_media_get_time_scale(qtime_media_t *qtmd)
{
  return mdia_get_media_time_scale(qtmd->mdia);
}

int
qtime_media_set_time_scale(qtime_media_t *qtmd, uint32_t time_scale)
{
  mdhd_set_time_scale(qtmd->mdhd, time_scale);
  return QTIME_OK;
}

uint32_t
qtime_media_get_max_count(qtime_media_t *qtmd)
{
  return qtime_sample_get_max_count(&qtmd->qtsp);
}

uint32_t
qtime_media_get_max_duration(qtime_media_t *qtmd)
{
  return qtime_sample_get_max_duration(&qtmd->qtsp);
}

uint32_t
qtime_media_get_count(qtime_media_t *qtmd)
{
  return qtmd->spinfo_rd.count;
}

uint32_t
qtime_media_set_count(qtime_media_t *qtmd, uint32_t count)
{
  return qtmd->spinfo_rd.count = count;
}

uint32_t
qtime_media_get_duration(qtime_media_t *qtmd)
{
  return qtmd->spinfo_rd.duration;
}

uint32_t
qtime_media_set_duration(qtime_media_t *qtmd, uint32_t duration)
{
  uint32_t count;
  count = qtime_sample_duration_to_count(&qtmd->qtsp, duration);
  qtmd->spinfo_rd.count = count;
  duration = qtime_sample_count_to_duration(&qtmd->qtsp, count);
  return duration;
}

int
qtime_media_read_video(qtime_media_t *qtmd, uint8_t *buf, int *keyframe)
{
  qtime_io_data_t qtio_data;
  int size;

  qtmd->spinfo_rd.samples = 1;
  if (qtime_sample_get_sample_info(&qtmd->qtsp, &qtmd->spinfo_rd) < 0)
    return QTIME_ERROR;
  if (qtmd->spinfo_rd.sample_description_id != qtmd->sample_description_id) {
    qtmd->sample_description_id = qtmd->spinfo_rd.sample_description_id;
    qtmd->data_reference_id = qtmd->spinfo_rd.sample_desc->data_reference_id;
  }
  qtio_data.size = qtmd->spinfo_rd.sample_size;
  qtio_data.start_offset = qtmd->spinfo_rd.data_start_offset;
  qtio_data.data = buf;
  size = qtime_io_read_data(qtmd->moio, &qtio_data);
  if (qtmd->spinfo_rd.flag & QTIME_SAMPLE_INFO_FLAG_KEYFRAME)
    *keyframe = 1;
  else
    *keyframe = 0;
//  fprintf(stdout, "qtmd: ct %u, size %u, off %lld, key %d.\n", qtmd->spinfo_rd.count, qtmd->spinfo_rd.sample_size, qtmd->spinfo_rd.data_start_offset, *keyframe);
  qtmd->spinfo_rd.count += qtmd->spinfo_rd.samples;
  return size;
}

int
qtime_media_read_audio(qtime_media_t *qtmd, uint8_t *buf, int *num_samples)
{
  qtime_io_data_t qtio_data;
  int size;

  qtmd->spinfo_rd.samples = 1;
  qtmd->spinfo_rd.sample_duration = *num_samples;
  if (qtime_sample_get_sample_info(&qtmd->qtsp, &qtmd->spinfo_rd) < 0)
    return QTIME_ERROR;
  if (qtmd->spinfo_rd.sample_description_id != qtmd->sample_description_id) {
    qtmd->sample_description_id = qtmd->spinfo_rd.sample_description_id;
    qtmd->data_reference_id = qtmd->spinfo_rd.sample_desc->data_reference_id;
  }
  qtio_data.size = qtmd->spinfo_rd.sample_size;
  qtio_data.start_offset = qtmd->spinfo_rd.data_start_offset;
  qtio_data.data = buf;
  size = qtime_io_read_data(qtmd->moio, &qtio_data);
  *num_samples = qtmd->spinfo_rd.sample_duration;
//  fprintf(stdout, "qtmd: ct %u, size %u, off %lld, key %d.\n", qtmd->spinfo_rd.count, qtmd->spinfo_rd.sample_size, qtmd->spinfo_rd.data_start_offset, *keyframe);
  qtmd->spinfo_rd.count += qtmd->spinfo_rd.samples;
  return size;
}

int
qtime_media_write_video(qtime_media_t *qtmd, uint8_t *buf, int size, int keyframe)
{
  qtime_io_data_t qtio_data;
  qtio_data.size = size;
  qtio_data.data = buf;
  if (qtime_io_write_data(qtmd->moio, &qtio_data) < 0) {
    return QTIME_ERROR;
  }
  if (qtio_data.size != size) {
    qtime_error_debug_info(QTIME_ERROR_IO);
    return QTIME_ERROR_IO;
  }

  qtmd->spinfo_wr.samples = 1;
  qtmd->spinfo_wr.sample_size = qtio_data.size;
  qtmd->spinfo_wr.sample_duration = qtmd->sample_duration;
  qtmd->spinfo_wr.sample_description_id = qtmd->sample_description_id;
  if (keyframe)
    qtmd->spinfo_wr.flag = QTIME_SAMPLE_INFO_FLAG_KEYFRAME;
  else
    qtmd->spinfo_wr.flag = 0;
  qtmd->spinfo_wr.data_start_offset = qtio_data.start_offset;
  qtmd->spinfo_wr.data_end_offset = qtio_data.end_offset;
  qtmd->spinfo_wr.media_type = qtmd->media_type;

  qtime_sample_add_sample_info(&qtmd->qtsp, &qtmd->spinfo_wr);

  return qtmd->spinfo_wr.sample_size;
}

int
qtime_media_write_audio(qtime_media_t *qtmd, uint8_t *buf, int size, int number_of_samples)
{
  qtime_io_data_t qtio_data;
  qtio_data.size = size;
  qtio_data.data = buf;
  if (qtime_io_write_data(qtmd->moio, &qtio_data) < 0) {
    return QTIME_ERROR;
  }
  if (qtio_data.size != size) {
    qtime_error_debug_info(QTIME_ERROR_IO);
    return QTIME_ERROR_IO;
  }

  qtmd->spinfo_wr.samples = 1;
  qtmd->spinfo_wr.sample_size = qtio_data.size;
  qtmd->spinfo_wr.sample_duration = number_of_samples;
  qtmd->spinfo_wr.sample_description_id = qtmd->sample_description_id;
  qtmd->spinfo_wr.flag = 0;
  qtmd->spinfo_wr.data_start_offset = qtio_data.start_offset;
  qtmd->spinfo_wr.data_end_offset = qtio_data.end_offset;
  qtmd->spinfo_wr.media_type = qtmd->media_type;

  qtime_sample_add_sample_info(&qtmd->qtsp, &qtmd->spinfo_wr);

  return qtmd->spinfo_wr.sample_size;
}

int
qtime_media_get_compressor(qtime_media_t *qtmd, char compressor[5])
{
  sample_desc_t *desc;
  int32_t desc_id;

  desc_id = 1;
  desc =qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!desc) {
    compressor[0] = compressor[1] = compressor[2] = compressor[3] = '\0';
    return QTIME_ERROR;
  }
  qtime_type_to_char(desc->data_format, compressor);
  compressor[4] = '\0';

  return QTIME_OK;
}

// video

int
qtime_media_get_video_width(qtime_media_t *qtmd)
{
  video_desc_t *vdesc;
  int32_t desc_id;
  int width;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_VIDEO, QTIME_ERROR_INVALID_MEDIA)
  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_VIDEO, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = qtmd->sample_description_id;
  vdesc =(video_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!vdesc)
    return QTIME_ERROR;
  width = vdesc->width;
  return width;
}

int
qtime_media_get_video_height(qtime_media_t *qtmd)
{
  video_desc_t *vdesc;
  int32_t desc_id;
  int height;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_VIDEO, QTIME_ERROR_INVALID_MEDIA)
  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_VIDEO, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = qtmd->sample_description_id;
  vdesc =(video_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!vdesc)
    return QTIME_ERROR;
  height = vdesc->height;
  return height;
}

double
qtime_media_get_video_fps(qtime_media_t *qtmd)
{
  double duration, count, time_scale, fps;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_VIDEO, 0)

  duration = qtime_sample_get_max_duration(&qtmd->qtsp);
  count = qtime_sample_get_max_count(&qtmd->qtsp);
  time_scale = qtime_media_get_time_scale(qtmd);
  fps = duration / count;
  fps = time_scale / fps;

  return fps;
}

int
qtime_media_set_video_fps(qtime_media_t *qtmd, double fps)
{
  uint32_t time_scale, duration;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_VIDEO, QTIME_ERROR_INVALID_MEDIA)

  qtime_fps_to_time_scale_duration(fps, &time_scale, &duration);
  qtmd->sample_duration = duration;
  qtime_media_set_time_scale(qtmd, time_scale);
  qtime_sample_set_constant_duration(&qtmd->qtsp, duration);
  return QTIME_OK;
}

int
qtime_media_set_video(qtime_media_t *qtmd, int width, int height, uint8_t *compressor)
{
  video_desc_t *vdesc;
  int32_t desc_id, dref_id;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_VIDEO, QTIME_ERROR_INVALID_MEDIA)

  if (qtmd->media_type != QTIME_MEDIA_TYPE_VIDEO &&
      qtime_media_set_media_type(qtmd, QTIME_MEDIA_TYPE_VIDEO) != QTIME_OK) {
    return QTIME_ERROR_INVALID_MEDIA;
  }

  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_VIDEO, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = 1;
  vdesc =(video_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!vdesc) {
    vdesc = video_desc_create(NULL);
    desc_id = qtime_sample_add_sample_desc(&qtmd->qtsp, (sample_desc_t*)vdesc);
  }
  vdesc->width = width;
  vdesc->height = height;
  qtime_char_to_type(compressor, &vdesc->sample_desc.data_format);
  qtmd->sample_description_id = desc_id;
  qtmd->sample_desc = (sample_desc_t*)vdesc;

  if (vdesc->sample_desc.data_reference_id == 0) {
    dref_id = qtime_media_set_data_reference(qtmd, NULL);
    vdesc->sample_desc.data_reference_id = dref_id;
    qtmd->data_reference_id = dref_id;
  }

  return desc_id;
}

int
qtime_media_video_is_keyframe(qtime_media_t *qtmd, uint32_t count)
{
  return qtime_sample_is_keyframe(&qtmd->qtsp, count);
}

uint32_t
qtime_media_video_next_keyframe(qtime_media_t *qtmd, uint32_t count)
{
  return qtime_sample_next_keyframe(&qtmd->qtsp, count);
}

uint32_t
qtime_media_video_prev_keyframe(qtime_media_t *qtmd, uint32_t count)
{
  return qtime_sample_prev_keyframe(&qtmd->qtsp, count);
}

// audio

int
qtime_media_get_sound_channels(qtime_media_t *qtmd)
{
  sound_desc_t *sdesc;
  int32_t desc_id;
  int channels;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_SOUND, QTIME_ERROR_INVALID_MEDIA)
  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_SOUND, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = qtmd->sample_description_id;
  sdesc =(sound_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!sdesc)
    return QTIME_ERROR;
  channels = sdesc->number_of_channels;
  return channels;
}

int
qtime_media_get_sound_bits(qtime_media_t *qtmd)
{
  sound_desc_t *sdesc;
  int32_t desc_id;
  int bits;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_SOUND, QTIME_ERROR_INVALID_MEDIA)
  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_SOUND, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = qtmd->sample_description_id;
  sdesc =(sound_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!sdesc)
    return QTIME_ERROR;
  bits = sdesc->sample_size;
  return bits;
}

int
qtime_media_get_sound_rate(qtime_media_t *qtmd)
{
  sound_desc_t *sdesc;
  int32_t desc_id;
  int rate;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_SOUND, QTIME_ERROR_INVALID_MEDIA)
  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_SOUND, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = qtmd->sample_description_id;
  sdesc =(sound_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!sdesc)
    return QTIME_ERROR;
  rate = ((uint32_t)sdesc->sample_rate >> 16);

  return rate;
}

int
qtime_media_set_sound(qtime_media_t *qtmd, int channels, int rate, int bits, uint8_t *compressor)
{
  sound_desc_t *sdesc;
  int32_t desc_id, dref_id;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_SOUND, QTIME_ERROR_INVALID_MEDIA)

  if (qtmd->media_type != QTIME_MEDIA_TYPE_SOUND &&
      qtime_media_set_media_type(qtmd, QTIME_MEDIA_TYPE_SOUND) != QTIME_OK) {
    return QTIME_ERROR_INVALID_MEDIA;
  }

  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_SOUND, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = 1;
  sdesc =(sound_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!sdesc) {
    sdesc = sound_desc_create(NULL);
    desc_id = qtime_sample_add_sample_desc(&qtmd->qtsp, (sample_desc_t*)sdesc);
  }
  qtmd->sample_description_id = desc_id;
  qtmd->sample_desc = (sample_desc_t*)sdesc;
  qtime_char_to_type(compressor, &sdesc->sample_desc.data_format);
  sdesc->number_of_channels = channels;
  sdesc->sample_size = bits;
  if (rate > 65535) {
    sdesc->sample_rate = rate;
  } else {
    sdesc->sample_rate = rate<<16;
  }
  qtime_media_set_time_scale(qtmd, rate);
  qtmd->sample_duration = 0;

  if (sdesc->sample_desc.data_reference_id == 0) {
    dref_id = qtime_media_set_data_reference(qtmd, NULL);
    sdesc->sample_desc.data_reference_id = dref_id;
    qtmd->data_reference_id = dref_id;
  }

  return desc_id;
}

int
qtime_media_get_sound_ext(qtime_media_t *qtmd, int8_t *version, int16_t *compression_id, uint32_t *samples_per_packet, uint32_t *bytes_per_packet, uint32_t *bytes_per_frame, uint32_t *bytes_per_sample)
{
  sound_desc_t *sdesc;
  int32_t desc_id;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_SOUND, QTIME_ERROR_INVALID_MEDIA)
  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_SOUND, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = qtmd->sample_description_id;
  sdesc =(sound_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!sdesc)
    return QTIME_ERROR;
  *version = sdesc->version;
  *compression_id = sdesc->compression_id;
  *samples_per_packet = sdesc->samples_per_packet;
  *bytes_per_packet = sdesc->bytes_per_packet;
  *bytes_per_frame = sdesc->bytes_per_frame;
  *bytes_per_sample = sdesc->bytes_per_sample;
  return QTIME_OK;
}

int
qtime_media_set_sound_ext(qtime_media_t *qtmd, int8_t version, int16_t compression_id, uint32_t samples_per_packet, uint32_t bytes_per_packet, uint32_t bytes_per_frame, uint32_t bytes_per_sample)
{
  sound_desc_t *sdesc;
  int32_t desc_id;

  qtime_error_media_check_v(QTIME_TYPE_QTMD, qtmd->media_type, QTIME_MEDIA_TYPE_SOUND, QTIME_ERROR_INVALID_MEDIA)
  qtime_error_media_check_v(QTIME_TYPE_QTMD, QTIME_MEDIA_TYPE_SOUND, qtime_sample_get_media_type(&qtmd->qtsp), QTIME_ERROR_INVALID_MEDIA)

  desc_id = qtmd->sample_description_id;
  sdesc =(sound_desc_t*)qtime_sample_get_sample_desc(&qtmd->qtsp, desc_id);
  if (!sdesc)
    return QTIME_ERROR;
  sdesc->version = version;
  sdesc->compression_id = compression_id;
  sdesc->samples_per_packet = samples_per_packet;
  sdesc->bytes_per_packet = bytes_per_packet;
  sdesc->bytes_per_frame = bytes_per_frame;
  sdesc->bytes_per_sample = bytes_per_sample;
  return QTIME_OK;
}

