/* qtime.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime.h"
#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "mdat.h"
#include "moov.h"


#define QTIME_TYPE_ROOT  QTIME_TYPE('r','o','o','t')


void
qtime_init(qtime_t *qt)
{
  qtime_io_init(&qt->qtio);
  moov_init(&qt->moov);
  mdat_init(&qt->mdat);

  qt->qtrk_num = 0;
  qt->qtrk_list = NULL;
}

qtime_t*
qtime_new(void)
{
  qtime_t *qt;

  qt = (qtime_t*) qtime_malloc(sizeof(qtime_t));
  if (!qt) return NULL;
  qtime_init(qt);
  return qt;
}

void
qtime_delete(qtime_t *qt)
{
  int i;

  if (!qt)          return;
  qtime_io_clean(&qt->qtio);
  mdat_clean(&qt->mdat);
  moov_clean(&qt->moov);
  if (qt->qtrk_list) {
    for (i = 0; i < qt->qtrk_num; i++)
      qtime_track_delete(qt->qtrk_list[i]);
    qtime_free(qt->qtrk_list);
  }
  qtime_free(qt);
}

qtime_t*
qtime_open_read(const char *file_name)
{
  qtime_t *qt;
  atom_head_t atom_head;
  moov_t *moov;
  int moov_index, mdat_index;
  atom_list_t trak_list;
  int i, trak_num;

  if (!file_name || file_name[0] == '\0') {
    qtime_error_empty_filename();
    return NULL;
  }

  qt = qtime_new();
  if (!qt)
    return NULL;

  if (qtime_io_open(&qt->qtio, file_name, QTIME_IO_MODE_READ) < 0) {
    qtime_delete(qt);
    return NULL;
  }

  atom_head_init(&atom_head);
  atom_head.parent = NULL;

  moov = &qt->moov;
  moov_index = 0;
  mdat_index = 0;

  while (1) {
    int ret;

    ret = atom_read_header(&qt->qtio, &atom_head);
    if (ret != QTIME_OK) {
      if (ret != QTIME_END_OF_FILE) {
	qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
        qtime_error_atom_read(QTIME_TYPE_ROOT);
	qtime_close(qt);
	return NULL;
      }
      break;
    }

    switch (atom_head.type) {
      case QTIME_TYPE_MOOV:
	if (moov_index > 0) {
	  qtime_error_debug_info(QTIME_ERROR_TOO_MANY_ATOM);
          qtime_error_too_many_atom(QTIME_TYPE_ROOT, atom_head.type);
	} else {
	  if (moov_read_atom(&qt->qtio, &atom_head, moov) == NULL) {
	    qtime_error_debug_info(atom_head.error_code);
            qtime_error_atom_read(QTIME_TYPE_ROOT);
	    qtime_close(qt);
	    return NULL;
	  }
	  moov_index++;
	}
	break;
      case QTIME_TYPE_MDAT:
	if (mdat_index > 0) {
	  qtime_error_debug_info(QTIME_ERROR_TOO_MANY_ATOM);
          qtime_error_too_many_atom(QTIME_TYPE_ROOT, atom_head.type);
	} else {
	  if (mdat_read_atom(&qt->qtio, &atom_head, &qt->mdat) != QTIME_OK) {
	    qtime_error_debug_info(atom_head.error_code);
            qtime_error_atom_read(QTIME_TYPE_ROOT);
	    qtime_close(qt);
	    return NULL;
	  }
	  mdat_index++;
	}
	break;
      case QTIME_TYPE_FREE:
      case QTIME_TYPE_WIDE:
#ifndef NDEBUG
        qtime_error_unknown_atom(QTIME_TYPE_ROOT, atom_head.type, (int32_t)atom_head.size);
#endif
	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_ROOT, atom_head.type, (int32_t)atom_head.size);
	break;
    }

    if (atom_read_footer(&qt->qtio, &atom_head) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      qtime_error_atom_read(QTIME_TYPE_ROOT);
      qtime_close(qt);
      return NULL;
    }
  }

  if (moov_index == 0) {
    fprintf(stderr, "QTIME_ERROR: cannot find 'moov' atom.\n");
    qtime_close(qt);
    return NULL;
  }

  trak_list = atom_find_child_atoms((atom_t*)&qt->moov, QTIME_TYPE_TRAK, &trak_num);
  qt->qtrk_list =(qtime_track_t**)qtime_malloc(sizeof(qtime_track_t*)*trak_num);
  if (!qt->qtrk_list) {
    qtime_close(qt);
    return NULL;
  }

  for (i = 0; i < trak_num; i++) {
    qt->qtrk_list[i] = qtime_track_create(NULL, (trak_t*)trak_list[i]);
    qtime_track_set_moov_io(qt->qtrk_list[i], &qt->qtio);
  }
  qt->qtrk_num = i;

  atom_list_free(trak_list);

  return qt;
}

qtime_t*
qtime_open_write(const char *file_name)
{
  qtime_t *qt;

  if (!file_name || file_name[0] == '\0') {
    qtime_error_empty_filename();
    return NULL;
  }

  qt = qtime_new();
  if (!qt)
    return NULL;

  if (qtime_io_open(&qt->qtio, file_name, QTIME_IO_MODE_WRITE) < 0) {
    qtime_delete(qt);
    return NULL;
  }

  if (moov_create(&qt->moov) == NULL) {
    qtime_delete(qt);
    return NULL;
  }

  if (mdat_write_init(&qt->qtio, &qt->mdat) < 0) {
    qtime_delete(qt);
    return NULL;
  }

  return qt;
}

int
qtime_close(qtime_t *qt)
{
  if (!qt)
    return QTIME_ERROR;

  if (qt->qtio.mode == QTIME_IO_MODE_WRITE) {
//    int i;
//    int track_num;
//    qtime_track_t *qtrk;

    mdat_write_end(&qt->qtio, &qt->mdat);

#if 0
    track_num = qtime_get_track_num(qt);
    {
      double time[track_num];
      double max_time;
      uint32_t scale, duration, max_scale, max_duration;

      max_scale = max_duration = 0;
      max_time = 0;

      for (i = 0; i < track_num; i++) {
        qtrk = qtime_get_index_track(qt, i);
        scale = qtime_track_get_media_time_scale(qtrk);
        duration = qtime_track_get_media_duration(qtrk);
        time[i] = (double)duration / (double)scale;
        if (time[i] > max_time) {
	  max_time = time[i];
	  max_scale = scale;
	  max_duration = duration;
        }
      }

      qt->moov.mvhd.time_scale = max_scale;
      qt->moov.mvhd.duration = max_duration;

      for (i = 0; i < track_num; i++) {
        qtrk = qtime_get_index_track(qt, i);
	duration = (uint32_t)(time[i] * (double)max_scale + 0.5);
        qtime_track_set_movie_duration(qtrk, duration);
      }
    }
#endif

    moov_write_atom(&qt->qtio, &qt->moov);
  }
  qtime_io_close(&qt->qtio);
  qtime_delete(qt);
  return QTIME_OK;
}

void
qtime_dump(qtime_t *qt)
{
  printf("file name: %s\n", qt->qtio.name);
  mdat_dump(&qt->mdat);
  moov_dump(&qt->moov);
}

int
qtime_get_track_num(qtime_t *qt)
{
  return qt->qtrk_num;
}

qtime_track_t*
qtime_get_track(qtime_t *qt)
{
  return qt->qtrk_list[0];
}

qtime_track_t*
qtime_get_index_track(qtime_t *qt, int index)
{
  if (index < 0 || index >= qt->qtrk_num)
    return NULL;
  return qt->qtrk_list[index];
}

qtime_track_t*
qtime_get_next_track(qtime_t *qt, qtime_track_t *qtrk)
{
  int i;
  qtime_track_t *next_qtrk;

  if (!qtrk)
    return qt->qtrk_list[0];

  next_qtrk = NULL;

  for (i = 0; i < qt->qtrk_num-1; i++) {
    if (qtrk == qt->qtrk_list[i]) {
      next_qtrk = qt->qtrk_list[i+1];
      break;
    }
  }

  return next_qtrk;
}

qtime_track_t*
qtime_get_video_track(qtime_t *qt)
{
  int i;
  for (i = 0; i < qt->qtrk_num; i++)
    if (qtime_track_get_media_type(qt->qtrk_list[i]) == QTIME_MEDIA_TYPE_VIDEO)
      return qt->qtrk_list[i];
  return NULL;
}

qtime_track_t*
qtime_get_next_video_track(qtime_t *qt, qtime_track_t *qtrk)
{
  int i;

  i = 0;
  if (qtrk) {
    for (; i < qt->qtrk_num-1; i++) {
      if (qtrk == qt->qtrk_list[i]) {
	i++;
	break;
      }
    }
  }
  for (; i < qt->qtrk_num; i++) {
    if (qtime_track_get_media_type(qt->qtrk_list[i]) == QTIME_MEDIA_TYPE_VIDEO)
      return qt->qtrk_list[i];
  }

  return NULL;
}

qtime_track_t*
qtime_get_sound_track(qtime_t *qt)
{
  int i;
  for (i = 0; i < qt->qtrk_num; i++)
    if (qtime_track_get_media_type(qt->qtrk_list[i]) == QTIME_MEDIA_TYPE_SOUND)
      return qt->qtrk_list[i];
  return NULL;
}

qtime_track_t*
qtime_get_next_sound_track(qtime_t *qt, qtime_track_t *qtrk)
{
  int i;

  i = 0;
  if (qtrk) {
    for (; i < qt->qtrk_num-1; i++) {
      if (qtrk == qt->qtrk_list[i]) {
	i++;
	break;
      }
    }
  }
  for (; i < qt->qtrk_num; i++) {
    if (qtime_track_get_media_type(qt->qtrk_list[i]) == QTIME_MEDIA_TYPE_SOUND)
      return qt->qtrk_list[i];
  }

  return NULL;
}

qtime_track_t*
qtime_create_track(qtime_t *qt)
{
  //int i, id, max_id, trak_id;
  int qtrk_num;
  qtime_track_t **qtrk_list;
  qtime_track_t *qtrk;
  trak_t *trak;

  qtrk_num = qt->qtrk_num;
  qtrk_list = qt->qtrk_list;

  trak = moov_create_trak(&qt->moov, NULL);

  qtrk_list = (qtime_track_t**)qtime_realloc(qtrk_list, sizeof(qtime_track_t*) * (qtrk_num+1));
  if (!qtrk_list) {
    trak_delete(trak);
    return NULL;
  }

  qtrk = qtime_track_create(NULL, trak);
  if (!qtrk) {
    trak_delete(trak);
    return NULL;
  }

  qtrk_list[qtrk_num] = qtrk;
  qtrk_num++;

//  qtime_media_set_data_reference(&qtrk->qtmd, NULL);
  qtime_track_set_moov_io(qtrk, &qt->qtio);

  qt->qtrk_num = qtrk_num;
  qt->qtrk_list = qtrk_list;

  return qtrk;
}

