/* moov.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "moov.h"
#include "mvhd.h"
#include "trak.h"
#include "udta.h"
#include "ctab.h"
//#include "clip.h"

#ifndef NDEBUG
int
moov_check_atoms_count(moov_t *moov)
{
  int c, ret;

  ret = QTIME_OK;
  if ((c = atom_count_child((atom_t*)moov, QTIME_TYPE_MVHD)) != 1) {
    fprintf(stderr, "QTIME_ERROR: 'mvhd' atom's count, %d.\n", c);
    ret = QTIME_ERROR;
  }
  if ((c = atom_count_child((atom_t*)moov, QTIME_TYPE_UDTA)) > 1) {
    fprintf(stderr, "QTIME_ERROR: 'udta' atom's count, %d.\n", c);
    ret = QTIME_ERROR;
  }
  if ((c = atom_count_child((atom_t*)moov, QTIME_TYPE_TRAK)) < 1) {
    fprintf(stderr, "QTIME_ERROR: 'trak' atom's count, %d.\n", c);
    ret = QTIME_ERROR;
  }
  return ret;
}
#endif

void
moov_init(moov_t *moov)
{
  atom_init(&moov->atom);
  moov->atom.type = QTIME_TYPE_MOOV;
  mvhd_init(&moov->mvhd);
}

void
moov_clean(moov_t *moov)
{
  int i;
  atom_t *atom = &moov->atom;

  qtime_error_type_check(QTIME_TYPE_MOOV, moov->atom.type)

  for (i = 0; i < atom->number_of_childs; i++) {
    if (atom->childs[i] == (atom_t*)&moov->mvhd) {
      mvhd_clean((mvhd_t*)atom->childs[i]);
      continue;
    }

    switch (atom->childs[i]->type) {
      case QTIME_TYPE_MVHD:
        mvhd_delete((mvhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        udta_delete((udta_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_CTAB:
        ctab_delete((ctab_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_TRAK:
        trak_delete((trak_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_CLIP:
        clip_delete((clip_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
	qtime_error_illegal_atom(moov->atom.type, atom->childs[i]->type);
	break;
    }
  }
  atom_remove_child_all(&moov->atom);
  //moov_init(moov);
  atom_init(&moov->atom);
  moov->atom.type = QTIME_TYPE_MOOV;
}

moov_t*
moov_new(void)
{
  moov_t *moov;
  moov = (moov_t*)qtime_malloc(sizeof(moov_t));
  if (!moov) return NULL;
  moov_init(moov);
  return moov;
}

void
moov_delete(moov_t *moov)
{
  qtime_error_type_check(QTIME_TYPE_MOOV, moov->atom.type)
  moov_clean(moov);
  qtime_free(moov);
}

moov_t*
moov_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, moov_t *moov_ptr)
{
  atom_head_t subatom;
  atom_t *atom;
  moov_t *moov;
  mvhd_t *mvhd;
  trak_t *trak;
  udta_t *udta;
  ctab_t *ctab;
  int trak_index, mvhd_index, udta_index, ctab_index;
//  clip_t *clip;
//  int clip_index;

#if ATOM_SIZE_T_LENGTH == 4
  if (atom_head->size & SIZE64MASK) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_SIZE);
    fprintf(stderr, "QTIME_ERROR_ATOM_SIZE: atom size too large, %lld, atom size max %u.\n", atom_head->size, ATOM_SIZE_MAX);
  }
#endif

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_MOOV, atom_head->type, NULL)
  if (moov_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MOOV, moov_ptr->atom.type, NULL)
    moov = moov_ptr;
    moov_clean(moov);
  } else {
    if ((moov = moov_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  moov->atom.size = atom_head->size;
  moov->atom.type = QTIME_TYPE_MOOV;
  moov->atom.parent = atom_head->parent;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)moov;

  mvhd = &moov->mvhd;
  trak = NULL;
  udta = NULL;
  ctab = NULL;
  mvhd_index = trak_index = udta_index = ctab_index = 0;
//  clip = NULL;
//  clip_index = 0;

  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    atom = NULL;
    switch (subatom.type) {
      case QTIME_TYPE_TRAK:
	if ((trak = trak_read_atom(qtio, &subatom, trak)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)trak;
	atom->index = ++trak_index;
	trak = NULL;
	break;
      case QTIME_TYPE_MVHD:
	if ((mvhd = mvhd_read_atom(qtio, &subatom, mvhd)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)mvhd;
	atom->index = ++mvhd_index;
	mvhd = NULL;
	break;
      case QTIME_TYPE_CTAB:
	if ((ctab = ctab_read_atom(qtio, &subatom, ctab)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)ctab;
	atom->index = ++ctab_index;
	ctab = NULL;
	break;
      case QTIME_TYPE_UDTA:
	if ((udta = udta_read_atom(qtio, &subatom, udta)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)udta;
	atom->index = ++udta_index;
	udta = NULL;
	break;
#if 0
      case QTIME_TYPE_CLIP:
	if ((clip = clip_read_atom(qtio, &subatom, clip)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)clip;
	atom->index = ++clip_index;
	clip = NULL;
	break;
#endif
      default:
        qtime_error_unknown_atom(QTIME_TYPE_MOOV, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom) {
      if ((atom_head->error_code = atom_add_child((atom_t*)moov, atom)) < 0) {
        qtime_error_debug_info(atom_head->error_code);
	goto fail;
      }
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

#ifndef NDEBUG
  if (moov_check_atoms_count(moov) != QTIME_OK) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number of atoms is wrong.\n");
  }
#endif

  return moov;

fail:
  if (moov_ptr)
    moov_clean(moov);
  else
    moov_delete(moov);
  qtime_error_atom_read(QTIME_TYPE_MOOV);
  return NULL;
}

moov_t*
moov_create(moov_t *moov_ptr)
{
  moov_t *moov;
  int error_code;

  if (moov_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MOOV, moov_ptr->atom.type, NULL)
    moov = moov_ptr;
    moov_clean(moov);
  } else {
    if ((moov = moov_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  if ((error_code = atom_add_child((atom_t*)moov, (atom_t*)&moov->mvhd)) < 0) {
    qtime_error_debug_info(error_code);
    if (moov_ptr)
      moov_clean(moov);
    else
      moov_delete(moov);
    return NULL;
  }
  return moov;
}

void
moov_dump(moov_t *moov)
{
  int i;
  uint8_t type_str[5];
  atom_t *atom = &moov->atom;

  qtime_type_to_str(atom->type, type_str);
  fprintf(stdout, "%s: size         %lld\n", type_str, (int64_t)atom->size);
  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_MVHD:
        mvhd_dump(type_str, (mvhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        udta_dump(type_str, (udta_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_CTAB:
        ctab_dump(type_str, (ctab_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_TRAK:
        trak_dump(type_str, (trak_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_CLIP:
        clip_dump(type_str, (clip_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
	qtime_error_illegal_atom(QTIME_TYPE_MOOV, atom->childs[i]->type);
	break;
    }
  }
}

int
moov_valid(moov_t *moov)
{
  if (moov->atom.number_of_childs <= 0)
    return 0;
  else if (atom_count_child((atom_t*)moov, QTIME_TYPE_TRAK) <= 0)
    return 0;
  return 1;
}

int64_t
moov_calc_size(moov_t *moov)
{
  int i;
  int64_t size = 0;
  atom_t *atom = &moov->atom;

  qtime_error_type_check_v(QTIME_TYPE_MOOV, moov->atom.type, 0)

  if (!moov_valid(moov))
    return 0;

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_MVHD:
        size += mvhd_calc_size((mvhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        size += udta_calc_size((udta_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_CTAB:
        size += ctab_calc_size((ctab_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_TRAK:
        size += trak_calc_size((trak_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_CLIP:
        size += clip_calc_size((clip_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
	qtime_error_illegal_atom(QTIME_TYPE_MOOV, atom->childs[i]->type);
	break;
    }
  }

  if (size & SIZE64MASK)
    size += 8 + 8;
  else
    size += 8;

#if ATOM_SIZE_T_LENGTH == 4
  if (size & SIZE64MASK) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_SIZE);
    fprintf(stderr, "QTIME_ERROR_ATOM_SIZE: atom size too large, %lld, atom size max %u.\n", size, ATOM_SIZE_MAX);
  }
#endif
  moov->atom.size = size;

  return size;
}

int
moov_write_atom_pre(moov_t *moov)
{
  int i, trak_num;
  mvhd_t *mvhd;
  atom_t *atom = &moov->atom;

  mvhd = (mvhd_t*)atom_find_child((atom_t*)moov, QTIME_TYPE_MVHD);
  trak_num = atom_count_child((atom_t*)moov, QTIME_TYPE_TRAK);
  {
    double time[trak_num];
    double max_time;
    uint32_t scale, duration, max_scale, max_duration;
    int idx;

    max_scale = max_duration = 0;
    max_time = 0;

    for (i = idx = 0; i < atom->number_of_childs; i++) {
      if (atom->childs[i]->type != QTIME_TYPE_TRAK)
	continue;
      scale = trak_get_media_time_scale((trak_t*)atom->childs[i]);
      duration = trak_get_media_duration((trak_t*)atom->childs[i]);
      time[idx] = (double)duration / (double)scale;
      if (time[idx] > max_time) {
	max_time = time[idx];
	max_scale = scale;
	max_duration = duration;
      }
      idx++;
    }

    mvhd_set_time_scale(mvhd, max_scale);
    mvhd_set_duration(mvhd, max_duration);

    for (i = idx = 0; i < atom->number_of_childs; i++) {
      if (atom->childs[i]->type != QTIME_TYPE_TRAK)
	continue;
      duration = (uint32_t)(time[idx] * (double)max_scale + 0.5);
      trak_set_movie_duration((trak_t*)atom->childs[i], duration);
      idx++;
    }
  }

  mvhd_update_modification_time(mvhd);

  return QTIME_OK;
}

int
moov_write_atom(qtime_io_t *qtio, moov_t *moov)
{
  int i;
  atom_head_t atom_head;
  atom_t *atom;

  qtime_error_type_check_v(QTIME_TYPE_MOOV, moov->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

#ifndef NDEBUG
  if (moov_check_atoms_count(moov) != QTIME_OK) {
    qtime_error_debug_info(QTIME_ERROR);
    fprintf(stderr, "QTIME_ERROR: number of atoms is wrong.\n");
  }
#endif

  if (!moov_valid)
    return QTIME_OK;

  moov_write_atom_pre(moov);
  moov->atom.size = moov_calc_size(moov);

  atom_head_init(&atom_head);
  atom_head.size = moov->atom.size;
  atom_head.type = moov->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MOOV);
    return QTIME_ERROR_ATOM_WRITE;
  }

  atom = &moov->atom;
  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_MVHD:
        mvhd_write_atom(qtio, (mvhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_UDTA:
        udta_write_atom(qtio, (udta_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_CTAB:
        ctab_write_atom(qtio, (ctab_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_TRAK:
        trak_write_atom(qtio, (trak_t*)atom->childs[i]);
	break;
#if 0
      case QTIME_TYPE_CLIP:
        clip_write_atom(qtio, (clip_t*)atom->childs[i]);
	break;
#endif
      default:
	qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
	qtime_error_illegal_atom(QTIME_TYPE_MOOV, atom->childs[i]->type);
	break;
    }
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MOOV);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

static int
sort_compar(const void *v0, const void *v1)
{
  if (*((int*)v0) < *((int*)v1))
    return -1;
  else if (*((int*)v0) == *((int*)v1))
    return 0;
  else
    return 1;
}

trak_t*
moov_create_trak(moov_t *moov, trak_t *trak_ptr)
{
  trak_t *trak;
  mvhd_t *mvhd;
  int error_code;
  int i;
  int id;
  int next_id;
  int trak_num;

  qtime_error_type_check_v(QTIME_TYPE_MOOV, moov->atom.type, NULL)

  mvhd = (mvhd_t*)atom_find_child((atom_t*)moov, QTIME_TYPE_MVHD);
  id = mvhd_get_next_track_id(mvhd);
  next_id = id+1;
  if (id > QTIME_TRAK_MAX) {
    trak_num = atom_count_child((atom_t*)moov, QTIME_TYPE_TRAK);
    if (trak_num >= QTIME_TRAK_MAX) {
      qtime_error_debug_info(QTIME_ERROR);
      fprintf(stderr, "QTIME_ERROR: number of tracks reached the maximum, %d\n", QTIME_TRAK_MAX);
      return NULL;
    } else {
      int ids[trak_num];
      int idx;
      for (i = idx = 0; i < moov->atom.number_of_childs; i++) {
	if (moov->atom.childs[i]->type != QTIME_TYPE_TRAK)
	  continue;
	ids[idx++] = trak_get_id((trak_t*)moov->atom.childs[i]);
      }
      qsort(ids, idx, sizeof(int), sort_compar);
      id = 0;
      for (i = 0; i < idx-1; i++) {
	if ((ids[i+1] - ids[i]) > 1) {
	  id = ids[i]+1;
	  break;
	}
      }
      if (id == 0) {
        qtime_error_debug_info(QTIME_ERROR);
	fprintf(stderr, "QTIME_ERROR: cannot find trak id number.\n");
	return NULL;
      }
      next_id = ids[idx-1] + 1;
    }
  }

  if ((trak = trak_create(trak_ptr)) == NULL) {
    qtime_error_debug_info(QTIME_ERROR);
    return NULL;
  }

  if ((error_code = atom_add_child((atom_t*)moov, (atom_t*)trak)) < 0) {
    qtime_error_debug_info(error_code);
    if (trak_ptr)
      trak_clean(trak);
    else
      trak_delete(trak);
    return NULL;
  }

  trak_set_id(trak, id);
  mvhd_set_next_track_id(mvhd, next_id);

  return trak;
}

