/* minf.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "minf.h"
#include "vmhd.h"
#include "smhd.h"
#include "gmhd.h"
#include "gmin.h"
#include "hdlr.h"
#include "dinf.h"
#include "stbl.h"
#include "stsd.h"


#ifndef NDEBUG

int
minf_check_media_type(minf_t *minf)
{
  vmhd_t *vmhd;
  smhd_t *smhd;
  gmhd_t *gmhd;
  hdlr_t *hdlr;
  uint32_t media_type;
  uint32_t hdlr_type;

  vmhd = (vmhd_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_VMHD);
  smhd = (smhd_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_SMHD);
  gmhd = (gmhd_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_GMHD);

  if ((vmhd && smhd) || (vmhd && gmhd) || (smhd && gmhd)) {
    fprintf(stderr, "QTIME_ERROR: multiple media header atoms exist in 'minf' atom.\n");
    return QTIME_ERROR_INVALID_MEDIA;
  }

  if (vmhd) {
    media_type = QTIME_MEDIA_TYPE_VIDEO;
  } else if (smhd) {
    media_type = QTIME_MEDIA_TYPE_SOUND;
//  } else if (gmhd) {
  } else {
    media_type = QTIME_MEDIA_TYPE_BASE;
  }

  hdlr = (hdlr_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_HDLR);
  if (hdlr) {
    hdlr_type = hdlr_get_media_type(hdlr);
    if (hdlr_type != QTIME_MEDIA_TYPE_UNKNOWN && hdlr_type != media_type) {
      fprintf(stderr, "QTIME_ERROR: 'hdlr' atom's media_type is invalid in 'minf' atom.\n");
      return QTIME_ERROR_INVALID_MEDIA;
    }
  }

  return QTIME_OK;
}

#endif


void
minf_init(minf_t *minf)
{
  atom_init(&minf->atom);
  minf->atom.type = QTIME_TYPE_MINF;
  minf->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
  minf->vmhd = NULL;
  minf->smhd = NULL;
  minf->gmhd = NULL;
  minf->gmin = NULL;
  hdlr_init(&minf->hdlr);
  dinf_init(&minf->dinf);
  stbl_init(&minf->stbl);
}

void
minf_clean(minf_t *minf)
{
  int i;
  atom_t *atom = &minf->atom;

  qtime_error_type_check(QTIME_TYPE_MINF, minf->atom.type)

  for (i = 0; i < atom->number_of_childs; i++) {
    if (atom->childs[i] == (atom_t*)&minf->hdlr) {
      hdlr_clean((hdlr_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&minf->dinf) {
      dinf_clean((dinf_t*)atom->childs[i]);
      continue;
    } else if (atom->childs[i] == (atom_t*)&minf->stbl) {
      stbl_clean((stbl_t*)atom->childs[i]);
      continue;
    }

    switch (atom->childs[i]->type) {
      case QTIME_TYPE_VMHD:
        vmhd_delete((vmhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_SMHD:
        smhd_delete((smhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_GMHD:
        gmhd_delete((gmhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_GMIN:
        gmin_delete((gmin_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_HDLR:
        hdlr_delete((hdlr_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_DINF:
        dinf_clean((dinf_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STBL:
        stbl_delete((stbl_t*)atom->childs[i]);
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_MINF, atom->childs[i]->type);
	break;
    }
  }

  atom_clean(atom);
  minf->atom.type = QTIME_TYPE_MINF;
  minf->media_type = QTIME_MEDIA_TYPE_UNKNOWN;
  minf->vmhd = NULL;
  minf->smhd = NULL;
  minf->gmhd = NULL;
  minf->gmin = NULL;
}

minf_t*
minf_new(void)
{
  minf_t *minf;
  minf = (minf_t*)qtime_malloc(sizeof(minf_t));
  if (!minf) return NULL;
  minf_init(minf);
  return minf;
}

void
minf_delete(minf_t *minf)
{
  qtime_error_type_check(QTIME_TYPE_MINF, minf->atom.type)
  minf_clean(minf);
  qtime_free(minf);
}

minf_t*
minf_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, minf_t *minf_ptr)
{
  atom_head_t subatom;
  atom_t *atom;
  int vmhd_index, smhd_index, gmhd_index, gmin_index, hdlr_index, dinf_index;
  int stbl_index;
  minf_t *minf;
  vmhd_t *vmhd;
  smhd_t *smhd;
  gmhd_t *gmhd;
  gmin_t *gmin;
  hdlr_t *hdlr;
  dinf_t *dinf;
  stbl_t *stbl;
  uint32_t media_type;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_MINF, atom_head->type, NULL)
  if (minf_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MINF, minf_ptr->atom.type, NULL)
    minf = minf_ptr;
    minf_clean(minf);
  } else {
    if ((minf = minf_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  minf->atom.size = atom_head->size;
  minf->atom.type = QTIME_TYPE_MINF;
  minf->atom.parent = atom_head->parent;
  minf->media_type = atom_head->media_type;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)minf;
  subatom.media_type = atom_head->media_type;

  media_type = atom_head->media_type;
  vmhd_index = smhd_index = gmhd_index = gmin_index = hdlr_index =
    dinf_index = stbl_index = 0;
  vmhd = NULL;
  smhd = NULL;
  gmhd = NULL;
  gmin = NULL;
  hdlr = &minf->hdlr;
  dinf = &minf->dinf;
  stbl = &minf->stbl;

  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    atom = NULL;
    switch (subatom.type) {
      case QTIME_TYPE_VMHD:
	if ((vmhd = vmhd_read_atom(qtio, &subatom, vmhd)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
          atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)vmhd;
	atom->index = ++vmhd_index;
	vmhd = NULL;
	break;
      case QTIME_TYPE_SMHD:
	if ((smhd = smhd_read_atom(qtio, &subatom, smhd)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
          atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)smhd;
	atom->index = ++smhd_index;
	smhd = NULL;
	break;
      case QTIME_TYPE_GMHD:
	if ((gmhd = gmhd_read_atom(qtio, &subatom, gmhd)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
          atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)gmhd;
	atom->index = ++gmhd_index;
	gmhd = NULL;
	break;
      case QTIME_TYPE_GMIN:
	if ((gmin = gmin_read_atom(qtio, &subatom, gmin)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
          atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)gmin;
	atom->index = ++gmin_index;
	gmin = NULL;
	break;
      case QTIME_TYPE_HDLR:
	if ((hdlr = hdlr_read_atom(qtio, &subatom, hdlr)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
          atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)hdlr;
	atom->index = ++hdlr_index;
	hdlr = NULL;
	break;
      case QTIME_TYPE_DINF:
	if ((dinf = dinf_read_atom(qtio, &subatom, dinf)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
          atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)dinf;
	atom->index = ++dinf_index;
	dinf = NULL;
	break;
      case QTIME_TYPE_STBL:
	if ((stbl = stbl_read_atom(qtio, &subatom, stbl)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
          atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	atom = (atom_t*)stbl;
	atom->index = ++stbl_index;
	stbl = NULL;
	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_MINF, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (minf->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
      minf->media_type = media_type = subatom.media_type;
    } else if (subatom.media_type != QTIME_MEDIA_TYPE_UNKNOWN &&
               subatom.media_type != media_type) {
      qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
      qtime_error_invalid_media(QTIME_TYPE_MINF,media_type,subatom.media_type);
    }

    if (atom) {
      if ((atom_head->error_code = atom_add_child((atom_t*)minf, atom)) < 0) {
	qtime_error_debug_info(atom_head->error_code);
	goto fail;
      }
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  minf->vmhd = (vmhd_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_VMHD);
  minf->smhd = (smhd_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_SMHD);
  minf->gmhd = (gmhd_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_GMHD);
  minf->gmin = (gmin_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_GMIN);

  stbl = (stbl_t*)atom_find_child((atom_t*)minf, QTIME_TYPE_STBL);
  if (stbl && stbl->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
    stsd_t *stsd;
    stbl->media_type = minf->media_type;
    stsd = (stsd_t*)atom_find_child((atom_t*)stbl, QTIME_TYPE_STSD);
    if (stsd && stsd->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
      stsd->media_type = minf->media_type;
      stsd_read_atom_post(qtio, stsd);
    }
  }

#ifndef NDEBUG
  if (minf_check_media_type(minf) != QTIME_OK) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: 'minf' atom's media_type is invalid.\n");
  }
#endif

  return minf;

fail:
  if (minf_ptr)
    minf_clean(minf);
  else
    minf_delete(minf);
  qtime_error_atom_read(QTIME_TYPE_MINF);
  return NULL;
}

minf_t*
minf_create(minf_t *minf_ptr)
{
  minf_t *minf;
  int error_code;

  if (minf_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_MINF, minf_ptr->atom.type, NULL)
    minf = minf_ptr;
    minf_clean(minf);
  } else {
    if ((minf = minf_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  hdlr_create(&minf->hdlr);
  dinf_create(&minf->dinf);
  stbl_create(&minf->stbl);

  if ((error_code = atom_add_child((atom_t*)minf, (atom_t*)&minf->hdlr)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)minf, (atom_t*)&minf->dinf)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }
  if ((error_code = atom_add_child((atom_t*)minf, (atom_t*)&minf->stbl)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }

  return minf;

fail:
  if (minf_ptr)
    minf_clean(minf);
  else
    minf_delete(minf);
  return NULL;
}

int64_t
minf_calc_size(minf_t *minf)
{
  int i;
  int64_t size = 0;
  atom_t *atom = &minf->atom;

  qtime_error_type_check_v(QTIME_TYPE_MINF, minf->atom.type, 0)
  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_VMHD:
        size += vmhd_calc_size((vmhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_SMHD:
        size += smhd_calc_size((smhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_GMHD:
        size += gmhd_calc_size((gmhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_GMIN:
        size += gmin_calc_size((gmin_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_HDLR:
        size += hdlr_calc_size((hdlr_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_DINF:
        size += dinf_calc_size((dinf_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STBL:
        size += stbl_calc_size((stbl_t*)atom->childs[i]);
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_MINF, atom->childs[i]->type);
	break;
    }
  }
  if (size & SIZE64MASK)
    size += 8 + 8;
  else
    size += 8;
  minf->atom.size = size;
  return size;
}

int
minf_write_atom(qtime_io_t *qtio, minf_t *minf)
{
  int i;
  atom_head_t atom_head;
  atom_t *atom = &minf->atom;

  qtime_error_type_check_v(QTIME_TYPE_MINF, minf->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
#ifndef NDEBUG
  if (minf_check_media_type(minf) != QTIME_OK) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    fprintf(stderr, "QTIME_ERROR_INVALID_MEDIA: 'minf' atom's media_type is invalid.\n");
  }
#endif

  atom_head_init(&atom_head);
  atom_head.size = minf->atom.size;
  atom_head.type = minf->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MINF);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_VMHD:
        vmhd_write_atom(qtio, (vmhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_SMHD:
        smhd_write_atom(qtio, (smhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_GMHD:
        gmhd_write_atom(qtio, (gmhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_GMIN:
        gmin_write_atom(qtio, (gmin_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_HDLR:
        hdlr_write_atom(qtio, (hdlr_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_DINF:
        dinf_write_atom(qtio, (dinf_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STBL:
        stbl_write_atom(qtio, (stbl_t*)atom->childs[i]);
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_MINF, atom->childs[i]->type);
	break;
    }
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_MINF);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
minf_dump(const char *parent_types, minf_t *minf)
{
  int i, len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  atom_t *atom = &minf->atom;

  qtime_type_to_str(minf->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)minf->atom.size);
  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_VMHD:
        vmhd_dump(types, (vmhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_SMHD:
        smhd_dump(types, (smhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_GMHD:
        gmhd_dump(types, (gmhd_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_GMIN:
        gmin_dump(types, (gmin_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_HDLR:
        hdlr_dump(types, (hdlr_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_DINF:
        dinf_dump(types, (dinf_t*)atom->childs[i]);
	break;
      case QTIME_TYPE_STBL:
        stbl_dump(types, (stbl_t*)atom->childs[i]);
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_MINF, atom->childs[i]->type);
	break;
    }
  }
}

uint32_t
minf_get_media_type(minf_t *minf)
{
  qtime_error_type_check_v(QTIME_TYPE_MINF, minf->atom.type, QTIME_MEDIA_TYPE_UNKNOWN)

  if (atom_find_child((atom_t*)minf, QTIME_TYPE_VMHD) != NULL)
    return QTIME_MEDIA_TYPE_VIDEO;
  else if (atom_find_child((atom_t*)minf, QTIME_TYPE_SMHD) != NULL)
    return QTIME_MEDIA_TYPE_SOUND;
  else if (atom_find_child((atom_t*)minf, QTIME_TYPE_GMHD) != NULL)
    return QTIME_MEDIA_TYPE_BASE;

  return QTIME_MEDIA_TYPE_UNKNOWN;
}

int
minf_set_media_type(minf_t *minf, uint32_t media_type)
{
  vmhd_t *vmhd;
  smhd_t *smhd;
//  gmhd_t *gmhd;

  qtime_error_type_check_v(QTIME_TYPE_MINF, minf->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  switch (media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      if (atom_find_child((atom_t*)minf, QTIME_TYPE_VMHD) == NULL) {
	vmhd = vmhd_create(NULL);
        qtime_flags_set(vmhd->flags, 1);
	atom_insert_child((atom_t*)minf, (atom_t*)vmhd, 0);
      }
#ifndef NDEBUG
      if ((atom_find_child((atom_t*)minf, QTIME_TYPE_SMHD) != NULL) ||
          (atom_find_child((atom_t*)minf, QTIME_TYPE_GMHD) != NULL)) {
	qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
        fprintf(stderr, "QTIME_ERROR: multiple media headers are exist in 'minf' atom.\n");
      }
#endif
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      if (atom_find_child((atom_t*)minf, QTIME_TYPE_SMHD) == NULL) {
	smhd = smhd_create(NULL);
        qtime_flags_set(smhd->flags, 0);
	atom_insert_child((atom_t*)minf, (atom_t*)smhd, 0);
      }
#ifndef NDEBUG
      if ((atom_find_child((atom_t*)minf, QTIME_TYPE_VMHD) != NULL) ||
          (atom_find_child((atom_t*)minf, QTIME_TYPE_GMHD) != NULL)) {
	qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
        fprintf(stderr, "QTIME_ERROR: multiple media headers are exist in 'minf' atom.\n");
      }
#endif
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR);
      fprintf(stderr, "QTIME_ERROR: unsupported media_type.\n");
      break;
  }

  return QTIME_OK;
}

