/* matt.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "matt.h"
#include "kmat.h"

void
matt_init(matt_t *matt)
{
  atom_init(&matt->atom);
  matt->atom.type = QTIME_TYPE_MATT;
  matt->kmat = NULL;
}

void
matt_clean(matt_t *matt)
{
  matt->atom.size = 0;
  matt->atom.type = QTIME_TYPE_MATT;
  if (matt->kmat)
    kmat_delete(matt->kmat);
  matt->kmat = NULL;
}

matt_t*
matt_new(void)
{
  matt_t *matt;
  matt = (matt_t*)qtime_malloc(sizeof(matt_t));
  if (!matt)
    return NULL;
  matt_init(matt);
  return matt;
}

void
matt_delete(matt_t *matt)
{
  if (!matt) return;
  if (matt->kmat)
    kmat_delete(matt->kmat);
  qtime_free(matt);
}

int
matt_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, matt_t *matt)
{
  const char *func = "matt_read_atom";
  atom_head_t subatom;

#ifdef DEBUG
  if (atom_head->type != QTIME_TYPE_MATT) {
    qtime_print_error(QTIME_ERROR_ILLEGAL_ATOM, func, atom_head->type);
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif /* DEBUG */

  matt->atom.size = atom_head->size;
  matt->atom.type = QTIME_TYPE_MATT;

  subatom.size = 0;
  subatom.type = 0;
  subatom.start_offset = 0;
  subatom.end_offset = 0;
  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_print_error(QTIME_ERROR_ATOM_READ, func, atom_head->type);
      matt_clean(matt);
      return QTIME_ERROR_ATOM_READ;
    }

    switch (subatom.type) {
      case QTIME_TYPE_KMAT:
	if (matt->kmat)
          qtime_print_error(QTIME_ERROR_TOO_MANY_ATOM, func, subatom.type);
	else {
	  if ((matt->kmat = kmat_new()) == NULL) {
            matt_clean(matt);
            return QTIME_ERROR_MEMORY;
	  }
	  if (kmat_read_atom(qtio, &subatom, matt->kmat) != QTIME_OK) {
            qtime_print_error(QTIME_ERROR_ATOM_READ, func, subatom.type);
            matt_clean(matt);
            return QTIME_ERROR_ATOM_READ;
	  }
	}
	break;
      default:
        qtime_print_error(QTIME_ERROR_UNKNOWN_ATOM, func, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_print_error(QTIME_ERROR_ATOM_READ, func, atom_head->type);
      matt_clean(matt);
      return QTIME_ERROR_ATOM_READ;
    }
  }

  return QTIME_OK;
}

int64_t
matt_calc_size(matt_t *matt)
{
  int64_t size = 0;

  if (matt->kmat)
    kmat_calc_size(matt->kmat);
  size += 8;
  matt->atom.size = size;
  return size;
}

int
matt_write_atom(qtime_io_t *qtio, matt_t *matt)
{
  atom_head_t atom_head;

  matt->atom.size = matt_calc_size(matt);
  atom_head.size = matt->atom.size;
  atom_head.type = matt->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    return QTIME_ERROR_ATOM_WRITE;
  }

  if (matt->kmat)
    kmat_write_atom(qtio, matt->kmat);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
matt_dump(const char *parent_types, matt_t *matt)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];

  qtime_type_to_str(matt->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);

  fprintf(stdout, "%s: size         %lld\n", types, matt->atom.size);
  if (matt->kmat)
    kmat_dump(types, matt->kmat);
}

