/* hdlr.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "hdlr.h"

void
hdlr_init(hdlr_t *hdlr)
{
  atom_init(&hdlr->atom);
  hdlr->atom.type = QTIME_TYPE_HDLR;
  hdlr->version = 0;
  qtime_flags_set(hdlr->flags, 0);
  hdlr->component_type = 0;
  hdlr->component_subtype = 0;
  hdlr->component_manufacturer = 0;
  hdlr->component_flags = 0;
  hdlr->component_flags_mask = 0;
  hdlr->component_name = NULL;
}

void
hdlr_clean(hdlr_t *hdlr)
{
  qtime_error_type_check(QTIME_TYPE_HDLR, hdlr->atom.type)
  hdlr_init(hdlr);
}

hdlr_t*
hdlr_new(void)
{
  hdlr_t *hdlr;
  hdlr = (hdlr_t*)qtime_malloc(sizeof(hdlr_t));
  if (!hdlr) return NULL;
  hdlr_init(hdlr);
  return hdlr;
}

void
hdlr_delete(hdlr_t *hdlr)
{
  qtime_error_type_check(QTIME_TYPE_HDLR, hdlr->atom.type)
  if (hdlr->component_name) qtime_free(hdlr->component_name);
  qtime_free(hdlr);
}

hdlr_t*
hdlr_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, hdlr_t *hdlr_ptr)
{
  uint32_t media_type;
  int data_size;
  hdlr_t *hdlr;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_HDLR, atom_head->type, NULL)
  if (hdlr_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr_ptr->atom.type, NULL)
    hdlr = hdlr_ptr;
    hdlr_clean(hdlr);
  } else {
    if ((hdlr = hdlr_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  hdlr->atom.size = atom_head->size;
  hdlr->atom.type = QTIME_TYPE_HDLR;
  hdlr->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &hdlr->version, 1);
  qtime_io_read(qtio, hdlr->flags, 3);
  qtime_io_read_type(qtio, &hdlr->component_type);
  qtime_io_read_type(qtio, &hdlr->component_subtype);
  qtime_io_read32(qtio, &hdlr->component_manufacturer);
  qtime_io_read32(qtio, &hdlr->component_flags);
  qtime_io_read32(qtio, &hdlr->component_flags_mask);

  data_size = atom_head->body_size - HDLR_PROP_SIZE;
  hdlr->component_name = (uint8_t*)qtime_malloc(data_size+1);
  if (!hdlr->component_name) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    atom_head->error_code = QTIME_ERROR_MEMORY;
    goto fail;
  }
  qtime_io_read(qtio, hdlr->component_name, data_size);
  hdlr->component_name[data_size] = '\0';

#ifndef NDEBUG
  if (hdlr->component_name[0] != data_size-1) {
    fprintf(stderr, "hdlr_read_atom: component_name size doesn't suit, %d, %d\n", hdlr->component_name[0], data_size);
  }
#endif

  media_type = hdlr_get_media_type(hdlr);
  if (media_type != QTIME_MEDIA_TYPE_UNKNOWN) {
    if (atom_head->media_type == QTIME_MEDIA_TYPE_UNKNOWN) {
      atom_head->media_type = media_type;
    } else if (atom_head->media_type != media_type) {
      qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
      qtime_error_invalid_media(QTIME_TYPE_HDLR, media_type, atom_head->media_type);
    }
  }

  return hdlr;

fail:
  if (hdlr_ptr)
    hdlr_clean(hdlr);
  else
    hdlr_delete(hdlr);
  qtime_error_atom_read(QTIME_TYPE_HDLR);
  return NULL;
}

hdlr_t*
hdlr_create(hdlr_t *hdlr_ptr)
{
  hdlr_t *hdlr;

  if (hdlr_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr_ptr->atom.type, NULL)
    hdlr = hdlr_ptr;
    hdlr_clean(hdlr);
  } else {
    if ((hdlr = hdlr_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  return hdlr;
}

int
hdlr_valid(hdlr_t *hdlr)
{
  if (hdlr->component_type == 0)
    return 0;
  if (hdlr->component_subtype == 0)
    return 0;
  return 1;
}

int64_t
hdlr_calc_size(hdlr_t *hdlr)
{
  int64_t size;

  qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr->atom.type, 0)

  if (!hdlr_valid(hdlr))
    return 0;

  size = 4 + 4 + HDLR_PROP_SIZE;
  if (hdlr->component_name)
    size += hdlr->component_name[0] + 1;
  hdlr->atom.size = size;
  return size;
}

int
hdlr_write_atom(qtime_io_t *qtio, hdlr_t *hdlr)
{
  atom_head_t atom_head;
  int length;

  qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  if (!hdlr_valid(hdlr))
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = hdlr->atom.size;
  atom_head.type = hdlr->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_HDLR);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &hdlr->version, 1);
  qtime_io_write(qtio, hdlr->flags, 3);
  qtime_io_write_type(qtio, &hdlr->component_type);
  qtime_io_write_type(qtio, &hdlr->component_subtype);
  qtime_io_write32(qtio, &hdlr->component_manufacturer);
  qtime_io_write32(qtio, &hdlr->component_flags);
  qtime_io_write32(qtio, &hdlr->component_flags_mask);

  if (hdlr->component_name) {
    length = hdlr->component_name[0];
    qtime_io_write(qtio, hdlr->component_name, length + 1);
  } else {
    uint8_t buf[1];
    buf[0] = 0;
    qtime_io_write(qtio, buf, 1);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_HDLR);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
hdlr_dump(const char *parent_types, hdlr_t *hdlr)
{
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint8_t buf[5];
  uint32_t u32;
  uint32_t flags;

  buf[4] = 0;
  qtime_type_to_str(hdlr->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(hdlr->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)hdlr->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, hdlr->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  qtime_type_to_str(hdlr->component_type, buf);
  fprintf(stdout, "%s: component type %.4s\n", types, buf);
  qtime_type_to_str(hdlr->component_subtype, buf);
  fprintf(stdout, "%s: component subtype %.4s\n", types, buf);
  //qtime_type_to_str(hdlr->component_manufacturer, buf);
  //fprintf(stdout, "%s: component manufacturer %.4s\n", types, buf);
  u32 = hdlr->component_manufacturer;
  fprintf(stdout, "%s: component manufacturer 0x%x\n", types, u32);
  fprintf(stdout, "%s: component flags 0x%x\n", types, hdlr->component_flags);
  fprintf(stdout, "%s: component flags mask 0x%x\n", types, hdlr->component_flags_mask);
  fprintf(stdout, "%s: component name  '%s'\n", types, hdlr->component_name+1);
}

uint32_t
hdlr_get_component_type(hdlr_t *hdlr)
{
  qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr->atom.type, 0)
  return hdlr->component_type;
}

void
hdlr_set_component_type(hdlr_t *hdlr, uint32_t component_type)
{
  qtime_error_type_check(QTIME_TYPE_HDLR, hdlr->atom.type)
  hdlr->component_type = component_type;
}

uint32_t
hdlr_get_component_subtype(hdlr_t *hdlr)
{
  qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr->atom.type, 0)
  return hdlr->component_subtype;
}

void
hdlr_set_component_subtype(hdlr_t *hdlr, uint32_t component_subtype)
{
  qtime_error_type_check(QTIME_TYPE_HDLR, hdlr->atom.type)
  hdlr->component_subtype = component_subtype;
}

uint8_t*
hdlr_get_component_name(hdlr_t *hdlr)
{
  qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr->atom.type, NULL)
  if (!hdlr->component_name)
    return NULL;
  return hdlr->component_name+1;
}

int
hdlr_set_component_name(hdlr_t *hdlr, uint8_t *component_name)
{
  int length;
  uint8_t *name;

  qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  length = strlen(component_name);
  length = (length < 255) ? length : 255;

  name = (uint8_t*)qtime_realloc(hdlr->component_name, length+2);
  if (!name)
    return QTIME_ERROR_MEMORY;
  name[0] = length;
  strncpy(name+1, component_name, length);
  name[length+1] = '\0';
  hdlr->component_name = name;
  return QTIME_OK;
}

uint32_t
hdlr_get_media_type(hdlr_t *hdlr)
{
  qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr->atom.type, QTIME_MEDIA_TYPE_UNKNOWN)
  if (hdlr->component_type == QTIME_TYPE_MHLR) {
    switch (hdlr->component_subtype) {
      case QTIME_TYPE_VIDE:
	return QTIME_MEDIA_TYPE_VIDEO;
	break;
      case QTIME_TYPE_SOUN:
	return QTIME_MEDIA_TYPE_SOUND;
	break;
      default:
	qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
	fprintf(stderr, "QTIME_ERROR_HDLR: media type cannot be specified in 'hdlr' atom.\n");
	break;
    }
  }
  return QTIME_MEDIA_TYPE_UNKNOWN;
}

int
hdlr_set_media_handler(hdlr_t *hdlr, uint32_t media_type)
{
  qtime_error_type_check_v(QTIME_TYPE_HDLR, hdlr->atom.type, QTIME_ERROR_ILLEGAL_ATOM)
  if (hdlr->component_type == QTIME_TYPE_DHLR) {
    fprintf(stderr, "QTIME_ERROR_HDLR: handler type doesn't suit.\n");
    return QTIME_ERROR_INVALID_MEDIA;
  }
  hdlr->component_type = QTIME_TYPE_MHLR;

  if (hdlr->component_subtype != 0 && hdlr->component_subtype != media_type) {
    fprintf(stderr, "QTIME_ERROR_HDLR: media type doesn't suit.\n");
  }

  switch (media_type) {
    case QTIME_MEDIA_TYPE_VIDEO:
      hdlr->component_subtype = QTIME_TYPE_VIDE;
      break;
    case QTIME_MEDIA_TYPE_SOUND:
      hdlr->component_subtype = QTIME_TYPE_SOUN;
      break;
    default:
      fprintf(stderr, "QTIME_ERROR_HDLR: unsupported media_type.\n");
      return QTIME_ERROR;
      break;
  }

  if (!hdlr->component_name) {
    hdlr_set_component_name(hdlr, "libqtime media handler");
  }

  return QTIME_OK;
}

int
hdlr_set_data_handler(hdlr_t *hdlr, uint32_t subtype)
{
  if (hdlr->component_type == QTIME_TYPE_MHLR) {
    fprintf(stderr, "QTIME_ERROR_HDLR: handler type doesn't suit.\n");
    return QTIME_ERROR;
  }

  if (hdlr->component_subtype != 0 && hdlr->component_subtype != subtype) {
    fprintf(stderr, "QTIME_ERROR_HDLR: subtype doesn't suit.\n");
  }

  switch (subtype) {
    case QTIME_TYPE_ALIS:
      break;
    default:
      fprintf(stderr, "QTIME_ERROR_HDLR: unsupported subtype.\n");
      return QTIME_ERROR;
      break;
  }

  hdlr->component_type = QTIME_TYPE_DHLR;
  hdlr->component_subtype = subtype;

  if (!hdlr->component_name) {
    hdlr_set_component_name(hdlr, "libqtime data handler");
  }

  return QTIME_OK;
}

