/* gmin.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_media_type.h"
#include "atom.h"
#include "gmin.h"

void
gmin_init(gmin_t *gmin)
{
  atom_init(&gmin->atom);
  gmin->atom.type = QTIME_TYPE_GMIN;
  gmin->version = 0;
  qtime_flags_set(gmin->flags, 0);
  gmin->graphics_mode = 0;
  gmin->opcolor[0] = 0;
  gmin->opcolor[1] = 0;
  gmin->opcolor[2] = 0;
  gmin->balance = 0;
  gmin->reserved = 0;
}

void
gmin_clean(gmin_t *gmin)
{
  qtime_error_type_check(QTIME_TYPE_GMIN, gmin->atom.type)
  gmin_init(gmin);
}

gmin_t*
gmin_new(void)
{
  gmin_t *gmin;
  gmin = (gmin_t*)qtime_malloc(sizeof(gmin_t));
  if (!gmin) return NULL;
  gmin_init(gmin);
  return gmin;
}

void
gmin_delete(gmin_t *gmin)
{
  qtime_error_type_check(QTIME_TYPE_GMIN, gmin->atom.type)
  qtime_free(gmin);
}

gmin_t*
gmin_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, gmin_t *gmin_ptr)
{
  gmin_t *gmin;

#ifndef NDEBUG
  if (atom_head->media_type == QTIME_MEDIA_TYPE_VIDEO ||
      atom_head->media_type == QTIME_MEDIA_TYPE_SOUND) {
    qtime_error_debug_info(QTIME_ERROR_INVALID_MEDIA);
    qtime_error_invalid_media(QTIME_TYPE_GMIN, QTIME_MEDIA_TYPE_BASE, atom_head->media_type);
    return NULL;
  }
#endif

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_GMIN, atom_head->type, NULL)
  if (gmin_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_GMIN, gmin_ptr->atom.type, NULL)
    gmin = gmin_ptr;
    gmin_clean(gmin);
  } else {
    if ((gmin = gmin_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  gmin->atom.size = atom_head->size;
  gmin->atom.type = QTIME_TYPE_GMIN;
  gmin->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &gmin->version, 1);
  qtime_io_read(qtio, gmin->flags, 3);
  qtime_io_read16(qtio, &gmin->graphics_mode);
  qtime_io_read16(qtio, &gmin->opcolor[0]);
  qtime_io_read16(qtio, &gmin->opcolor[1]);
  qtime_io_read16(qtio, &gmin->opcolor[2]);
  qtime_io_read16(qtio, &gmin->balance);
  qtime_io_read16(qtio, &gmin->reserved);

  atom_head->media_type = QTIME_MEDIA_TYPE_BASE;

  return gmin;
}

gmin_t*
gmin_create(gmin_t *gmin_ptr)
{
  gmin_t *gmin;

  if (gmin_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_GMIN, gmin_ptr->atom.type, NULL)
    gmin = gmin_ptr;
    gmin_clean(gmin);
  } else {
    if ((gmin = gmin_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return gmin;
}

int64_t
gmin_calc_size(gmin_t *gmin)
{
  int64_t size = 0;
  qtime_error_type_check_v(QTIME_TYPE_GMIN, gmin->atom.type, 0)
  size = 8 + GMIN_PROP_SIZE;
  gmin->atom.size = size;
  return size;
}

int
gmin_write_atom(qtime_io_t *qtio, gmin_t *gmin)
{
  atom_head_t atom_head;

  qtime_error_type_check_v(QTIME_TYPE_GMIN, gmin->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  atom_head_init(&atom_head);
  atom_head.size = gmin->atom.size;
  atom_head.type = gmin->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_GMIN);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &gmin->version, 1);
  qtime_io_write(qtio, gmin->flags, 3);
  qtime_io_write16(qtio, &gmin->graphics_mode);
  qtime_io_write16(qtio, &gmin->opcolor[0]);
  qtime_io_write16(qtio, &gmin->opcolor[1]);
  qtime_io_write16(qtio, &gmin->opcolor[2]);
  qtime_io_write16(qtio, &gmin->balance);
  qtime_io_write16(qtio, &gmin->reserved);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_GMIN);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
gmin_dump(const char *parent_types, gmin_t *gmin)
{
  int i, len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  qtime_type_to_str(gmin->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(gmin->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)gmin->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, gmin->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: graphics mode %u\n", types, gmin->graphics_mode);
  for (i = 0; i < 3; i++) {
    fprintf(stdout, "%s: opcolor[%d]  %u\n", types, i, gmin->opcolor[i]);
  }
  fprintf(stdout, "%s: balance      %f\n", types, qtime_fixed16_to_float(gmin->balance));
}

