/* edts.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "edts.h"
#include "elst.h"

void
edts_init(edts_t *edts)
{
  atom_init(&edts->atom);
  edts->atom.type = QTIME_TYPE_EDTS;
  elst_init(&edts->elst);
}

void
edts_clean(edts_t *edts)
{
  int i;
  atom_t *atom = &edts->atom;

  qtime_error_type_check(QTIME_TYPE_EDTS, edts->atom.type)

  for (i = 0; i < atom->number_of_childs; i++) {
    if (atom->childs[i]->type == QTIME_TYPE_ELST) {
      elst_clean((elst_t*)atom->childs[i]);
    } else {
      qtime_error_debug_info(QTIME_ERROR_UNKNOWN_ATOM);
      qtime_error_illegal_atom(QTIME_TYPE_EDTS, atom->childs[i]->type);
    }
  }
  atom_clean(&edts->atom);
  edts->atom.type = QTIME_TYPE_EDTS;
}

edts_t*
edts_new(void)
{
  edts_t *edts;
  edts = (edts_t*)qtime_malloc(sizeof(edts_t));
  if (!edts)
    return NULL;
  edts_init(edts);
  return edts;
}

void
edts_delete(edts_t *edts)
{
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_EDTS, edts->atom.type)) {
    return;
  }
#endif
  edts_clean(edts);
  edts->atom.type = QTIME_TYPE_EDTS;
  qtime_free(edts);
}

edts_t*
edts_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, edts_t *edts_ptr)
{
  atom_t *atom;
  atom_head_t subatom;
  edts_t *edts;
  elst_t *elst;
  int elst_index;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_EDTS, atom_head->type)) {
    atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
    return NULL;
  }
#endif

  if (edts_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_EDTS, edts_ptr->atom.type)) {
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
    }
#endif
    edts = edts_ptr;
    edts_clean(edts);
  } else {
    if ((edts = edts_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
    }
  }

  edts->atom.size = atom_head->size;
  edts->atom.type = QTIME_TYPE_EDTS;
  edts->atom.parent = atom_head->parent;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)edts;

  elst = &edts->elst;
  elst_index = 0;

  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    atom = NULL;
    switch (subatom.type) {
      case QTIME_TYPE_ELST:
	if ((elst = elst_read_atom(qtio, &subatom, elst)) == NULL) {
          qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
          atom_head->error_code = QTIME_ERROR_ATOM_READ;
	  goto fail;
	}
	atom = (atom_t*)elst;
	atom->index = ++elst_index;
	elst = NULL;
	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_EDTS, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom) {
      atom->parent = (atom_t*)edts;
      if ((atom_head->error_code = atom_add_child((atom_t*)edts, atom)) < 0) {
	qtime_error_debug_info(atom_head->error_code);
	goto fail;
      }
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  return edts;

fail:
  if (edts_ptr)
    edts_clean(edts);
  else
    edts_delete(edts);
  qtime_error_atom_read(QTIME_TYPE_EDTS);
  return NULL;
}

edts_t*
edts_create(edts_t *edts_ptr)
{
  edts_t *edts;
  int error_code;

  if (edts_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_EDTS, edts_ptr->atom.type)) {
      return NULL;
    }
#endif
    edts = edts_ptr;
    edts_clean(edts);
  } else {
    if ((edts = edts_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  elst_create(&edts->elst);

  if ((error_code = atom_add_child((atom_t*)edts, (atom_t*)&edts->elst)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }

  return edts;

fail:
  if (edts_ptr)
    edts_clean(edts);
  else
    edts_delete(edts);
  qtime_error_debug_info(error_code);
  return NULL;
}

int64_t
edts_calc_size(edts_t *edts)
{
  int i;
  int64_t size = 0;
  atom_t *atom = &edts->atom;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_EDTS, edts->atom.type)) {
    return 0;
  }
#endif
  for (i = 0; i < atom->number_of_childs; i++)
    if (atom->childs[i]->type == QTIME_TYPE_ELST)
      size += elst_calc_size((elst_t*)atom->childs[i]);
  if (size <= 0)
    return 0;
  size += 8;
  edts->atom.size = size;
  return size;
}

int
edts_write_atom(qtime_io_t *qtio, edts_t *edts)
{
  int i;
  atom_head_t atom_head;
  atom_t *atom = &edts->atom;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_EDTS, edts->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif

  if (edts->atom.number_of_childs <= 0)
    return QTIME_OK;

  if (edts->atom.size <= 8)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = edts->atom.size;
  atom_head.type = edts->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_EDTS);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < atom->number_of_childs; i++) {
    if (atom->childs[i]->type == QTIME_TYPE_ELST) {
      elst_write_atom(qtio, (elst_t*)atom->childs[i]);
    } else {
      qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
      qtime_error_illegal_atom(QTIME_TYPE_EDTS, atom->childs[i]->type);
    }
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_EDTS);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
edts_dump(const char *parent_types, edts_t *edts)
{
  int i;
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];

  if (edts->atom.number_of_childs <= 0)
    return;

  qtime_type_to_str(edts->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)edts->atom.size);
  for (i = 0; i < edts->atom.number_of_childs; i++) {
    if (edts->atom.childs[i]->type == QTIME_TYPE_ELST) {
      elst_dump(types, (elst_t*)edts->atom.childs[i]);
    } else {
      qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
      qtime_error_illegal_atom(QTIME_TYPE_EDTS, edts->atom.childs[i]->type);
    }
  }
}

