/* dref.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "qtime_be.h"
#include "atom.h"
#include "dref.h"

/* data_ref */

int
data_ref_type_check(uint32_t type)
{
  switch (type) {
    case QTIME_TYPE_ALIS:
    case QTIME_TYPE_RSRC:
    case QTIME_TYPE_URL:
      return 0;
      break;
  }
  return QTIME_ERROR_ILLEGAL_ATOM;
}

void
data_ref_init(data_ref_t *data_ref)
{
  data_ref->size = 0;
  data_ref->type = 0;
  data_ref->version = 0;
//  qtime_flags_set(data_ref->flags, DREF_FLAG_SELF_REFERENCE);
  qtime_flags_set(data_ref->flags, 0);
  data_ref->data_size = 0;
  data_ref->data = NULL;
}

void
data_ref_clean(data_ref_t *data_ref)
{
#ifndef NDEBUG
  if (data_ref_type_check(data_ref->type) < 0) {
    fprintf(stderr, "QTIME_ERROR_TYPE_CHECK:%s:%d:%s\n",__FILE__,__LINE__,__FUNCTION__);
    qtime_error_illegal_atom(QTIME_TYPE_DREF, data_ref->type);
    return;
  }
#endif

  if (data_ref->data)
    qtime_free(data_ref->data);
  data_ref_init(data_ref);
}

data_ref_t*
data_ref_new(void)
{
  data_ref_t *data_ref;
  data_ref = (data_ref_t*)qtime_malloc(sizeof(data_ref_t));
  if (!data_ref) return NULL;
  data_ref_init(data_ref);
  return data_ref;
}

void
data_ref_delete(data_ref_t *data_ref)
{
#ifndef NDEBUG
  if (data_ref_type_check(data_ref->type) < 0) {
    fprintf(stderr, "QTIME_ERROR_TYPE_CHECK:%s:%d:%s\n",__FILE__,__LINE__,__FUNCTION__);
    qtime_error_illegal_atom(QTIME_TYPE_DREF, data_ref->type);
    return;
  }
#endif

  if (data_ref->data)
    qtime_free(data_ref->data);
  qtime_free(data_ref);
}

data_ref_t*
data_ref_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, data_ref_t *data_ref_ptr)
{
  int data_size = 0;
  uint8_t *data = NULL;
  data_ref_t *data_ref = NULL;

#ifndef NDEBUG
  if (data_ref_type_check(atom_head->type) < 0) {
    fprintf(stderr, "QTIME_ERROR_TYPE_CHECK:%s:%d:%s\n",__FILE__,__LINE__,__FUNCTION__);
    qtime_error_illegal_atom(QTIME_TYPE_DREF, atom_head->type);
    atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
    return NULL;
  }
#endif

  if (data_ref_ptr) {
#ifndef NDEBUG
    if (data_ref_type_check(data_ref->type) < 0) {
      fprintf(stderr, "QTIME_ERROR_TYPE_CHECK:%s:%d:%s\n",__FILE__,__LINE__,__FUNCTION__);
      qtime_error_illegal_atom(QTIME_TYPE_DREF, data_ref->type);
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
    }
#endif
    data_ref = data_ref_ptr;
    data_ref_clean(data_ref);
  } else {
    if ((data_ref = data_ref_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  data_size = data_ref->data_size;
  data      = data_ref->data;
  data_size = atom_head->body_size;
  data_size -= DATA_REF_PROP_SIZE;
  if (data_size < 0) {
    data_ref->data_size = 0;
    if (data_ref->data)
      qtime_free(data_ref->data);
    data_ref->data = NULL;
    qtime_error_debug_info(QTIME_ERROR_ATOM_SIZE);
    qtime_error_atom_size(atom_head->type, (int32_t)atom_head->body_size, 0);
    atom_head->error_code = QTIME_ERROR_ATOM_SIZE;
    goto fail;
  }

  data_ref->size = atom_head->size;
  data_ref->type = atom_head->type;

  qtime_io_read(qtio, &data_ref->version, 1);
  qtime_io_read(qtio, data_ref->flags, 3);

  if (data_size > 0) {
    if ((data = (uint8_t*)qtime_realloc(data, data_size+1)) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      goto fail;
    }
    qtime_io_read(qtio, data, data_size);
    data[data_size] = '\0';
    data_ref->data      = data;
    data_ref->data_size = data_size;
  }

  return data_ref;

fail:
  if (data_ref_ptr)
    data_ref_clean(data_ref);
  else
    data_ref_delete(data_ref);
  qtime_error_atom_read(atom_head->type);
  return NULL;
}

data_ref_t*
data_ref_create(data_ref_t* data_ref_ptr)
{
  data_ref_t *data_ref = NULL;

  if (data_ref_ptr) {
#ifndef NDEBUG
    if (data_ref_type_check(data_ref->type) < 0) {
      fprintf(stderr, "QTIME_ERROR_TYPE_CHECK:%s:%d:%s\n",__FILE__,__LINE__,__FUNCTION__);
      qtime_error_illegal_atom(QTIME_TYPE_DREF, data_ref->type);
      return NULL;
    }
#endif
    data_ref = data_ref_ptr;
    data_ref_clean(data_ref);
  } else {
    if ((data_ref = data_ref_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  return data_ref;
}

int
data_ref_valid(data_ref_t *data_ref)
{
  uint32_t flags;
  qtime_flags_get(data_ref->flags, &flags);
  if (flags == DREF_FLAG_SELF_REFERENCE)
    return 1;
  if (data_ref->data && data_ref->data_size > 0)
    return 1;
  return 0;
}

int64_t
data_ref_calc_size(data_ref_t *data_ref)
{
  int64_t size;

#ifndef NDEBUG
  if (data_ref_type_check(data_ref->type) < 0) {
    fprintf(stderr, "QTIME_ERROR_TYPE_CHECK:%s:%d:%s\n",__FILE__,__LINE__,__FUNCTION__);
    qtime_error_illegal_atom(QTIME_TYPE_DREF, data_ref->type);
    return 0;
  }
#endif

  if (!data_ref_valid(data_ref))
    return 0;
  size = DATA_REF_PROP_SIZE + data_ref->data_size;
  size += 8;
  data_ref->size = size;
  return size;
}

int
data_ref_write_atom(qtime_io_t *qtio, data_ref_t *data_ref)
{
  atom_head_t atom_head;

#ifndef NDEBUG
  if (data_ref_type_check(data_ref->type) < 0) {
    fprintf(stderr, "QTIME_ERROR_TYPE_CHECK:%s:%d:%s\n",__FILE__,__LINE__,__FUNCTION__);
    qtime_error_illegal_atom(QTIME_TYPE_DREF, data_ref->type);
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif

  if (!data_ref_valid(data_ref))
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = data_ref->size;
  atom_head.type = data_ref->type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(data_ref->type);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &data_ref->version, 1);
  qtime_io_write(qtio, data_ref->flags, 3);
  if (data_ref->data_size > 0)
    qtime_io_write(qtio, data_ref->data, data_ref->data_size);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(data_ref->type);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
data_ref_dump(const char *parent_types, data_ref_t *data_ref)
{
  int len = strlen(parent_types);
  uint8_t type_str[5];
  uint8_t types[len+6];
  uint32_t flags;
  uint8_t buf[data_ref->data_size+1];

  qtime_type_to_str(data_ref->type, type_str);
  sprintf(types, "%s.%.4s", parent_types, type_str);
  qtime_flags_get(data_ref->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)data_ref->size);
  fprintf(stdout, "%s: version      %d\n", types, data_ref->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: data size    %d\n", types, data_ref->data_size);
  if (data_ref->type == QTIME_TYPE_ALIS || data_ref->type == QTIME_TYPE_URL) {
    memcpy(buf, data_ref->data, data_ref->data_size);
    buf[data_ref->data_size] = '\0';
    fprintf(stdout, "%s: data = '%s'\n", types, buf);
  } else if (data_ref->type == QTIME_TYPE_RSRC) {
    if (data_ref->data_size > 0) {
      int16_t id;
      uint32_t res;
      memcpy(buf, data_ref->data, data_ref->data_size-4-2);
      buf[data_ref->data_size-4-2] = '\0';
      fprintf(stdout, "%s: data = '%s'\n", types, buf);
      qtime_type_to_str(*((uint32_t*)&data_ref->data[data_ref->data_size-4-2]), type_str);
      QTIME_BE32((*(uint32_t*)&data_ref->data[data_ref->data_size-4-2]), res);
      fprintf(stdout, "%s: resource type '%.4s' 0x%x\n", types, type_str, res);
      QTIME_BE16((*(uint16_t*)&data_ref->data[data_ref->data_size-2]), id);
      fprintf(stdout, "%s: resource id   %d\n", types, id);
    }
  }
}

/* dref */

void
dref_init(dref_t *dref)
{
  atom_init(&dref->atom);
  dref->atom.type = QTIME_TYPE_DREF;
  dref->version = 0;
  qtime_flags_set(dref->flags, 0);
  dref->number_of_entries = 0;
  dref->data_ref = NULL;
}

void
dref_clean(dref_t *dref)
{
  int i;

  qtime_error_type_check(QTIME_TYPE_DREF, dref->atom.type)

  for (i = 0; i < dref->number_of_entries; i++)
    data_ref_delete(dref->data_ref[i]);
  if (dref->data_ref)
    qtime_free(dref->data_ref);
  dref_init(dref);
}

dref_t*
dref_new(void)
{
  dref_t *dref;
  dref = (dref_t*)qtime_malloc(sizeof(dref_t));
  if (!dref) return NULL;
  dref_init(dref);
  return dref;
}

void
dref_delete(dref_t *dref)
{
  qtime_error_type_check(QTIME_TYPE_DREF, dref->atom.type)
  dref_clean(dref);
  qtime_free(dref);
}

int
dref_add_list(dref_t *dref, data_ref_t *data_ref)
{
  int num_ent = 0;
  data_ref_t **list = NULL;

  num_ent = dref->number_of_entries;
  list = dref->data_ref;

  list =(data_ref_t**)qtime_realloc(list,sizeof(data_ref_t*)*(num_ent+1));
  if (!list) {
    qtime_error_debug_info(QTIME_ERROR_MEMORY);
    return QTIME_ERROR_MEMORY;
  }
  list[num_ent] = data_ref;
  num_ent++;
  dref->number_of_entries = num_ent;
  dref->data_ref = list;
  return num_ent;
}

dref_t*
dref_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, dref_t *dref_ptr)
{
  data_ref_t *data_ref;
  atom_head_t subatom;
  int num_ent;
  dref_t *dref;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  if (dref_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_DREF, dref_ptr->atom.type, NULL)
    dref = dref_ptr;
    dref_clean(dref);
  } else {
    if ((dref = dref_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  dref->atom.size = atom_head->size;
  dref->atom.type = QTIME_TYPE_DREF;
  dref->atom.parent = atom_head->parent;

  qtime_io_read(qtio, &dref->version, 1);
  qtime_io_read(qtio, dref->flags, 3);
//  qtime_io_read32(qtio, &dref->number_of_entries);
  qtime_io_read32(qtio, &num_ent);

  atom_head_init(&subatom);

  data_ref = NULL;

  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    switch (subatom.type) {
      case QTIME_TYPE_ALIS:
      case QTIME_TYPE_RSRC:
      case QTIME_TYPE_URL:
	if ((data_ref = data_ref_read_atom(qtio, &subatom, data_ref)) == NULL) {
	  qtime_error_debug_info(subatom.error_code);
	  atom_head->error_code = subatom.error_code;
	  goto fail;
	}
	if ((atom_head->error_code = dref_add_list(dref, data_ref)) < 0) {
	  qtime_error_debug_info(atom_head->error_code);
	  goto fail;
	}
	data_ref = NULL;
	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_DREF, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  if (num_ent != dref->number_of_entries) {
    fprintf(stderr, "dref_read_atom: The number_of_entries doesn't suit. num_ent %d, entries %d\n", num_ent, dref->number_of_entries);
  }

  return dref;

fail:
  if (dref_ptr)
    dref_clean(dref);
  else
    dref_delete(dref);
  qtime_error_atom_read(QTIME_TYPE_DREF);
  return NULL;
}

dref_t*
dref_create(dref_t *dref_ptr)
{
  dref_t *dref;

  if (dref_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_DREF, dref_ptr->atom.type, NULL)
    dref = dref_ptr;
    dref_clean(dref);
  } else {
    if ((dref = dref_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }
  return dref;
}

int64_t
dref_calc_size(dref_t *dref)
{
  int64_t size = 0;
  int i;

  qtime_error_type_check_v(QTIME_TYPE_DREF, dref->atom.type, 0)

  if (dref->number_of_entries <= 0)
    return 0;

  size = DREF_PROP_SIZE;

  for (i = 0; i < dref->number_of_entries; i++)
    size += data_ref_calc_size(dref->data_ref[i]);

  size += 8;
  dref->atom.size = size;

  return size;
}

int
dref_write_atom(qtime_io_t *qtio, dref_t *dref)
{
  atom_head_t atom_head;
  int i;

  qtime_error_type_check_v(QTIME_TYPE_DREF, dref->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  if (dref->number_of_entries <= 0)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = dref->atom.size;
  atom_head.type = dref->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_DREF);
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write(qtio, &dref->version, 1);
  qtime_io_write(qtio, dref->flags, 3);
  qtime_io_write32(qtio, &dref->number_of_entries);

  for (i = 0; i < dref->number_of_entries; i++) {
    data_ref_write_atom(qtio, dref->data_ref[i]);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_DREF);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
dref_dump(const char *parent_types, dref_t *dref)
{
  int i;
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  uint32_t flags;

  qtime_type_to_str(dref->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);
  qtime_flags_get(dref->flags, &flags);
  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)dref->atom.size);
  fprintf(stdout, "%s: version      %d\n", types, dref->version);
  fprintf(stdout, "%s: flags        0x%x\n", types, flags);
  fprintf(stdout, "%s: number of entries %d\n", types, dref->number_of_entries);
  for (i = 0; i < dref->number_of_entries; i++) {
    data_ref_dump(types, dref->data_ref[i]);
  }
}

int
dref_add_data_ref(dref_t *dref, uint32_t type, uint32_t flags, uint8_t *data, int data_size)
{
  data_ref_t *data_ref;
  uint8_t *d;
  int id;

  qtime_error_type_check_v(QTIME_TYPE_DREF, dref->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  switch (type) {
    case QTIME_TYPE_ALIS:
    case QTIME_TYPE_RSRC:
    case QTIME_TYPE_URL:
      break;
    default:
      qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
      qtime_error_illegal_atom(QTIME_TYPE_DREF, type);
      return QTIME_ERROR_ILLEGAL_ATOM;
      break;
  }

  d = (uint8_t*)qtime_malloc(data_size+1);
  if (!d) {
    return QTIME_ERROR_MEMORY;
  }
  memcpy(d, data, data_size);
  d[data_size] = '\0';

  data_ref = data_ref_new();
  if (!data_ref) {
    qtime_free(d);
    return QTIME_ERROR_MEMORY;
  }
  data_ref->type = type;
  if (flags != DREF_FLAG_SELF_REFERENCE)
    flags = 0;
  qtime_flags_set(data_ref->flags, flags);
  data_ref->data_size = data_size;
  data_ref->data = d;

  if ((id = dref_add_list(dref, data_ref)) < 0) {
    data_ref_delete(data_ref);
    return QTIME_ERROR;
  }

  return id;
}

int
dref_add_alis(dref_t *dref, uint32_t flags, uint8_t *path)
{
  int data_size;
  data_size = strlen(path);
  return dref_add_data_ref(dref, QTIME_TYPE_ALIS, flags, path, data_size);
}

