/* dinf.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "dinf.h"
#include "dref.h"

void
dinf_init(dinf_t *dinf)
{
  atom_init(&dinf->atom);
  dinf->atom.type = QTIME_TYPE_DINF;
  dref_init(&dinf->dref);
}

void
dinf_clean(dinf_t *dinf)
{
  int i;
  atom_t *atom = &dinf->atom;

  qtime_error_type_check(QTIME_TYPE_DINF, dinf->atom.type);

  for (i = 0; i < atom->number_of_childs; i++) {
    if (atom->childs[i] == (atom_t*)&dinf->dref) {
      dref_clean((dref_t*)atom->childs[i]);
      continue;
    }

    switch (atom->childs[i]->type) {
      case QTIME_TYPE_DREF:
	dref_delete((dref_t*)atom->childs[i]);
	break;
      default:
        qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_DINF, atom->childs[i]->type);
	break;
    }
  }

  atom_clean(&dinf->atom);
  dinf->atom.type = QTIME_TYPE_DINF;
}

dinf_t*
dinf_new(void)
{
  dinf_t *dinf;
  dinf = (dinf_t*)qtime_malloc(sizeof(dinf_t));
  if (!dinf) return NULL;
  dinf_init(dinf);
  return dinf;
}

void
dinf_delete(dinf_t *dinf)
{
  qtime_error_type_check(QTIME_TYPE_DINF, dinf->atom.type);
  dinf_clean(dinf);
  qtime_free(dinf);
}

dinf_t*
dinf_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, dinf_t *dinf_ptr)
{
  atom_head_t subatom;
  atom_t *atom;
  dinf_t *dinf;
  dref_t *dref;
  int dref_index;

  atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
  qtime_error_type_check_v(QTIME_TYPE_DINF, atom_head->type, NULL)
  if (dinf_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_DINF, dinf_ptr->atom.type, NULL)
    dinf = dinf_ptr;
    dinf_clean(dinf);
  } else {
    if ((dinf = dinf_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }
  atom_head->error_code = QTIME_OK;

  dinf->atom.size = atom_head->size;
  dinf->atom.type = QTIME_TYPE_DINF;
  dinf->atom.parent = atom_head->parent;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)dinf;

  dref = &dinf->dref;
  dref_index = 0;

  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }

    atom = NULL;
    switch (subatom.type) {
      case QTIME_TYPE_DREF:
        if ((dref = dref_read_atom(qtio, &subatom, dref)) == NULL) {
          qtime_error_debug_info(subatom.error_code);
          atom_head->error_code = subatom.error_code;
	  goto fail;
        }
        atom = (atom_t*)dref;
        atom->index = ++dref_index;
	dref = NULL;
	break;
      default:
        qtime_error_unknown_atom(QTIME_TYPE_DINF, subatom.type, (int32_t)subatom.size);
	break;
    }

    if (atom) {
      if ((atom_head->error_code = atom_add_child((atom_t*)dinf, atom)) < 0) {
        qtime_error_debug_info(atom_head->error_code);
        goto fail;
      }
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      goto fail;
    }
  }

  return dinf;

fail:
  if (dinf_ptr)
    dinf_clean(dinf);
  else
    dinf_delete(dinf);
  qtime_error_atom_read(QTIME_TYPE_DINF);
  return NULL;
}

dinf_t*
dinf_create(dinf_t *dinf_ptr)
{
  dinf_t *dinf;
  int error_code;

  if (dinf_ptr) {
    qtime_error_type_check_v(QTIME_TYPE_DINF, dinf_ptr->atom.type, NULL)
    dinf = dinf_ptr;
    dinf_clean(dinf);
  } else {
    if ((dinf = dinf_new()) == NULL) {
      qtime_error_debug_info(QTIME_ERROR_MEMORY);
      return NULL;
    }
  }

  dref_create(&dinf->dref);

  if ((error_code = atom_add_child((atom_t*)dinf, (atom_t*)&dinf->dref)) < 0) {
    qtime_error_debug_info(error_code);
    goto fail;
  }

  return dinf;

fail:
  if (dinf_ptr)
    dinf_clean(dinf);
  else
    dinf_delete(dinf);
  qtime_error_debug_info(error_code);
  return NULL;
}

int64_t
dinf_calc_size(dinf_t *dinf)
{
  int i;
  int64_t size = 0;
  atom_t *atom = &dinf->atom;

  qtime_error_type_check_v(QTIME_TYPE_DINF, dinf->atom.type, 0)

  if (atom->number_of_childs <= 0)
    return 0;

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_DREF:
	size += dref_calc_size((dref_t*)atom->childs[i]);
	break;
      default:
        qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_DINF, atom->childs[i]->type);
	break;
    }
  }

  size += 8;
  dinf->atom.size = size;
  return size;
}

int
dinf_write_atom(qtime_io_t *qtio, dinf_t *dinf)
{
  atom_head_t atom_head;
  int i;
  atom_t *atom = &dinf->atom;

  qtime_error_type_check_v(QTIME_TYPE_DINF, dinf->atom.type, QTIME_ERROR_ILLEGAL_ATOM)

  if (dinf->atom.number_of_childs <= 0)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = dinf->atom.size;
  atom_head.type = dinf->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_DINF);
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_DREF:
	dref_write_atom(qtio, (dref_t*)atom->childs[i]);
	break;
      default:
        qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_DINF, atom->childs[i]->type);
	break;
    }
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    qtime_error_debug_info(QTIME_ERROR_ATOM_WRITE);
    qtime_error_atom_write(QTIME_TYPE_DINF);
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
dinf_dump(const char *parent_types, dinf_t *dinf)
{
  int i, len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];
  atom_t *atom;

  qtime_type_to_str(dinf->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)dinf->atom.size);
  atom = &dinf->atom;
  for (i = 0; i < atom->number_of_childs; i++) {
    switch (atom->childs[i]->type) {
      case QTIME_TYPE_DREF:
	dref_dump(types, (dref_t*)atom->childs[i]);
	break;
      default:
        qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
        qtime_error_illegal_atom(QTIME_TYPE_DINF, atom->childs[i]->type);
	break;
    }
  }
}

