/* ctab.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "ctab.h"

void
ctab_init(ctab_t *ctab)
{
  atom_init(&ctab->atom);
  ctab->atom.type = QTIME_TYPE_CTAB;
  ctab->color_table_seed = 0;
  ctab->color_table_flags = 0x8000;
  ctab->color_table_size = -1;
  ctab->color_array = NULL;
  ctab->color_array_max = 0;
  return;
}

void
ctab_clean(ctab_t *ctab)
{
  if (!ctab) return;
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_CTAB, ctab->atom.type)) {
    return;
  }
#endif
  if (!ctab->color_array)
    qtime_free(ctab->color_array);
  ctab_init(ctab);
}

ctab_t*
ctab_new(void)
{
  ctab_t *ctab;

  ctab = qtime_malloc(sizeof(ctab_t));
  if (!ctab)
    return NULL;
  ctab_init(ctab);
  return ctab;
}

void
ctab_delete(ctab_t *ctab)
{
  if (!ctab) return;
#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_CTAB, ctab->atom.type)) {
    return;
  }
#endif
  if (!ctab->color_array)
    qtime_free(ctab->color_array);
  qtime_free(ctab);
}

int
ctab_add_table(ctab_t *ctab, uint16_t alpha, uint16_t red, uint16_t green, uint16_t blue)
{
  color_array_t *cary;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_CTAB, ctab->atom.type)) {
    return QTIME_ERROR;
  }
#endif
  if ((ctab->color_table_size+2) > ctab->color_array_max) {
    ctab->color_array_max += 8;
    cary = (color_array_t*)qtime_realloc(ctab->color_array,
        sizeof(color_array_t) * ctab->color_array_max);
    if (!cary)
      return QTIME_ERROR_MEMORY;
    ctab->color_array = cary;
  }
#ifndef NDEBUG
  if (alpha != 0) {
   fprintf(stderr,"ctab_add_table: warnning, alpha value isn't 0, %d.\n",alpha);
   alpha = 0;
  }
#endif
  ctab->color_table_size++;
  ctab->color_array[ctab->color_table_size].alpha = alpha;
  ctab->color_array[ctab->color_table_size].red   = red;
  ctab->color_array[ctab->color_table_size].green = green;
  ctab->color_array[ctab->color_table_size].blue  = blue;
  return QTIME_OK;
}

ctab_t*
ctab_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, ctab_t *ctab_ptr)
{
  color_array_t *color_array;
  int i, table_num, table_byte_size;
  ctab_t *ctab;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_CTAB, atom_head->type)) {
    atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
    return NULL;
  }
#endif

  if (ctab_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_CTAB, ctab_ptr->atom.type)) {
      atom_head->error_code = QTIME_ERROR_ILLEGAL_ATOM;
      return NULL;
    }
#endif
    ctab = ctab_ptr;
    ctab_clean(ctab);
  } else {
    if ((ctab = ctab_new()) == NULL) {
      atom_head->error_code = QTIME_ERROR_MEMORY;
      return NULL;
    }
  }

  ctab->atom.size = atom_head->size;
  ctab->atom.type = QTIME_TYPE_CTAB;
  ctab->atom.parent = atom_head->parent;

  qtime_io_read32(qtio, &ctab->color_table_seed);
  qtime_io_read16(qtio, &ctab->color_table_flags);
  qtime_io_read16(qtio, &ctab->color_table_size);

#ifndef NDEBUG
  {
    int ctab_atom_size, size;
    ctab_atom_size = CTAB_PROP_SIZE;
    ctab_atom_size += (ctab->color_table_size+1) * sizeof(color_array_t);
    size = atom_head->body_size;
    if (size != ctab_atom_size) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_SIZE);
      qtime_error_atom_size(QTIME_TYPE_CTAB, ctab_atom_size, size);
    }
  }
#endif

  table_num = ctab->color_table_size + 1;
  table_byte_size = table_num * sizeof(color_array_t);
  color_array = (color_array_t*)qtime_malloc(table_byte_size);
  if (!color_array) {
    if (ctab_ptr)
      ctab_clean(ctab);
    else
      ctab_delete(ctab);
    atom_head->error_code = QTIME_ERROR_MEMORY;
    return NULL;
  }
  ctab->color_array = color_array;

  for (i = 0; i < table_num; i++) {
    qtime_io_read16(qtio, &color_array[i].alpha);
    qtime_io_read16(qtio, &color_array[i].red);
    qtime_io_read16(qtio, &color_array[i].green);
    qtime_io_read16(qtio, &color_array[i].blue);
  }
  ctab->color_array_max = table_num;

  return ctab;
}

ctab_t*
ctab_create(ctab_t *ctab_ptr)
{
  ctab_t *ctab = NULL;

  if (ctab_ptr) {
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_CTAB, ctab->atom.type)) {
      return NULL;
    }
#endif
    ctab = ctab_ptr;
    ctab_clean(ctab);
  } else {
    if ((ctab = ctab_new()) == NULL) {
      return NULL;
    }
  }
  return ctab;
}

int32_t
ctab_calc_size(ctab_t *ctab)
{
  int32_t size;

  if (!ctab) return 0;
#ifndef NDEBUG
    if (qtime_type_check(QTIME_TYPE_CTAB, ctab->atom.type)) {
      return 0;
    }
#endif
  if (ctab->color_table_size < 0)
    return 0;
  size = 8 + CTAB_PROP_SIZE;
  size += (ctab->color_table_size+1) * COLOR_ARRAY_SIZE;
  ctab->atom.size = size;
  return size;
}

int
ctab_write_atom(qtime_io_t *qtio, ctab_t *ctab)
{
  color_array_t *color_array;
  int i, table_num;
  atom_head_t atom_head;

#ifndef NDEBUG
  if (qtime_type_check(QTIME_TYPE_CTAB, ctab->atom.type)) {
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif

  atom_head_init(&atom_head);
  atom_head.size = ctab->atom.size;
  atom_head.type = ctab->atom.type;

  if (atom_write_header(qtio, &atom_head) < 0) {
    return QTIME_ERROR_ATOM_WRITE;
  }

  qtime_io_write32(qtio, &ctab->color_table_seed);
  qtime_io_write16(qtio, &ctab->color_table_flags);
  qtime_io_write16(qtio, &ctab->color_table_size);

  table_num = ctab->color_table_size + 1;
  color_array = ctab->color_array;

  for (i = 0; i < table_num; i++) {
    qtime_io_write16(qtio, &color_array[i].alpha);
    qtime_io_write16(qtio, &color_array[i].red);
    qtime_io_write16(qtio, &color_array[i].green);
    qtime_io_write16(qtio, &color_array[i].blue);
  }

  if (atom_write_footer(qtio, &atom_head) < 0) {
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
ctab_dump(const char *parent_types, ctab_t *ctab)
{
  int i;
  int len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];

  qtime_type_to_str(ctab->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);

  fprintf(stdout, "%s: size         %lld\n", types, (int64_t)ctab->atom.size);
  fprintf(stdout, "%s: color_table_seed  %d\n", types, ctab->color_table_seed);
  fprintf(stdout, "%s: color_table_flags 0x%x\n", types, ctab->color_table_flags);
  fprintf(stdout, "%s: color_table_size  %d\n", types, ctab->color_table_size);

  for (i = 0; i <= ctab->color_table_size; i++) {
    fprintf(stdout, "%s: color_table_array[%d] alpha = %d, red = %d, green = %d, blue = %d\n", types, i, ctab->color_array[i].alpha, ctab->color_array[i].red, ctab->color_array[i].green, ctab->color_array[i].blue);
  }
}

