/* clip.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include "qtime_io.h"
#include "qtime_util.h"
#include "qtime_error.h"
#include "atom.h"
#include "clip.h"
#include "crgn.h"

void
clip_init(clip_t *clip)
{
  atom_init(&clip->atom);
  clip->atom.type = QTIME_TYPE_CLIP;
  clip->crgn = NULL;
}

void
clip_clean(clip_t *clip)
{
  int i;
  if (!clip) return;
  for (i = 0; i < clip->atom.number_of_childs; i++)
    crgn_delete((crgn_t*)clip->atom.childs[i]);
  atom_childs_free(&clip->atom);
  clip_init(clip);
}

clip_t*
clip_create(void)
{
  clip_t *clip;
  clip = (clip_t*)qtime_malloc(sizeof(clip_t));
  if (!clip)
    return NULL;
  clip_init(clip);
  return clip;
}

void
clip_delete(clip_t *clip)
{
  int i;
  if (!clip) return;
  for (i = 0; i < clip->atom.number_of_childs; i++)
    crgn_delete((crgn_t*)clip->atom.childs[i]);
  atom_childs_free(&clip->atom);
  qtime_free(clip);
}

int
clip_read_atom(qtime_io_t *qtio, atom_head_t *atom_head, clip_t *clip)
{
  atom_t *atom;
  atom_head_t subatom;
  int crgn_index;

#ifdef DEBUG
  if (atom_head->type != QTIME_TYPE_CLIP) {
    qtime_error_debug_info(QTIME_ERROR_ILLEGAL_ATOM);
    qtime_error_print(QTIME_ERROR_ILLEGAL_ATOM,QTIME_TYPE_CLIP,atom_head->type);
    return QTIME_ERROR_ILLEGAL_ATOM;
  }
#endif /* DEBUG */

  clip->atom.size = atom_head->size;
  clip->atom.type = QTIME_TYPE_CLIP;
  clip->atom.parent = atom_head->parent;

  atom_head_init(&subatom);
  subatom.parent = (atom_t*)clip;

  crgn_index = 0;

  while (atom_head->end_offset > subatom.end_offset) {
    if (atom_read_header(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      qtime_error_print(QTIME_ERROR_ATOM_READ, atom_head->type);
      clip_clean(clip);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      return QTIME_ERROR_ATOM_READ;
    }

    atom = NULL;
    if (subatom.type == QTIME_TYPE_CRGN) {
      if (crgn_index > 0) {
        qtime_error_debug_info(QTIME_ERROR_TOO_MANY_ATOM);
        qtime_error_print(QTIME_ERROR_TOO_MANY_ATOM, clip->atom.type, subatom.type);
      }
      if ((atom = (atom_t*)crgn_create_read_atom(qtio, &subatom)) == NULL) {
	clip_clean(clip);
        atom_head->error_code = subatom.error_code;
	return subatom.error_code;
      }
      atom->index = ++crgn_index;
    } else {
      qtime_error_print(QTIME_ERROR_UNKNOWN_ATOM, clip->atom.type, subatom.type, (int32_t)subatom.size);
    }

    if (atom_add_child((atom_t*)clip, atom) < 0) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      qtime_error_print(QTIME_ERROR_ATOM_READ, atom_head->type);
      clip_clean(clip);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      return QTIME_ERROR_ATOM_READ;
    }

    if (atom_read_footer(qtio, &subatom) != QTIME_OK) {
      qtime_error_debug_info(QTIME_ERROR_ATOM_READ);
      qtime_error_print(QTIME_ERROR_ATOM_READ, atom_head->type);
      clip_clean(clip);
      atom_head->error_code = QTIME_ERROR_ATOM_READ;
      return QTIME_ERROR_ATOM_READ;
    }
  }

  return QTIME_OK;
}

clip_t*
clip_create_read_atom(qtime_io_t *qtio, atom_head_t *atom_head)
{
  clip_t *clip;
  if ((clip = clip_create()) == NULL) {
    return NULL;
  }
  if (clip_read_atom(qtio, atom_head, clip) < 0) {
    clip_delete(clip);
    return NULL;
  }
  return clip;
}

int64_t
clip_calc_size(clip_t *clip)
{
  int i;
  int64_t size = 0;

  for (i = 0; i < clip->atom.number_of_childs; i++)
    size += crgn_calc_size((crgn_t*)clip->atom.childs[i]);
  size += 8;
  if (size <= 16)
    size = 0;
  clip->atom.size = size;
  return size;
}

int
clip_write_atom(qtime_io_t *qtio, clip_t *clip)
{
  int i;
  atom_head_t atom_head;

  if (clip->atom.number_of_childs <= 0 || clip->atom.size <= 8)
    return QTIME_OK;

  atom_head_init(&atom_head);
  atom_head.size = clip->atom.size;
  atom_head.type = clip->atom.type;
  if (atom_write_header(qtio, &atom_head) < 0) {
    return QTIME_ERROR_ATOM_WRITE;
  }

  for (i = 0; i < clip->atom.number_of_childs; i++)
    crgn_write_atom(qtio, (crgn_t*)clip->atom.childs[i]);

  if (atom_write_footer(qtio, &atom_head) < 0) {
    return QTIME_ERROR_ATOM_WRITE;
  }

  return QTIME_OK;
}

void
clip_dump(const char *parent_types, clip_t *clip)
{
  int i, len = strlen(parent_types);
  uint8_t types[len+6];
  uint8_t type[5];

  qtime_type_to_str(clip->atom.type, type);
  sprintf(types, "%s.%.4s", parent_types, type);

  fprintf(stdout, "%s: size         %lld\n", types, clip->atom.size);
  for (i = 0; i < clip->atom.number_of_childs; i++)
    crgn_dump(types, (crgn_t*)clip->atom.childs[i]);
}

