/* movtoavi.c */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>

#include <avilib.h>
#include <qtime.h>

uint8_t *vbuf = NULL;
uint8_t *abuf = NULL;
int channels, bits, format, rate, bitrate;
int abuf_size = 0;
int abuf_pos = 0;
int bytespspl = 0;
int audio_bytes = 0;
int audio_samples = 0;

int video_frames = 0;
int sound_frames = 0;

int
audio_write(avi_t *avifile, qtime_track_t *aqtrk, int num_samples)
{
  int samples;
  int bytes = 0;
  int result;
  unsigned short *sp;
  int i, spls;

  if (num_samples <= 0)
    return 0;

  result = 0;
  spls = 0;

  if (format == 0x1) {
    samples = abuf_pos / bytespspl;
    while (samples < num_samples) {
      bytes = qtime_track_read_audio(aqtrk, &abuf[abuf_pos], &spls);
      if (bytes <= 0)
	break;
      abuf_pos += bytes;
      samples += spls;
    }

    if (samples <= 0)
      return result;
    if (samples < num_samples)
      num_samples = samples;

    sp = (unsigned short*)abuf;
    for (i = 0; i < num_samples * channels; i++) {
      *sp = ((*sp)>>8)|((*sp)<<8);
      sp++;
    }
    bytes = num_samples * bytespspl;
    AVI_write_audio(avifile, abuf, bytes);
//    printf("sound_frames %d\n", ++sound_frames);
    if (bytes < abuf_pos)
      memmove(abuf, &abuf[bytes], abuf_pos - bytes);
    abuf_pos -= bytes;
    result = num_samples;
  } else if (format == 0x55) {
    samples = 0;
    while (samples < num_samples) {
      bytes = qtime_track_read_audio(aqtrk, &abuf[abuf_pos], &spls);
      if (bytes <= 0)
	break;
      abuf_pos += bytes;
      samples += spls;
    }

    if (samples <= 0)
      return result;

    AVI_write_audio(avifile, abuf, abuf_pos);
//    printf("sound_frames %d\n", ++sound_frames);
    bytes = abuf_pos;
    abuf_pos = 0;
    result = samples;
  }

  audio_bytes += bytes;
  audio_samples += result;

  return result;
}

int
audio_write_end(avi_t *avifile, qtime_track_t *aqtrk)
{
  int samples = 0;
  int bytes;
  int result = 0;

  while (1) {
    bytes = qtime_track_read_audio(aqtrk, &abuf[abuf_pos], &samples);
//      printf("samples %d\n", samples);
    if (bytes <= 0)
      break;
    abuf_pos += bytes;

    AVI_write_audio(avifile, abuf, abuf_pos);
//    printf("sound_frames %d\n", ++sound_frames);
    abuf_pos = 0;

    audio_bytes += bytes;
    audio_samples += samples;
    result += samples;
  }

  return result;
}

int main (int argc, char *argv[])
{
  char *avifile_name;
  char *qtfile_name;
  avi_t *avifile;
  qtime_t *qt;
  qtime_track_t *qtrk;
  qtime_track_t *vqtrk;
  qtime_track_t *aqtrk;
  int frames, width, height;
  uint32_t audio_samples;
  double fps;
  char *compressor;
  char *audio_compressor;
  int i;
  int start_sample, end_sample;
  int total_size;

  if (argc <= 1) {
    printf("usage: %s <input avi file> <output mov file>\n", argv[0]);
    exit(0);
  }

  if (argc > 1) {
    qtfile_name = argv[1];
  }
  if (argc > 2) {
    avifile_name = argv[2];
  } else {
    avifile_name = "out.avi";
  }

  qt = qtime_open_read(qtfile_name);
  if (!qt) {
    fprintf(stderr, "mov file open failed.\n");
    exit(1);
  }

  qtrk = vqtrk = aqtrk = NULL;
  vqtrk = qtime_get_video_track(qt);
  aqtrk = qtime_get_sound_track(qt);

  if (!vqtrk && !aqtrk) {
    fprintf(stderr, "cannot find media track.\n");
    exit(1);
  }

  if (!vqtrk) {
    fprintf(stderr, "qtime video track was not found.\n");
    frames = 0;
    width = 0;
    height = 0;
    fps = 0;
    compressor = NULL;

    vbuf = NULL;
  } else {
    frames = qtime_track_get_video_frames(vqtrk);
    width = qtime_track_get_video_width(vqtrk);
    height = qtime_track_get_video_height(vqtrk);
    fps = qtime_track_get_video_fps(vqtrk);
    compressor = qtime_track_get_compressor(vqtrk);

    vbuf = (uint8_t*) malloc(width * height * 3);
  }

  if (!aqtrk) {
    fprintf(stderr, "qtime audio track was not found.\n");
    audio_samples = 0;
    channels = 0;
    bits = 0;
    rate = 0;
    audio_compressor = NULL;

    bytespspl = 0;
    abuf_size = 0;
    abuf = NULL;
  } else {
    audio_samples = qtime_track_get_audio_samples(aqtrk);
    channels = qtime_track_get_audio_channels(aqtrk);
    bits = qtime_track_get_audio_bits(aqtrk);
    rate = qtime_track_get_audio_rate(aqtrk);
    audio_compressor = qtime_track_get_compressor(aqtrk);

    bytespspl = ((bits+7)/8) * channels;
    abuf_size = (int)((double)rate / fps * 1.2) * bytespspl;
    abuf = (uint8_t*) malloc(abuf_size);
  }

  avifile = AVI_open_output_file(avifile_name);
  if (!avifile) {
    AVI_print_error("avi file open failed");
    exit(1);
  }
  if (vqtrk) {
    AVI_set_video(avifile, width, height, fps, compressor);
  }
  if (aqtrk) {
    if (strncasecmp(audio_compressor, "twos", 4) == 0) {
      format = 0x1;
      bitrate = 0;
    } else if (strncasecmp(audio_compressor, ".mp3", 4) == 0) {
      format = 0x55;
      bitrate = 0;
    }
    AVI_set_audio(avifile, channels, rate, bits, format, bitrate);
  }

  total_size = 0;
  start_sample = 0;
  for (i = 0; i < frames; i++) {
    int size, keyframe;
    size = qtime_track_read_video(vqtrk, vbuf, &keyframe);
    //printf("%5d: size = %6d, key = %d\n", i, size, keyframe);
    if (size > 0) {
      total_size += size + 16;
      if (total_size > (INT_MAX-(20*1024*1024)))
        break;
      AVI_write_frame(avifile, vbuf, size, keyframe);
//      printf("video_frames %d\n", ++video_frames);
    }
    if (aqtrk) {
      end_sample = (int)((double)(i+1) * (double)rate / fps);
      //printf("%5d: start_sample %d, end_sample %d, diff %d.\n", i, start_sample, end_sample, end_sample - start_sample);
      start_sample += audio_write(avifile, aqtrk, end_sample - start_sample);
    }
  }

  if (aqtrk) {
    double b;
    audio_write_end(avifile, aqtrk);
    bitrate = (int)(((double)audio_bytes * 8.0 / (double)audio_samples * (double)rate + 1) / 1000.);
//    bitrate /= 1000;
    b = (((double)audio_bytes * 8.0 / (double)audio_samples * (double)rate + 1) / 1000.);
//    printf("b %f\n", b);
    AVI_set_audio(avifile, channels, rate, bits, format, bitrate);
    printf("audio_bytes %d, audio_samples %d\n", audio_bytes, audio_samples);
    printf("bitrate %d\n", bitrate);
  }

  qtime_close(qt);
  AVI_close(avifile);

  return 0;
}

