/* aread.c */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
//#include <string.h>
#include <pthread.h>

#include "util.h"
#include "oss.h"
#include "aread.h"
#include "aframe_mgr.h"

#ifndef NDEBUG
#define debug(s) fprintf(stderr,s"\n")
#else
//#define debug(s) fprintf(stderr,s"\n")
#define debug(s)
#endif /* NDEBUG */

#ifndef NDEBUG
void print_flag(const char *mess, uint32_t flag)
{
  if (flag == (NORMAL_AFRAME|CHANGE_AFRAME) ||
      flag == (NORMAL_AFRAME))
    return;
  printf("%s", mess);
  if (flag & START_AFRAME)
    printf("aread print_flag: START_AFRAME\n");
  if (flag & DROP_AFRAME)
    printf("aread print_flag: DROP_AFRAME\n");
  if (flag & END_AFRAME)
    printf("aread print_flag: END_AFRAME\n");
  if (flag & CHANGE_AFRAME)
    printf("aread print_flag: CHANGE_AFRAME\n");
  if (flag & QUIT_AFRAME)
    printf("aread print_flag: QUIT_AFRAME\n");
}
#endif /* NDEBUG */

#define BUFFERED_TIME 3.0

static AREAD_CONFIG aread_conf;

static int aread_initialized = 0;
static int aread_thread_running = 0;

/* VIDEO FRAME infomation functions */

#ifndef NDEBUG
static int total_put_samples = 0;
static int total_gap_count = 0;
#endif /* NDEBUG */

int
aread_set_vinfo(uint32_t flag, u_time time)
{
  AREAD_CONFIG *conf = &aread_conf;
  int ret = AREAD_OK;

  pthread_mutex_lock(&conf->mutex);
#ifndef NDEBUG
  if ((conf->vinfo_top < conf->vinfo_bottom) ||
      (conf->vinfo_top - conf->vinfo_bottom) >= conf->vinfo_num) {
#else
  if ((conf->vinfo_top - conf->vinfo_bottom) >= conf->vinfo_num) {
#endif /* NDEBUG */
    fprintf(stderr, "aread_set_vinfo: vinfo position invalid, top %d, bottom %d.\n", conf->vinfo_top, conf->vinfo_bottom);
    conf->vinfo_bottom++;
    conf->vinfo[conf->vinfo_bottom%conf->vinfo_num].flag |= AREAD_FLAG_OVERRUN;
    ret = AREAD_FAIL;
  }
  conf->vinfo[conf->vinfo_top%conf->vinfo_num].flag = flag;
  conf->vinfo[conf->vinfo_top%conf->vinfo_num].time = time;
  conf->vinfo_top++;
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

static VINFO*
aread_get_vinfo(AREAD_CONFIG *conf)
{
  VINFO *info;
  pthread_mutex_lock(&conf->mutex);
  if (conf->vinfo_top <= conf->vinfo_bottom) {
    info = NULL;
  } else {
    info = &conf->vinfo[conf->vinfo_bottom%conf->vinfo_num];
    conf->vinfo_bottom++;
//    if (conf->vinfo_bottom > conf->vinfo_num) {
    if (conf->vinfo_bottom > conf->vinfo_num*2) {
      conf->vinfo_top -= conf->vinfo_num;
      conf->vinfo_bottom -= conf->vinfo_num;
    }
  }
  pthread_mutex_unlock(&conf->mutex);
  return info;
}

/* Thread helper functions */

static int
init_read_time(AREAD_CONFIG *conf)
{
  int i;
  double ttime;
  double rtime;
  u_time lasttime;
  u_time curtime;
  int buf_size = conf->read_frame_buf_size;
  char buf[buf_size];
  double frame_spl = conf->read_frame_spl_size;
  double rate = conf->rate;
  double frame_time;

  frame_time = 1000000.0 * frame_spl / rate;

  ttime = frame_time;
  curtime = get_cur_u_time();
  for (i = 0; i < conf->read_frame_num; i++) {
    lasttime = curtime;
    oss_read_data(buf, buf_size);
    curtime = get_cur_u_time();
    rtime = curtime - lasttime;
#ifndef NDEBUG
    //printf("init_read_time: i %d, rtime %f\n", i, rtime);
#endif /* NDEBUG */
    if (rtime > ttime * 0.75) {
#ifndef NDEBUG
      //printf("init_read_time: i %d, break loop ttime %f, rtime %f, diff %f\n", i, ttime, rtime, ttime - rtime);
#endif /* NDEBUG */
      break;
    }
  }
  if (i == conf->read_frame_num) {
    fprintf(stderr, "init_read_time: sample read failed.\n");
    return AREAD_FAIL;
  }

  conf->time_chunk_num = 64;
  conf->time_chunk = frame_time * conf->time_chunk_num;
  conf->time_length = conf->time_chunk / conf->time_chunk_num;
  conf->time_chunk = conf->time_chunk - conf->time_length;
  conf->read_time = curtime;
  return AREAD_OK;
}

static double
get_read_time(AREAD_CONFIG *conf)
{
  u_time cur_time;

  cur_time = get_cur_u_time();
  if (cur_time < conf->read_time)
    conf->read_time = cur_time;
  conf->last_read_time = conf->read_time;
#ifndef NDEBUG
//  fprintf(stderr, "get_read_time: time diff %lld\n", cur_time - conf->read_time);
  if (cur_time < conf->read_time) {
    fprintf(stderr, "get_read_time: time error.\n");
    fprintf(stderr, "get_read_time: cur_time %llu, read_time %llu\n", cur_time, conf->read_time);
    exit(1);
  }
#endif /* NDEBUG */
#if 1
  conf->time_chunk = conf->time_chunk + (cur_time - conf->read_time);
  conf->time_length = conf->time_chunk / conf->time_chunk_num;
  conf->time_chunk = conf->time_chunk - conf->time_length;
  conf->read_time = conf->read_time + conf->time_length;
#else
  conf->read_time = cur_time;
#endif
//  fprintf(stderr, "get_read_time: read_time diff %lld\n", conf->read_time - conf->last_read_time);
//  conf->read_time =
//    conf->read_time + ((cur_time - conf->read_time) / conf->time_chunk_num);
  return conf->read_time;
}

#define ROTATE_ON  1
#define ROTATE_OFF 0
static int
sample_read(AREAD_CONFIG *conf, int rotate_pos)
{
  u_time time;
  int top_pos;

#ifndef NDEBUG
  if (conf->read_frame_top > 0 &&
      conf->read_frame_top <= conf->read_frame_bottom) {
    fprintf(stderr, "sample_read: read_frame position invalid, top %d, bottom %d.\n", conf->read_frame_top, conf->read_frame_bottom);
    exit(1);
  }
#endif /* NDEBUG */
  if ((conf->read_frame_top - conf->read_frame_bottom) >= conf->read_frame_num){
    if (conf->record_flag)
//#ifndef NDEBUG
      fprintf(stderr, "sample_read: %d, read_frame overrun. top %d, bottom %d, num %d\n", rotate_pos, conf->read_frame_top, conf->read_frame_bottom, conf->read_frame_num);
//#else
//      fprintf(stderr, "sample_read: read_frame overrun.\n");
//#endif /* NDEBUG */
    conf->read_frame_bottom = conf->read_frame_top - conf->read_frame_num + 1;
    if (conf->read_frame_bottom < 0)
      conf->read_frame_bottom = 0;
    conf->read_frame_cur = conf->read_frame_bottom;
    conf->read_frame_bottom_offset = 0;
    conf->read_frame_cur_offset = 0;
  }

  top_pos = conf->read_frame_top % conf->read_frame_num;
  oss_read_data(conf->read_frame[top_pos].data,
      conf->read_frame_buf_size);
  time = get_read_time(conf);
  conf->read_frame[top_pos].time = time;

  conf->read_frame_top++;
  if (rotate_pos == ROTATE_ON &&
//      conf->read_frame_top    >= conf->read_frame_num &&
      conf->read_frame_bottom > conf->read_frame_num*2) {
    conf->read_frame_top    -= conf->read_frame_num;
    conf->read_frame_bottom -= conf->read_frame_num;
    conf->read_frame_cur    -= conf->read_frame_num;
  }
  return AREAD_OK;
}

static int
set_pos(AREAD_CONFIG *conf, u_time time)
{
  int pos;
  double t;
  int spl, offset;

#ifndef NDEBUG
  if (conf->read_frame_top < conf->read_frame_bottom ||
//    (conf->read_frame_top-conf->read_frame_bottom) >= conf->read_frame_num) {
     (conf->read_frame_top-conf->read_frame_bottom) > conf->read_frame_num) {
    fprintf(stderr, "set_pos: read_frame position invalid, top %d, bottom %d\n", conf->read_frame_top, conf->read_frame_bottom);
  }
#endif /* NDEBUG */
  for (pos = conf->read_frame_bottom; pos < conf->read_frame_top; pos++) {
    if (conf->read_frame[pos%conf->read_frame_num].time > time)
      break;
  }
  while (pos >= conf->read_frame_top) {
    sample_read(conf, ROTATE_OFF);
    debug("set_pos 1.");
    fprintf(stderr, "pos %d, top %d, time %llu, frame time %llu, d %llu\n", pos, conf->read_frame_top, time, conf->read_frame[pos%conf->read_frame_num].time, time-conf->read_frame[pos%conf->read_frame_num].time);
//    fprintf(stderr, "pos %d, top %d, time %llu, frame time %llu\n", pos, conf->read_frame_top, time, conf->read_frame[pos%conf->read_frame_num].time);
    if (conf->read_frame[pos%conf->read_frame_num].time > time)
      break;
    pos++;
  }

  conf->read_frame_cur = pos;
  t = conf->read_frame[pos%conf->read_frame_num].time - time;
#ifndef NDEBUG
  if (t < 0) {
    fprintf(stderr, "set_pos: time diff invalid, time %f\n", t);
    exit(1);
  }
#endif /* NDEBUG */
  offset = (conf->rate * t / 1000000.0 + 0.5);
  if (offset < 0) {
    offset = -offset;
#ifndef NDEBUG
    fprintf(stderr, "set_pos: offset %d invalid\n", offset);
    exit(1);
#endif /* NDEBUG */
    pos += (offset / conf->read_frame_spl_size) + 1;
    offset = offset % conf->read_frame_spl_size;
    while (pos >= conf->read_frame_top) {
      sample_read(conf, ROTATE_OFF);
      debug("set_pos 2.");
    }
  } else if (offset > conf->read_frame_spl_size) {
#ifndef NDEBUG
    fprintf(stderr, "set_pos: offset %d invalid, frame_size %d\n", offset, conf->read_frame_spl_size);
#endif /* NDEBUG */
    offset = 0;
  } else if (offset == 0) {
    pos++;
    while (pos >= conf->read_frame_top) {
      sample_read(conf, ROTATE_OFF);
      debug("set_pos 3.");
    }
  } else {
    offset = conf->read_frame_spl_size - offset;
  }

  if (pos < conf->read_frame_bottom) {
#ifndef NDEBUG
    fprintf(stderr, "set_pos: position error, pos %d, bottom %d\n",pos, conf->read_frame_bottom);
    exit(1);
#endif /* NDEBUG */
    pos = conf->read_frame_bottom;
    offset = conf->read_frame_bottom_offset;
  } else if (pos == conf->read_frame_bottom &&
      offset < conf->read_frame_bottom_offset) {
#ifndef NDEBUG
    fprintf(stderr, "set_pos: position error, offset %d, bottom %d\n",offset, conf->read_frame_bottom_offset);
#endif /* NDEBUG */
    offset = conf->read_frame_bottom_offset;
  }

  conf->read_frame_cur_offset = offset;
  conf->read_frame_cur = pos;

  if (conf->record_flag) {
    spl = (pos - conf->read_frame_bottom) * conf->read_frame_spl_size;
    spl -= conf->read_frame_bottom_offset;
    spl += offset;
  } else {
    conf->read_frame_bottom = pos;
    conf->read_frame_bottom_offset = offset;
    spl = 0;
  }
  conf->vfrm_spl = spl;

#ifndef NDEBUG
  if (conf->read_frame_bottom < 0) {
    fprintf(stderr, "set_pos: read_frame_bottom %d\n", conf->read_frame_bottom);
    exit(1);
  }
#endif /* NDEBUG */
  return spl;
}

static int
sample_cpy(AREAD_CONFIG *conf)
{
  int spl_gap, gap_count;
  int rd_length, rd_len;
  int af_length, af_len;
  int sp_length, offset;
  int interp_count;
#define INTERP_THRESHOLD_MIN -32
#define INTERP_THRESHOLD_MAX  32 

#ifndef NDEBUG
  if (conf->read_frame_bottom < 0) {
    fprintf(stderr, "sample_cpy: read_frame_bottom %d\n", conf->read_frame_bottom);
    exit(1);
  }
#endif /* NDEBUG */
  rd_length = conf->read_frame_spl_size - conf->read_frame_bottom_offset;
  if (conf->read_frame_bottom == conf->read_frame_cur)
    rd_length -= conf->read_frame_spl_size - conf->read_frame_cur_offset;
  af_length = conf->aframe->sample_buf_size - conf->aframe->sample_length;
  interp_count = conf->interp_count;

  rd_len = rd_length;
  af_len = af_length;
  spl_gap = (int)conf->spl_gap;
  gap_count = 0;
  if (conf->bitspspl == 16) {
    short *sp;
    short *dp[2];
    int channels = conf->channels;

    sp = (short*)
      conf->read_frame[conf->read_frame_bottom % conf->read_frame_num].data;
    sp += conf->read_frame_bottom_offset * conf->channels;

    dp[0] = (short*)conf->aframe->sample[0];
    dp[0] += conf->aframe->sample_length;
    if (channels > 1) {
      dp[1] = (short*)conf->aframe->sample[1];
      dp[1] += conf->aframe->sample_length;
    }

    while (rd_len > 0 && af_len > 0) {
      if (interp_count >= INTERP_INTERVAL) {
        interp_count = 0;
        if (spl_gap < INTERP_THRESHOLD_MIN) {
	  sp += channels;
	  rd_len--;
	  spl_gap++;
	  gap_count++;
#ifndef NDEBUG
	  total_gap_count++;
#endif /* NDEBUG */
	  continue;
	} else if (spl_gap > INTERP_THRESHOLD_MAX) {
          *dp[0]++ = *sp;
          if (conf->channels > 1)
            *dp[1]++ = *(sp+1);
	  af_len--;
          spl_gap--;
	  gap_count--;
#ifndef NDEBUG
	  total_gap_count--;
#endif /* NDEBUG */
          continue;
        }
      }
      interp_count++;

      *dp[0]++ = *sp++;
      if (conf->channels > 1)
        *dp[1]++ = *sp++;
      rd_len--;
      af_len--;
    }
  } else if (conf->bitspspl == 8) {
    fprintf(stderr, "sample_cpy: bitspspl == 8 not implement.\n");
    exit(1);
  }

  sp_length = rd_length - rd_len;
  offset = conf->read_frame_bottom_offset + sp_length;
  conf->read_frame_bottom += offset / conf->read_frame_spl_size;
  conf->read_frame_bottom_offset = offset % conf->read_frame_spl_size;

  conf->aframe->sample_length += af_length - af_len;

  conf->interp_count = interp_count;
  conf->spl_gap = conf->spl_gap + gap_count;

  return sp_length;
}

static int
sample_put(AREAD_CONFIG *conf, uint32_t flag)
{
  int len;
  int frame_spl = conf->vfrm_spl;

  while (frame_spl) {
    if (!conf->aframe) {
      conf->aframe = aframe_src_frame_get();
      while (!conf->aframe) {
#ifndef NDEBUG
	fprintf(stderr, "sample_put: aframe_src_frame_get() failed.\n");
#endif /* NDEBUG */
	sample_read(conf, ROTATE_OFF);
        debug("sample_put");
        conf->aframe = aframe_src_frame_get();
      }
      conf->aframe->sample_length = 0;
      conf->afrm_spl_size = conf->aframe->sample_buf_size;
    }

#ifndef NDEBUG
    if (conf->read_frame_bottom < 0) {
      fprintf(stderr, "sample_put: read_frame_bottom %d\n", conf->read_frame_bottom);
      exit(1);
    }
#endif /* NDEBUG */

    len = sample_cpy(conf);
    frame_spl -= len;

    if (frame_spl == 0) {
      conf->aframe->flag |= CHANGE_AFRAME;
    }

    if (conf->aframe->sample_buf_size == conf->aframe->sample_length) {
#ifndef NDEBUG
      total_put_samples += conf->aframe->sample_buf_size;
      //print_flag("sample_put: flag ", conf->aframe->flag);
#endif /* NDEBUG */
      aframe_src_frame_update(conf->aframe);
      conf->aframe = NULL;
    }

#ifndef NDEBUG
    if (conf->read_frame_cur        == conf->read_frame_bottom &&
	conf->read_frame_cur_offset == conf->read_frame_bottom_offset) {
      if (frame_spl > 0) {
        fprintf(stderr, "sample_put_end: break frame_spl %d.\n", frame_spl);
        exit(1);
      }
      break;
    }
#endif /* NDEBUG */
  }
  conf->vfrm_spl = frame_spl;

  return flag;
}

static int
sample_put_end(AREAD_CONFIG *conf, uint32_t flag)
{
  double spl_gap;
  int gap;
  int len;
  int frame_spl;
  int bottom_pos, bottom_offset;
  int offset;
  int afrm_spl_length = 0;

  if (!conf->record_flag) {
    fprintf(stderr, "sample_put_end: record_flag = 0.\n");
    if (flag & (AREAD_FLAG_START|AREAD_FLAG_STOP|AREAD_FLAG_QUIT)) {
      if (conf->aframe) {
#ifndef NDEBUG
        total_put_samples += conf->aframe->sample_buf_size;
#endif /* NDEBUG */
        conf->aframe->sample_length = conf->aframe->sample_buf_size;
        aframe_src_frame_update(conf->aframe);
        conf->aframe = NULL;
      }
      return flag;
    } else {
      return flag;
    }
  }

  gap = (int)(conf->spl_gap);

#ifndef NDEBUG
  fprintf(stderr, "sample_put_end: gap %d !\n", gap);
#endif /* NDEBUG */

  if (gap < 0) {
    bottom_offset = conf->read_frame_bottom_offset + gap;
    bottom_pos = conf->read_frame_bottom;
    if (bottom_offset < 0) {
      int top_pos;
      bottom_offset = -bottom_offset;
      bottom_pos -= bottom_offset / conf->read_frame_spl_size;
      bottom_pos--;
      bottom_offset = bottom_offset % conf->read_frame_spl_size;
      bottom_offset = conf->read_frame_spl_size - bottom_offset;
      if (bottom_pos < 0) {
	bottom_pos += conf->read_frame_num;
	conf->read_frame_bottom += conf->read_frame_num;
	conf->read_frame_top += conf->read_frame_num;
      }
      top_pos = conf->read_frame_top;
      if ((top_pos-bottom_pos) >= conf->read_frame_num) {
	fprintf(stderr, "sample_put_end: restart position failed.\n");
	bottom_pos = top_pos - conf->read_frame_num + 1;
	bottom_offset = 0;
      }
    }
  } else {
    bottom_offset = conf->read_frame_bottom_offset + gap;
    bottom_pos = conf->read_frame_bottom;
    bottom_pos += bottom_offset / conf->read_frame_spl_size;
    bottom_offset = bottom_offset % conf->read_frame_spl_size;
  }

  if (gap <= -64) {
    if (!conf->aframe)
      return flag;
    gap = 0;
  } else {
    conf->spl_gap = -64;
    gap += 64;
  }

  if (!conf->aframe) {
    conf->aframe = aframe_src_frame_get();
    while (!conf->aframe) {
      fprintf(stderr, "sample_put_end: aframe_src_frame_get() failed.\n");
      sample_read(conf, ROTATE_OFF);
      debug("sample_put_end");
      conf->aframe = aframe_src_frame_get();
    }
    conf->aframe->sample_length = 0;
//    conf->aframe->flag = NORMAL_AFRAME;
  }

  if (!(flag & (AREAD_FLAG_START|AREAD_FLAG_STOP|AREAD_FLAG_QUIT))) {
    afrm_spl_length = conf->aframe->sample_length + gap;
    afrm_spl_length = afrm_spl_length % conf->aframe->sample_buf_size;
  }

  frame_spl = conf->aframe->sample_buf_size - conf->aframe->sample_length;
  while (frame_spl < gap)
    frame_spl += conf->aframe->sample_buf_size;

  len = (conf->read_frame_top - conf->read_frame_bottom) *
    conf->read_frame_spl_size;
  len -= conf->read_frame_bottom_offset;
  while (frame_spl > len) {
    sample_read(conf, ROTATE_OFF);
    debug("sample_put_end");
    len = (conf->read_frame_top - conf->read_frame_bottom) *
      conf->read_frame_spl_size;
    len -= conf->read_frame_bottom_offset;
  }

  offset = conf->read_frame_bottom_offset + frame_spl;
  conf->read_frame_cur = conf->read_frame_bottom +
    (offset / conf->read_frame_spl_size);
  conf->read_frame_cur_offset = offset % conf->read_frame_spl_size;

  spl_gap = conf->spl_gap;
  conf->spl_gap = INTERP_THRESHOLD_MAX;
//  conf->spl_gap = (INTERP_THRESHOLD_MAX + INTERP_THRESHOLD_MIN) / 2;

#ifndef NDEBUG
  if (frame_spl <= 0) {
    fprintf(stderr, "sample_put_end: end frame_spl invalid, %d.\n", frame_spl);
    exit(1);
  }
#endif /* NDEBUG */
  while (frame_spl) {
    if (!conf->aframe) {
      conf->aframe = aframe_src_frame_get();
      while (!conf->aframe) {
	fprintf(stderr, "sample_put_end: aframe_src_frame_get() failed.\n");
	sample_read(conf, ROTATE_OFF);
        debug("sample_put_end");
        conf->aframe = aframe_src_frame_get();
      }
      conf->aframe->sample_length = 0;
//      conf->aframe->flag = NORMAL_AFRAME;
    }

    len = sample_cpy(conf);
    frame_spl -= len;

    if (conf->aframe->sample_buf_size == conf->aframe->sample_length) {
#ifndef NDEBUG
      total_put_samples += conf->aframe->sample_length;
        //print_flag("sample_put_end: flag ", conf->aframe->flag);
#endif /* NDEBUG */
      aframe_src_frame_update(conf->aframe);
      conf->aframe = NULL;
    }

#ifndef NDEBUG
    if (conf->read_frame_cur        == conf->read_frame_bottom &&
	conf->read_frame_cur_offset == conf->read_frame_bottom_offset) {
      if (frame_spl != 0) {
        fprintf(stderr, "sample_put_end: break frame_spl %d.\n", frame_spl);
        exit(1);
      }
      break;
    }
#endif /* NDEBUG */
  }

#ifndef NDEBUG
  if (flag & (AREAD_FLAG_START|AREAD_FLAG_STOP|AREAD_FLAG_QUIT)) {
    if (conf->aframe) {
      fprintf(stderr, "sample_put_end: aframe remains. sample_length %d.\n", conf->aframe->sample_length);
      conf->aframe->sample_length = conf->aframe->sample_buf_size;
      print_flag("sample_put_end: error flag ", conf->aframe->flag);
      aframe_src_frame_update(conf->aframe);
      exit(1);
    }
  }
#endif /* NDEBUG */

  if (!(flag & (AREAD_FLAG_START|AREAD_FLAG_STOP|AREAD_FLAG_QUIT))) {
    if (conf->aframe)
      conf->aframe->sample_length = afrm_spl_length;
  }

  if ((conf->read_frame_top - bottom_pos) >= conf->read_frame_num) {
    fprintf(stderr, "sample_put_end: restart pos failed.\n");
    bottom_pos = conf->read_frame_top - conf->read_frame_num + 1;
    bottom_offset = 0;
  }
  conf->read_frame_bottom = bottom_pos;
  conf->read_frame_bottom_offset = bottom_offset;
  conf->read_frame_cur = bottom_pos;
  conf->read_frame_cur_offset = bottom_offset;
  conf->spl_gap = spl_gap;

  return flag;
}

static int
put_ext_aframe(AREAD_CONFIG *conf, uint32_t flag)
{
  AFRAME *afrm;
//#ifndef NDEBUG
  if (conf->aframe) {
    fprintf(stderr, "put_ext_aframe: aframe remains.\n");
    exit(1);
  }
//#endif /* NDEBUG */
  afrm = aframe_ext_frame_get();
#ifndef NDEBUG
  if (!afrm || !(afrm->flag & EXT_AFRAME)) {
    fprintf(stderr, "put_ext_aframe: invalid frame. %u !\n", afrm->flag);
    exit(1);
  }
#endif /* NDEBUG */
  afrm->flag |= flag;
  aframe_src_frame_update(afrm);
  return AREAD_OK;
}

static inline int
get_state(AREAD_CONFIG *conf)
{
  int state;
  pthread_mutex_lock(&conf->mutex);
  state = conf->state;
  pthread_mutex_unlock(&conf->mutex);
  return state;
}

static inline void
set_state(AREAD_CONFIG *conf, int state)
{
  pthread_mutex_lock(&conf->mutex);
  conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP|STATE_QUIT);
  conf->state |= state;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

static inline void
state_change_done(AREAD_CONFIG *conf)
{
  pthread_mutex_lock(&conf->mutex);
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
}

static void
gap_calc(AREAD_CONFIG *conf)
{
  int frate;
  double fps;
  double rate;
  double video_frames;
  double record_time;
  double samples;
  double read_samples;
  double gap, last_gap, gap_dif;
#define SCALE 10000.0

  rate = conf->rate;
  video_frames = conf->video_frames;
  record_time = conf->record_time / 1000000.0;
  read_samples = conf->read_samples;
  last_gap = conf->last_gap;

  fps = video_frames / record_time;
  frate = (int) (SCALE * fps + 0.5);
  fps = (double)frate / SCALE;
  record_time = (double)video_frames / fps;
  samples = rate * record_time;
  gap = samples - read_samples;
  gap_dif = gap - last_gap;

  conf->spl_gap += gap_dif;
  conf->last_gap = gap;

//  fprintf(stderr, "gap_calc: spl_gap %f\n", conf->spl_gap);
}

/* thread function */

static void*
aread_thread(void *data)
{
  AREAD_CONFIG *conf = &aread_conf;
  int state;
  VINFO *vinfo = NULL;
  u_time last_vinfo_time = 0;
  uint32_t flag = 0;
  int quit_flag = 0;
  int rec_start_flag = 0;

  // avoid compile warnning, unnecessary
  data = NULL;
  // avoid compile warnning, unnecessary

  conf->vfrm_spl = 0;

  init_read_time(conf);

  pthread_mutex_lock(&conf->mutex);
  aread_thread_running = 1;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);

  while (1) {
    state = get_state(conf);
    if (quit_flag || state & STATE_ERROR)
      break;
    //else
    //  state_change_done(conf);

    sample_read(conf, ROTATE_ON);

    while (1) {
      int vfrm_spl_size;

#ifndef NDEBUG
      if (vinfo) {
	fprintf(stderr, "aread_thread: vinfo is not NULL.\n");
	exit(1);
      }
#endif /* NDEBUG */
      if ((vinfo = aread_get_vinfo(conf)) == NULL)
        break;

      if (vinfo->flag & AREAD_FLAG_OVERRUN) {
	conf->record_flag = 0;
      }

      vfrm_spl_size = set_pos(conf, vinfo->time);
#ifndef NDEBUG
      if (conf->record_flag && vfrm_spl_size <= 0) {
	fprintf(stderr, "aread_thread: vfrm_spl_size %d.\n", vfrm_spl_size);
      }
#endif /* NDEBUG */

      if (conf->record_flag) {
#if 0
        double frame_time, time_spl;
        frame_time = vinfo->time - last_vinfo_time;
        time_spl = conf->rate * frame_time / 1000000.0;
	conf->spl_gap += time_spl - (double)vfrm_spl_size;
#else
	conf->record_time += vinfo->time - last_vinfo_time;
	conf->video_frames++;
	conf->read_samples += vfrm_spl_size;
	gap_calc(conf);
#endif

	if (vfrm_spl_size > 0)
	  flag = sample_put(conf, flag);

#ifndef NDEBUG
	if (conf->read_frame_bottom != conf->read_frame_cur ||
	    conf->read_frame_bottom_offset != conf->read_frame_cur_offset) {
	  fprintf(stderr, "aread_thread: read_frame position error, bottom %d, cur %d, bottom_offset %d, cur_offset %d.\n", conf->read_frame_bottom, conf->read_frame_cur, conf->read_frame_bottom_offset, conf->read_frame_cur_offset);
	  exit(1);
	}
#endif /* NDEBUG */
      }

      if (rec_start_flag && (vinfo->flag &
	  (AREAD_FLAG_START|AREAD_FLAG_STOP|AREAD_FLAG_QUIT))){
	flag = sample_put_end(conf, vinfo->flag);
      }

      if (vinfo->flag & AREAD_FLAG_START) {
	if (rec_start_flag)
	  put_ext_aframe(conf, END_AFRAME);
	put_ext_aframe(conf, START_AFRAME);
	rec_start_flag = 1;
	set_state(conf, STATE_START);
//	conf->record_flag = 0;
	conf->read_frame_bottom = conf->read_frame_top-conf->read_frame_num;
	if (conf->read_frame_bottom < 0)
	  conf->read_frame_bottom = 0;
	conf->read_frame_bottom_offset = 0;
        set_pos(conf, vinfo->time);
	conf->record_flag = 1;
	conf->spl_gap = 0;
	conf->last_gap = 0;
	conf->record_time = 0;
	conf->video_frames = 0;
	conf->read_samples = 0;
      } else if (vinfo->flag & AREAD_FLAG_RECORD) {
	conf->record_flag = 1;
      } else if (vinfo->flag & AREAD_FLAG_DROP) {
	if (rec_start_flag)
	  set_state(conf, STATE_PAUSE);
	conf->record_flag = 0;
      } else if (vinfo->flag & AREAD_FLAG_STOP) {
	if (rec_start_flag)
	  put_ext_aframe(conf, END_AFRAME);
	rec_start_flag = 0;
	conf->record_flag = 0;
	set_state(conf, STATE_STOP);
	conf->read_frame_bottom = conf->read_frame_top - conf->read_frame_num;
	if (conf->read_frame_bottom < 0)
	  conf->read_frame_bottom = 0;
	conf->read_frame_bottom_offset = 0;
      } else if (vinfo->flag & AREAD_FLAG_QUIT) {
	if (rec_start_flag)
	  put_ext_aframe(conf, END_AFRAME);
	rec_start_flag = 0;
	conf->record_flag = 0;
	put_ext_aframe(conf, QUIT_AFRAME);
	rec_start_flag = 0;
	conf->record_flag = 0;
	quit_flag = 1;
	vinfo = NULL;
	set_state(conf, STATE_QUIT);
	break;
      } else if (vinfo->flag & AREAD_FLAG_ERROR) {
	vinfo = NULL;
	set_state(conf, STATE_ERROR);
	break;
      }

      flag = vinfo->flag;
      last_vinfo_time = vinfo->time;
      vinfo = NULL;
    }
  }

  pthread_mutex_lock(&conf->mutex);
  aread_thread_running = 0;
  if (conf->state & STATE_INTERFACE_WAIT) {
    conf->state &= ~STATE_INTERFACE_WAIT;
    pthread_cond_signal(&conf->interface_wait_cond);
  }
  pthread_mutex_unlock(&conf->mutex);
//  state_change_done(conf);

#ifndef NDEBUG
  fprintf(stderr, "aread_thread: done\n");
#endif
  return NULL;
}

/* interface functions */

int
aread_state_error(void)
{
  AREAD_CONFIG *conf = &aread_conf;
  int ret = AREAD_FAIL;

  if (!aread_initialized)
    return AREAD_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!aread_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return AREAD_OK;
  }
  if (conf->state & STATE_THREAD_WAIT) {
    conf->state &= ~STATE_THREAD_WAIT;
    pthread_cond_signal(&conf->thread_wait_cond);
  }
  if (conf->state & STATE_ERROR) {
    ret = AREAD_OK;
  } else {
    conf->state &= ~(STATE_START|STATE_PAUSE|STATE_STOP);
    conf->state |= STATE_ERROR;
#if 0
    conf->state |= STATE_INTERFACE_WAIT;
    while (conf->state & STATE_INTERFACE_WAIT) {
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
#endif
    ret = AREAD_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
aread_state_start(void)
{
  AREAD_CONFIG *conf = &aread_conf;
  int ret = AREAD_FAIL;

  if (!aread_initialized)
    return AREAD_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!aread_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return AREAD_OK;
  }
  if (conf->state & (STATE_START|STATE_QUIT|STATE_ERROR)) {
    ret = AREAD_OK;
  } else {
    while (!(conf->state & STATE_START)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = AREAD_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
aread_state_stop(void)
{
  AREAD_CONFIG *conf = &aread_conf;
  int ret = AREAD_FAIL;

  if (!aread_initialized)
    return AREAD_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!aread_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return AREAD_OK;
  }
  if (conf->state & (STATE_STOP|STATE_QUIT|STATE_ERROR)) {
    ret = AREAD_OK;
  } else {
    while (!(conf->state & STATE_STOP)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = AREAD_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
aread_state_quit(void)
{
  AREAD_CONFIG *conf = &aread_conf;
  int ret = AREAD_FAIL;

  if (!aread_initialized)
    return AREAD_OK;

  pthread_mutex_lock(&conf->mutex);
  if (!aread_thread_running) {
    pthread_mutex_unlock(&conf->mutex);
    return AREAD_OK;
  }
  if (conf->state & (STATE_QUIT|STATE_ERROR)) {
    ret = AREAD_OK;
  } else {
    while (!(conf->state & STATE_QUIT)) {
      conf->state |= STATE_INTERFACE_WAIT;
      pthread_cond_wait(&conf->interface_wait_cond, &conf->mutex);
    }
    ret = AREAD_OK;
  }
  pthread_mutex_unlock(&conf->mutex);
  return ret;
}

int
aread_state_pause(void)
{
  if (!aread_initialized)
    return AREAD_OK;
  return AREAD_OK;
}

int
aread_state_toggle_pause(void)
{
  if (!aread_initialized)
    return AREAD_OK;
  return AREAD_OK;
}

int
aread_error_quit (void)
{
  int retcode;
  void* retval;

  if (!aread_initialized) {
#ifndef NDEBUG
    if (aread_conf.read_bufp || aread_conf.read_frame ||
       	aread_conf.vinfo || aread_conf.aframe) {
      fprintf(stderr, "aread_quit: pointer not NULL.\n");
      exit(1);
    }
#endif /* NDEBUG */
    return AREAD_OK;
  }

  aread_state_error();

  retcode = pthread_join(aread_conf.aread_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "aread_quit() thread join error %d\n", retcode);
//  pthread_mutex_destroy(&conf->mutex);
  
  free(aread_conf.read_bufp);
  aread_conf.read_bufp = NULL;
  free(aread_conf.read_frame);
  aread_conf.read_frame = NULL;
  free(aread_conf.vinfo);
  aread_conf.vinfo = NULL;
  aread_conf.aframe = NULL;

  aread_initialized = 0;

//  fprintf (stderr, "aread_quit: done\n");
  return AREAD_OK;
}

int
aread_quit (void)
{
  int retcode;
  void* retval;

  if (!aread_initialized) {
#ifndef NDEBUG
    if (aread_conf.read_bufp || aread_conf.read_frame ||
       	aread_conf.vinfo || aread_conf.aframe) {
      fprintf(stderr, "aread_quit: pointer not NULL.\n");
      exit(1);
    }
#endif /* NDEBUG */
    return AREAD_OK;
  }

  aread_state_quit();

  retcode = pthread_join(aread_conf.aread_th, &retval);
  if (retcode != 0)
    fprintf (stderr, "aread_quit() thread join error %d\n", retcode);
//  pthread_mutex_destroy(&conf->mutex);
  
  free(aread_conf.read_bufp);
  aread_conf.read_bufp = NULL;
  free(aread_conf.read_frame);
  aread_conf.read_frame = NULL;
  free(aread_conf.vinfo);
  aread_conf.vinfo = NULL;
  aread_conf.aframe = NULL;

  aread_initialized = 0;

//  fprintf (stderr, "aread_quit: done\n");
  return AREAD_OK;
}

int
aread_init (int rate, int channels, int bitspspl)
{
  int retcode;
  int fragsize;
  int fragstotal;
  int bytespspl;
  double buffered_time = BUFFERED_TIME;
  int buffer_size;
  int buf_num;
  int i;
  int need_config = 0;
static int first = 1;

  if (first) {
    aread_initialized = 0;
    aread_thread_running = 0;
    aread_conf.rate = 0;
    aread_conf.channels = 0;
//    aread_conf.format = 0;
    aread_conf.read_bufp = NULL;
    aread_conf.read_frame = NULL;
    aread_conf.vinfo = NULL;
    aread_conf.aframe = NULL;
    first = 0;
    need_config = 1;
  }

#ifndef NDEBUG
  if (aread_initialized) {
    fprintf(stderr, "aread_init: aread_initialized TRUE.\n");
  }
  if (aread_conf.read_frame || aread_conf.vinfo) {
    fprintf(stderr, "aread_init: pointer not NULL.\n");
    aread_quit();
  }
#endif /* NDEBUG */

  if (!aread_initialized)
    need_config = 1;
  else if (aread_conf.rate != rate)
    need_config = 1;
  else if (aread_conf.channels != channels)
    need_config = 1;

  if (!need_config) {
    return AREAD_OK;
  }

  if (rate != oss_get_rate()) {
    fprintf(stderr, "aread_init: rate error, rate %d, %d\n", rate, oss_get_rate());
    return AREAD_FAIL;
  }
  if (channels != oss_get_channels()) {
    fprintf(stderr, "aread_init: channels error, channels %d, %d\n", channels, oss_get_channels());
    return AREAD_FAIL;
  }
  if (bitspspl != oss_get_bitspspl()) {
    fprintf(stderr, "aread_init: bitspspl error, bitspspl %d, %d\n", bitspspl, oss_get_bitspspl());
    return AREAD_FAIL;
  }

  fragsize = oss_get_fragsize();
  fragstotal = oss_get_fragstotal();

  aread_conf.rate = rate;
  aread_conf.channels = channels;
//  aread_conf.format = format;
  aread_conf.bitspspl = bitspspl;
  bytespspl = channels * ((bitspspl+7)/8);
  aread_conf.bytespspl = bytespspl;

  buffer_size = (int)((double)(rate * bytespspl) * buffered_time);
  buffer_size = buffer_size + (fragsize * fragstotal);
  buffer_size = buffer_size - (buffer_size % (fragsize * fragstotal));
  buf_num = buffer_size / fragsize;

  aread_conf.read_frame_num = buf_num;
  aread_conf.read_frame_buf_size = fragsize;
  aread_conf.read_frame_spl_size = fragsize / bytespspl;
  aread_conf.read_frame =
    (READ_FRAME*)mem_malloc(sizeof(READ_FRAME) * buf_num, "aread_init");
  aread_conf.read_bufp = (uint8_t*)mem_malloc(buffer_size, "aread_init");
  for (i = 0; i < buf_num; i++) {
    aread_conf.read_frame[i].time = 0;
    aread_conf.read_frame[i].data = aread_conf.read_bufp + (i * fragsize);
  }
  aread_conf.read_frame_pos = 0;
  aread_conf.read_frame_top = 0;
  aread_conf.read_frame_bottom = 0;
  aread_conf.read_frame_bottom_offset = 0;
  aread_conf.read_frame_cur = 0;
  aread_conf.read_frame_cur_offset = 0;

  aread_conf.vinfo_num = VINFO_NUM;
  aread_conf.vinfo = (VINFO*)mem_malloc(sizeof(VINFO)*VINFO_NUM, "aread_init");
  for (i = 0; i < aread_conf.vinfo_num; i++) {
    aread_conf.vinfo[i].flag = 0;
    aread_conf.vinfo[i].time = 0;
  }
  aread_conf.vinfo_top    = 0;
  aread_conf.vinfo_bottom = 0;

  aread_conf.record_flag = 0;

  aread_conf.aframe = NULL;

  aread_conf.fragsize = fragsize;
  aread_conf.fragstotal = fragstotal;
  init_read_time(&aread_conf);

  pthread_mutex_init (&aread_conf.mutex, NULL);
  pthread_cond_init (&aread_conf.interface_wait_cond, NULL);
  pthread_cond_init (&aread_conf.thread_wait_cond, NULL);
  retcode = pthread_create (&aread_conf.aread_th, NULL, aread_thread, NULL);
  if (retcode != 0) {
    fprintf (stderr, "aread_init() thread create failed %d\n", retcode);
    free(aread_conf.read_bufp);
    aread_conf.read_bufp = NULL;
    free(aread_conf.read_frame);
    aread_conf.read_frame = NULL;
    free(aread_conf.vinfo);
    aread_conf.vinfo = NULL;
    return AREAD_FAIL;
  }

  pthread_mutex_lock(&aread_conf.mutex);
  if (!aread_thread_running) {
    aread_conf.state |= STATE_INTERFACE_WAIT;
    pthread_cond_wait(&aread_conf.interface_wait_cond, &aread_conf.mutex);
  }
  pthread_mutex_unlock(&aread_conf.mutex);

  aread_initialized = 1;

  return AREAD_OK;
}

