/* aframe_mgr.c */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "util.h"
#include "aframe_mgr.h"

static AFRAME_MGR aframe_mgr;

#define AFRAME_NUM_MAX 128
#define AFRAME_NUM_MIN 5
#define AFRAME_MAX_LAST_FRAME_ALLOC_COUNT (64)
#define AFRAME_MAX_EXT_FRAME_NUM 256

#ifdef DEBUG
static void
print_flag(uint32_t flag)
{
  if (flag & START_AFRAME)
    printf("afrm_mgr: START_AFRAME !!\n");
  if (flag & END_AFRAME)
    printf("afrm_mgr: END_AFRAME !!\n");
  if (flag & QUIT_AFRAME)
    printf("afrm_mgr: QUIT_AFRAME !!\n");
}
#endif /* DEBUG */

static AFRAME*
alloc_aframe(AFRAME_MGR *afrm_mgr)
{
  AFRAME *afrm;
  const char proc_name[] = "alloc_aframe";

  if (afrm_mgr->alloc_frame_num >= afrm_mgr->max_frame_num)
    return NULL;

  afrm = (AFRAME*)mem_malloc(sizeof(AFRAME), proc_name);
  if (!afrm) {
    pthread_mutex_unlock(&afrm_mgr->mutex);
    fprintf(stderr, "%s: memory allocate error.\n", proc_name);
    exit(1);
  }
#ifdef ARCH_X86
  afrm->bufp =
    (uint8_t*)malloc(afrm_mgr->aframe_buf_byte_size+31);
  afrm->sample[0] = (uint8_t*)((((uint32_t)(afrm->bufp)) + 15) & 0xfffffff0);
#if 0
  if (afrm_mgr->channels == 2)
    afrm->sample[1] = (uint8_t*)((((uint32_t)(afrm->sample[0] + afrm_mgr->aframe_buf_byte_size/2)) + 15) & 0xfffffff0);
  else
    afrm->sample[1] = NULL;
#else
  afrm->sample[1] = (uint8_t*)((((uint32_t)(afrm->sample[0] + afrm_mgr->aframe_buf_byte_size/2)) + 15) & 0xfffffff0);
#endif
  afrm->stream = afrm->sample[0];
#else
  afrm->bufp =
    (uint8_t*)malloc(afrm_mgr->aframe_buf_byte_size);
  afrm->sample[0] = (uint8_t*)afrm->bufp;
#if 0
  if (afrm_mgr->channels == 2)
    afrm->sample[1] = (uint8_t*)(afrm->bufp + afrm_mgr->aframe_buf_byte_size/2);
  else
    afrm->sample[1] = NULL;
#else
  afrm->sample[1] = (uint8_t*)(afrm->bufp + afrm_mgr->aframe_buf_byte_size/2);
#endif
  afrm->stream = afrm->sample[0];
#endif /* ARCH_X86 */
  if (!afrm->bufp) {
    pthread_mutex_unlock(&afrm_mgr->mutex);
    fprintf(stderr, "%s: memory allocate error.\n", proc_name);
    exit(1);
  }
  afrm->flag = NORMAL_AFRAME;
//  afrm->bitspspl = afrm_mgr->bitspspl;
//  afrm->channels = afrm_mgr->channels;
  afrm->sample_length = 0;
  afrm->sample_buf_size = afrm_mgr->aframe_buf_sample_size;
  afrm->stream_length = 0;
  afrm->stream_buf_size = afrm_mgr->aframe_buf_byte_size;
  afrm->prev = afrm->next = NULL;
  afrm_mgr->alloc_frame_num++;
  afrm_mgr->last_frame_alloc_count = 0;
#ifdef DEBUG
#ifdef ARCH_X86
  if ((int)(afrm->sample[0]) & 0xf) {
    fprintf(stderr, "alloc_aframe: data pointer address is not aligned 16.\n");
    exit(1);
  }
#endif /* ARCH_X86 */
#endif /* DEBUG */
  return afrm;
}

static void
free_aframe(AFRAME_MGR *afrm_mgr, AFRAME *afrm)
{
#ifdef DEBUG
  if (afrm->prev || afrm->next) {
    fprintf(stderr, "free_aframe: frame is not removed chain.\n");
    exit(1);
  }
#endif /* DEBUG */
  if (afrm->bufp)
    free (afrm->bufp);
  free(afrm);
  afrm_mgr->alloc_frame_num--;
  afrm_mgr->last_frame_alloc_count = 0;
#ifdef DEBUG
  if (afrm_mgr->alloc_frame_num < 0) {
    fprintf(stderr, "free_aframe: alloc_frame_num < 0\n");
    exit(1);
  }
#endif /* DEBUG */
}

static AFRAME*
alloc_ext_aframe(AFRAME_MGR *afrm_mgr)
{
  AFRAME *afrm;
  const char proc_name[] = "alloc_ext_aframe";

  if (afrm_mgr->alloc_ext_frame_num >= afrm_mgr->max_ext_frame_num) {
    fprintf(stderr, "%s: ext_frame_num over flow.\n", proc_name);
    return NULL;
  }

  afrm = (AFRAME*)malloc(sizeof(AFRAME));
  if (!afrm) {
    pthread_mutex_unlock(&afrm_mgr->mutex);
    fprintf(stderr, "%s: memory allocate error.\n", proc_name);
    exit(1);
  }
  afrm->sample[0] = afrm->sample[1] = afrm->bufp = NULL;
  afrm->flag = EXT_AFRAME;
//  afrm->bitspspl = afrm_mgr->bitspspl;
//  afrm->channels = afrm_mgr->channels;
  afrm->sample_length = 0;
  afrm->sample_buf_size = 0;
  afrm->stream_length = 0;
  afrm->stream_buf_size = 0;
  afrm->stream = NULL;
  afrm->prev = afrm->next = NULL;
  afrm_mgr->alloc_ext_frame_num++;
//  afrm_mgr->last_ext_frame_alloc_count = 0;
  return afrm;
}

static void
free_ext_aframe(AFRAME_MGR *afrm_mgr, AFRAME *afrm)
{
#ifdef DEBUG
  if (afrm->prev || afrm->next) {
    fprintf(stderr, "free_ext_aframe: frame doesn't removed chain.\n");
    exit(1);
  } else if (afrm->bufp) {
    fprintf(stderr, "free_ext_aframe: data pointer not NULL.\n");
    exit(1);
  }
#endif /* DEBUG */
  free(afrm);
  afrm_mgr->alloc_ext_frame_num--;
#ifdef DEBUG
  if (afrm_mgr->alloc_ext_frame_num < 0) {
    fprintf(stderr, "free_aframe: alloc_frame_num < 0\n");
    exit(1);
  }
#endif /* DEBUG */
}

static inline AFRAME*
add_chain(AFRAME **chain, AFRAME *afrm)
{
  if (*chain) {
    afrm->prev = (*chain)->prev;
    afrm->next = (*chain);
    (*chain)->prev->next = afrm;
    (*chain)->prev = afrm;
  } else {
    afrm->prev = afrm->next = afrm;
    *chain = afrm;
  }
  return afrm;
}

static inline AFRAME*
del_chain(AFRAME **chain)
{
  AFRAME *afrm;
  if (*chain == NULL)
    return NULL;
  afrm = *chain;
  if (afrm->next == afrm) {
    *chain = NULL;
  } else {
    *chain = (*chain)->next;
    afrm->prev->next = afrm->next;
    afrm->next->prev = afrm->prev;
  }
  afrm->prev = afrm->next = NULL;
  return afrm;
}

static AFRAME*
get_free_frame(AFRAME_MGR *afrm_mgr)
{
  AFRAME *afrm;
  if (afrm_mgr->free_chain) {
    afrm = del_chain(&(afrm_mgr->free_chain));
    afrm_mgr->free_frame_num--;
  } else {
    afrm = alloc_aframe(afrm_mgr);
  }
  return afrm;
}

static AFRAME*
get_ext_frame(AFRAME_MGR *afrm_mgr)
{
  AFRAME *afrm;
  afrm = alloc_ext_aframe(afrm_mgr);
  return afrm;
}

static void
add_free_frame(AFRAME_MGR *afrm_mgr, AFRAME *afrm)
{
#ifdef DEBUG
  if (afrm == NULL) {
    fprintf(stderr, "add_free_frame: invalid frame\n");
    exit(1);
  } else if (!(afrm->flag & NORMAL_AFRAME)) {
    fprintf(stderr, "add_free_frame: invalid normal frame\n");
    exit(1);
  }
#endif /* DEBUG */
  afrm->flag = NORMAL_AFRAME;
  afrm->sample_length = 0;
  afrm->stream_length = 0;
  if (afrm_mgr->aframe_flag & ENC_DST_WAIT) {
    add_chain(&afrm_mgr->free_chain, afrm);
    afrm_mgr->free_frame_num++;
//    afrm_mgr->enc_dst_frame = afrm;
    if (afrm_mgr->enc_wait_cb) {
      afrm_mgr->enc_wait_cb();
    }
    afrm_mgr->aframe_flag &= ~ENC_DST_WAIT;
    afrm_mgr->last_frame_alloc_count = 0;
  } else if (afrm_mgr->aframe_flag & SRC_WAIT) {
    add_chain(&afrm_mgr->free_chain, afrm);
    afrm_mgr->free_frame_num++;
    if (afrm_mgr->src_wait_cb) {
      afrm_mgr->src_wait_cb();
    }
    afrm_mgr->aframe_flag &= ~SRC_WAIT;
    afrm_mgr->last_frame_alloc_count = 0;
  } else if (
      afrm_mgr->alloc_frame_num > AFRAME_NUM_MIN &&
      afrm_mgr->last_frame_alloc_count > afrm_mgr->max_last_frame_alloc_count &&
      afrm_mgr->free_frame_num > 0) {
    free_aframe(afrm_mgr, afrm);
    afrm_mgr->last_frame_alloc_count = 0;
  } else {
    add_chain(&afrm_mgr->free_chain, afrm);
    afrm_mgr->free_frame_num++;
    afrm_mgr->last_frame_alloc_count++;
  }
}

AFRAME*
aframe_ext_frame_get(void)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;
  AFRAME *afrm = NULL;
  pthread_mutex_lock(&afrm_mgr->mutex);
  afrm = get_ext_frame(afrm_mgr);
  pthread_mutex_unlock(&afrm_mgr->mutex);
  return afrm;
}

AFRAME*
aframe_src_frame_get(void)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;
  AFRAME *afrm = NULL;

  pthread_mutex_lock(&afrm_mgr->mutex);
#ifdef DEBUG
  if (afrm_mgr->src_frame != NULL) {
    fprintf(stderr, "aframe_src_frame_get: invalid frame\n");
    exit(1);
  }
  if (afrm_mgr->used_frame_num < 0) {
    fprintf(stderr, "aframe_src_frame_get: used_frame_num invalid, %d.\n", afrm_mgr->used_frame_num);
    exit(1);
  }
#endif /* DEBUG */
  if (afrm_mgr->used_frame_num < afrm_mgr->max_frame_num) {
    afrm = get_free_frame(afrm_mgr);
    if (afrm) {
      afrm_mgr->src_frame = afrm;
      afrm_mgr->src_frame_num++;
      afrm_mgr->used_frame_num++;
      afrm_mgr->aframe_flag &= ~SRC_WAIT;
    } else {
      //afrm_mgr->src_frame = NULL;
      afrm_mgr->aframe_flag |= SRC_WAIT;
    }
  }
#ifdef DEBUG
  if (!afrm) {
    fprintf(stderr, "aframe_src_frame_get: src_frame_num %d, dst_frame_num %d, free_frame_num %d\n", afrm_mgr->src_frame_num, afrm_mgr->dst_frame_num, afrm_mgr->free_frame_num);
    fprintf(stderr, "aframe_src_frame_get: max_frame_num %d \n", afrm_mgr->max_frame_num);
  }
#endif /* DEBUG */
  pthread_mutex_unlock(&afrm_mgr->mutex);

  return afrm;
}

void
aframe_src_frame_update(AFRAME *afrm)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;

  pthread_mutex_lock(&afrm_mgr->mutex);
#ifdef DEBUG
  if (afrm->flag & NORMAL_AFRAME && afrm_mgr->src_frame != afrm) {
    fprintf(stderr, "aframe_src_frame_update: invalid frame\n");
    exit(1);
  } else if ((afrm->flag & EXT_AFRAME) && afrm_mgr->src_frame) {
    fprintf(stderr, "aframe_src_frame_update: src frame was not updated\n");
    exit(1);
  } else if (!(afrm->flag & (NORMAL_AFRAME|EXT_AFRAME))) {
    fprintf(stderr, "aframe_src_frame_update: src frame was not updated\n");
    exit(1);
  }
#endif /* DEBUG */
  add_chain(&afrm_mgr->src_chain, afrm);
  afrm_mgr->src_frame = NULL;
  if (afrm_mgr->aframe_flag & ENC_SRC_WAIT) {
    if (afrm_mgr->enc_wait_cb) {
      afrm_mgr->enc_wait_cb();
    }
    afrm_mgr->aframe_flag &= ~ENC_SRC_WAIT;
  }
  pthread_mutex_unlock(&afrm_mgr->mutex);
}

AFRAME *
aframe_enc_src_frame_get(void)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;
  AFRAME *afrm = NULL;

  pthread_mutex_lock(&afrm_mgr->mutex);
#ifdef DEBUG
  if (afrm_mgr->enc_src_frame) {
    fprintf(stderr, "aframe_enc_src_frame_get: enc_src_frame remains\n");
    exit(1);
  }
#endif /* DEBUG */
  if (afrm_mgr->src_chain) {
    afrm = del_chain(&afrm_mgr->src_chain);
    afrm_mgr->enc_src_frame = afrm;
    afrm_mgr->aframe_flag &= ~ENC_SRC_WAIT;
  } else {
    //afrm_mgr->enc_src_frame = NULL;
    afrm_mgr->aframe_flag |= ENC_SRC_WAIT;
  }
  pthread_mutex_unlock(&afrm_mgr->mutex);

  return afrm;
}

void
aframe_enc_src_frame_update(AFRAME *afrm)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;

  pthread_mutex_lock(&afrm_mgr->mutex);
#ifdef DEBUG
  if (afrm_mgr->enc_src_frame != afrm) {
    fprintf(stderr, "aframe_enc_src_frame_update: invalid frame\n");
    exit(1);
  }
#endif /* DEBUG */
  afrm_mgr->enc_src_frame = NULL;
  if (afrm->flag & NORMAL_AFRAME) {
    add_free_frame(afrm_mgr, afrm);
    afrm_mgr->used_frame_num--;
    afrm_mgr->src_frame_num--;
  } else {
#ifdef DEBUG
    if (!(afrm->flag & EXT_AFRAME)) {
      fprintf(stderr, "aframe_enc_src_frame_update: unknown frame\n");
      exit(1);
    }
//    fprintf(stderr, "aframe_enc_src_frame_update: ");
//    print_flag(afrm->flag);
#endif /* DEBUG */
    add_chain(&afrm_mgr->dst_chain, afrm);
    if (afrm_mgr->aframe_flag & DST_WAIT) {
      if (afrm_mgr->dst_wait_cb) {
        afrm_mgr->dst_wait_cb();
      }
      afrm_mgr->aframe_flag &= ~DST_WAIT;
    }
  }
  pthread_mutex_unlock(&afrm_mgr->mutex);
}

AFRAME*
aframe_enc_dst_frame_get(void)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;
  AFRAME *afrm = NULL;

  pthread_mutex_lock(&afrm_mgr->mutex);
#ifdef DEBUG
  if (afrm_mgr->enc_dst_frame != NULL) {
    fprintf(stderr, "aframe_enc_dst_frame_get: enc_dst_frame remains.\n");
    exit(1);
  }
#endif /* DEBUG */
  afrm = get_free_frame(afrm_mgr);
  if (afrm) {
    afrm_mgr->enc_dst_frame = afrm;
    afrm_mgr->dst_frame_num++;
    afrm_mgr->used_frame_num++;
    afrm_mgr->aframe_flag &= ~ENC_DST_WAIT;
  } else {
    //afrm_mgr->enc_dst_frame = NULL;
    afrm_mgr->aframe_flag |= ENC_DST_WAIT;
  }
  pthread_mutex_unlock(&afrm_mgr->mutex);

#ifdef DEBUG
  if (!afrm) {
    fprintf(stderr, "aframe_enc_dst_frame_get: src_frame_num %d, dst_frame_num %d, free_frame_num %d\n", afrm_mgr->src_frame_num, afrm_mgr->dst_frame_num, afrm_mgr->free_frame_num);
  }
#endif /* DEBUG */
  return afrm;
}

void
aframe_enc_dst_frame_update(AFRAME *afrm)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;

  pthread_mutex_lock(&afrm_mgr->mutex);
#ifdef DEBUG
  if (!(afrm->flag & NORMAL_AFRAME)) {
    fprintf(stderr,"aframe_enc_dst_frame_update: frame is not NORMAL_AFRAME\n");
    print_flag(afrm->flag);
    exit(1);
  } else if (afrm_mgr->enc_dst_frame != afrm) {
    fprintf(stderr, "aframe_enc_dst_frame_update: invalid frame\n");
    exit(1);
  }
//  print_flag(afrm->flag);
#endif /* DEBUG */
  add_chain(&afrm_mgr->dst_chain, afrm);
  afrm_mgr->enc_dst_frame = NULL;
  if (afrm_mgr->aframe_flag & DST_WAIT) {
    if (afrm_mgr->dst_wait_cb) {
      afrm_mgr->dst_wait_cb();
    }
    afrm_mgr->aframe_flag &= ~DST_WAIT;
  }
  pthread_mutex_unlock(&afrm_mgr->mutex);
}

AFRAME*
aframe_dst_frame_get(void)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;
  AFRAME *afrm = NULL;

  pthread_mutex_lock(&afrm_mgr->mutex);
#ifdef DEBUG
  if (afrm_mgr->dst_frame) {
    fprintf(stderr, "aframe_dst_frame_get: dst_frame remains.\n");
    exit(1);
  }
#endif /* DEBUG */
  if (afrm_mgr->dst_chain) {
    afrm = del_chain(&afrm_mgr->dst_chain);
    afrm_mgr->dst_frame = afrm;
    afrm_mgr->aframe_flag &= ~DST_WAIT;
  } else {
    //afrm_mgr->dst_frame = NULL;
    afrm_mgr->aframe_flag |= DST_WAIT;
  }
  pthread_mutex_unlock(&afrm_mgr->mutex);

  return afrm;
}

void
aframe_dst_frame_update(AFRAME *afrm)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;

  pthread_mutex_lock(&afrm_mgr->mutex);
#ifdef DEBUG
  if (afrm_mgr->dst_frame != afrm) {
    fprintf(stderr, "aframe_dst_frame_update: invalid frame\n");
    exit(1);
  }
#endif /* DEBUG */
  if (afrm->flag & EXT_AFRAME) {
    free_ext_aframe(afrm_mgr, afrm);
  } else {
#ifdef DEBUG
    if (!(afrm->flag & NORMAL_AFRAME)) {
      fprintf(stderr, "aframe_dst_frame_update: invalid normal frame\n");
      exit(1);
    }
#endif /* DEBUG */
    add_free_frame(afrm_mgr, afrm);
    afrm_mgr->dst_frame_num--;
    afrm_mgr->used_frame_num--;
  }
  afrm_mgr->dst_frame = NULL;
  pthread_mutex_unlock(&afrm_mgr->mutex);
}

int
aframe_get_src_frame_num(void)
{
  return aframe_mgr.src_frame_num;
}

int
aframe_get_dst_frame_num(void)
{
  return aframe_mgr.dst_frame_num;
}

int
aframe_get_used_frame_num(void)
{
  return aframe_mgr.used_frame_num;
}

int
aframe_get_rate(void)
{
  return aframe_mgr.rate;
}

int
aframe_get_channels(void)
{
  return aframe_mgr.channels;
}

int
aframe_get_bitspspl(void)
{
  return aframe_mgr.bitspspl;
}

int
aframe_get_bytespspl(void)
{
  return aframe_mgr.bytespspl;
}

int
aframe_get_aframe_buf_sample_size(void)
{
  return aframe_mgr.aframe_buf_sample_size;
}

void
aframe_set_src_wait_cb(void (*wait_cb)(void))
{
  aframe_mgr.src_wait_cb = wait_cb;
}

void
aframe_set_enc_wait_cb(void (*wait_cb)(void))
{
  aframe_mgr.enc_wait_cb = wait_cb;
}

void
aframe_set_dst_wait_cb(void (*wait_cb)(void))
{
  aframe_mgr.dst_wait_cb = wait_cb;
}

static void
aframe_free(AFRAME *afrm)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;
  if (afrm->flag & EXT_AFRAME)
    free_ext_aframe(afrm_mgr, afrm);
  else if (afrm->flag & NORMAL_AFRAME)
    free_aframe(afrm_mgr, afrm);
#ifdef DEBUG
  else {
    fprintf(stderr, "aframe_free: unknown frame\n");
    exit(1);
  }
#endif /* DEBUG */
}

int
aframe_mgr_error_quit(void)
{
  AFRAME *afrm;
  AFRAME_MGR *afrm_mgr = &aframe_mgr;

  pthread_mutex_lock(&afrm_mgr->mutex);
  if (afrm_mgr->src_frame) {
    aframe_free(afrm_mgr->src_frame);
    afrm_mgr->src_frame = NULL;
  }
  if (afrm_mgr->dst_frame) {
    aframe_free(afrm_mgr->dst_frame);
    afrm_mgr->dst_frame = NULL;
  }
  if (afrm_mgr->enc_src_frame) {
    aframe_free(afrm_mgr->enc_src_frame);
    afrm_mgr->enc_src_frame = NULL;
  }
  if (afrm_mgr->enc_dst_frame) {
    aframe_free(afrm_mgr->enc_dst_frame);
    afrm_mgr->enc_dst_frame = NULL;
  }

  while (afrm_mgr->src_chain != NULL) {
    afrm = del_chain(&afrm_mgr->src_chain);
    aframe_free(afrm);
  }
  afrm_mgr->src_frame_num = 0;
  while (afrm_mgr->dst_chain != NULL) {
    afrm = del_chain(&afrm_mgr->dst_chain);
    aframe_free(afrm);
  }
  afrm_mgr->dst_frame_num = 0;
  while (afrm_mgr->free_chain != NULL) {
    afrm = del_chain(&afrm_mgr->free_chain);
    aframe_free(afrm);
  }
  afrm_mgr->free_frame_num = 0;
  pthread_mutex_unlock(&afrm_mgr->mutex);
//  pthread_mutex_destroy(&afrm_mgr->mutex);
//  memset(afrm_mgr, 0, sizeof(AFRAME_MGR));
  return AFRAME_MGR_OK;
}

int
aframe_mgr_quit(void)
{
  AFRAME *afrm;
  AFRAME_MGR *afrm_mgr = &aframe_mgr;

  pthread_mutex_lock(&afrm_mgr->mutex);
  if (afrm_mgr->src_frame_num > 0 || afrm_mgr->dst_frame_num > 0) {
    pthread_mutex_unlock(&afrm_mgr->mutex);
    return AFRAME_MGR_BUSY;
  }
  while (afrm_mgr->free_chain != NULL) {
    afrm = del_chain(&afrm_mgr->free_chain);
    free_aframe(afrm_mgr, afrm);
  }
  pthread_mutex_unlock(&afrm_mgr->mutex);
//  pthread_mutex_destroy(&afrm_mgr->mutex);
//  memset(afrm_mgr, 0, sizeof(AFRAME_MGR));
  return AFRAME_MGR_OK;
}

int
aframe_mgr_init(int rate, int channels, int bitspspl, int aframe_buf_sample_size)
{
  AFRAME_MGR *afrm_mgr = &aframe_mgr;
  AFRAME *afrm;
  int i;
  int bytespspl;
  int max_buf;
  int need_config = 0;
  double buf_time;
  static int first = 1;

  if (first) {
    afrm_mgr->last_frame_alloc_count = 0;
    afrm_mgr->max_last_frame_alloc_count = 0;
    afrm_mgr->max_frame_num = 0;
    afrm_mgr->max_ext_frame_num = 0;
    afrm_mgr->alloc_frame_num = 0;
    afrm_mgr->alloc_ext_frame_num = 0;
    afrm_mgr->used_frame_num = 0;
    afrm_mgr->free_frame_num = 0;
    afrm_mgr->src_frame_num = 0;
    afrm_mgr->dst_frame_num = 0;
    afrm_mgr->src_frame = NULL;
    afrm_mgr->dst_frame = NULL;
    afrm_mgr->enc_src_frame = NULL;
    afrm_mgr->enc_dst_frame = NULL;
    afrm_mgr->src_chain = NULL;
    afrm_mgr->dst_chain = NULL;
    afrm_mgr->free_chain = NULL;
    afrm_mgr->src_wait_cb = NULL;
    afrm_mgr->enc_wait_cb = NULL;
    afrm_mgr->dst_wait_cb = NULL;
    afrm_mgr->aframe_buf_byte_size = 0;
    afrm_mgr->aframe_buf_sample_size = 0;
    first = 0;
    need_config = 1;

    pthread_mutex_init(&afrm_mgr->mutex, NULL);
  }

  pthread_mutex_lock(&afrm_mgr->mutex);

  if (afrm_mgr->rate != rate)
    need_config = 1;
  else if (afrm_mgr->channels != channels)
    need_config = 1;
  else if (afrm_mgr->bitspspl != bitspspl)
    need_config = 1;
  else if (afrm_mgr->aframe_buf_sample_size != aframe_buf_sample_size)
    need_config = 1;

  if (need_config == 0) {
    pthread_mutex_unlock(&afrm_mgr->mutex);
    return AFRAME_MGR_OK;
  }

  if (afrm_mgr->src_chain != NULL || afrm_mgr->dst_chain != NULL ||
      afrm_mgr->src_frame != NULL || afrm_mgr->dst_frame != NULL ||
      afrm_mgr->enc_src_frame != NULL || afrm_mgr->enc_dst_frame != NULL ||
      afrm_mgr->src_frame_num > 0 || afrm_mgr->dst_frame_num > 0) {
    fprintf(stderr, "aframe_mgr_init: src or dst frame busy.\n");
    pthread_mutex_unlock(&afrm_mgr->mutex);
    return AFRAME_MGR_FAIL;
  }
  while (afrm_mgr->free_chain != NULL) {
    afrm = del_chain(&afrm_mgr->free_chain);
    free_aframe(afrm_mgr, afrm);
  }

#define BUF_TIME 3.0
  buf_time = BUF_TIME;
  max_buf = (int)(buf_time * (double)rate);
  max_buf = (max_buf + aframe_buf_sample_size) / aframe_buf_sample_size;

  if (max_buf < AFRAME_NUM_MIN || max_buf > AFRAME_NUM_MAX)
    max_buf = AFRAME_NUM_MAX;
  bytespspl = ((bitspspl+7)/8) * channels;

  afrm_mgr->rate = rate;
  afrm_mgr->channels = channels;
  afrm_mgr->bitspspl = bitspspl;
  afrm_mgr->aframe_buf_sample_size = aframe_buf_sample_size;
  afrm_mgr->bytespspl = bytespspl;
  afrm_mgr->aframe_buf_byte_size = aframe_buf_sample_size * bytespspl;
//  afrm_mgr->aframe_buf_byte_size = aframe_buf_sample_size * ((bitspspl+7)/8)*2;

  afrm_mgr->last_frame_alloc_count = 0;
  afrm_mgr->max_last_frame_alloc_count = AFRAME_MAX_LAST_FRAME_ALLOC_COUNT;
  afrm_mgr->max_frame_num = max_buf;
  afrm_mgr->max_ext_frame_num = AFRAME_MAX_EXT_FRAME_NUM;
  afrm_mgr->alloc_frame_num = 0;
  afrm_mgr->alloc_ext_frame_num = 0;
  afrm_mgr->used_frame_num = 0;
  afrm_mgr->src_frame_num = 0;
  afrm_mgr->dst_frame_num = 0;
  afrm_mgr->free_frame_num = 0;
  afrm_mgr->src_frame = NULL;
  afrm_mgr->dst_frame = NULL;
  afrm_mgr->enc_src_frame = NULL;
  afrm_mgr->enc_dst_frame = NULL;
  afrm_mgr->free_chain = NULL;
  afrm_mgr->src_chain = NULL;
  afrm_mgr->dst_chain = NULL;

  for (i = 0; i < AFRAME_NUM_MIN; i++) {
    afrm = alloc_aframe(afrm_mgr);
    add_chain(&afrm_mgr->free_chain, afrm);
    afrm_mgr->free_frame_num++;
  }

#if 0
  afrm_mgr->src_wait_cb = NULL;
  afrm_mgr->enc_wait_cb = NULL;
  afrm_mgr->dst_wait_cb = NULL;
#endif

  afrm_mgr->state = 0;
  afrm_mgr->aframe_flag = 0;

  pthread_mutex_unlock(&afrm_mgr->mutex);

  return AFRAME_MGR_OK;
}

